from notecoin.huobi.constant import SubUidState
from notecoin.huobi.constant import *
from notecoin.huobi.utils import default_parse, default_parse_list_dict


class AccountAssetValuationResult:
    """
    The account information for spot account, margin account etc.

    :member
        balance: balance valuation bases on given valuation currency.
        timestamp: unix timestamp from server.

    """

    def __init__(self):
        self.balance = ""
        self.timestamp = 0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.balance, format_data + "balance")
        PrintBasic.print_basic(self.timestamp, format_data + "timestamp")


class AccountBalanceReq:
    """
    The account change information received by subscription of account.

    :member
        ts: The UNIX formatted timestamp generated by server in UTC.
        cid: client request ID
        topic: request Channel or Topic
        data: The list of account and balance

    """

    def __init__(self):
        self.ts = 0
        self.cid = ""
        self.topic = ""
        self.data = list()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ts, format_data + "Timestamp")
        PrintBasic.print_basic(self.cid, format_data + "Client Order ID")
        PrintBasic.print_basic(self.topic, format_data + "Topic")
        print()
        if len(self.data):
            for account_balance in self.data:
                account_balance.print_object()
                print()


class AccountBalance:
    """
    The account information for spot account, margin account etc.

    :member
        id: The unique account id.
        account_type: The type of this account, possible value: spot, margin, otc, point.
        account_state: The account state, possible value: working, lock.
        list: The balance list of the specified currency. The content is Balance class

    """

    def __init__(self):
        self.id = 0
        self.type = AccountType.INVALID
        self.state = AccountState.INVALID
        self.subtype = ""
        self.list = list()

    @staticmethod
    def json_parse(data_dict):
        if data_dict and len(data_dict):
            balance_list = data_dict.get("list")
            data_dict.pop("list")
            account_balance_obj = default_parse(data_dict, AccountBalance, Balance)
            account_balance_obj.subtype = data_dict.get("subtype", data_dict.get("symbol"))
            account_balance_obj.list = default_parse_list_dict(balance_list, Balance, [])
            return account_balance_obj

        return None

    @staticmethod
    def json_parse_list(data_list):
        account_balance_list = []
        if data_list and len(data_list):
            for item in data_list:
                item_obj = AccountBalance.json_parse(item)
                if item_obj:
                    account_balance_list.append(item_obj)
        return account_balance_list

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.id, format_data + "Account ID")
        PrintBasic.print_basic(self.type, format_data + "Account Type")
        PrintBasic.print_basic(self.state, format_data + "Account State")
        PrintBasic.print_basic(self.subtype, format_data + "Subtype")

        print()
        if len(self.list):
            for row in self.list:
                row.print_object(format_data+"\t")
                print()


class AccountHistory:
    """
    The account information for spot account, margin account etc.

    :member
        account_id: Account Id.
        currency: Currency name
        transact_amt: Amount change (positive value if income, negative value if outcome)
        transact-type: Amount change type
        avail_balance: Available balance
        acct_balance: Account balance
        transact_time: Transaction time (database time)
        record_id: Unique record ID in the database

    """

    def __init__(self):
        self.account_id = 0
        self.currency = ""
        self.transact_amt = ""
        self.transact_type = ""
        self.avail_balance = ""
        self.acct_balance = ""
        self.transact_time = 0
        self.record_id = 0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.account_id, format_data + "Account Id")
        PrintBasic.print_basic(self.currency, format_data + "Currency")
        PrintBasic.print_basic(self.transact_amt, format_data + "Transact Amount")
        PrintBasic.print_basic(self.transact_type, format_data + "Transact Type")
        PrintBasic.print_basic(self.avail_balance, format_data + "Avail Balance")
        PrintBasic.print_basic(self.acct_balance, format_data + "Account Balance")
        PrintBasic.print_basic(self.transact_time, format_data + "Transact Time")
        PrintBasic.print_basic(self.record_id, format_data + "Record Id")


class AccountLedger:
    """
    The account ledger information.

    :member
        accountId: Account ID.
        currency: Cryptocurrency.
        transactAmt: Transaction amount (income positive, expenditure negative).
        transactType: Transaction type.
        transferType: Transfer type (only valid for transactType=transfer).
        transactId: Transaction ID.
        transactTime: Transaction time.
        transferer: Transferer’s account ID.
        transferee: Transferee’s account ID.
    """

    def __init__(self):
        self.accountId = 0
        self.currency = ""
        self.transactAmt = 0.0
        self.transactType = ""
        self.transferType = ""
        self.transactId = 0
        self.transactTime = 0
        self.transferer = 0
        self.transferee = 0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.accountId, format_data + "Account ID")
        PrintBasic.print_basic(self.currency, format_data + "Currency")
        PrintBasic.print_basic(self.transactAmt, format_data + "Transaction Amount")
        PrintBasic.print_basic(self.transactType, format_data + "Transaction Type")
        PrintBasic.print_basic(self.transferType, format_data + "Transfer Type")
        PrintBasic.print_basic(self.transactId, format_data + "Transaction ID")
        PrintBasic.print_basic(self.transactTime, format_data + "Transaction Time")
        PrintBasic.print_basic(self.transferer, format_data + "Transferer’s Account ID")
        PrintBasic.print_basic(self.transferee, format_data + "Transferee’s Account ID")


class AccountPointGroup:
    """
    The account information for spot account, margin account etc.

    :member
        id: The unique account id.
        account_type: The type of this account, possible value: spot, margin, otc, point.
        account_state: The account state, possible value: working, lock.
        list: The balance list of the specified currency. The content is Balance class

    """

    def __init__(self):
        self.groupId = ""
        self.expiryDate = ""
        self.remainAmt = ""

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.groupId, format_data + "Group Id")
        PrintBasic.print_basic(self.expiryDate, format_data + "Expiration date")
        PrintBasic.print_basic(self.remainAmt, format_data + "Remain Amount")


class AccountPointResult:
    """
    The account information for spot account, margin account etc.

    :member
        id: The unique account id.
        account_type: The type of this account, possible value: spot, margin, otc, point.
        account_state: The account state, possible value: working, lock.
        list: The balance list of the specified currency. The content is Balance class

    """

    def __init__(self):
        self.accountId = ""
        self.accountStatus = AccountPointState.INVALID
        self.groupIds = list()
        self.acctBalance = ""

    @staticmethod
    def json_parse(data_dict):
        if data_dict and len(data_dict):
            group_ids = data_dict.get("groupIds")
            data_dict.pop("groupIds")
            account_point_obj = default_parse(data_dict, AccountPointResult, AccountPointGroup)
            account_point_obj.subtype = data_dict.get("subtype", data_dict.get("symbol"))
            account_point_obj.list = default_parse_list_dict(group_ids, AccountPointGroup, [])
            return account_point_obj

        return None

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.accountId, format_data + "Account ID")
        PrintBasic.print_basic(self.accountStatus, format_data + "Account Status")
        PrintBasic.print_basic(self.acctBalance, format_data + "Account Balance")

        print()
        if len(self.groupIds):
            for row in self.groupIds:
                row.print_object(format_data + "\t")
                print()


class AccountPointTransferResult:
    """
    The account change information received by subscription of account.

    :member
        timestamp: The UNIX formatted timestamp generated by server in UTC.
        change_type: The event that asset change notification related.
        account_change_list: The list of account change, the content is AccountChange class

    """

    def __init__(self):
        self.transactId = ""
        self.transactTime = 0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.transactId, format_data + "transactId")
        PrintBasic.print_basic(self.transactTime, format_data + "transactTime")
        self.data.print_object()


class AccountTransferResult:
    """
    The account information for spot account, margin account etc.

    :member
        id: The unique account id.
        account_type: The type of this account, possible value: spot, margin, otc, point.
        account_state: The account state, possible value: working, lock.
        balances: The balance list of the specified currency. The content is Balance class

    """

    def __init__(self):
        self.transact_id = 0
        self.transact_time = 0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.transact_id, format_data + "TransactionId")
        PrintBasic.print_basic(self.transact_time, format_data + "TransactionTime")


class AccountUpdateEvent:
    """
    The account change information received by subscription of account.

    :member
        timestamp: The UNIX formatted timestamp generated by server in UTC.
        change_type: The event that asset change notification related.
        account_change_list: The list of account change, the content is AccountChange class

    """

    def __init__(self):
        self.ch = 0
        self.data = AccountUpdate()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.ch, format_data + "Topic")
        self.data.print_object()


class AccountUpdate:
    """
    The account change information received by subscription of account.

    :member
        currency: The currency of the change.
        accountId: The account id.
        balance: Account balance (only exists when account balance changed)
        available:	Available balance (only exists when available balance changed)
        changeType:	Change type see AccountChangeType, valid value: order-place,order-match,order-refund,order-cancel,order-fee-refund,margin-transfer,margin-loan,margin-interest,margin-repay,other,
        accountType: Account type see AccountBalanceUpdateType, valid value: trade, frozen, loan, interest
        changeTime:	Change time, unix time in millisecond
    """

    def __init__(self):
        self.currency = ""
        self.accountId = 0
        self.balance = ""
        self.available = ""
        self.changeType = AccountChangeType.INVALID
        self.accountType = AccountBalanceUpdateType.INVALID
        self.changeTime = 0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.currency, format_data + "Currency")
        PrintBasic.print_basic(self.accountId, format_data + "Account ID")
        PrintBasic.print_basic(self.balance, format_data + "Balance")
        PrintBasic.print_basic(self.available, format_data + "Available")
        PrintBasic.print_basic(self.changeType, format_data + "Account Change Type")
        PrintBasic.print_basic(self.accountType, format_data + "Account Balance Change Type")
        PrintBasic.print_basic(self.changeTime, format_data + "Account Timestamp")


class Account:
    """
    The account information for spot account, margin account etc.

    :member
        id: The unique account id.
        account_type: The type of this account, possible value: spot, margin, otc, point.
        account_state: The account state, possible value: working, lock.
        balances: The balance list of the specified currency. The content is Balance class

    """

    def __init__(self):
        self.id = 0
        self.type = AccountType.INVALID
        self.state = AccountState.INVALID
        self.subtype = ""

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.id, format_data + "ID")
        PrintBasic.print_basic(self.type, format_data + "Account Type")
        PrintBasic.print_basic(self.state, format_data + "Account State")
        PrintBasic.print_basic(self.subtype, format_data + "Subtype")


class Balance:
    """
    The balance of specified account.

    :member
        currency: The currency of this balance.
        balance_type: The balance type, trade or frozen.
        balance: The balance in the main currency unit.

    """

    def __init__(self):
        self.currency = ""
        self.type = AccountBalanceUpdateType.INVALID
        self.balance = 0.0

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.currency, format_data + "Currency")
        PrintBasic.print_basic(self.type, format_data + "Balance Type")
        PrintBasic.print_basic(self.balance, format_data + "Balance")


class CompleteSubAccount:
    """
    Sub-account completed info

    :member
        id: The sub-id.
        account_type: The sub account type.
        balances: The balance list, the content is Balance class.
    """

    def __init__(self):
        self.id = 0
        self.account_type = AccountType.INVALID
        self.balances = list()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.id, format_data + "ID")
        PrintBasic.print_basic(self.account_type, format_data + "Account Type")
        if len(self.balances):
            for row in self.balances:
                row.print_object()
                print()


class MarginBalanceDetail:
    def __init__(self):
        self.id = 0
        self.symbol = 0
        self.state = AccountState.INVALID
        self.type = AccountType.INVALID
        self.risk_rate = 0.0
        self.fl_price = 0.0
        self.fl_type = ""
        self.sub_account_balance_list = list()

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.id, format_data + "ID")
        PrintBasic.print_basic(self.type, format_data + "Account Type")
        PrintBasic.print_basic(self.symbol, format_data + "Symbol")
        PrintBasic.print_basic(self.state, format_data + "Account State")
        PrintBasic.print_basic(self.fl_price, format_data + "Burst Price")
        PrintBasic.print_basic(self.fl_type, format_data + "Burst Type")
        PrintBasic.print_basic(self.risk_rate, format_data + "Risk Rate")


class SubUidManagement:
    """
    The trade information with price and amount etc.

    :member
        subUid: sub user ID.
        userState: sub user account state, states see SubUidState.
    """

    def __init__(self):
        self.subUid = 0
        self.userState = SubUidState.INVALID

    def print_object(self, format_data=""):
        from huobi.utils.print_mix_object import PrintBasic
        PrintBasic.print_basic(self.subUid, format_data + "subUid")
        PrintBasic.print_basic(self.userState, format_data + "userState")
