#!/usr/bin/env python
"""
Color palettes from ColorBrewer2

It includes palettes from colorbrewer2.org.

Colorbrewer License
-------------------
Copyright (c) 2002 Cynthia Brewer, Mark Harrower,
and The Pennsylvania State University.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
either express or implied. See the License for the specific
language governing permissions and limitations under the License.

.. moduleauthor:: Matthias Cuntz

History
    * Written Sep 2012, Matthias Cuntz
    * Made dictionaries with color palettes,
      Mar 2015, Matthias Cuntz
    * Ported to pyjams, Nov 2021, Matthias Cuntz
    * flake8 compatible, Nov 2021, Matthias Cuntz
    * Prepend brewer to all palette names, Nov 2021, Matthias Cuntz

"""

__all__ = ['brewer_sequential', 'brewer_diverging', 'brewer_qualitative']


# sequential color maps
brewer_sequential = {
    'brewer_ylgn3': [(247, 252, 185), (173, 221, 142), (49, 163, 84)],
    'brewer_ylgn4': [(255, 255, 204), (194, 230, 153), (120, 198, 121),
                     (35, 132, 67)],
    'brewer_ylgn5': [(255, 255, 204), (194, 230, 153), (120, 198, 121),
                     (49, 163, 84), (0, 104, 55)],
    'brewer_ylgn6': [(255, 255, 204), (217, 240, 163), (173, 221, 142),
                     (120, 198, 121), (49, 163, 84), (0, 104, 55)],
    'brewer_ylgn7': [(255, 255, 204), (217, 240, 163), (173, 221, 142),
                     (120, 198, 121), (65, 171, 93), (35, 132, 67),
                     (0, 90, 50)],
    'brewer_ylgn8': [(255, 255, 229), (247, 252, 185), (217, 240, 163),
                     (173, 221, 142), (120, 198, 121), (65, 171, 93),
                     (35, 132, 67), (0, 90, 50)],
    'brewer_ylgn9': [(255, 255, 229), (247, 252, 185), (217, 240, 163),
                     (173, 221, 142), (120, 198, 121), (65, 171, 93),
                     (35, 132, 67), (0, 104, 55), (0, 69, 41)],
    'brewer_ylgnbu3': [(237, 248, 177), (127, 205, 187), (44, 127, 184)],
    'brewer_ylgnbu4': [(255, 255, 204), (161, 218, 180), (65, 182, 196),
                       (34, 94, 168)],
    'brewer_ylgnbu5': [(255, 255, 204), (161, 218, 180), (65, 182, 196),
                       (44, 127, 184), (37, 52, 148)],
    'brewer_ylgnbu6': [(255, 255, 204), (199, 233, 180), (127, 205, 187),
                       (65, 182, 196), (44, 127, 184), (37, 52, 148)],
    'brewer_ylgnbu7': [(255, 255, 204), (199, 233, 180), (127, 205, 187),
                       (65, 182, 196), (29, 145, 192), (34, 94, 168),
                       (12, 44, 132)],
    'brewer_ylgnbu8': [(255, 255, 217), (237, 248, 177), (199, 233, 180),
                       (127, 205, 187), (65, 182, 196), (29, 145, 192),
                       (34, 94, 168), (12, 44, 132)],
    'brewer_ylgnbu9': [(255, 255, 217), (237, 248, 177), (199, 233, 180),
                       (127, 205, 187), (65, 182, 196), (29, 145, 192),
                       (34, 94, 168), (37, 52, 148), (8, 29, 88)],
    'brewer_gnbu3': [(224, 243, 219), (168, 221, 181), (67, 162, 202)],
    'brewer_gnbu4': [(240, 249, 232), (186, 228, 188), (123, 204, 196),
                     (43, 140, 190)],
    'brewer_gnbu5': [(240, 249, 232), (186, 228, 188), (123, 204, 196),
                     (67, 162, 202), (8, 104, 172)],
    'brewer_gnbu6': [(240, 249, 232), (204, 235, 197), (168, 221, 181),
                     (123, 204, 196), (67, 162, 202), (8, 104, 172)],
    'brewer_gnbu7': [(240, 249, 232), (204, 235, 197), (168, 221, 181),
                     (123, 204, 196), (78, 179, 211), (43, 140, 190),
                     (8, 88, 158)],
    'brewer_gnbu8': [(247, 252, 240), (224, 243, 219), (204, 235, 197),
                     (168, 221, 181), (123, 204, 196), (78, 179, 211),
                     (43, 140, 190), (8, 88, 158)],
    'brewer_gnbu9': [(247, 252, 240), (224, 243, 219), (204, 235, 197),
                     (168, 221, 181), (123, 204, 196), (78, 179, 211),
                     (43, 140, 190), (8, 104, 172), (8, 64, 129)],
    'brewer_bugn3': [(229, 245, 249), (153, 216, 201), (44, 162, 95)],
    'brewer_bugn4': [(237, 248, 251), (178, 226, 226), (102, 194, 164),
                     (35, 139, 69)],
    'brewer_bugn5': [(237, 248, 251), (178, 226, 226), (102, 194, 164),
                     (44, 162, 95), (0, 109, 44)],
    'brewer_bugn6': [(237, 248, 251), (204, 236, 230), (153, 216, 201),
                     (102, 194, 164), (44, 162, 95), (0, 109, 44)],
    'brewer_bugn7': [(237, 248, 251), (204, 236, 230), (153, 216, 201),
                     (102, 194, 164), (65, 174, 118), (35, 139, 69),
                     (0, 88, 36)],
    'brewer_bugn8': [(247, 252, 253), (229, 245, 249), (204, 236, 230),
                     (153, 216, 201), (102, 194, 164), (65, 174, 118),
                     (35, 139, 69), (0, 88, 36)],
    'brewer_bugn9': [(247, 252, 253), (229, 245, 249), (204, 236, 230),
                     (153, 216, 201), (102, 194, 164), (65, 174, 118),
                     (35, 139, 69), (0, 109, 44), (0, 68, 27)],
    'brewer_pubugn3': [(236, 226, 240), (166, 189, 219), (28, 144, 153)],
    'brewer_pubugn4': [(246, 239, 247), (189, 201, 225), (103, 169, 207),
                       (2, 129, 138)],
    'brewer_pubugn5': [(246, 239, 247), (189, 201, 225), (103, 169, 207),
                       (28, 144, 153), (1, 108, 89)],
    'brewer_pubugn6': [(246, 239, 247), (208, 209, 230), (166, 189, 219),
                       (103, 169, 207), (28, 144, 153), (1, 108, 89)],
    'brewer_pubugn7': [(246, 239, 247), (208, 209, 230), (166, 189, 219),
                       (103, 169, 207), (54, 144, 192), (2, 129, 138),
                       (1, 100, 80)],
    'brewer_pubugn8': [(255, 247, 251), (236, 226, 240), (208, 209, 230),
                       (166, 189, 219), (103, 169, 207), (54, 144, 192),
                       (2, 129, 138), (1, 100, 80)],
    'brewer_pubugn9': [(255, 247, 251), (236, 226, 240), (208, 209, 230),
                       (166, 189, 219), (103, 169, 207), (54, 144, 192),
                       (2, 129, 138), (1, 108, 89), (1, 70, 54)],
    'brewer_pubu3': [(236, 231, 242), (166, 189, 219), (43, 140, 190)],
    'brewer_pubu4': [(241, 238, 246), (189, 201, 225), (116, 169, 207),
                     (5, 112, 176)],
    'brewer_pubu5': [(241, 238, 246), (189, 201, 225), (116, 169, 207),
                     (43, 140, 190), (4, 90, 141)],
    'brewer_pubu6': [(241, 238, 246), (208, 209, 230), (166, 189, 219),
                     (116, 169, 207), (43, 140, 190), (4, 90, 141)],
    'brewer_pubu7': [(241, 238, 246), (208, 209, 230), (166, 189, 219),
                     (116, 169, 207), (54, 144, 192), (5, 112, 176),
                     (3, 78, 123)],
    'brewer_pubu8': [(255, 247, 251), (236, 231, 242), (208, 209, 230),
                     (166, 189, 219), (116, 169, 207), (54, 144, 192),
                     (5, 112, 176), (3, 78, 123)],
    'brewer_pubu9': [(255, 247, 251), (236, 231, 242), (208, 209, 230),
                     (166, 189, 219), (116, 169, 207), (54, 144, 192),
                     (5, 112, 176), (4, 90, 141), (2, 56, 88)],
    'brewer_bupu3': [(224, 236, 244), (158, 188, 218), (136, 86, 167)],
    'brewer_bupu4': [(237, 248, 251), (179, 205, 227), (140, 150, 198),
                     (136, 65, 157)],
    'brewer_bupu5': [(237, 248, 251), (179, 205, 227), (140, 150, 198),
                     (136, 86, 167), (129, 15, 124)],
    'brewer_bupu6': [(237, 248, 251), (191, 211, 230), (158, 188, 218),
                     (140, 150, 198), (136, 86, 167), (129, 15, 124)],
    'brewer_bupu7': [(237, 248, 251), (191, 211, 230), (158, 188, 218),
                     (140, 150, 198), (140, 107, 177), (136, 65, 157),
                     (110, 1, 107)],
    'brewer_bupu8': [(247, 252, 253), (224, 236, 244), (191, 211, 230),
                     (158, 188, 218), (140, 150, 198), (140, 107, 177),
                     (136, 65, 157), (110, 1, 107)],
    'brewer_bupu9': [(247, 252, 253), (224, 236, 244), (191, 211, 230),
                     (158, 188, 218), (140, 150, 198), (140, 107, 177),
                     (136, 65, 157), (129, 15, 124), (77, 0, 75)],
    'brewer_rdpu3': [(253, 224, 221), (250, 159, 181), (197, 27, 138)],
    'brewer_rdpu4': [(254, 235, 226), (251, 180, 185), (247, 104, 161),
                     (174, 1, 126)],
    'brewer_rdpu5': [(254, 235, 226), (251, 180, 185), (247, 104, 161),
                     (197, 27, 138), (122, 1, 119)],
    'brewer_rdpu6': [(254, 235, 226), (252, 197, 192), (250, 159, 181),
                     (247, 104, 161), (197, 27, 138), (122, 1, 119)],
    'brewer_rdpu7': [(254, 235, 226), (252, 197, 192), (250, 159, 181),
                     (247, 104, 161), (221, 52, 151), (174, 1, 126),
                     (122, 1, 119)],
    'brewer_rdpu8': [(255, 247, 243), (253, 224, 221), (252, 197, 192),
                     (250, 159, 181), (247, 104, 161), (221, 52, 151),
                     (174, 1, 126), (122, 1, 119)],
    'brewer_rdpu9': [(255, 247, 243), (253, 224, 221), (252, 197, 192),
                     (250, 159, 181), (247, 104, 161), (221, 52, 151),
                     (174, 1, 126), (122, 1, 119), (73, 0, 106)],
    'brewer_purd3': [(231, 225, 239), (201, 148, 199), (221, 28, 119)],
    'brewer_purd4': [(241, 238, 246), (215, 181, 216), (223, 101, 176),
                     (206, 18, 86)],
    'brewer_purd5': [(241, 238, 246), (215, 181, 216), (223, 101, 176),
                     (221, 28, 119), (152, 0, 67)],
    'brewer_purd6': [(241, 238, 246), (212, 185, 218), (201, 148, 199),
                     (223, 101, 176), (221, 28, 119), (152, 0, 67)],
    'brewer_purd7': [(241, 238, 246), (212, 185, 218), (201, 148, 199),
                     (223, 101, 176), (231, 41, 138), (206, 18, 86),
                     (145, 0, 63)],
    'brewer_purd8': [(247, 244, 249), (231, 225, 239), (212, 185, 218),
                     (201, 148, 199), (223, 101, 176), (231, 41, 138),
                     (206, 18, 86), (145, 0, 63)],
    'brewer_purd9': [(247, 244, 249), (231, 225, 239), (212, 185, 218),
                     (201, 148, 199), (223, 101, 176), (231, 41, 138),
                     (206, 18, 86), (152, 0, 67), (103, 0, 31)],
    'brewer_orrd3': [(254, 232, 200), (253, 187, 132), (227, 74, 51)],
    'brewer_orrd4': [(254, 240, 217), (253, 204, 138), (252, 141, 89),
                     (215, 48, 31)],
    'brewer_orrd5': [(254, 240, 217), (253, 204, 138), (252, 141, 89),
                     (227, 74, 51), (179, 0, 0)],
    'brewer_orrd6': [(254, 240, 217), (253, 212, 158), (253, 187, 132),
                     (252, 141, 89), (227, 74, 51), (179, 0, 0)],
    'brewer_orrd7': [(254, 240, 217), (253, 212, 158), (253, 187, 132),
                     (252, 141, 89), (239, 101, 72), (215, 48, 31),
                     (153, 0, 0)],
    'brewer_orrd8': [(255, 247, 236), (254, 232, 200), (253, 212, 158),
                     (253, 187, 132), (252, 141, 89), (239, 101, 72),
                     (215, 48, 31), (153, 0, 0)],
    'brewer_orrd9': [(255, 247, 236), (254, 232, 200), (253, 212, 158),
                     (253, 187, 132), (252, 141, 89), (239, 101, 72),
                     (215, 48, 31), (179, 0, 0), (127, 0, 0)],
    'brewer_ylorrd3': [(255, 237, 160), (254, 178, 76), (240, 59, 32)],
    'brewer_ylorrd4': [(255, 255, 178), (254, 204, 92), (253, 141, 60),
                       (227, 26, 28)],
    'brewer_ylorrd5': [(255, 255, 178), (254, 204, 92), (253, 141, 60),
                       (240, 59, 32), (189, 0, 38)],
    'brewer_ylorrd6': [(255, 255, 178), (254, 217, 118), (254, 178, 76),
                       (253, 141, 60), (240, 59, 32), (189, 0, 38)],
    'brewer_ylorrd7': [(255, 255, 178), (254, 217, 118), (254, 178, 76),
                       (253, 141, 60), (252, 78, 42), (227, 26, 28),
                       (177, 0, 38)],
    'brewer_ylorrd8': [(255, 255, 204), (255, 237, 160), (254, 217, 118),
                       (254, 178, 76), (253, 141, 60), (252, 78, 42),
                       (227, 26, 28), (177, 0, 38)],
    'brewer_ylorrd9': [(255, 255, 204), (255, 237, 160), (254, 217, 118),
                       (254, 178, 76), (253, 141, 60), (252, 78, 42),
                       (227, 26, 28), (189, 0, 38), (128, 0, 38)],
    'brewer_ylorbr3': [(255, 247, 188), (254, 196, 79), (217, 95, 14)],
    'brewer_ylorbr4': [(255, 255, 212), (254, 217, 142), (254, 153, 41),
                       (204, 76, 2)],
    'brewer_ylorbr5': [(255, 255, 212), (254, 217, 142), (254, 153, 41),
                       (217, 95, 14), (153, 52, 4)],
    'brewer_ylorbr6': [(255, 255, 212), (254, 227, 145), (254, 196, 79),
                       (254, 153, 41), (217, 95, 14), (153, 52, 4)],
    'brewer_ylorbr7': [(255, 255, 212), (254, 227, 145), (254, 196, 79),
                       (254, 153, 41), (236, 112, 20), (204, 76, 2),
                       (140, 45, 4)],
    'brewer_ylorbr8': [(255, 255, 229), (255, 247, 188), (254, 227, 145),
                       (254, 196, 79), (254, 153, 41), (236, 112, 20),
                       (204, 76, 2), (140, 45, 4)],
    'brewer_ylorbr9': [(255, 255, 229), (255, 247, 188), (254, 227, 145),
                       (254, 196, 79), (254, 153, 41), (236, 112, 20),
                       (204, 76, 2), (153, 52, 4), (102, 37, 6)],
    'brewer_purples3': [(239, 237, 245), (188, 189, 220), (117, 107, 177)],
    'brewer_purples4': [(242, 240, 247), (203, 201, 226), (158, 154, 200),
                        (106, 81, 163)],
    'brewer_purples5': [(242, 240, 247), (203, 201, 226), (158, 154, 200),
                        (117, 107, 177), (84, 39, 143)],
    'brewer_purples6': [(242, 240, 247), (218, 218, 235), (188, 189, 220),
                        (158, 154, 200), (117, 107, 177), (84, 39, 143)],
    'brewer_purples7': [(242, 240, 247), (218, 218, 235), (188, 189, 220),
                        (158, 154, 200), (128, 125, 186), (106, 81, 163),
                        (74, 20, 134)],
    'brewer_purples8': [(252, 251, 253), (239, 237, 245), (218, 218, 235),
                        (188, 189, 220), (158, 154, 200), (128, 125, 186),
                        (106, 81, 163), (74, 20, 134)],
    'brewer_purples9': [(252, 251, 253), (239, 237, 245), (218, 218, 235),
                        (188, 189, 220), (158, 154, 200), (128, 125, 186),
                        (106, 81, 163), (84, 39, 143), (63, 0, 125)],
    'brewer_blues3': [(222, 235, 247), (158, 202, 225), (49, 130, 189)],
    'brewer_blues4': [(239, 243, 255), (189, 215, 231), (107, 174, 214),
                      (33, 113, 181)],
    'brewer_blues5': [(239, 243, 255), (189, 215, 231), (107, 174, 214),
                      (49, 130, 189), (8, 81, 156)],
    'brewer_blues6': [(239, 243, 255), (198, 219, 239), (158, 202, 225),
                      (107, 174, 214), (49, 130, 189), (8, 81, 156)],
    'brewer_blues7': [(239, 243, 255), (198, 219, 239), (158, 202, 225),
                      (107, 174, 214), (66, 146, 198), (33, 113, 181),
                      (8, 69, 148)],
    'brewer_blues8': [(247, 251, 255), (222, 235, 247), (198, 219, 239),
                      (158, 202, 225), (107, 174, 214), (66, 146, 198),
                      (33, 113, 181), (8, 69, 148)],
    'brewer_blues9': [(247, 251, 255), (222, 235, 247), (198, 219, 239),
                      (158, 202, 225), (107, 174, 214), (66, 146, 198),
                      (33, 113, 181), (8, 81, 156), (8, 48, 107)],
    'brewer_greens3': [(229, 245, 224), (161, 217, 155), (49, 163, 84)],
    'brewer_greens4': [(237, 248, 233), (186, 228, 179), (116, 196, 118),
                       (35, 139, 69)],
    'brewer_greens5': [(237, 248, 233), (186, 228, 179), (116, 196, 118),
                       (49, 163, 84), (0, 109, 44)],
    'brewer_greens6': [(237, 248, 233), (199, 233, 192), (161, 217, 155),
                       (116, 196, 118), (49, 163, 84), (0, 109, 44)],
    'brewer_greens7': [(237, 248, 233), (199, 233, 192), (161, 217, 155),
                       (116, 196, 118), (65, 171, 93), (35, 139, 69),
                       (0, 90, 50)],
    'brewer_greens8': [(247, 252, 245), (229, 245, 224), (199, 233, 192),
                       (161, 217, 155), (116, 196, 118), (65, 171, 93),
                       (35, 139, 69), (0, 90, 50)],
    'brewer_greens9': [(247, 252, 245), (229, 245, 224), (199, 233, 192),
                       (161, 217, 155), (116, 196, 118), (65, 171, 93),
                       (35, 139, 69), (0, 109, 44), (0, 68, 27)],
    'brewer_oranges3': [(254, 230, 206), (253, 174, 107), (230, 85, 13)],
    'brewer_oranges4': [(254, 237, 222), (253, 190, 133), (253, 141, 60),
                        (217, 71, 1)],
    'brewer_oranges5': [(254, 237, 222), (253, 190, 133), (253, 141, 60),
                        (230, 85, 13), (166, 54, 3)],
    'brewer_oranges6': [(254, 237, 222), (253, 208, 162), (253, 174, 107),
                        (253, 141, 60), (230, 85, 13), (166, 54, 3)],
    'brewer_oranges7': [(254, 237, 222), (253, 208, 162), (253, 174, 107),
                        (253, 141, 60), (241, 105, 19), (217, 72, 1),
                        (140, 45, 4)],
    'brewer_oranges8': [(255, 245, 235), (254, 230, 206), (253, 208, 162),
                        (253, 174, 107), (253, 141, 60), (241, 105, 19),
                        (217, 72, 1), (140, 45, 4)],
    'brewer_oranges9': [(255, 245, 235), (254, 230, 206), (253, 208, 162),
                        (253, 174, 107), (253, 141, 60), (241, 105, 19),
                        (217, 72, 1), (166, 54, 3), (127, 39, 4)],
    'brewer_reds3': [(254, 224, 210), (252, 146, 114), (222, 45, 38)],
    'brewer_reds4': [(254, 229, 217), (252, 174, 145), (251, 106, 74),
                     (203, 24, 29)],
    'brewer_reds5': [(254, 229, 217), (252, 174, 145), (251, 106, 74),
                     (222, 45, 38), (165, 15, 21)],
    'brewer_reds6': [(254, 229, 217), (252, 187, 161), (252, 146, 114),
                     (251, 106, 74), (222, 45, 38), (165, 15, 21)],
    'brewer_reds7': [(254, 229, 217), (252, 187, 161), (252, 146, 114),
                     (251, 106, 74), (239, 59, 44), (203, 24, 29),
                     (153, 0, 13)],
    'brewer_reds8': [(255, 245, 240), (254, 224, 210), (252, 187, 161),
                     (252, 146, 114), (251, 106, 74), (239, 59, 44),
                     (203, 24, 29), (153, 0, 13)],
    'brewer_reds9': [(255, 245, 240), (254, 224, 210), (252, 187, 161),
                     (252, 146, 114), (251, 106, 74), (239, 59, 44),
                     (203, 24, 29), (165, 15, 21), (103, 0, 13)],
    'brewer_greys3': [(240, 240, 240), (189, 189, 189), (99, 99, 99)],
    'brewer_greys4': [(247, 247, 247), (204, 204, 204), (150, 150, 150),
                      (82, 82, 82)],
    'brewer_greys5': [(247, 247, 247), (204, 204, 204), (150, 150, 150),
                      (99, 99, 99), (37, 37, 37)],
    'brewer_greys6': [(247, 247, 247), (217, 217, 217), (189, 189, 189),
                      (150, 150, 150), (99, 99, 99), (37, 37, 37)],
    'brewer_greys7': [(247, 247, 247), (217, 217, 217), (189, 189, 189),
                      (150, 150, 150), (115, 115, 115), (82, 82, 82),
                      (37, 37, 37)],
    'brewer_greys8': [(255, 255, 255), (240, 240, 240), (217, 217, 217),
                      (189, 189, 189), (150, 150, 150), (115, 115, 115),
                      (82, 82, 82), (37, 37, 37)],
    'brewer_greys9': [(255, 255, 255), (240, 240, 240), (217, 217, 217),
                      (189, 189, 189), (150, 150, 150), (115, 115, 115),
                      (82, 82, 82), (37, 37, 37), (0, 0, 0)],
}


# diverging color maps
brewer_diverging = {
    'brewer_puor3': [(241, 163, 64), (247, 247, 247), (153, 142, 195)],
    'brewer_puor4': [(230, 97, 1), (253, 184, 99), (178, 171, 210),
                     (94, 60, 153)],
    'brewer_puor5': [(230, 97, 1), (253, 184, 99), (247, 247, 247),
                     (178, 171, 210), (94, 60, 153)],
    'brewer_puor6': [(179, 88, 6), (241, 163, 64), (254, 224, 182),
                     (216, 218, 235), (153, 142, 195), (84, 39, 136)],
    'brewer_puor7': [(179, 88, 6), (241, 163, 64), (254, 224, 182),
                     (247, 247, 247), (216, 218, 235), (153, 142, 195),
                     (84, 39, 136)],
    'brewer_puor8': [(179, 88, 6), (224, 130, 20), (253, 184, 99),
                     (254, 224, 182), (216, 218, 235), (178, 171, 210),
                     (128, 115, 172), (84, 39, 136)],
    'brewer_puor9': [(179, 88, 6), (224, 130, 20), (253, 184, 99),
                     (254, 224, 182), (247, 247, 247), (216, 218, 235),
                     (178, 171, 210), (128, 115, 172), (84, 39, 136)],
    'brewer_puor10': [(127, 59, 8), (179, 88, 6), (224, 130, 20),
                      (253, 184, 99), (254, 224, 182), (216, 218, 235),
                      (178, 171, 210), (128, 115, 172), (84, 39, 136),
                      (45, 0, 75)],
    'brewer_puor11': [(127, 59, 8), (179, 88, 6), (224, 130, 20),
                      (253, 184, 99), (254, 224, 182), (247, 247, 247),
                      (216, 218, 235), (178, 171, 210), (128, 115, 172),
                      (84, 39, 136), (45, 0, 75)],
    'brewer_brbg3': [(216, 179, 101), (245, 245, 245), (90, 180, 172)],
    'brewer_brbg4': [(166, 97, 26), (223, 194, 125), (128, 205, 193),
                     (1, 133, 113)],
    'brewer_brbg5': [(166, 97, 26), (223, 194, 125), (245, 245, 245),
                     (128, 205, 193), (1, 133, 113)],
    'brewer_brbg6': [(140, 81, 10), (216, 179, 101), (246, 232, 195),
                     (199, 234, 229), (90, 180, 172), (1, 102, 94)],
    'brewer_brbg7': [(140, 81, 10), (216, 179, 101), (246, 232, 195),
                     (245, 245, 245), (199, 234, 229), (90, 180, 172),
                     (1, 102, 94)],
    'brewer_brbg8': [(140, 81, 10), (191, 129, 45), (223, 194, 125),
                     (246, 232, 195), (199, 234, 229), (128, 205, 193),
                     (53, 151, 143), (1, 102, 94)],
    'brewer_brbg9': [(140, 81, 10), (191, 129, 45), (223, 194, 125),
                     (246, 232, 195), (245, 245, 245), (199, 234, 229),
                     (128, 205, 193), (53, 151, 143), (1, 102, 94)],
    'brewer_brbg10': [(84, 48, 5), (140, 81, 10), (191, 129, 45),
                      (223, 194, 125), (246, 232, 195), (199, 234, 229),
                      (128, 205, 193), (53, 151, 143), (1, 102, 94),
                      (0, 60, 48)],
    'brewer_brbg11': [(84, 48, 5), (140, 81, 10), (191, 129, 45),
                      (223, 194, 125), (246, 232, 195), (245, 245, 245),
                      (199, 234, 229), (128, 205, 193), (53, 151, 143),
                      (1, 102, 94), (0, 60, 48)],
    'brewer_prgn3': [(175, 141, 195), (247, 247, 247), (127, 191, 123)],
    'brewer_prgn4': [(123, 50, 148), (194, 165, 207), (166, 219, 160),
                     (0, 136, 55)],
    'brewer_prgn5': [(123, 50, 148), (194, 165, 207), (247, 247, 247),
                     (166, 219, 160), (0, 136, 55)],
    'brewer_prgn6': [(118, 42, 131), (175, 141, 195), (231, 212, 232),
                     (217, 240, 211), (127, 191, 123), (27, 120, 55)],
    'brewer_prgn7': [(118, 42, 131), (175, 141, 195), (231, 212, 232),
                     (247, 247, 247), (217, 240, 211), (127, 191, 123),
                     (27, 120, 55)],
    'brewer_prgn8': [(118, 42, 131), (153, 112, 171), (194, 165, 207),
                     (231, 212, 232), (217, 240, 211), (166, 219, 160),
                     (90, 174, 97), (27, 120, 55)],
    'brewer_prgn9': [(118, 42, 131), (153, 112, 171), (194, 165, 207),
                     (231, 212, 232), (247, 247, 247), (217, 240, 211),
                     (166, 219, 160), (90, 174, 97), (27, 120, 55)],
    'brewer_prgn10': [(64, 0, 75), (118, 42, 131), (153, 112, 171),
                      (194, 165, 207), (231, 212, 232), (217, 240, 211),
                      (166, 219, 160), (90, 174, 97), (27, 120, 55),
                      (0, 68, 27)],
    'brewer_prgn11': [(64, 0, 75), (118, 42, 131), (153, 112, 171),
                      (194, 165, 207), (231, 212, 232), (247, 247, 247),
                      (217, 240, 211), (166, 219, 160), (90, 174, 97),
                      (27, 120, 55), (0, 68, 27)],
    'brewer_piyg3': [(233, 163, 201), (247, 247, 247), (161, 215, 106)],
    'brewer_piyg4': [(208, 28, 139), (241, 182, 218), (184, 225, 134),
                     (77, 172, 38)],
    'brewer_piyg5': [(208, 28, 139), (241, 182, 218), (247, 247, 247),
                     (184, 225, 134), (77, 172, 38)],
    'brewer_piyg6': [(197, 27, 125), (233, 163, 201), (253, 224, 239),
                     (230, 245, 208), (161, 215, 106), (77, 146, 33)],
    'brewer_piyg7': [(197, 27, 125), (233, 163, 201), (253, 224, 239),
                     (247, 247, 247), (230, 245, 208), (161, 215, 106),
                     (77, 146, 33)],
    'brewer_piyg8': [(197, 27, 125), (222, 119, 174), (241, 182, 218),
                     (253, 224, 239), (230, 245, 208), (184, 225, 134),
                     (127, 188, 65), (77, 146, 33)],
    'brewer_piyg9': [(197, 27, 125), (222, 119, 174), (241, 182, 218),
                     (253, 224, 239), (247, 247, 247), (230, 245, 208),
                     (184, 225, 134), (127, 188, 65), (77, 146, 33)],
    'brewer_piyg10': [(142, 1, 82), (197, 27, 125), (222, 119, 174),
                      (241, 182, 218), (253, 224, 239), (230, 245, 208),
                      (184, 225, 134), (127, 188, 65), (77, 146, 33),
                      (39, 100, 25)],
    'brewer_piyg11': [(142, 1, 82), (197, 27, 125), (222, 119, 174),
                      (241, 182, 218), (253, 224, 239), (247, 247, 247),
                      (230, 245, 208), (184, 225, 134), (127, 188, 65),
                      (77, 146, 33), (39, 100, 25)],
    'brewer_rdbu3': [(239, 138, 98), (247, 247, 247), (103, 169, 207)],
    'brewer_rdbu4': [(202, 0, 32), (244, 165, 130), (146, 197, 222),
                     (5, 113, 176)],
    'brewer_rdbu5': [(202, 0, 32), (244, 165, 130), (247, 247, 247),
                     (146, 197, 222), (5, 113, 176)],
    'brewer_rdbu6': [(178, 24, 43), (239, 138, 98), (253, 219, 199),
                     (209, 229, 240), (103, 169, 207), (33, 102, 172)],
    'brewer_rdbu7': [(178, 24, 43), (239, 138, 98), (253, 219, 199),
                     (247, 247, 247), (209, 229, 240), (103, 169, 207),
                     (33, 102, 172)],
    'brewer_rdbu8': [(178, 24, 43), (214, 96, 77), (244, 165, 130),
                     (253, 219, 199), (209, 229, 240), (146, 197, 222),
                     (67, 147, 195), (33, 102, 172)],
    'brewer_rdbu9': [(178, 24, 43), (214, 96, 77), (244, 165, 130),
                     (253, 219, 199), (247, 247, 247), (209, 229, 240),
                     (146, 197, 222), (67, 147, 195), (33, 102, 172)],
    'brewer_rdbu10': [(103, 0, 31), (178, 24, 43), (214, 96, 77),
                      (244, 165, 130), (253, 219, 199), (209, 229, 240),
                      (146, 197, 222), (67, 147, 195), (33, 102, 172),
                      (5, 48, 97)],
    'brewer_rdbu11': [(103, 0, 31), (178, 24, 43), (214, 96, 77),
                      (244, 165, 130), (253, 219, 199), (247, 247, 247),
                      (209, 229, 240), (146, 197, 222), (67, 147, 195),
                      (33, 102, 172), (5, 48, 97)],
    'brewer_rdgy3': [(239, 138, 98), (255, 255, 255), (153, 153, 153)],
    'brewer_rdgy4': [(202, 0, 32), (244, 165, 130), (186, 186, 186),
                     (64, 64, 64)],
    'brewer_rdgy5': [(202, 0, 32), (244, 165, 130), (255, 255, 255),
                     (186, 186, 186), (64, 64, 64)],
    'brewer_rdgy6': [(178, 24, 43), (239, 138, 98), (253, 219, 199),
                     (224, 224, 224), (153, 153, 153), (77, 77, 77)],
    'brewer_rdgy7': [(178, 24, 43), (239, 138, 98), (253, 219, 199),
                     (255, 255, 255), (224, 224, 224), (153, 153, 153),
                     (77, 77, 77)],
    'brewer_rdgy8': [(178, 24, 43), (214, 96, 77), (244, 165, 130),
                     (253, 219, 199), (224, 224, 224), (186, 186, 186),
                     (135, 135, 135), (77, 77, 77)],
    'brewer_rdgy9': [(178, 24, 43), (214, 96, 77), (244, 165, 130),
                     (253, 219, 199), (255, 255, 255), (224, 224, 224),
                     (186, 186, 186), (135, 135, 135), (77, 77, 77)],
    'brewer_rdgy10': [(103, 0, 31), (178, 24, 43), (214, 96, 77),
                      (244, 165, 130), (253, 219, 199), (224, 224, 224),
                      (186, 186, 186), (135, 135, 135), (77, 77, 77),
                      (26, 26, 26)],
    'brewer_rdgy11': [(103, 0, 31), (178, 24, 43), (214, 96, 77),
                      (244, 165, 130), (253, 219, 199), (255, 255, 255),
                      (224, 224, 224), (186, 186, 186), (135, 135, 135),
                      (77, 77, 77), (26, 26, 26)],
    'brewer_rdylbu3': [(252, 141, 89), (255, 255, 191), (145, 191, 219)],
    'brewer_rdylbu4': [(215, 25, 28), (253, 174, 97), (171, 217, 233),
                       (44, 123, 182)],
    'brewer_rdylbu5': [(215, 25, 28), (253, 174, 97), (255, 255, 191),
                       (171, 217, 233), (44, 123, 182)],
    'brewer_rdylbu6': [(215, 48, 39), (252, 141, 89), (254, 224, 144),
                       (224, 243, 248), (145, 191, 219), (69, 117, 180)],
    'brewer_rdylbu7': [(215, 48, 39), (252, 141, 89), (254, 224, 144),
                       (255, 255, 191), (224, 243, 248), (145, 191, 219),
                       (69, 117, 180)],
    'brewer_rdylbu8': [(215, 48, 39), (244, 109, 67), (253, 174, 97),
                       (254, 224, 144), (224, 243, 248), (171, 217, 233),
                       (116, 173, 209), (69, 117, 180)],
    'brewer_rdylbu9': [(215, 48, 39), (244, 109, 67), (253, 174, 97),
                       (254, 224, 144), (255, 255, 191), (224, 243, 248),
                       (171, 217, 233), (116, 173, 209), (69, 117, 180)],
    'brewer_rdylbu10': [(165, 0, 38), (215, 48, 39), (244, 109, 67),
                        (253, 174, 97), (254, 224, 144), (224, 243, 248),
                        (171, 217, 233), (116, 173, 209), (69, 117, 180),
                        (49, 54, 149)],
    'brewer_rdylbu11': [(165, 0, 38), (215, 48, 39), (244, 109, 67),
                        (253, 174, 97), (254, 224, 144), (255, 255, 191),
                        (224, 243, 248), (171, 217, 233), (116, 173, 209),
                        (69, 117, 180), (49, 54, 149)],
    'brewer_spectral3': [(252, 141, 89), (255, 255, 191), (153, 213, 148)],
    'brewer_spectral4': [(215, 25, 28), (253, 174, 97), (171, 221, 164),
                         (43, 131, 186)],
    'brewer_spectral5': [(215, 25, 28), (253, 174, 97), (255, 255, 191),
                         (171, 221, 164), (43, 131, 186)],
    'brewer_spectral6': [(213, 62, 79), (252, 141, 89), (254, 224, 139),
                         (230, 245, 152), (153, 213, 148), (50, 136, 189)],
    'brewer_spectral7': [(213, 62, 79), (252, 141, 89), (254, 224, 139),
                         (255, 255, 191), (230, 245, 152), (153, 213, 148),
                         (50, 136, 189)],
    'brewer_spectral8': [(213, 62, 79), (244, 109, 67), (253, 174, 97),
                         (254, 224, 139), (230, 245, 152), (171, 221, 164),
                         (102, 194, 165), (50, 136, 189)],
    'brewer_spectral9': [(213, 62, 79), (244, 109, 67), (253, 174, 97),
                         (254, 224, 139), (255, 255, 191), (230, 245, 152),
                         (171, 221, 164), (102, 194, 165), (50, 136, 189)],
    'brewer_spectral10': [(158, 1, 66), (213, 62, 79), (244, 109, 67),
                          (253, 174, 97), (254, 224, 139), (230, 245, 152),
                          (171, 221, 164), (102, 194, 165), (50, 136, 189),
                          (94, 79, 162)],
    'brewer_spectral11': [(158, 1, 66), (213, 62, 79), (244, 109, 67),
                          (253, 174, 97), (254, 224, 139), (255, 255, 191),
                          (230, 245, 152), (171, 221, 164), (102, 194, 165),
                          (50, 136, 189), (94, 79, 162)],
    'brewer_rdylgn3': [(252, 141, 89), (255, 255, 191), (145, 207, 96)],
    'brewer_rdylgn4': [(215, 25, 28), (253, 174, 97), (166, 217, 106),
                       (26, 150, 65)],
    'brewer_rdylgn5': [(215, 25, 28), (253, 174, 97), (255, 255, 191),
                       (166, 217, 106), (26, 150, 65)],
    'brewer_rdylgn6': [(215, 48, 39), (252, 141, 89), (254, 224, 139),
                       (217, 239, 139), (145, 207, 96), (26, 152, 80)],
    'brewer_rdylgn7': [(215, 48, 39), (252, 141, 89), (254, 224, 139),
                       (255, 255, 191), (217, 239, 139), (145, 207, 96),
                       (26, 152, 80)],
    'brewer_rdylgn8': [(215, 48, 39), (244, 109, 67), (253, 174, 97),
                       (254, 224, 139), (217, 239, 139), (166, 217, 106),
                       (102, 189, 99), (26, 152, 80)],
    'brewer_rdylgn9': [(215, 48, 39), (244, 109, 67), (253, 174, 97),
                       (254, 224, 139), (255, 255, 191), (217, 239, 139),
                       (166, 217, 106), (102, 189, 99), (26, 152, 80)],
    'brewer_rdylgn10': [(165, 0, 38), (215, 48, 39), (244, 109, 67),
                        (253, 174, 97), (254, 224, 139), (217, 239, 139),
                        (166, 217, 106), (102, 189, 99), (26, 152, 80),
                        (0, 104, 55)],
    'brewer_rdylgn11': [(165, 0, 38), (215, 48, 39), (244, 109, 67),
                        (253, 174, 97), (254, 224, 139), (255, 255, 191),
                        (217, 239, 139), (166, 217, 106), (102, 189, 99),
                        (26, 152, 80), (0, 104, 55)]
}


# qualitative color maps
brewer_qualitative = {
    'brewer_set33': [(141, 211, 199), (255, 255, 179), (190, 186, 218)],
    'brewer_set34': [(141, 211, 199), (255, 255, 179), (190, 186, 218),
                     (251, 128, 114)],
    'brewer_set35': [(141, 211, 199), (255, 255, 179), (190, 186, 218),
                     (251, 128, 114), (128, 177, 211)],
    'brewer_set36': [(141, 211, 199), (255, 255, 179), (190, 186, 218),
                     (251, 128, 114), (128, 177, 211), (253, 180, 98)],
    'brewer_set37': [(141, 211, 199), (255, 255, 179), (190, 186, 218),
                     (251, 128, 114), (128, 177, 211), (253, 180, 98),
                     (179, 222, 105)],
    'brewer_set38': [(141, 211, 199), (255, 255, 179), (190, 186, 218),
                     (251, 128, 114), (128, 177, 211), (253, 180, 98),
                     (179, 222, 105), (252, 205, 229)],
    'brewer_set39': [(141, 211, 199), (255, 255, 179), (190, 186, 218),
                     (251, 128, 114), (128, 177, 211), (253, 180, 98),
                     (179, 222, 105), (252, 205, 229), (217, 217, 217)],
    'brewer_set310': [(141, 211, 199), (255, 255, 179), (190, 186, 218),
                      (251, 128, 114), (128, 177, 211), (253, 180, 98),
                      (179, 222, 105), (252, 205, 229), (217, 217, 217),
                      (188, 128, 189)],
    'brewer_set311': [(141, 211, 199), (255, 255, 179), (190, 186, 218),
                      (251, 128, 114), (128, 177, 211), (253, 180, 98),
                      (179, 222, 105), (252, 205, 229), (217, 217, 217),
                      (188, 128, 189), (204, 235, 197)],
    'brewer_set312': [(141, 211, 199), (255, 255, 179), (190, 186, 218),
                      (251, 128, 114), (128, 177, 211), (253, 180, 98),
                      (179, 222, 105), (252, 205, 229), (217, 217, 217),
                      (188, 128, 189), (204, 235, 197), (255, 237, 111)],
    'brewer_pastel13': [(251, 180, 174), (179, 205, 227), (204, 235, 197)],
    'brewer_pastel14': [(251, 180, 174), (179, 205, 227), (204, 235, 197),
                        (222, 203, 228)],
    'brewer_pastel15': [(251, 180, 174), (179, 205, 227), (204, 235, 197),
                        (222, 203, 228), (254, 217, 166)],
    'brewer_pastel16': [(251, 180, 174), (179, 205, 227), (204, 235, 197),
                        (222, 203, 228), (254, 217, 166), (255, 255, 204)],
    'brewer_pastel17': [(251, 180, 174), (179, 205, 227), (204, 235, 197),
                        (222, 203, 228), (254, 217, 166), (255, 255, 204),
                        (229, 216, 189)],
    'brewer_pastel18': [(251, 180, 174), (179, 205, 227), (204, 235, 197),
                        (222, 203, 228), (254, 217, 166), (255, 255, 204),
                        (229, 216, 189), (253, 218, 236)],
    'brewer_pastel19': [(251, 180, 174), (179, 205, 227), (204, 235, 197),
                        (222, 203, 228), (254, 217, 166), (255, 255, 204),
                        (229, 216, 189), (253, 218, 236), (242, 242, 242)],
    'brewer_set13': [(228, 26, 28), (55, 126, 184), (77, 175, 74)],
    'brewer_set14': [(228, 26, 28), (55, 126, 184), (77, 175, 74),
                     (152, 78, 163)],
    'brewer_set15': [(228, 26, 28), (55, 126, 184), (77, 175, 74),
                     (152, 78, 163), (255, 127, 0)],
    'brewer_set16': [(228, 26, 28), (55, 126, 184), (77, 175, 74),
                     (152, 78, 163), (255, 127, 0), (255, 255, 51)],
    'brewer_set17': [(228, 26, 28), (55, 126, 184), (77, 175, 74),
                     (152, 78, 163), (255, 127, 0), (255, 255, 51),
                     (166, 86, 40)],
    'brewer_set18': [(228, 26, 28), (55, 126, 184), (77, 175, 74),
                     (152, 78, 163), (255, 127, 0), (255, 255, 51),
                     (166, 86, 40), (247, 129, 191)],
    'brewer_set19': [(228, 26, 28), (55, 126, 184), (77, 175, 74),
                     (152, 78, 163), (255, 127, 0), (255, 255, 51),
                     (166, 86, 40), (247, 129, 191), (153, 153, 153)],
    'brewer_pastel23': [(179, 226, 205), (253, 205, 172), (203, 213, 232)],
    'brewer_pastel24': [(179, 226, 205), (253, 205, 172), (203, 213, 232),
                        (244, 202, 228)],
    'brewer_pastel25': [(179, 226, 205), (253, 205, 172), (203, 213, 232),
                        (244, 202, 228), (230, 245, 201)],
    'brewer_pastel26': [(179, 226, 205), (253, 205, 172), (203, 213, 232),
                        (244, 202, 228), (230, 245, 201), (255, 242, 174)],
    'brewer_pastel27': [(179, 226, 205), (253, 205, 172), (203, 213, 232),
                        (244, 202, 228), (230, 245, 201), (255, 242, 174),
                        (241, 226, 204)],
    'brewer_pastel28': [(179, 226, 205), (253, 205, 172), (203, 213, 232),
                        (244, 202, 228), (230, 245, 201), (255, 242, 174),
                        (241, 226, 204), (204, 204, 204)],
    'brewer_set23': [(102, 194, 165), (252, 141, 98), (141, 160, 203)],
    'brewer_set24': [(102, 194, 165), (252, 141, 98), (141, 160, 203),
                     (231, 138, 195)],
    'brewer_set25': [(102, 194, 165), (252, 141, 98), (141, 160, 203),
                     (231, 138, 195), (166, 216, 84)],
    'brewer_set26': [(102, 194, 165), (252, 141, 98), (141, 160, 203),
                     (231, 138, 195), (166, 216, 84), (255, 217, 47)],
    'brewer_set27': [(102, 194, 165), (252, 141, 98), (141, 160, 203),
                     (231, 138, 195), (166, 216, 84), (255, 217, 47),
                     (229, 196, 148)],
    'brewer_set28': [(102, 194, 165), (252, 141, 98), (141, 160, 203),
                     (231, 138, 195), (166, 216, 84), (255, 217, 47),
                     (229, 196, 148), (179, 179, 179)],
    'brewer_dark23': [(27, 158, 119), (217, 95, 2), (117, 112, 179)],
    'brewer_dark24': [(27, 158, 119), (217, 95, 2), (117, 112, 179),
                      (231, 41, 138)],
    'brewer_dark25': [(27, 158, 119), (217, 95, 2), (117, 112, 179),
                      (231, 41, 138), (102, 166, 30)],
    'brewer_dark26': [(27, 158, 119), (217, 95, 2), (117, 112, 179),
                      (231, 41, 138), (102, 166, 30), (230, 171, 2)],
    'brewer_dark27': [(27, 158, 119), (217, 95, 2), (117, 112, 179),
                      (231, 41, 138), (102, 166, 30), (230, 171, 2),
                      (166, 118, 29)],
    'brewer_dark28': [(27, 158, 119), (217, 95, 2), (117, 112, 179),
                      (231, 41, 138), (102, 166, 30), (230, 171, 2),
                      (166, 118, 29), (102, 102, 102)],
    'brewer_paired3': [(166, 206, 227), (31, 120, 180), (178, 223, 138)],
    'brewer_paired4': [(166, 206, 227), (31, 120, 180), (178, 223, 138),
                       (51, 160, 44)],
    'brewer_paired5': [(166, 206, 227), (31, 120, 180), (178, 223, 138),
                       (51, 160, 44), (251, 154, 153)],
    'brewer_paired6': [(166, 206, 227), (31, 120, 180), (178, 223, 138),
                       (51, 160, 44), (251, 154, 153), (227, 26, 28)],
    'brewer_paired7': [(166, 206, 227), (31, 120, 180), (178, 223, 138),
                       (51, 160, 44), (251, 154, 153), (227, 26, 28),
                       (253, 191, 111)],
    'brewer_paired8': [(166, 206, 227), (31, 120, 180), (178, 223, 138),
                       (51, 160, 44), (251, 154, 153), (227, 26, 28),
                       (253, 191, 111), (255, 127, 0)],
    'brewer_paired9': [(166, 206, 227), (31, 120, 180), (178, 223, 138),
                       (51, 160, 44), (251, 154, 153), (227, 26, 28),
                       (253, 191, 111), (255, 127, 0), (202, 178, 214)],
    'brewer_paired10': [(166, 206, 227), (31, 120, 180), (178, 223, 138),
                        (51, 160, 44), (251, 154, 153), (227, 26, 28),
                        (253, 191, 111), (255, 127, 0), (202, 178, 214),
                        (106, 61, 154)],
    'brewer_paired11': [(166, 206, 227), (31, 120, 180), (178, 223, 138),
                        (51, 160, 44), (251, 154, 153), (227, 26, 28),
                        (253, 191, 111), (255, 127, 0), (202, 178, 214),
                        (106, 61, 154), (255, 255, 153)],
    'brewer_paired12': [(166, 206, 227), (31, 120, 180), (178, 223, 138),
                        (51, 160, 44), (251, 154, 153), (227, 26, 28),
                        (253, 191, 111), (255, 127, 0), (202, 178, 214),
                        (106, 61, 154), (255, 255, 153), (177, 89, 40)],
    'brewer_accent3': [(127, 201, 127), (190, 174, 212), (253, 192, 134)],
    'brewer_accent4': [(127, 201, 127), (190, 174, 212), (253, 192, 134),
                       (255, 255, 153)],
    'brewer_accent5': [(127, 201, 127), (190, 174, 212), (253, 192, 134),
                       (255, 255, 153), (56, 108, 176)],
    'brewer_accent6': [(127, 201, 127), (190, 174, 212), (253, 192, 134),
                       (255, 255, 153), (56, 108, 176), (240, 2, 127)],
    'brewer_accent7': [(127, 201, 127), (190, 174, 212), (253, 192, 134),
                       (255, 255, 153), (56, 108, 176), (240, 2, 127),
                       (191, 91, 23)],
    'brewer_accent8': [(127, 201, 127), (190, 174, 212), (253, 192, 134),
                       (255, 255, 153), (56, 108, 176), (240, 2, 127),
                       (191, 91, 23), (102, 102, 102)],
}


if __name__ == '__main__':
    import doctest
    doctest.testmod(optionflags=doctest.NORMALIZE_WHITESPACE)
