# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['camply',
 'camply.config',
 'camply.notifications',
 'camply.providers',
 'camply.providers.recreation_dot_gov',
 'camply.providers.xanterra',
 'camply.search',
 'camply.utils']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML', 'pandas', 'python-dotenv', 'pytz', 'requests', 'tenacity']

entry_points = \
{'console_scripts': ['camply = camply.utils.camply_cli:main']}

setup_kwargs = {
    'name': 'camply',
    'version': '0.1.4',
    'description': 'camply, the campsite finder ⛺️',
    'long_description': '<div align="center">\n<a href="https://github.com/juftin/camply">\n  <img src="https://raw.githubusercontent.com/juftin/camply/main/docs/static/camply.svg" \n    width="400" height="400" alt="camply">\n</a>\n</div>\n\n`camply`, the campsite finder ⛺️, is a tool to help you book a campground online. Finding\nreservations at sold out campgrounds can be tough. That\'s where camply comes in. It searches the\nAPIs of booking services like https://recreation.gov (which indexes thousands of campgrounds across\nthe USA) to continuously check for cancellations and availabilities to pop up. Once a campsite\nbecomes available, camply sends you a notification to book your spot!\n\n___________\n___________\n\n[![PyPI](https://img.shields.io/pypi/v/camply?color=blue&label=⛺️camply)](https://github.com/juftin/camply)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/camply)](https://pypi.python.org/pypi/camply/)\n[![Docker Image Version](https://img.shields.io/docker/v/juftin/camply?color=blue&label=docker&logo=docker)](https://hub.docker.com/r/juftin/camply)\n[![Testing Status](https://github.com/juftin/camply/actions/workflows/pytest.yml/badge.svg?branch=camply)](https://github.com/juftin/camply/actions/workflows/pytest.yml)\n[![GitHub License](https://img.shields.io/github/license/juftin/camply?color=blue&label=License)](https://github.com/juftin/camply/blob/main/LICENSE)\n\n## Table of Contents\n\n- [Installation](#installation)\n    * [PyPI](#pypi)\n    * [Docker](#docker)\n    * [Building Locally](#building-locally)\n- [Command Line Usage](#command-line-usage)\n    * [`campsites`](#campsites)\n    * [`recreation-areas`](#recreation-areas)\n    * [`campgrounds`](#campgrounds)\n    * [`configure`](#configure)\n    * [Examples](#examples)\n        + [Searching for a Campsite](#searching-for-a-campsite)\n        + [Searching for a Campsite by Campground ID](#searching-for-a-campsite-by-campground-id)\n        + [Continuously Searching for A Campsite](#continuously-searching-for-a-campsite)\n        + [Continue Looking After The First Match Is Found](#continue-looking-after-the-first-match-is-found)\n        + [Look for Weekend Campsite Availabilities](#look-for-weekend-campsite-availabilities)\n        + [Look for a Campsite Inside of Yellowstone](#look-for-a-campsite-inside-of-yellowstone)\n        + [Look for a Campsite Across Multiple Recreation areas](#look-for-a-campsite-across-multiple-recreation-areas)\n        + [Using a YML Configuration file to search for campsites](#using-a-yml-configuration-file-to-search-for-campsites)\n        + [Search for Recreation Areas by Query String](#search-for-recreation-areas-by-query-string)\n        + [Look for Specific Campgrounds Within a Recreation Area](#look-for-specific-campgrounds-within-a-recreation-area)\n        + [Look for Specific Campgrounds by Query String](#look-for-specific-campgrounds-by-query-string)\n- [Finding Recreation Areas IDs and Campground IDs To Search Without Using the Command Line](#finding-recreation-areas-ids-and-campground-ids-to-search-without-using-the-command-line)\n- [Object-Oriented Usage (Python)](#object-oriented-usage-python)\n    * [Search for a Recreation.gov Campsite](#search-for-a-recreationgov-campsite)\n    * [Continuously Search for Recreation.gov Campsites](#continuously-search-for-recreationgov-campsites)\n- [Running in Docker](#running-in-docker)\n- [Dependencies](#dependencies)\n\n## Installation\n\n### <a name="pypi"></a>[PyPI](https://pypi.python.org/pypi/camply/)\n\n```text\npip install camply\n```\n\n### <a name="docker"></a>[Docker](https://hub.docker.com/r/juftin/camply)\n\n```text\ndocker pull juftin/camply\n```\n\n**_see [Running in Docker](#running-in-docker) below._\n\n### Building Locally\n\n```text\ngit clone https://github.com/juftin/camply.git\ncd camply\npython setup.py install\n```\n\n## Command Line Usage\n\nWhen installed, `camply`\'s command line utility can be invoked with the command, `camply`. The CLI\ntool accepts one of four sub-arguments: `campsites`, `recreation-areas`, `campgrounds`,\nand `configure`.\n\n```text\n❯ camply --help\n\n2021-05-17 19:11:59,858 [  CAMPLY]: camply, the campsite finder ⛺️\nusage: camply [-h] [--version]\n              {campsites,recreation-areas,campgrounds,configure} ...\n\nWelcome to camply, the campsite finder. Finding reservations at sold out\ncampgrounds can be tough. That\'s where camply comes in. It searches the APIs\nof booking services like https://recreation.gov (which indexes thousands of\ncampgrounds across the USA) to continuously check for cancellations and\navailabilities to pop up. Once a campsite becomes available, camply sends you\na notification to book your spot!\n\npositional arguments:\n  {campsites,recreation-areas,campgrounds,configure}\n    campsites           Find available Campsites using search criteria\n    recreation-areas    Search for Recreation Areas and list them\n    campgrounds         Search for Campgrounds (inside of Recreation Areas)\n                        and list them\n    configure           Set up camply configuration file with an interactive\n                        console\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --version             show program\'s version number and exit\n\nvisit the camply documentation at https://github.com/juftin/camply\n\n2021-05-17 19:11:59,863 [  CAMPLY]: Exiting camply 👋\n```\n\n### `campsites`\n\nSearch for a campsite within camply. Campsites are returned based on the search criteria provided.\nCampsites contain properties like booking date, site type (tent, RV, cabin, etc), capacity, price,\nand a link to make the booking. Required parameters include `--start-date`, `--end-date`,\n`--rec-area` / `--campground`. Constant searching functionality can be enabled with\n`--continuous` and notifications via Email, Pushover, and Pushbullet can be enabled using\n`--notifications`.\n\n#### Arguments:\n\n* `--rec-area`: `RECREATION_AREA_ID`\n    + Add Recreation Areas (comprised of campgrounds) by ID. [**_example_](#searching-for-a-campsite)\n* `--campground`: `CAMPGROUND_ID`\n    + Add individual Campgrounds by ID. [**_example_](#searching-for-a-campsite-by-campground-id)\n* `--start-date`: `START_DATE`\n    + `YYYY-MM-DD`: Start of Search window. You will be arriving this day. [**_example_](#searching-for-a-campsite)\n* `--end-date`: `END_DATE`\n    + `YYYY-MM-DD`: End of Search window. You will be leaving the following day. [**_example_](#searching-for-a-campsite)\n* `--weekends`\n    + Only search for weekend bookings (Fri/Sat nights). [**_example_](#look-for-weekend-campsite-availabilities)\n* `--provider`: `PROVIDER`\n    + Camping Search Provider. Options available are \'Yellowstone\' and \'RecreationDotGov\'. Defaults\n      to \'RecreationDotGov\', not case-sensitive. [**_example_](#look-for-a-campsite-inside-of-yellowstone)\n* `--continuous`\n    + Continuously check for a campsite to become available, and quit once at least one campsite is\n      found. [**_example_](#continuously-searching-for-a-campsite)\n* `--polling-interval`: `POLLING_INTERVAL`\n    + If `--continuous` is activated, how often to wait in between checks (in minutes). Defaults to\n      10, cannot be less than 5. [**_example_](#look-for-weekend-campsite-availabilities)\n* `--notifications`: `NOTIFICATIONS`\n    + If `--continuous` is activated, types of notifications to receive. Options available are\n      `email`, `pushover`, `pushbullet`, or `silent`. Defaults to `silent` - which just logs\n      messages to console. [**_example_](#continuously-searching-for-a-campsite)\n* `--notify-first-try`\n    + If `--continuous` is activated, whether to send all non-silent notifications if more than 5\n      matching campsites are found on the first try. Defaults to false which only sends the first\n      5. [**_example_](#continuously-searching-for-a-campsite)\n* `--search-forever`\n    + If `--continuous` is activated, this method continues to search after the first availability\n      has been found. The one caveat is that it will never notify about the same identical campsite\n      for the same booking date. [**_example_](#continue-looking-after-the-first-match-is-found)\n* `--yml-config`\n    + Rather than provide arguments to the command line utility, instead pass a file path to a YAML\n      configuration file. See the documentation for more information on how to structure your\n      configuration file. [**_example_](#using-a-yml-configuration-file-to-search-for-campsites)\n\n```text\ncamply campsites \\\n    --rec-area 2725 \\\n    --start-date 2021-07-10 \\\n    --end-date 2021-07-17\n```\n\n### `recreation-areas`\n\nSearch for Recreation Areas and their IDs. Recreation Areas are places like National Parks and\nNational Forests that can contain one or many campgrounds.\n\n#### Arguments:\n\n* `--search` `SEARCH`\n    + Search for Campgrounds or Recreation Areas by search string.\n* `--state` `STATE`\n    + Filter by US state code.\n\n```text\ncamply recreation-areas --search "Yosemite National Park"\n```\n\n**_see the [examples](#search-for-recreation-areas-by-query-string) for more information_\n\n### `campgrounds`\n\nSearch for Campgrounds and their IDs. Campgrounds are facilities inside of Recreation Areas that\ncontain campsites. Most \'campgrounds\' are areas made up of multiple campsites, others are facilities\nlike fire towers or cabins that might only contain a single \'campsite\' to book.\n\n#### Arguments:\n\n* `--search` `SEARCH`\n    + Search for Campgrounds or Recreation Areas by search string.\n* `--state` `STATE`\n    + Filter by US state code.\n* `--rec-area`: `RECREATION_AREA_ID`\n    + Add Recreation Areas (comprised of campgrounds) by ID.\n* `--campground`: `CAMPGROUND_ID`\n    + Add individual Campgrounds by ID.\n\n```text\ncamply campgrounds --search "Fire Tower Lookout" --state CA\n```\n\n**_see the [examples](#look-for-specific-campgrounds-by-query-string) for more information_\n\n### `configure`\n\nSet up `camply` configuration file with an interactive console\n\nIn order to send notifications through `camply` you must set up some authorization values. Whether\nyou need to set up [Pushover notifications](https://pushover.net)\n, [PushBullet](https://www.pushbullet.com/#settings/account) or Email messages, everything can be\ndone through the `configure` command. The end result is a file called\n[`.camply`](docs/examples/example.camply) in your home folder. See\nthe [Running in Docker](#running-in-docker) section to see how you can use environment variables\ninstead of a config file.\n\n```text\ncamply configure\n```\n\n### Examples\n\nRead through the examples below to get a better understanding of `camply`, its features, and the\nfunctionality of the different arguments provided to the CLI.\n\n#### Searching for a Campsite\n\nThe below search looks for campsites inside of Recreation Area ID #2725 (Glacier National Park)\nbetween 2021-07-10 and 2021-07-17. The search will be performed once and any results will be logged\nto the console. camply searches for campsites inside of search windows in increments of one night.\n`--start-date` and `--end-date` define the bounds of the search window, you will be leaving the day\nafter `--end-date`.\n\n```text\ncamply campsites \\\n    --rec-area 2725 \\\n    --start-date 2021-07-10 \\\n    --end-date 2021-07-17\n```\n\n#### Searching for a Campsite by Campground ID\n\nThe below search looks for across three campgrounds (all inside Glacier National Park)\nbetween 2021-07-10 and 2021-07-17. Multiple Campgrounds (and Recreation Areas too) can be found by\nsupplying the arguments more than once.\n\n```text\ncamply campsites \\\n    --campground 232493 \\\n    --campground 251869 \\\n    --campground 232492 \\\n    --start-date 2021-07-10 \\\n    --end-date 2021-07-17\n```\n\n#### Continuously Searching for A Campsite\n\nSometimes you want to look for campgrounds until an eventual match is found. The below snippet will\nsearch for matching campsites until it finds a match. It also sends a notification via `pushover`\nonce matches are found. Alternate notification methods are `email`, `pushbullet`, and `silent` (\ndefault).\n\n__Important Note__: When `camply` is told to run `--continuous` with non-silent notifications set up\nand it finds more than 5 matching campsites on the first try, it will only send notifications for\nthe first 5 campsites. This is to prevent thousands of campsites flooding your notifications. It\'s\nalways encouraged to perform an initial online search before setting up a `camply` search. To bypass\nthis behavior and send all notifications, pass the `--notify-first-try` argument.\n\n```text\ncamply campsites \\\n    --rec-area 2725 \\\n    --start-date 2021-07-12 \\\n    --end-date 2021-07-12 \\\n    --continuous \\\n    --notifications pushover \\\n    --notify-first-try\n```\n\n#### Continue Looking After The First Match Is Found\n\nSometimes you want to search for all possible matches up until your arrival date. No problem. Add\nthe `--search-forever` and `camply` won\'t stop sending notifications after the first match is found.\nOne important note, `camply` will save and store all previous notifications when `--search-forever`\nis enabled, so it won\'t notify you about the exact same campsite availability twice. This can be\nproblematic when certain campsites become available more than once.\n\n```text\ncamply campsites \\\n    --rec-area 2725 \\\n    --start-date 2021-07-01 \\\n    --end-date 2021-07-31 \\\n    --continuous \\\n    --notifications pushover \\\n    --search-forever\n```\n\n#### Look for Weekend Campsite Availabilities\n\nThis below search looks across larger periods of time, but only if a campground is available to book\non a Friday or Saturday night (`--weekends`). It also uses the `--polling-interval` argument which\nchecks every 5 minutes instead of the default 10 minutes.\n\n```text\ncamply campsites \\\n    --rec-area 2991 \\\n    --start-date 2021-05-01 \\\n    --end-date 2021-07-31 \\\n    --weekends \\\n    --continuous \\\n    --notifications email \\\n    --polling-interval 5\n```\n\n#### Look for a Campsite Inside of Yellowstone\n\nYellowstone doesn\'t use https://recreation.gov to manage its campgrounds, instead it uses its own\nproprietary system. In order to search the Yellowstone API for campsites, make sure to pass\nthe `--provider "yellowstone"` argument. This flag disables `--rec-area` and `--campground`\narguments.\n\n```text\ncamply campsites \\\n    --provider yellowstone \\\n    --start-date 2021-07-09 \\\n    --end-date 2021-07-16 \\\n    --continuous\n```\n\n#### Look for a Campsite Across Multiple Recreation areas\n\nYou don\'t have to confine your search to a single Recreation or Campground ID. Adding multiple\narguments to the command line will search across multiple IDs. Keep in mind that any `--campground`\narguments will overwrite all `--rec-area` arguments.\n\n```text\ncamply campsites \\\n    --rec-area 2991 \\\n    --rec-area 1074 \\\n    --start-date 2021-07-09 \\\n    --end-date 2021-07-16\n```\n\n#### Using a YML Configuration file to search for campsites\n\nSometimes, using a YAML configuration file is easier to manage all of your search options. See the\nbelow [YML example file](docs/examples/example_search.yml) and corresponding camply command:\n\n```yaml\nprovider:         RecreationDotGov # RecreationDotGov IF NOT PROVIDED\nrecreation_area: # (LIST OR SINGLE ENTRY)\n    - 2991 # Yosemite National Park, CA (All Campgrounds)\n    - 1074 # Sierra National Forest, CA (All Campgrounds)\ncampgrounds:      null # ENTIRE FIELD CAN BE OMITTED IF NOT USED. # (LIST OR SINGLE ENTRY)\nstart_date:       2021-09-12 # YYYY-MM-DD\nend_date:         2021-09-12 # YYYY-MM-DD\nweekends:         False # FALSE BY DEFAULT\ncontinuous:       True # DEFAULTS TO TRUE\npolling_interval: 5 # DEFAULTS TO 10 , CAN\'T BE LESS THAN 5\nnotifications:    email # (silent, email, pushover, pushbullet), DEFAULTS TO `silent`\nsearch_forever:   True # FALSE BY DEFAULT\nnotify_first_try: False # FALSE BY DEFAULT\n```\n\n```text\ncamply campsites --yml-config example_search.yml \n```\n\n#### Search for Recreation Areas by Query String\n\nJust need to find what your local Recreation Area ID number is? This simple command allows you to\nsearch and list recreation areas. It accepts `--search` and `--state` arguments.\n\n```text\ncamply recreation-areas --search "Yosemite National Park"\n```\n\n#### Look for Specific Campgrounds Within a Recreation Area\n\nNeed to get even more specific and search for a particular campground? This search lists campgrounds\nattached to a recreation area id `--rec-area`. It also accepts `--search` and `--state`\narguments.\n\n```text\ncamply campgrounds --rec-area 2991\n```\n\n#### Look for Specific Campgrounds by Query String\n\nThe below search looks for Fire Lookout Towers to stay in inside of California.\n\n```text\ncamply campgrounds --search "Fire Tower Lookout" --state CA\n```\n\n## Finding Recreation Areas IDs and Campground IDs To Search Without Using the Command Line\n\nYou can uncover campground and recreation area IDs just by using the https://recreation.gov search\nfunctionality. Use the below example for a campground within Glacier National Park.\n\nFirst, perform your search on https://recreation.gov.\n\n<div>\n<img src="https://raw.githubusercontent.com/juftin/camply/main/docs/static/recreation_dot_gov.png" \n    width="500" alt="recreation_dot_gov search">\n</div>\n\nThe above search will take you to a URL like this:\nhttps://www.recreation.gov/search?q=Glacier%20National%20Park&entity_id=2725&entity_type=recarea.\nTaking a closer look at the URL components you can see that Glacier National Park has the Recreation\nArea ID #2725.\n\nSearching deeper into campgrounds inside of Glacier National Park you might find Fish Creek\nCampground at a URL like https://www.recreation.gov/camping/campgrounds/232493. Here, we can see\nthat this campground has a Campground ID of #232493.\n\n## Object-Oriented Usage (Python)\n\n### Search for a Recreation.gov Campsite\n\n```python\nfrom datetime import datetime\nimport logging\nfrom typing import List\n\nfrom camply.containers import AvailableCampsite, SearchWindow\nfrom camply.search import SearchRecreationDotGov\n\nlogging.basicConfig(format="%(asctime)s [%(levelname)8s]: %(message)s",\n                    level=logging.INFO)\n\nmonth_of_june = SearchWindow(start_date=datetime(year=2021, month=6, day=1),\n                             end_date=datetime(year=2021, month=6, day=30))\ncamping_finder = SearchRecreationDotGov(search_window=month_of_june,\n                                        recreation_area=2725,  # Glacier Ntl Park\n                                        weekends_only=False)\nmatches: List[AvailableCampsite] = camping_finder.get_matching_campsites(log=True, verbose=True,\n                                                                         continuous=False)\n```\n\nThe above script returns a list of any matching `AvailableCampsite` namedtuple objects:\n\n```python\n[\n    AvailableCampsite(campsite_id="5391",\n                      booking_date=datetime.datetime(2021, 6, 13, 0, 0),\n                      campsite_site_name="B37",\n                      campsite_loop_name="Loop B",\n                      campsite_type="STANDARD NONELECTRIC",\n                      campsite_occupancy=(0, 8),\n                      campsite_use_type="Overnight",\n                      availability_status="Available",\n                      recreation_area="Glacier National Park, MT",\n                      recreation_area_id="2725",\n                      facility_name="Fish Creek Campground",\n                      facility_id="232493",\n                      booking_url="https://www.recreation.gov/camping/campsites/5391")\n]\n```\n\n### Continuously Search for Recreation.gov Campsites\n\nYou\'ll notice that the `get_matching_campsites` function takes accepts parameter values very similar\nto the commandline arguments.\n\n```python\nfrom datetime import datetime\nimport logging\n\nfrom camply.containers import SearchWindow\nfrom camply.search import SearchRecreationDotGov\n\nlogging.basicConfig(format="%(asctime)s [%(levelname)8s]: %(message)s",\n                    level=logging.INFO)\n\nmonth_of_june = SearchWindow(start_date=datetime(year=2021, month=6, day=1),\n                             end_date=datetime(year=2021, month=6, day=30))\ncamping_finder = SearchRecreationDotGov(search_window=month_of_june,\n                                        recreation_area=[2991, 1074],  # Multiple Rec Areas\n                                        weekends_only=False)\ncamping_finder.get_matching_campsites(log=True, verbose=True,\n                                      continuous=True,\n                                      polling_interval=5,\n                                      notification_provider="pushover",\n                                      search_forever=True,\n                                      notify_first_try=False)\n```\n\n## Running in Docker\n\nHere\'s an example of a detached container searching in the background (notice the `-d` flag, the\ncontainer will run detached).\n\n```text\ndocker run -d \\\n  --name camply-detached-example \\\n  --env PUSHOVER_PUSH_TOKEN=${PUSHOVER_PUSH_TOKEN} \\\n  --env PUSHOVER_PUSH_USER=${PUSHOVER_PUSH_USER} \\\n  --env TZ="America/Denver" \\\n  juftin/camply \\\n  camply campsites \\\n      --rec-area 2991 \\\n      --start-date 2021-08-01 \\\n      --end-date 2021-08-31 \\\n      --continuous \\\n      --notifications pushover\n```\n\nThe docker image accepts the following environment variables:\n\n- Pushover Notifications\n    * `PUSHOVER_PUSH_TOKEN`\n    * `PUSHOVER_PUSH_USER`\n- Email Notifications\n    * `EMAIL_TO_ADDRESS`\n    * `EMAIL_USERNAME`\n    * `EMAIL_PASSWORD`\n    * `EMAIL_FROM_ADDRESS` (defaults to "camply@juftin.com")\n    * `EMAIL_SUBJECT_LINE` (defaults to "camply Notification")\n    * `EMAIL_SMTP_SERVER` (defaults to "smtp.gmail.com")\n    * `EMAIL_SMTP_PORT` (defaults to 465)\n- Optional Environment Variables\n    * `LOG_LEVEL` (sets logging level, defaults to "INFO")\n    * `RIDB_API_KEY` (Personal API Key\n      for [Recreation.gov API](https://ridb.recreation.gov/profile))\n    * `TZ` ([TZ Database Name](https://en.wikipedia.org/wiki/List_of_tz_database_time_zones) for\n      logging, defaults to UTC)\n\nAlternatively, if you have already run `camply configure` locally, you can share\nyour [`.camply`](docs/examples/example.camply) file inside the docker container.\n\n```text\ndocker run \\\n  --name camply-file-share-example \\\n  --env TZ="America/Denver" \\\n  --volume ${HOME}/.camply:/home/camply/.camply \\\n  juftin/camply \\\n  camply campsites \\\n      --provider yellowstone \\\n      --start-date 2021-07-22 \\\n      --end-date 2021-07-26 \\\n      --continuous \\\n      --notifications email\n```\n\nTo manage multiple searches (with different notification preferences) I like to use YML\nconfiguration files:\n\n```text\ndocker run -d \\\n  --name camply-email-example \\\n  --env TZ="America/Denver" \\\n  --env EMAIL_TO_ADDRESS=${EMAIL_TO_ADDRESS} \\\n  --env EMAIL_USERNAME=${EMAIL_USERNAME} \\\n  --env EMAIL_PASSWORD=${EMAIL_PASSWORD} \\\n  --volume example_search.yml:/home/camply/example_search.yml \\\n  juftin/camply:latest \\\n  camply campsites \\\n      --yml-config /home/camply/example_search.yml\n```\n\n## Dependencies\n\n`camply` is compatible with any Python version >= `3.6`. Currently, there are four required\ndependencies:\n\n- [requests](https://docs.python-requests.org/en/master/)\n    - The `requests` package is used to fetch data from the APIs of Camping Booking Providers.\n- [pandas](https://pandas.pydata.org/)\n    - The `pandas` package is to group and aggregate across large data sets of campsites,\n      campgrounds, and recreation areas.\n- [tenacity](https://tenacity.readthedocs.io/en/latest/)\n    - The `tenacity` package is used for retrying data searches on the underlying campsite APIs.\n      This retrying methodology handles exceptions allowing for API downtime and facilitating\n      exponential backoff.\n- [python-dotenv](https://github.com/theskumar/python-dotenv)\n    - The `python-dotenv` package reads key-value pairs from a `.env` file and can set them as\n      environment variables - this helps with the `.camply` configuration file.\n\n___________\n___________\n\n<br/>\n\n[<p align="center" ><img src="https://raw.githubusercontent.com/juftin/juftin/main/static/juftin.png" width="120" height="120"  alt="juftin logo"> </p>](https://github.com/juftin)\n\n',
    'author': 'Justin Flannery',
    'author_email': 'juftin@juftin.com',
    'maintainer': 'Justin Flannery',
    'maintainer_email': 'juftin@juftin.com',
    'url': 'https://github.com/juftin/camply',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6.0,<3.10.0',
}


setup(**setup_kwargs)
