"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_apigateway_1 = require("@aws-cdk/aws-apigateway");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * Rest APIs have request validation enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_apigateway_1.CfnRestApi) {
        const apiLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of core_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_apigateway_1.CfnRequestValidator) {
                if (isMatchingRequestValidator(child, apiLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Request Validator is associated with the given Rest API
 * @param node the CfnRequestValidator to check
 * @param apiLogicalId the Cfn Logical ID of the REST API
 * returns whether the CfnRequestValidator is associated with the given Rest API
 */
function isMatchingRequestValidator(node, apiLogicalId) {
    const resourceLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.restApiId);
    if (resourceLogicalId === apiLogicalId) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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