"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * CloudFront distributions use an origin access identity for S3 origins
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cloudfront_1.CfnDistribution) {
        const distributionConfig = core_1.Stack.of(node).resolve(node.distributionConfig);
        if (distributionConfig.origins != undefined) {
            const origins = core_1.Stack.of(node).resolve(distributionConfig.origins);
            for (const origin of origins) {
                const resolvedOrigin = core_1.Stack.of(node).resolve(origin);
                const resolvedDomainName = core_1.Stack.of(node).resolve(resolvedOrigin.domainName);
                const s3Regex = /^.+\.s3(?:-website)?(?:\..+)?(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))$/;
                if (s3Regex.test(resolvedDomainName)) {
                    if (resolvedOrigin.s3OriginConfig == undefined) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    const resolvedConfig = core_1.Stack.of(node).resolve(resolvedOrigin.s3OriginConfig);
                    if (resolvedConfig.originAccessIdentity == undefined ||
                        resolvedConfig.originAccessIdentity.replace(/\s/g, '').length == 0) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_cloudfront_1.CfnStreamingDistribution) {
        const distributionConfig = core_1.Stack.of(node).resolve(node.streamingDistributionConfig);
        const resolvedOrigin = core_1.Stack.of(node).resolve(distributionConfig.s3Origin);
        if (resolvedOrigin.originAccessIdentity.replace(/\s/g, '').length == 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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