"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * Cognito user pools have password policies that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cognito_1.CfnUserPool) {
        const policies = core_1.Stack.of(node).resolve(node.policies);
        if (policies == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const passwordPolicy = core_1.Stack.of(node).resolve(policies.passwordPolicy);
        if (passwordPolicy == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const minimumLength = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.minimumLength);
        if (minimumLength == undefined || minimumLength < 8) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requireUppercase = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.requireUppercase);
        if (requireUppercase !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requireNumbers = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.requireNumbers);
        if (requireNumbers !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const requireSymbols = nag_rules_1.NagRules.resolveIfPrimitive(node, passwordPolicy.requireSymbols);
        if (requireSymbols !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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