"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
const EBS_OPTIMIZED_SUPPORTED = [
    'c1.xlarge',
    'c3.xlarge',
    'c3.2xlarge',
    'c3.4xlarge',
    'g2.2xlarge',
    'i2.xlarge',
    'i2.2xlarge',
    'i2.4xlarge',
    'm1.large',
    'm1.xlarge',
    'm2.2xlarge',
    'm2.4xlarge',
    'm3.xlarge',
    'm3.2xlarge',
    'r3.xlarge',
    'r3.2xlarge',
    'r3.4xlarge',
];
const DEFAULT_TYPE = 'm1.small';
/**
 * EC2 instance types that support EBS optimization and are not EBS optimized by default have EBS optimization enabled
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html#previous
 *  @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnInstance) {
        const instanceType = node.instanceType
            ? nag_rules_1.NagRules.resolveIfPrimitive(node, node.instanceType)
            : DEFAULT_TYPE;
        const ebsOptimized = core_1.Stack.of(node).resolve(node.ebsOptimized);
        if (EBS_OPTIMIZED_SUPPORTED.includes(instanceType) &&
            ebsOptimized !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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