"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * Security Groups do not allow for unrestricted SSH traffic
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = core_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = core_1.Stack.of(node).resolve(rule);
                const ipProtocol = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.ipProtocol);
                const cidrIp = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.cidrIp);
                const cidrIpv6 = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.cidrIpv6);
                const fromPort = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.fromPort);
                const toPort = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.toPort);
                if ((cidrIp != undefined && cidrIp.includes('/0')) ||
                    (cidrIpv6 != undefined && cidrIpv6.includes('/0'))) {
                    if (fromPort != undefined && toPort != undefined) {
                        if ((fromPort <= 22 && toPort >= 22) ||
                            fromPort == -1 ||
                            toPort == -1 ||
                            ipProtocol == '-1') {
                            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                        }
                    }
                    else {
                        if (fromPort == 22 || ipProtocol == '-1') {
                            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                        }
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        const ipProtocol = nag_rules_1.NagRules.resolveIfPrimitive(node, node.ipProtocol);
        const cidrIp = nag_rules_1.NagRules.resolveIfPrimitive(node, node.cidrIp);
        const cidrIpv6 = nag_rules_1.NagRules.resolveIfPrimitive(node, node.cidrIpv6);
        const fromPort = nag_rules_1.NagRules.resolveIfPrimitive(node, node.fromPort);
        const toPort = nag_rules_1.NagRules.resolveIfPrimitive(node, node.toPort);
        if ((cidrIp != undefined && cidrIp.includes('/0')) ||
            (cidrIpv6 != undefined && cidrIpv6.includes('/0'))) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= 22 && toPort >= 22) ||
                    fromPort == -1 ||
                    toPort == -1 ||
                    ipProtocol == '-1') {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
            else {
                if (fromPort == 22 || ipProtocol == '-1') {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiRUMyUmVzdHJpY3RlZFNTSC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9ydWxlcy9lYzIvRUMyUmVzdHJpY3RlZFNTSC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBOzs7RUFHRTtBQUNGLCtCQUE2QjtBQUM3Qiw4Q0FBNkU7QUFDN0Usd0NBQW1EO0FBQ25ELCtDQUE4RDtBQUU5RDs7O0dBR0c7QUFDSCxrQkFBZSxNQUFNLENBQUMsY0FBYyxDQUNsQyxDQUFDLElBQWlCLEVBQXFCLEVBQUU7SUFDdkMsSUFBSSxJQUFJLFlBQVksMEJBQWdCLEVBQUU7UUFDcEMsTUFBTSxZQUFZLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDdkUsSUFBSSxZQUFZLElBQUksU0FBUyxFQUFFO1lBQzdCLGdGQUFnRjtZQUNoRixLQUFLLE1BQU0sSUFBSSxJQUFJLFlBQVksRUFBRTtnQkFDL0IsTUFBTSxZQUFZLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ2xELE1BQU0sVUFBVSxHQUFHLG9CQUFRLENBQUMsa0JBQWtCLENBQzVDLElBQUksRUFDSixZQUFZLENBQUMsVUFBVSxDQUN4QixDQUFDO2dCQUNGLE1BQU0sTUFBTSxHQUFHLG9CQUFRLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQztnQkFDdEUsTUFBTSxRQUFRLEdBQUcsb0JBQVEsQ0FBQyxrQkFBa0IsQ0FDMUMsSUFBSSxFQUNKLFlBQVksQ0FBQyxRQUFRLENBQ3RCLENBQUM7Z0JBQ0YsTUFBTSxRQUFRLEdBQUcsb0JBQVEsQ0FBQyxrQkFBa0IsQ0FDMUMsSUFBSSxFQUNKLFlBQVksQ0FBQyxRQUFRLENBQ3RCLENBQUM7Z0JBQ0YsTUFBTSxNQUFNLEdBQUcsb0JBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUN0RSxJQUNFLENBQUMsTUFBTSxJQUFJLFNBQVMsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO29CQUM5QyxDQUFDLFFBQVEsSUFBSSxTQUFTLElBQUksUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUNsRDtvQkFDQSxJQUFJLFFBQVEsSUFBSSxTQUFTLElBQUksTUFBTSxJQUFJLFNBQVMsRUFBRTt3QkFDaEQsSUFDRSxDQUFDLFFBQVEsSUFBSSxFQUFFLElBQUksTUFBTSxJQUFJLEVBQUUsQ0FBQzs0QkFDaEMsUUFBUSxJQUFJLENBQUMsQ0FBQzs0QkFDZCxNQUFNLElBQUksQ0FBQyxDQUFDOzRCQUNaLFVBQVUsSUFBSSxJQUFJLEVBQ2xCOzRCQUNBLE9BQU8sNkJBQWlCLENBQUMsYUFBYSxDQUFDO3lCQUN4QztxQkFDRjt5QkFBTTt3QkFDTCxJQUFJLFFBQVEsSUFBSSxFQUFFLElBQUksVUFBVSxJQUFJLElBQUksRUFBRTs0QkFDeEMsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7eUJBQ3hDO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtRQUNELE9BQU8sNkJBQWlCLENBQUMsU0FBUyxDQUFDO0tBQ3BDO1NBQU0sSUFBSSxJQUFJLFlBQVksaUNBQXVCLEVBQUU7UUFDbEQsTUFBTSxVQUFVLEdBQUcsb0JBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ3RFLE1BQU0sTUFBTSxHQUFHLG9CQUFRLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM5RCxNQUFNLFFBQVEsR0FBRyxvQkFBUSxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDbEUsTUFBTSxRQUFRLEdBQUcsb0JBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ2xFLE1BQU0sTUFBTSxHQUFHLG9CQUFRLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM5RCxJQUNFLENBQUMsTUFBTSxJQUFJLFNBQVMsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzlDLENBQUMsUUFBUSxJQUFJLFNBQVMsSUFBSSxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLEVBQ2xEO1lBQ0EsNEJBQTRCO1lBQzVCLElBQUksUUFBUSxJQUFJLFNBQVMsSUFBSSxNQUFNLElBQUksU0FBUyxFQUFFO2dCQUNoRCxJQUNFLENBQUMsUUFBUSxJQUFJLEVBQUUsSUFBSSxNQUFNLElBQUksRUFBRSxDQUFDO29CQUNoQyxRQUFRLElBQUksQ0FBQyxDQUFDO29CQUNkLE1BQU0sSUFBSSxDQUFDLENBQUM7b0JBQ1osVUFBVSxJQUFJLElBQUksRUFDbEI7b0JBQ0EsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7aUJBQ3hDO2FBQ0Y7aUJBQU07Z0JBQ0wsSUFBSSxRQUFRLElBQUksRUFBRSxJQUFJLFVBQVUsSUFBSSxJQUFJLEVBQUU7b0JBQ3hDLE9BQU8sNkJBQWlCLENBQUMsYUFBYSxDQUFDO2lCQUN4QzthQUNGO1NBQ0Y7UUFDRCxPQUFPLDZCQUFpQixDQUFDLFNBQVMsQ0FBQztLQUNwQztTQUFNO1FBQ0wsT0FBTyw2QkFBaUIsQ0FBQyxjQUFjLENBQUM7S0FDekM7QUFDSCxDQUFDLEVBQ0QsTUFBTSxFQUNOLEVBQUUsS0FBSyxFQUFFLFlBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FDbEMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuaW1wb3J0IHsgcGFyc2UgfSBmcm9tICdwYXRoJztcbmltcG9ydCB7IENmblNlY3VyaXR5R3JvdXBJbmdyZXNzLCBDZm5TZWN1cml0eUdyb3VwIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgeyBDZm5SZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IE5hZ1J1bGVDb21wbGlhbmNlLCBOYWdSdWxlcyB9IGZyb20gJy4uLy4uL25hZy1ydWxlcyc7XG5cbi8qKlxuICogU2VjdXJpdHkgR3JvdXBzIGRvIG5vdCBhbGxvdyBmb3IgdW5yZXN0cmljdGVkIFNTSCB0cmFmZmljXG4gKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAqL1xuZXhwb3J0IGRlZmF1bHQgT2JqZWN0LmRlZmluZVByb3BlcnR5KFxuICAobm9kZTogQ2ZuUmVzb3VyY2UpOiBOYWdSdWxlQ29tcGxpYW5jZSA9PiB7XG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5TZWN1cml0eUdyb3VwKSB7XG4gICAgICBjb25zdCBpbmdyZXNzUnVsZXMgPSBTdGFjay5vZihub2RlKS5yZXNvbHZlKG5vZGUuc2VjdXJpdHlHcm91cEluZ3Jlc3MpO1xuICAgICAgaWYgKGluZ3Jlc3NSdWxlcyAhPSB1bmRlZmluZWQpIHtcbiAgICAgICAgLy9Gb3IgZWFjaCBpbmdyZXNzIHJ1bGUsIGVuc3VyZSB0aGF0IGl0IGRvZXMgbm90IGFsbG93IHVucmVzdHJpY3RlZCBTU0ggdHJhZmZpYy5cbiAgICAgICAgZm9yIChjb25zdCBydWxlIG9mIGluZ3Jlc3NSdWxlcykge1xuICAgICAgICAgIGNvbnN0IHJlc29sdmVkUnVsZSA9IFN0YWNrLm9mKG5vZGUpLnJlc29sdmUocnVsZSk7XG4gICAgICAgICAgY29uc3QgaXBQcm90b2NvbCA9IE5hZ1J1bGVzLnJlc29sdmVJZlByaW1pdGl2ZShcbiAgICAgICAgICAgIG5vZGUsXG4gICAgICAgICAgICByZXNvbHZlZFJ1bGUuaXBQcm90b2NvbFxuICAgICAgICAgICk7XG4gICAgICAgICAgY29uc3QgY2lkcklwID0gTmFnUnVsZXMucmVzb2x2ZUlmUHJpbWl0aXZlKG5vZGUsIHJlc29sdmVkUnVsZS5jaWRySXApO1xuICAgICAgICAgIGNvbnN0IGNpZHJJcHY2ID0gTmFnUnVsZXMucmVzb2x2ZUlmUHJpbWl0aXZlKFxuICAgICAgICAgICAgbm9kZSxcbiAgICAgICAgICAgIHJlc29sdmVkUnVsZS5jaWRySXB2NlxuICAgICAgICAgICk7XG4gICAgICAgICAgY29uc3QgZnJvbVBvcnQgPSBOYWdSdWxlcy5yZXNvbHZlSWZQcmltaXRpdmUoXG4gICAgICAgICAgICBub2RlLFxuICAgICAgICAgICAgcmVzb2x2ZWRSdWxlLmZyb21Qb3J0XG4gICAgICAgICAgKTtcbiAgICAgICAgICBjb25zdCB0b1BvcnQgPSBOYWdSdWxlcy5yZXNvbHZlSWZQcmltaXRpdmUobm9kZSwgcmVzb2x2ZWRSdWxlLnRvUG9ydCk7XG4gICAgICAgICAgaWYgKFxuICAgICAgICAgICAgKGNpZHJJcCAhPSB1bmRlZmluZWQgJiYgY2lkcklwLmluY2x1ZGVzKCcvMCcpKSB8fFxuICAgICAgICAgICAgKGNpZHJJcHY2ICE9IHVuZGVmaW5lZCAmJiBjaWRySXB2Ni5pbmNsdWRlcygnLzAnKSlcbiAgICAgICAgICApIHtcbiAgICAgICAgICAgIGlmIChmcm9tUG9ydCAhPSB1bmRlZmluZWQgJiYgdG9Qb3J0ICE9IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgICBpZiAoXG4gICAgICAgICAgICAgICAgKGZyb21Qb3J0IDw9IDIyICYmIHRvUG9ydCA+PSAyMikgfHxcbiAgICAgICAgICAgICAgICBmcm9tUG9ydCA9PSAtMSB8fFxuICAgICAgICAgICAgICAgIHRvUG9ydCA9PSAtMSB8fFxuICAgICAgICAgICAgICAgIGlwUHJvdG9jb2wgPT0gJy0xJ1xuICAgICAgICAgICAgICApIHtcbiAgICAgICAgICAgICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVDtcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgaWYgKGZyb21Qb3J0ID09IDIyIHx8IGlwUHJvdG9jb2wgPT0gJy0xJykge1xuICAgICAgICAgICAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UO1xuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuQ09NUExJQU5UO1xuICAgIH0gZWxzZSBpZiAobm9kZSBpbnN0YW5jZW9mIENmblNlY3VyaXR5R3JvdXBJbmdyZXNzKSB7XG4gICAgICBjb25zdCBpcFByb3RvY29sID0gTmFnUnVsZXMucmVzb2x2ZUlmUHJpbWl0aXZlKG5vZGUsIG5vZGUuaXBQcm90b2NvbCk7XG4gICAgICBjb25zdCBjaWRySXAgPSBOYWdSdWxlcy5yZXNvbHZlSWZQcmltaXRpdmUobm9kZSwgbm9kZS5jaWRySXApO1xuICAgICAgY29uc3QgY2lkcklwdjYgPSBOYWdSdWxlcy5yZXNvbHZlSWZQcmltaXRpdmUobm9kZSwgbm9kZS5jaWRySXB2Nik7XG4gICAgICBjb25zdCBmcm9tUG9ydCA9IE5hZ1J1bGVzLnJlc29sdmVJZlByaW1pdGl2ZShub2RlLCBub2RlLmZyb21Qb3J0KTtcbiAgICAgIGNvbnN0IHRvUG9ydCA9IE5hZ1J1bGVzLnJlc29sdmVJZlByaW1pdGl2ZShub2RlLCBub2RlLnRvUG9ydCk7XG4gICAgICBpZiAoXG4gICAgICAgIChjaWRySXAgIT0gdW5kZWZpbmVkICYmIGNpZHJJcC5pbmNsdWRlcygnLzAnKSkgfHxcbiAgICAgICAgKGNpZHJJcHY2ICE9IHVuZGVmaW5lZCAmJiBjaWRySXB2Ni5pbmNsdWRlcygnLzAnKSlcbiAgICAgICkge1xuICAgICAgICAvL0lzIGEgcG9ydCByYW5nZSBzcGVjaWZpZWQ/XG4gICAgICAgIGlmIChmcm9tUG9ydCAhPSB1bmRlZmluZWQgJiYgdG9Qb3J0ICE9IHVuZGVmaW5lZCkge1xuICAgICAgICAgIGlmIChcbiAgICAgICAgICAgIChmcm9tUG9ydCA8PSAyMiAmJiB0b1BvcnQgPj0gMjIpIHx8XG4gICAgICAgICAgICBmcm9tUG9ydCA9PSAtMSB8fFxuICAgICAgICAgICAgdG9Qb3J0ID09IC0xIHx8XG4gICAgICAgICAgICBpcFByb3RvY29sID09ICctMSdcbiAgICAgICAgICApIHtcbiAgICAgICAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT05fQ09NUExJQU5UO1xuICAgICAgICAgIH1cbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBpZiAoZnJvbVBvcnQgPT0gMjIgfHwgaXBQcm90b2NvbCA9PSAnLTEnKSB7XG4gICAgICAgICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVDtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5DT01QTElBTlQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT1RfQVBQTElDQUJMRTtcbiAgICB9XG4gIH0sXG4gICduYW1lJyxcbiAgeyB2YWx1ZTogcGFyc2UoX19maWxlbmFtZSkubmFtZSB9XG4pO1xuIl19