"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticbeanstalk_1 = require("@aws-cdk/aws-elasticbeanstalk");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * EC2 instances in Elastic Beanstalk environments upload rotated logs to S3
 * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options-general.html#command-options-general-elasticbeanstalkhostmanager
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticbeanstalk_1.CfnEnvironment) {
        const optionSettings = core_1.Stack.of(node).resolve(node.optionSettings);
        if (optionSettings == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let foundEnabled = false;
        for (const optionSetting of optionSettings) {
            const resolvedOptionSetting = core_1.Stack.of(node).resolve(optionSetting);
            const namespace = resolvedOptionSetting.namespace;
            const optionName = resolvedOptionSetting.optionName;
            const value = resolvedOptionSetting.value;
            if (namespace === 'aws:elasticbeanstalk:hostmanager' &&
                optionName === 'LogPublicationControl' &&
                value === 'true') {
                foundEnabled = true;
                break;
            }
        }
        if (!foundEnabled) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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