"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticbeanstalk_1 = require("@aws-cdk/aws-elasticbeanstalk");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * Elastic Beanstalk environments have managed updates enabled
 * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options-general.html#command-options-general-elasticbeanstalkmanagedactions
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticbeanstalk_1.CfnEnvironment) {
        const optionSettings = core_1.Stack.of(node).resolve(node.optionSettings);
        if (optionSettings == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let foundEnabled = false;
        let foundLevel = false;
        for (const optionSetting of optionSettings) {
            const resolvedOptionSetting = core_1.Stack.of(node).resolve(optionSetting);
            const namespace = resolvedOptionSetting.namespace;
            const optionName = resolvedOptionSetting.optionName;
            const value = resolvedOptionSetting.value;
            if (namespace === 'aws:elasticbeanstalk:managedactions' &&
                optionName === 'ManagedActionsEnabled' &&
                (value === undefined || value === 'true')) {
                foundEnabled = true;
                if (foundLevel) {
                    break;
                }
            }
            else if (namespace === 'aws:elasticbeanstalk:managedactions:platformupdate' &&
                optionName === 'UpdateLevel' &&
                (value === 'minor' || value === 'patch')) {
                foundLevel = true;
                if (foundEnabled) {
                    break;
                }
            }
        }
        if (!foundEnabled || !foundLevel) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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