"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticloadbalancing_1 = require("@aws-cdk/aws-elasticloadbalancing");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * CLB listeners are configured for secure (HTTPs or SSL) protocols for client communication
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        const listeners = core_1.Stack.of(node).resolve(node.listeners);
        for (const listener of listeners) {
            const resolvedListener = core_1.Stack.of(node).resolve(listener);
            const protocol = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedListener.protocol);
            const instanceProtocol = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedListener.instanceProtocol);
            if (protocol.toLowerCase() == 'ssl') {
                if (!(instanceProtocol == undefined ||
                    instanceProtocol.toLowerCase() == 'ssl')) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
            else if (protocol.toLowerCase() == 'https') {
                if (!(instanceProtocol == undefined ||
                    instanceProtocol.toLowerCase() == 'https')) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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