"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_redshift_1 = require("@aws-cdk/aws-redshift");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * Redshift clusters have user user activity logging enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_redshift_1.CfnCluster) {
        const clusterParameterGroupName = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.clusterParameterGroupName);
        if (clusterParameterGroupName === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of core_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_redshift_1.CfnClusterParameterGroup) {
                const childParameterGroupName = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, child.ref);
                if (childParameterGroupName === clusterParameterGroupName) {
                    found = isCompliantClusterParameterGroup(child);
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given cluster parameter group is compliant
 * @param node the CfnClusterParameterGroup to check
 * returns whether the Cluster Parameter group is compliant
 */
function isCompliantClusterParameterGroup(node) {
    const resolvedParameters = core_1.Stack.of(node).resolve(node.parameters);
    if (resolvedParameters == undefined) {
        return false;
    }
    for (const parameter of resolvedParameters) {
        const resolvedParam = core_1.Stack.of(node).resolve(parameter);
        if (resolvedParam.parameterName === 'enable_user_activity_logging' &&
            resolvedParam.parameterValue === 'true') {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiUmVkc2hpZnRDbHVzdGVyVXNlckFjdGl2aXR5TG9nZ2luZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9ydWxlcy9yZWRzaGlmdC9SZWRzaGlmdENsdXN0ZXJVc2VyQWN0aXZpdHlMb2dnaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7OztFQUdFO0FBQ0YsK0JBQTZCO0FBQzdCLHdEQUE2RTtBQUM3RSx3Q0FBbUQ7QUFDbkQsK0NBQThEO0FBRTlEOzs7R0FHRztBQUNILGtCQUFlLE1BQU0sQ0FBQyxjQUFjLENBQ2xDLENBQUMsSUFBaUIsRUFBcUIsRUFBRTtJQUN2QyxJQUFJLElBQUksWUFBWSx5QkFBVSxFQUFFO1FBQzlCLE1BQU0seUJBQXlCLEdBQUcsb0JBQVEsQ0FBQyw2QkFBNkIsQ0FDdEUsSUFBSSxFQUNKLElBQUksQ0FBQyx5QkFBeUIsQ0FDL0IsQ0FBQztRQUNGLElBQUkseUJBQXlCLEtBQUssU0FBUyxFQUFFO1lBQzNDLE9BQU8sNkJBQWlCLENBQUMsYUFBYSxDQUFDO1NBQ3hDO1FBQ0QsSUFBSSxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ2xCLEtBQUssTUFBTSxLQUFLLElBQUksWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUU7WUFDakQsSUFBSSxLQUFLLFlBQVksdUNBQXdCLEVBQUU7Z0JBQzdDLE1BQU0sdUJBQXVCLEdBQzNCLG9CQUFRLENBQUMsNkJBQTZCLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDMUQsSUFBSSx1QkFBdUIsS0FBSyx5QkFBeUIsRUFBRTtvQkFDekQsS0FBSyxHQUFHLGdDQUFnQyxDQUFDLEtBQUssQ0FBQyxDQUFDO29CQUNoRCxNQUFNO2lCQUNQO2FBQ0Y7U0FDRjtRQUNELElBQUksQ0FBQyxLQUFLLEVBQUU7WUFDVixPQUFPLDZCQUFpQixDQUFDLGFBQWEsQ0FBQztTQUN4QztRQUNELE9BQU8sNkJBQWlCLENBQUMsU0FBUyxDQUFDO0tBQ3BDO1NBQU07UUFDTCxPQUFPLDZCQUFpQixDQUFDLGNBQWMsQ0FBQztLQUN6QztBQUNILENBQUMsRUFDRCxNQUFNLEVBQ04sRUFBRSxLQUFLLEVBQUUsWUFBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUNsQyxDQUFDO0FBRUY7Ozs7R0FJRztBQUNILFNBQVMsZ0NBQWdDLENBQ3ZDLElBQThCO0lBRTlCLE1BQU0sa0JBQWtCLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ25FLElBQUksa0JBQWtCLElBQUksU0FBUyxFQUFFO1FBQ25DLE9BQU8sS0FBSyxDQUFDO0tBQ2Q7SUFDRCxLQUFLLE1BQU0sU0FBUyxJQUFJLGtCQUFrQixFQUFFO1FBQzFDLE1BQU0sYUFBYSxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3hELElBQ0UsYUFBYSxDQUFDLGFBQWEsS0FBSyw4QkFBOEI7WUFDOUQsYUFBYSxDQUFDLGNBQWMsS0FBSyxNQUFNLEVBQ3ZDO1lBQ0EsT0FBTyxJQUFJLENBQUM7U0FDYjtLQUNGO0lBQ0QsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLypcbkNvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiovXG5pbXBvcnQgeyBwYXJzZSB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgQ2ZuQ2x1c3RlciwgQ2ZuQ2x1c3RlclBhcmFtZXRlckdyb3VwIH0gZnJvbSAnQGF3cy1jZGsvYXdzLXJlZHNoaWZ0JztcbmltcG9ydCB7IENmblJlc291cmNlLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgTmFnUnVsZUNvbXBsaWFuY2UsIE5hZ1J1bGVzIH0gZnJvbSAnLi4vLi4vbmFnLXJ1bGVzJztcblxuLyoqXG4gKiBSZWRzaGlmdCBjbHVzdGVycyBoYXZlIHVzZXIgdXNlciBhY3Rpdml0eSBsb2dnaW5nIGVuYWJsZWRcbiAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICovXG5leHBvcnQgZGVmYXVsdCBPYmplY3QuZGVmaW5lUHJvcGVydHkoXG4gIChub2RlOiBDZm5SZXNvdXJjZSk6IE5hZ1J1bGVDb21wbGlhbmNlID0+IHtcbiAgICBpZiAobm9kZSBpbnN0YW5jZW9mIENmbkNsdXN0ZXIpIHtcbiAgICAgIGNvbnN0IGNsdXN0ZXJQYXJhbWV0ZXJHcm91cE5hbWUgPSBOYWdSdWxlcy5yZXNvbHZlUmVzb3VyY2VGcm9tSW5zdHJpbnNpYyhcbiAgICAgICAgbm9kZSxcbiAgICAgICAgbm9kZS5jbHVzdGVyUGFyYW1ldGVyR3JvdXBOYW1lXG4gICAgICApO1xuICAgICAgaWYgKGNsdXN0ZXJQYXJhbWV0ZXJHcm91cE5hbWUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVDtcbiAgICAgIH1cbiAgICAgIGxldCBmb3VuZCA9IGZhbHNlO1xuICAgICAgZm9yIChjb25zdCBjaGlsZCBvZiBTdGFjay5vZihub2RlKS5ub2RlLmZpbmRBbGwoKSkge1xuICAgICAgICBpZiAoY2hpbGQgaW5zdGFuY2VvZiBDZm5DbHVzdGVyUGFyYW1ldGVyR3JvdXApIHtcbiAgICAgICAgICBjb25zdCBjaGlsZFBhcmFtZXRlckdyb3VwTmFtZSA9XG4gICAgICAgICAgICBOYWdSdWxlcy5yZXNvbHZlUmVzb3VyY2VGcm9tSW5zdHJpbnNpYyhub2RlLCBjaGlsZC5yZWYpO1xuICAgICAgICAgIGlmIChjaGlsZFBhcmFtZXRlckdyb3VwTmFtZSA9PT0gY2x1c3RlclBhcmFtZXRlckdyb3VwTmFtZSkge1xuICAgICAgICAgICAgZm91bmQgPSBpc0NvbXBsaWFudENsdXN0ZXJQYXJhbWV0ZXJHcm91cChjaGlsZCk7XG4gICAgICAgICAgICBicmVhaztcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIGlmICghZm91bmQpIHtcbiAgICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PTl9DT01QTElBTlQ7XG4gICAgICB9XG4gICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuQ09NUExJQU5UO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9UX0FQUExJQ0FCTEU7XG4gICAgfVxuICB9LFxuICAnbmFtZScsXG4gIHsgdmFsdWU6IHBhcnNlKF9fZmlsZW5hbWUpLm5hbWUgfVxuKTtcblxuLyoqXG4gKiBIZWxwZXIgZnVuY3Rpb24gdG8gY2hlY2sgd2hldGhlciBhIGdpdmVuIGNsdXN0ZXIgcGFyYW1ldGVyIGdyb3VwIGlzIGNvbXBsaWFudFxuICogQHBhcmFtIG5vZGUgdGhlIENmbkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCB0byBjaGVja1xuICogcmV0dXJucyB3aGV0aGVyIHRoZSBDbHVzdGVyIFBhcmFtZXRlciBncm91cCBpcyBjb21wbGlhbnRcbiAqL1xuZnVuY3Rpb24gaXNDb21wbGlhbnRDbHVzdGVyUGFyYW1ldGVyR3JvdXAoXG4gIG5vZGU6IENmbkNsdXN0ZXJQYXJhbWV0ZXJHcm91cFxuKTogYm9vbGVhbiB7XG4gIGNvbnN0IHJlc29sdmVkUGFyYW1ldGVycyA9IFN0YWNrLm9mKG5vZGUpLnJlc29sdmUobm9kZS5wYXJhbWV0ZXJzKTtcbiAgaWYgKHJlc29sdmVkUGFyYW1ldGVycyA9PSB1bmRlZmluZWQpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cbiAgZm9yIChjb25zdCBwYXJhbWV0ZXIgb2YgcmVzb2x2ZWRQYXJhbWV0ZXJzKSB7XG4gICAgY29uc3QgcmVzb2x2ZWRQYXJhbSA9IFN0YWNrLm9mKG5vZGUpLnJlc29sdmUocGFyYW1ldGVyKTtcbiAgICBpZiAoXG4gICAgICByZXNvbHZlZFBhcmFtLnBhcmFtZXRlck5hbWUgPT09ICdlbmFibGVfdXNlcl9hY3Rpdml0eV9sb2dnaW5nJyAmJlxuICAgICAgcmVzb2x2ZWRQYXJhbS5wYXJhbWV0ZXJWYWx1ZSA9PT0gJ3RydWUnXG4gICAgKSB7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9XG4gIH1cbiAgcmV0dXJuIGZhbHNlO1xufVxuIl19