"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_redshift_1 = require("@aws-cdk/aws-redshift");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * Redshift clusters require TLS/SSL encryption
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_redshift_1.CfnCluster) {
        const clusterParameterGroupName = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.clusterParameterGroupName);
        if (clusterParameterGroupName === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of core_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_redshift_1.CfnClusterParameterGroup) {
                if (isMatchingParameterGroup(child, clusterParameterGroupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Parameter Group enforces SSL and is associated with the given Redshift cluster
 * @param node the CfnClusterParameterGroup to check
 * @param parameterGroupName The name of the associated parameter group
 * returns whether the CfnClusterParameterGroup enforces SSL and is associated given Redshift cluster
 */
function isMatchingParameterGroup(node, parameterGroupName) {
    const parameterGroupLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
    if (parameterGroupName !== parameterGroupLogicalId ||
        node.parameters == undefined) {
        return false;
    }
    const parameters = core_1.Stack.of(node).resolve(node.parameters);
    for (const parameter of parameters) {
        const resolvedParameter = core_1.Stack.of(node).resolve(parameter);
        if (resolvedParameter.parameterName.toLowerCase() == 'require_ssl' &&
            resolvedParameter.parameterValue.toLowerCase() == 'true') {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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