"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * S3 Buckets prohibit public read access through their Block Public Access configurations and bucket ACLs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        const publicAccessBlockConfiguration = core_1.Stack.of(node).resolve(node.publicAccessBlockConfiguration);
        if (publicAccessBlockConfiguration === undefined ||
            nag_rules_1.NagRules.resolveIfPrimitive(node, publicAccessBlockConfiguration.blockPublicPolicy) !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const accessControl = nag_rules_1.NagRules.resolveIfPrimitive(node, node.accessControl);
        const blockPublicAcls = nag_rules_1.NagRules.resolveIfPrimitive(node, publicAccessBlockConfiguration.blockPublicAcls);
        if ((accessControl === 'PublicRead' ||
            accessControl === 'PublicReadWrite') &&
            blockPublicAcls !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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