"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * S3 Buckets prohibit public write access through their Block Public Access configurations and bucket ACLs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        const publicAccessBlockConfiguration = core_1.Stack.of(node).resolve(node.publicAccessBlockConfiguration);
        if (publicAccessBlockConfiguration === undefined ||
            nag_rules_1.NagRules.resolveIfPrimitive(node, publicAccessBlockConfiguration.blockPublicPolicy) !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const accessControl = nag_rules_1.NagRules.resolveIfPrimitive(node, node.accessControl);
        const blockPublicAcls = nag_rules_1.NagRules.resolveIfPrimitive(node, publicAccessBlockConfiguration.blockPublicAcls);
        if (accessControl === 'PublicReadWrite' && blockPublicAcls !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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