"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * Secrets have automatic rotation scheduled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_secretsmanager_1.CfnSecret) {
        const secretLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        const secretTargetAttachmentLogicalIds = Array();
        const cfnSecretTargetAttachments = Array();
        const cfnRotationSchedules = Array();
        for (const child of core_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_secretsmanager_1.CfnSecretTargetAttachment) {
                cfnSecretTargetAttachments.push(child);
            }
            else if (child instanceof aws_secretsmanager_1.CfnRotationSchedule) {
                cfnRotationSchedules.push(child);
            }
        }
        if (cfnRotationSchedules.length === 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of cfnSecretTargetAttachments) {
            const attachmentLogicalId = getMatchingSecretTargetAttachment(child, secretLogicalId);
            if (attachmentLogicalId) {
                secretTargetAttachmentLogicalIds.push(attachmentLogicalId);
            }
        }
        for (const child of cfnRotationSchedules) {
            if (isMatchingRotationSchedule(child, secretLogicalId, secretTargetAttachmentLogicalIds)) {
                found = true;
                break;
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Secret Target Attachment is associated with the given secret.
 * @param node The CfnTargetAttachment to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * Returns the Logical ID if the attachment if is associated with the secret, otherwise and empty string.
 */
function getMatchingSecretTargetAttachment(node, secretLogicalId) {
    const resourceSecretId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId) {
        return nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
    }
    return '';
}
/**
 * Helper function to check whether a given Rotation Schedule is associated with the given secret.
 * @param node The CfnRotationSchedule to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * @param secretTargetAttachmentLogicalIds The Cfn Logical IDs of any Secret Target Attachments associated with the given secret.
 * Returns whether the CfnRotationSchedule is associated with the given secret.
 */
function isMatchingRotationSchedule(node, secretLogicalId, secretTargetAttachmentLogicalIds) {
    const resourceSecretId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId ||
        secretTargetAttachmentLogicalIds.includes(resourceSecretId)) {
        if (core_1.Stack.of(node).resolve(node.hostedRotationLambda) === undefined &&
            core_1.Stack.of(node).resolve(node.rotationLambdaArn) === undefined) {
            return false;
        }
        const rotationRules = core_1.Stack.of(node).resolve(node.rotationRules);
        if (rotationRules !== undefined) {
            const automaticallyAfterDays = core_1.Stack.of(node).resolve(rotationRules.automaticallyAfterDays);
            if (automaticallyAfterDays !== undefined) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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