"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_wafv2_1 = require("@aws-cdk/aws-wafv2");
const core_1 = require("@aws-cdk/core");
const nag_rules_1 = require("../../nag-rules");
/**
 * WAFv2 web ACLs have logging enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_wafv2_1.CfnWebACL) {
        const webAclLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        const webAclName = core_1.Stack.of(node).resolve(node.name);
        let found = false;
        for (const child of core_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnLoggingConfiguration) {
                if (isMatchingLoggingConfiguration(child, webAclLogicalId, webAclName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Logging Configuration contains the given Web ACL
 * @param node the CfnLoggingConfiguration to check
 * @param webAclLogicalId the Cfn Logical ID of the Web ACL
 * @param webAclName the name of the Web ACL
 * returns whether the CfnLoggingConfiguration contains the given Web ACL
 */
function isMatchingLoggingConfiguration(node, webAclLogicalId, webAclName) {
    const resourceArn = JSON.stringify(core_1.Stack.of(node).resolve(node.resourceArn));
    if (new RegExp(`${webAclLogicalId}(?![\\w])`).test(resourceArn) ||
        (webAclName != undefined &&
            new RegExp(`webacl\/${webAclName}(?![\\w\\-_\\.])`).test(resourceArn))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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