# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['omc4py',
 'omc4py.parser',
 'omc4py.v_1_13',
 'omc4py.v_1_14',
 'omc4py.v_1_15',
 'omc4py.v_1_16',
 'omc4py.v_1_17',
 'omc4py.v_1_18',
 'omc4py.v_1_19',
 'omc4py.v_1_20']

package_data = \
{'': ['*']}

install_requires = \
['Arpeggio>=1.6',
 'modelicalang>=0.1.0a0,<0.2.0',
 'numpy>=1.7.0',
 'pyzmq>=17.0.0',
 'typing-extensions>=3.7.2']

setup_kwargs = {
    'name': 'openmodelicacompiler',
    'version': '0.2.1',
    'description': 'OpenModelica compiler (omc) interface for Python',
    'long_description': '\n# OpenModelicaCompilerForPython [![License: OSMC-PL](https://img.shields.io/badge/license-OSMC--PL-lightgrey.svg)](./COPYING) [![lint.yml](https://github.com/ijknabla/OpenModelicaCompilerForPython/actions/workflows/lint.yml/badge.svg)](https://github.com/ijknabla/OpenModelicaCompilerForPython/actions/workflows/lint.yml) [![Pytest](https://github.com/ijknabla/OpenModelicaCompilerForPython/actions/workflows/test.yml/badge.svg)](https://github.com/ijknabla/OpenModelicaCompilerForPython/actions/workflows/test.yml)\n\nOpenModelica compiler (omc) interface for Python\n\n## Change log\n\n[See CHANGELOG.md](./CHANGELOG.md)\n\n## Quick tour\n\n### Setup\n\nMake sure that OpenModelica is installed on your system.\n\n```bash\n$ omc --version\n```\n\nOpenModelica official page [https://openmodelica.org/](https://openmodelica.org/)\n\nInstall OpenModelicaCompiler with pip.\n\n```bash\n$ python3 -m pip install OpenModelicaCompiler\n```\n\n`omc4py` is acutual package name. `omc4py.open_session()` will return session object which interfaces to omc.\n\n```python3\n#!/usr/bin/env python3\nimport omc4py\n\nwith omc4py.open_session() as session:\n    print(session.getVersion())\n```\n\n### More usage about `open_session(...)`\n\nIf `omc4py.open_session` cannot find omc, such as if you have not added OpenModelica to your _PATH_ environment variable, you can specify a valid omc command name or omc executable path by `str`.\n\n```python3\nimport omc4py\n\nwith omc4py.open_session(\n    "C:/OpenModelica1.13.0-64bit/bin/omc.exe"\n) as session:\n    print(session.getVersion())\n```\n\nIt is also possible to open multiple sessions with different versions of omc at the same time by explicitly specifying omc.\n\n```python3\nimport omc4py\n\nwith \\\n    omc4py.open_session(\n        "C:/OpenModelica1.13.0-64bit/bin/omc.exe"\n    ) as session_13, \\\n    omc4py.open_session(\n        "C:/Program Files/OpenModelica1.14.0-64bit/bin/omc.exe"\n    ) as session_14:\n\n    print("v1.13.0:", session_13.getVersion())\n    print("v1.14.0:", session_14.getVersion())\n```\n\nAs shown above, __it is recommended to ensure that session is closed by calling `omc4py.open_session()` via with-statement__.\n\nHowever, sometimes you want to use session interactively, like OMShell. `omc4py` closes all unclosed sessions when exiting the python interpreter.\n\n```python3\n>>> from omc4py import *\n>>> session = open_session()\n>>> session.loadString("""\n... package A\n...     package B\n...             package C\n...             end C;\n...     end B;\n... end A;\n... """)\nTrue\n>>> list(session.getClassNames("A", recursive=True))\n[TypeName(\'A\'), TypeName(\'A.B\'), TypeName(\'A.B.C\')]\n>>>\n>>>\n>>> exit()  # session will be closed internally\n```\n\nBesides, session object has `__close__` method to explicitly close session.\n\n```python3\n>>> from omc4py import *\n>>> session = open_session()\n>>> session.__close__()\n>>>\n>>> exit()\n```\n\n### About session API\n\nAll session methods are _OpenModelica.Scripting.*_ functions. The names and types of arguments and return values are the same as the original modelica function, and session internally converts between the python class and the modelica class.\n\nIf you want to know more about each session method, you can display it with the `help ()` function.\n\n- [UserGuide for OpenModelica Scripting API (v1.14)](https://www.openmodelica.org/doc/OpenModelicaUsersGuide/1.14/scripting_api.html)\n- [UserGuide for OpenModelica Scripting API (v1.15)](https://www.openmodelica.org/doc/OpenModelicaUsersGuide/1.15/scripting_api.html)\n- [UserGuide for OpenModelica Scripting API (v1.16)](https://www.openmodelica.org/doc/OpenModelicaUsersGuide/1.16/scripting_api.html)\n- [UserGuide for OpenModelica Scripting API (v1.17)](https://www.openmodelica.org/doc/OpenModelicaUsersGuide/1.17/scripting_api.html)\n- [UserGuide for OpenModelica Scripting API (v1.18)](https://www.openmodelica.org/doc/OpenModelicaUsersGuide/1.18/scripting_api.html)\n\n- [UserGuide for OpenModelica Scripting API (latest)](https://www.openmodelica.org/doc/OpenModelicaUsersGuide/latest/scripting_api.html)\n\n\n[Documentation in OpenModelica build server](https://build.openmodelica.org/Documentation/OpenModelica.Scripting.html) shows exhaustive information about _OpenModelica.Scripting_. You will find sub-packages not explained user guide.\n\n- _OpenModelica.Scripting.Internal.*_\n- _OpenModelica.Scripting.Experimental.*_\n\nThey are available from absolute reference\n\n```python3\n# Example for "timerTick" and "timerTock"\n# in "OpenModelica.Scripting.Internal.Time"\nfrom omc4py import open_session\nfrom time import sleep\n\ntimer_index: int = 1\n\nwith open_session() as session:\n    session.OpenModelica.Scripting.Internal.Time.timerTick(timer_index)\n\n    sleep(0.1)\n\n    # show elapsed time from last timerTick\n    print(session.OpenModelica.Scripting.Internal.Time.timerTock(timer_index))\n```\n\n- - -\n\nLet me introduce typical API functions!\n\n#### `loadModel`\n\nLoad library and returns True if success. You can specify versions by second argument\n\n```python3\nimport omc4py\n\nwith omc4py.open_session() as session:\n    assert(session.loadModel("Modelica"))  # load MSL\n```\n\n```python3\nimport omc4py\n\nwith omc4py.open_session() as session:\n    assert(session.loadModel("Modelica", ["3.2.3"]))  # load MSL 3.2.3\n```\n\n#### `getClassNames`\n\nReturns array of class names in the given class\n\n```python3\nimport omc4py\n\nwith omc4py.open_session() as session:\n    assert(session.loadModel("Modelica"))\n    for className in session.getClassNames("Modelica"):\n        print(className)\n```\n\nBy default, `getClassNames()` only returns "sub" classes. If you want to know all classes belongs to the class set `recursive=True`.\n\n```python3\nimport omc4py\n\nwith omc4py.open_session() as session:\n    assert(session.loadModel("Modelica"))\n    for className in session.getClassNames("Modelica", recursive=True):\n        print(className)  # many class names will be printed\n```\n\n#### `getComponents`\n\nReturns array of component (variable, parameter, constant, ...etc) profiles\n\n```python3\nimport omc4py\n\nwith omc4py.open_session() as session:\n    assert(session.loadModel("Modelica", ["3.2.3"]))\n    for component in session.getComponents("Modelica.Constants"):\n        print(\n            f"{component.className.last_identifier!s:<20}"\n            f"{component.name!s:<15}"\n            f"{component.comment!r}"\n        )\n```\n\n- - -\n\n#### Exception handling\n\n<!--\n@startuml\nnamespace omc4py.exception {\n    OMCException <-- OMCNotification\n    OMCException <-- OMCWarning\n    OMCException <-- OMCError\n    OMCException <-- OMCRuntimeError\n}\n\nException <-- omc4py.exception.OMCException\n\nException <-ri- Warning\nWarning <-- omc4py.exception.OMCNotification\nWarning <-- omc4py.exception.OMCWarning\n\nException <-ri- RuntimeError\nRuntimeError <-- omc4py.exception.OMCRuntimeError\n@enduml\n-->\n![class diagram of omc4py.exception](http://www.plantuml.com/plantuml/svg/SoWkIImgAStDuSfBp4qjBaXCJbN8pqqsAQZKIwr8JYqeoSpFKwZcKW02VrzdLxYGZQukIC0lloGpBJCv4II6Kr5uOb5UPbuwJddNegBy8fooGQLv9PcvgH15jLnSA0emtAg7R0Igug9CNGMOKw0qTYFG_4LGCLGUqpOKfoDpS1g5eiCXDIy563C0)\n\n- `OMCNotification`, `OMCWarning`, `OMCError` are raised from _omc_\n- `OMCRuntimeError` is raised from `omc4py` python implementation (not from _omc_)\n\nWe are not sure about whole OpenModelica\'s exception handling policy.\nThrough `omc4py` project, We found that there are 4 situation for expection caused by function calls.\n\nomc behavior\n\n1. Function returns "\\n" instead of valid value (no exception info)\n1. Function returns formatted error messages (contains sourceInfo, level, kind, message) instead of valid value\n1. Function returns unformatted error message (typically, startswith "* Error") instead of valid value\n1. Function returns valid value and set exception messages internally\n\n`omc4py` behavior\n\n1) function returns `None` instead of valid result (no exception will be sent)\n1) function send `OMCNotification` or `OMCWarning`, or raise `OMCError`\n1) function raise `OMCRuntimeError` with the message returned by the omc\n1) function returns valid value. You can check exceptions explicitly by `session.__check__()`\n\nNormally, 4th case seems to be _notification_ or _warning_. If you want to be sure to check for exceptions, call `session.__check__()` before exit doubtful context.\n\n```python3\nfrom omc4py import open_session\n\ndef doubtful_task(session):\n    # session.doubtful_API1(...)\n    # session.doubtful_API2(...)\n    # session.doubtful_API3(...)\n    session.__check__()\n\nwith open_session() as session:\n    doubtful_task(session)\n```\n',
    'author': 'ijknabla',
    'author_email': 'ijknabla@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/ijknabla/OpenModelicaCompilerForPython',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
