# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['hexpex']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'hexpex',
    'version': '0.2.0',
    'description': 'A type-hinted, object-oriented Python implementation for working with hex grids',
    'long_description': '<!-- PROJECT TITLE -->\n<h1 align="center">Hexpex</h1>\n\n<!-- PROJECT BLURB -->\n<p align="center">\n  <em>A type-hinted, object-oriented Python implementation for working with hex grids</em>\n</p>\n\n<!-- PROJECT SHIELDS -->\n<div align="center">\n  <a href="https://github.com/solbero/hexpex/actions/workflows/build.yaml/" target="_blank">\n    <img src="https://img.shields.io/github/workflow/status/solbero/hexpex/Build?label=build" alt="Build action">\n  </a>\n  <a href="https://github.com/solbero/hexpex/actions/workflows/publish.yaml/" target="_blank">\n    <img src="https://img.shields.io/github/workflow/status/solbero/hexpex/Publish?label=publish" alt="Publish action">\n  </a>\n  <a href="https://app.codecov.io/gh/solbero/hexpex" target="_blank">\n    <img src="https://img.shields.io/codecov/c/github/solbero/hexpex" alt="Code coverage">\n  </a>\n  <a href="https://pypi.org/project/hexpex/" target="_blank">\n    <img src="https://img.shields.io/pypi/v/hexpex" alt="Package version">\n  </a>\n  <a href="https://pypi.org/project/hexpex/" target="_blank">\n    <img src="https://img.shields.io/pypi/pyversions/hexpex" alt="Supported Python versions">\n  </a>\n  <a href="https://github.com/solbero/hexpex/blob/master/LICENSE.txt" target="_blank">\n    <img src="https://img.shields.io/github/license/solbero/hexpex" alt="License">\n  </a>\n</div>\n\n<!-- TABLE OF CONTENTS -->\n<div align="center">\n  <details>\n    <summary>Table of Contents</summary>\n    <p>\n      <a href="#about-the-project">About the Project</a>\n      <br>\n      <a href="#prerequisites">Prerequisites</a>\n      <br>\n      <a href="#installation">Installation</a>\n      <br>\n      <a href="#usage">Usage</a>\n      <br>\n      <a href="#roadmap">Roadmap</a>\n      <br>\n      <a href="#contributing">Contributing</a>\n      <br>\n      <a href="#license">License</a>\n      <br>\n      <a href="#contact">Contact</a>\n      <br>\n      <a href="#project-links">Project Links</a>\n    </p>\n  </details>\n</div>\n\n<!-- ABOUT THE PROJECT -->\n## About the Project\n\nThis is a type-hinted, object-oriented implementation in Python of [hexagonal grids](https://www.redblobgames.com/grids/hexagons/) as described on [Red Blob Games](https://www.redblobgames.com/).\nThis package allows you to easily work with hexagonal grids in Python.\nAll of its classes, attributes and methods are type-hinted which allows your editor to autocomplete signatures and catch bugs and mistakes early.\n\n<p align="right">(<a href="#hexpex">back to top</a>)</p>\n\n<!-- PREREQUISITES -->\n## Prerequisites\n\nBecause this package uses type hints, keyword-only and positional-only arguments you must have Python 3.9 or greater installed.\n\n<p align="right">(<a href="#hexpex">back to top</a>)</p>\n\n<!-- INSTALLATION -->\n\n## Installation\n\n### Using PIP\n\n   ```sh\n   pip install hexpex\n   ```\n\n### Using Poetry\n\n   ```sh\n   poetry add hexpex\n   ```\n\n### Manually (for testing or development)\n\n  ```sh\n  git clone https://github.com/solbero/hexpex.git\n  cd hexpex\n  poetry install\n  ```\n\n<p align="right">(<a href="#hexpex">back to top</a>)</p>\n\n<!-- USAGE EXAMPLES -->\n## Usage\n\n### Classes\n\nHexpex provides classes for working with hexagonal grids in both the cube and axial coordinate systems.\nFor more information about the difference between these two coordinate systems see the writeup on [Red Blob Games](https://www.redblobgames.com/grids/hexagons/#coordinates).\n\n```python\nfrom hexpex import Axial, Cube\n\nCube(q=1, r=0, s=-1)\nAxial(q=1, r=0)\n```\n\n### Object Methods\n\n<dl>\n  <dt><code>adjacent()</code></dt>\n  <dd>Returns the hex coordinate in adjacent direction from self</dd>\n\n  <dt><code>diagonal()</code></dt>\n  <dd>Returns the hex coordinate in diagonal direction from self</dd>\n\n  <dt><code>distance()</code></dt>\n  <dd>Returns the distance between passed hex coordinate and self</dd>\n\n  <dt><code>range()</code></dt>\n  <dd>Returns a set of hex coordinates within passed distance of self</dd>\n\n  <dt><code>ring()</code></dt>\n  <dd>Returns a set of hex coordinates on a ring passed distance from self</dd>\n\n  <dt><code>rotation()</code></dt>\n  <dd>Returns a set of rotated hex coordinates rotated around self</dd>\n\n  <dt><code>spiral()</code></dt>\n  <dd>Yields hex positions in a spiral from self out to passed distance from self</dd>\n</dl>\n\n### Operations\n\nObjects can be added or subtracted from each other, and multiplied or divided by integers.\n\n```python\nfrom hexpex import Axial, Cube\n\ncube_1 = Cube(q=2, r=0, s=-2)\ncube_2 = Cube(q=-1, r=0, s=1)\n\ncube_1 + cube_2\n#> Cube(1, 0, -1)\ncube_1 - cube_2\n#> Cube(3, 0, -3)\ncube_1 * 2\n#> Cube(4, 0, -4)\ncube_1 // 2\n#> Cube(1, 0, -1)\n\naxial_1 = Axial(q=2, r=0)\naxial_2 = Axial(q=-1, r=0)\n\naxial_1 + axial_2\n#> Axial(1, 0)\naxial_1 - axial_2\n#> Axial(3, 0)\naxial_1 * 2\n#> Axial(4, 0)\naxial_1 // 2\n#> Axial(1, 0)\n\n```\n\n### Direction Vectors\n\nHexpex provides some helper enums for giving direction vectors to the methods `adjacent()`, `diagonal()` and `spiral()`.\nTo use them import the enums for your coordinate system and hex orientation (pointy or flat).\nFor more information on the difference between the two hex orientations see [Red Blob Games](https://www.redblobgames.com/grids/hexagons/#basics).\n\n```python\nfrom hexpex import Cube, CubeFlatAdjacentDirection as AdjacentDirection, CubeFlatDiagonalDirection as DiagonalDirection\n\ncube = Cube(0, 0, 0)\n\ncube.adjacent(AdjacentDirection.SE)\n#> Cube(1, 0, -1)\ncube.diagonal(DiagonalDirection.E)\n#> Cube(2, -1, -1)\n```\n\n### Conversion\n\nA cube object can be converted to an axial object using the `to_axial()` method.\nThe reverse is true for an axial object using the `to_cube()` method.\n\nBoth representations can also be converted to a tuple using the `to_tuple()` method and to a dict using the `to_dict()`method.\n\n```python\n\nfrom hexpex import Axial, Cube\n\ncube = Cube(1, 0, -1)\naxial = Axial(1, 0)\n\ncube.to_axial()\n#> Axial(1, 0)\naxial.to_cube()\n#> Cube(1, 0, -1)\n\ncube.to_tuple()\n#> (1, 0, -1)\naxial.to_tuple()\n#> (1, 0)\n\ncube.to_dict()\n#> {"q": 1, "r": 0, "s": -1}\naxial.to_dict()\n#> {"q": 1, "r": 0}\n```\n\n<p align="right">(<a href="#hexpex">back to top</a>)</p>\n\n<!-- ROADMAP -->\n## Roadmap\n\n### Coordinate systems\n\n* [x] Cube coordinates\n* [x] Axial coordinates\n* [ ] Double offset coordinates\n\n### Methods\n\n* [x] Distances\n* [x] Neighbors\n* [x] Range\n* [x] Rings\n* [x] Rotation\n* [x] Spiral\n* [ ] Line drawing\n* [ ] Reflection\n* [ ] Rounding\n* [ ] Hex to pixel\n* [ ] Pixel to hex\n\nSee the [open issues](https://github.com/solbero/hexpex/issues) for a full list of proposed features (and known issues).\n\n<p align="right">(<a href="#hexpex">back to top</a>)</p>\n\n<!-- CONTRIBUTING -->\n## Contributing\n\nIf you have a suggestion that would make this project better, please [fork the repo](https://github.com/solbero/hexpex/fork) and create a pull request.\nYou can also simply [open an issue](https://github.com/solbero/hexpex/issues/new/choose) with the label "enhancement".\n\n1. Fork the project\n2. Create your feature branch\n\n  ```sh\n  git checkout -b feature/AmazingFeature\n  ```\n\n3. Commit your changes\n\n  ```sh\n  git commit -m \'Add some AmazingFeature\'\n  ```\n\n4. Push to the branch\n\n```sh\ngit push origin feature/AmazingFeature\n```\n\n5. Open a pull request\n\n<p align="right">(<a href="#hexpex">back to top</a>)</p>\n\n<!-- LICENSE -->\n## License\n\nDistributed under the GPLv3 License.\nSee [`LICENSE.txt`](https://github.com/solbero/hexpex/blob/master/LICENSE.txt) for more information.\n\n<p align="right">(<a href="#hexpex">back to top</a>)</p>\n\n<!-- CONTACT -->\n## Contact\n\n* Email: [njord.solberg@gmail.com](mailto:njord.solberg@gmail.com)\n\n<p align="right">(<a href="#hexpex">back to top</a>)</p>\n\n<!-- PROJECT LINKS -->\n## Project Links\n\n* Github: [https://github.com/solbero/hexpex](https://github.com/solbero/hexpex)\n* PyPI: [https://pypi.org/project/hexpex/](https://pypi.org/project/hexpex/)\n\n<p align="right">(<a href="#hexpex">back to top</a>)</p>\n',
    'author': 'solbero',
    'author_email': 'njord.solberg@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/solbero/hexpex',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
