/*
 * Decompiled with CFR 0.152.
 */
package org.openrdf.sail.nativerdf.btree;

import info.aduna.io.ByteArrayUtil;
import info.aduna.io.NioFile;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.openrdf.sail.nativerdf.btree.AllocatedNodesList;
import org.openrdf.sail.nativerdf.btree.DefaultRecordComparator;
import org.openrdf.sail.nativerdf.btree.RecordComparator;
import org.openrdf.sail.nativerdf.btree.RecordIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BTree {
    private static final byte[] MAGIC_NUMBER = new byte[]{98, 116, 102};
    private static final byte[] OLD_MAGIC_NUMBER = new byte[]{0, 0, 0};
    private static final byte FILE_FORMAT_VERSION = 1;
    private static final int HEADER_LENGTH = 16;
    private static final int NODE_CACHE_SIZE = 10;
    private static final int MIN_MRU_CACHE_SIZE = 4;
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final NioFile nioFile;
    private final boolean forceSync;
    private final RecordComparator comparator;
    private final ReentrantReadWriteLock btreeLock = new ReentrantReadWriteLock();
    private final Map<Integer, Node> nodeCache = new HashMap<Integer, Node>(10);
    private final Map<Integer, Node> mruNodes = new LinkedHashMap<Integer, Node>(10);
    private final AllocatedNodesList allocatedNodesList;
    private final int blockSize;
    private final int valueSize;
    private final int slotSize;
    private final int branchFactor;
    private final int minValueCount;
    private final int nodeSize;
    private volatile int rootNodeID;
    private volatile int height = -1;
    private volatile boolean closed = false;

    public BTree(File dataDir, String filenamePrefix, int blockSize, int valueSize) throws IOException {
        this(dataDir, filenamePrefix, blockSize, valueSize, false);
    }

    public BTree(File dataDir, String filenamePrefix, int blockSize, int valueSize, boolean forceSync) throws IOException {
        this(dataDir, filenamePrefix, blockSize, valueSize, new DefaultRecordComparator(), forceSync);
    }

    public BTree(File dataDir, String filenamePrefix, int blockSize, int valueSize, RecordComparator comparator) throws IOException {
        this(dataDir, filenamePrefix, blockSize, valueSize, comparator, false);
    }

    public BTree(File dataDir, String filenamePrefix, int blockSize, int valueSize, RecordComparator comparator, boolean forceSync) throws IOException {
        if (dataDir == null) {
            throw new IllegalArgumentException("dataDir must not be null");
        }
        if (filenamePrefix == null) {
            throw new IllegalArgumentException("filenamePrefix must not be null");
        }
        if (blockSize < 16) {
            throw new IllegalArgumentException("block size must be at least 16 bytes");
        }
        if (valueSize <= 0) {
            throw new IllegalArgumentException("value size must be larger than 0");
        }
        if (blockSize < 3 * valueSize + 20) {
            throw new IllegalArgumentException("block size to small; must at least be able to store three values");
        }
        if (comparator == null) {
            throw new IllegalArgumentException("comparator muts not be null");
        }
        File file = new File(dataDir, filenamePrefix + ".dat");
        this.nioFile = new NioFile(file);
        this.comparator = comparator;
        this.forceSync = forceSync;
        File allocFile = new File(dataDir, filenamePrefix + ".alloc");
        this.allocatedNodesList = new AllocatedNodesList(allocFile, this);
        if (this.nioFile.size() == 0L) {
            this.blockSize = blockSize;
            this.valueSize = valueSize;
            this.rootNodeID = 0;
            this.height = 0;
            this.writeFileHeader();
        } else {
            ByteBuffer buf = ByteBuffer.allocate(16);
            this.nioFile.read(buf, 0L);
            buf.rewind();
            byte[] magicNumber = new byte[MAGIC_NUMBER.length];
            buf.get(magicNumber);
            byte version = buf.get();
            this.blockSize = buf.getInt();
            this.valueSize = buf.getInt();
            this.rootNodeID = buf.getInt();
            if (Arrays.equals(MAGIC_NUMBER, magicNumber)) {
                if (version > 1) {
                    throw new IOException("Unable to read BTree file " + file + "; it uses a newer file format");
                }
                if (version != 1) {
                    throw new IOException("Unable to read BTree file " + file + "; invalid file format version: " + version);
                }
            } else if (Arrays.equals(OLD_MAGIC_NUMBER, magicNumber)) {
                if (version != 1) {
                    throw new IOException("Unable to read BTree file " + file + "; invalid file format version: " + version);
                }
                this.logger.info("Updating file header for btree file '{}'", (Object)file.getAbsolutePath());
                this.writeFileHeader();
            } else {
                throw new IOException("File doesn't contain (compatible) BTree data: " + file);
            }
            if (this.valueSize != valueSize) {
                throw new IOException("Specified value size (" + valueSize + ") is different from what is stored on disk (" + this.valueSize + ") in " + file);
            }
        }
        this.slotSize = 4 + this.valueSize;
        this.branchFactor = 1 + (this.blockSize - 8) / this.slotSize;
        this.minValueCount = (this.branchFactor - 1) / 2;
        this.nodeSize = 8 + (this.branchFactor - 1) * this.slotSize;
    }

    public File getFile() {
        return this.nioFile.getFile();
    }

    public boolean delete() throws IOException {
        this.close(false);
        boolean success = this.allocatedNodesList.delete();
        return success &= this.nioFile.delete();
    }

    public void close() throws IOException {
        this.close(true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void close(boolean syncChanges) throws IOException {
        this.btreeLock.writeLock().lock();
        try {
            if (this.closed) {
                return;
            }
            if (syncChanges) {
                this.sync();
            }
            this.closed = true;
            Map<Integer, Node> map = this.nodeCache;
            synchronized (map) {
                this.nodeCache.clear();
                this.mruNodes.clear();
            }
            try {
                this.nioFile.close();
            }
            finally {
                this.allocatedNodesList.close(syncChanges);
            }
        }
        finally {
            this.btreeLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void sync() throws IOException {
        this.btreeLock.readLock().lock();
        try {
            Map<Integer, Node> map = this.nodeCache;
            synchronized (map) {
                for (Node node : this.nodeCache.values()) {
                    if (!node.dataChanged()) continue;
                    node.write();
                }
            }
            if (this.forceSync) {
                this.nioFile.force(false);
            }
            this.allocatedNodesList.sync();
        }
        finally {
            this.btreeLock.readLock().unlock();
        }
    }

    public byte[] get(byte[] key) throws IOException {
        this.btreeLock.readLock().lock();
        try {
            Node node = this.readRootNode();
            if (node == null) {
                byte[] byArray = null;
                return byArray;
            }
            while (true) {
                int valueIdx;
                if ((valueIdx = node.search(key)) >= 0) {
                    byte[] result = node.getValue(valueIdx);
                    node.release();
                    byte[] byArray = result;
                    return byArray;
                }
                if (node.isLeaf()) {
                    node.release();
                    byte[] byArray = null;
                    return byArray;
                }
                Node childNode = node.getChildNode(-valueIdx - 1);
                node.release();
                node = childNode;
            }
        }
        finally {
            this.btreeLock.readLock().unlock();
        }
    }

    public RecordIterator iterateAll() {
        return new RangeIterator(null, null, null, null);
    }

    public RecordIterator iterateRange(byte[] minValue, byte[] maxValue) {
        return new RangeIterator(null, null, minValue, maxValue);
    }

    public RecordIterator iterateValues(byte[] searchKey, byte[] searchMask) {
        return new RangeIterator(searchKey, searchMask, null, null);
    }

    public RecordIterator iterateRangedValues(byte[] searchKey, byte[] searchMask, byte[] minValue, byte[] maxValue) {
        return new RangeIterator(searchKey, searchMask, minValue, maxValue);
    }

    public long getValueCountEstimate() throws IOException {
        int allocatedNodesCount = this.allocatedNodesList.getNodeCount();
        return (long)((double)(allocatedNodesCount * (this.branchFactor - 1)) * 0.5);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getValueCountEstimate(byte[] minValue, byte[] maxValue) throws IOException {
        List<PathSegment> maxValuePath;
        List<PathSegment> minValuePath;
        assert (minValue != null) : "minValue must not be null";
        assert (maxValue != null) : "maxValue must not be null";
        this.btreeLock.readLock().lock();
        try {
            minValuePath = this.getPath(minValue);
            maxValuePath = this.getPath(maxValue);
        }
        finally {
            this.btreeLock.readLock().unlock();
        }
        return this.getValueCountEstimate(minValuePath, maxValuePath);
    }

    private List<PathSegment> getPath(byte[] key) throws IOException {
        assert (key != null) : "key must not be null";
        ArrayList<PathSegment> path = new ArrayList<PathSegment>(this.height());
        Node currentNode = this.readRootNode();
        if (currentNode != null) {
            while (true) {
                int keyIndex = currentNode.search(key);
                path.add(new PathSegment(keyIndex, currentNode.getValueCount()));
                if (keyIndex >= 0 || currentNode.isLeaf()) break;
                Node childNode = currentNode.getChildNode(-keyIndex - 1);
                currentNode.release();
                currentNode = childNode;
            }
            currentNode.release();
        }
        return path;
    }

    private long getValueCountEstimate(List<PathSegment> minValuePath, List<PathSegment> maxValuePath) throws IOException {
        PathSegment ps;
        int i;
        int splitIdx;
        int commonListSize = Math.min(minValuePath.size(), maxValuePath.size());
        if (commonListSize == 0) {
            return 0L;
        }
        PathSegment minNode = null;
        PathSegment maxNode = null;
        for (splitIdx = 0; splitIdx < commonListSize; ++splitIdx) {
            minNode = minValuePath.get(splitIdx);
            maxNode = maxValuePath.get(splitIdx);
            if (minNode.valueIndex != maxNode.valueIndex) break;
        }
        if (splitIdx >= commonListSize) {
            return minNode.valueIndex >= 0 ? 1L : 0L;
        }
        int minValueIndex = minNode.getMinValueIndex();
        int maxValueIndex = maxNode.getMaxValueIndex();
        long valueCount = (long)(maxValueIndex - minValueIndex) * this.getTreeSizeEstimate(splitIdx + 2);
        valueCount += (long)(maxValueIndex - minValueIndex + 1);
        for (i = splitIdx + 1; i < minValuePath.size(); ++i) {
            ps = minValuePath.get(i);
            minValueIndex = ps.getMinValueIndex();
            valueCount += (long)(ps.valueCount - minValueIndex);
            valueCount += (long)(ps.valueCount - minValueIndex) * this.getTreeSizeEstimate(i + 2);
        }
        for (i = splitIdx + 1; i < maxValuePath.size(); ++i) {
            ps = maxValuePath.get(i);
            maxValueIndex = ps.getMaxValueIndex();
            valueCount += (long)(maxValueIndex + 1);
            valueCount += (long)(maxValueIndex + 1) * this.getTreeSizeEstimate(i + 2);
        }
        return valueCount;
    }

    private long getTreeSizeEstimate(int nodeDepth) throws IOException {
        int fanOut = this.branchFactor / 2;
        long valueCount = 0L;
        for (int i = this.height() - nodeDepth; i >= 0; --i) {
            ++valueCount;
            valueCount *= (long)fanOut;
        }
        return valueCount;
    }

    private int height() throws IOException {
        if (this.height >= 0) {
            return this.height;
        }
        int nodeDepth = 0;
        Node currentNode = this.readRootNode();
        if (currentNode != null) {
            nodeDepth = 1;
            while (!currentNode.isLeaf()) {
                Node childNode = currentNode.getChildNode(0);
                currentNode.release();
                currentNode = childNode;
                ++nodeDepth;
            }
            currentNode.release();
        }
        this.height = nodeDepth;
        return this.height;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] insert(byte[] value) throws IOException {
        this.btreeLock.writeLock().lock();
        try {
            Node rootNode = this.readRootNode();
            if (rootNode == null) {
                rootNode = this.createNewNode();
                this.rootNodeID = rootNode.getID();
                this.writeFileHeader();
                this.height = 1;
            }
            InsertResult insertResult = this.insertInTree(value, 0, rootNode);
            if (insertResult.overflowValue != null) {
                Node newRootNode = this.createNewNode();
                newRootNode.setChildNodeID(0, rootNode.getID());
                newRootNode.insertValueNodeIDPair(0, insertResult.overflowValue, insertResult.overflowNodeID);
                this.rootNodeID = newRootNode.getID();
                this.writeFileHeader();
                newRootNode.release();
                if (this.height >= 0) {
                    ++this.height;
                }
            }
            rootNode.release();
            byte[] byArray = insertResult.oldValue;
            return byArray;
        }
        finally {
            this.btreeLock.writeLock().unlock();
        }
    }

    private InsertResult insertInTree(byte[] value, int nodeID, Node node) throws IOException {
        InsertResult insertResult = null;
        int valueIdx = node.search(value);
        if (valueIdx >= 0) {
            insertResult = new InsertResult();
            insertResult.oldValue = node.getValue(valueIdx);
            if (!Arrays.equals(value, insertResult.oldValue)) {
                node.setValue(valueIdx, value);
            }
        } else {
            valueIdx = -valueIdx - 1;
            if (node.isLeaf()) {
                insertResult = this.insertInNode(value, nodeID, valueIdx, node);
            } else {
                Node childNode = node.getChildNode(valueIdx);
                insertResult = this.insertInTree(value, nodeID, childNode);
                childNode.release();
                if (insertResult.overflowValue != null) {
                    byte[] oldValue = insertResult.oldValue;
                    insertResult = this.insertInNode(insertResult.overflowValue, insertResult.overflowNodeID, valueIdx, node);
                    insertResult.oldValue = oldValue;
                }
            }
        }
        return insertResult;
    }

    private InsertResult insertInNode(byte[] value, int nodeID, int valueIdx, Node node) throws IOException {
        InsertResult insertResult = new InsertResult();
        if (node.isFull()) {
            Node newNode = this.createNewNode();
            insertResult.overflowValue = node.splitAndInsert(value, nodeID, valueIdx, newNode);
            insertResult.overflowNodeID = newNode.getID();
            newNode.release();
        } else {
            node.insertValueNodeIDPair(valueIdx, value, nodeID);
        }
        return insertResult;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] remove(byte[] key) throws IOException {
        this.btreeLock.writeLock().lock();
        try {
            byte[] result = null;
            Node rootNode = this.readRootNode();
            if (rootNode != null) {
                result = this.removeFromTree(key, rootNode);
                if (rootNode.isEmpty()) {
                    if (rootNode.isLeaf()) {
                        this.rootNodeID = 0;
                    } else {
                        this.rootNodeID = rootNode.getChildNodeID(0);
                        rootNode.setChildNodeID(0, 0);
                    }
                    this.writeFileHeader();
                    if (this.height >= 0) {
                        --this.height;
                    }
                }
                rootNode.release();
            }
            byte[] byArray = result;
            return byArray;
        }
        finally {
            this.btreeLock.writeLock().unlock();
        }
    }

    private byte[] removeFromTree(byte[] key, Node node) throws IOException {
        byte[] value = null;
        int valueIdx = node.search(key);
        if (valueIdx >= 0) {
            if (node.isLeaf()) {
                value = node.removeValueRight(valueIdx);
            } else {
                value = node.getValue(valueIdx);
                Node leftChildNode = node.getChildNode(valueIdx);
                byte[] largestValue = this.removeLargestValueFromTree(leftChildNode);
                node.setValue(valueIdx, largestValue);
                this.balanceChildNode(node, leftChildNode, valueIdx);
                leftChildNode.release();
            }
        } else if (!node.isLeaf()) {
            valueIdx = -valueIdx - 1;
            Node childNode = node.getChildNode(valueIdx);
            value = this.removeFromTree(key, childNode);
            this.balanceChildNode(node, childNode, valueIdx);
            childNode.release();
        }
        return value;
    }

    private byte[] removeLargestValueFromTree(Node node) throws IOException {
        int nodeValueCount = node.getValueCount();
        if (node.isLeaf()) {
            if (node.isEmpty()) {
                throw new IllegalArgumentException("Trying to remove largest value from an empty node in " + this.getFile());
            }
            return node.removeValueRight(nodeValueCount - 1);
        }
        Node childNode = node.getChildNode(nodeValueCount);
        byte[] value = this.removeLargestValueFromTree(childNode);
        this.balanceChildNode(node, childNode, nodeValueCount);
        childNode.release();
        return value;
    }

    private void balanceChildNode(Node parentNode, Node childNode, int childIdx) throws IOException {
        if (childNode.getValueCount() < this.minValueCount) {
            Node rightSibling;
            Node node = rightSibling = childIdx < parentNode.getValueCount() ? parentNode.getChildNode(childIdx + 1) : null;
            if (rightSibling != null && rightSibling.getValueCount() > this.minValueCount) {
                parentNode.rotateLeft(childIdx, childNode, rightSibling);
            } else {
                Node leftSibling;
                Node node2 = leftSibling = childIdx > 0 ? parentNode.getChildNode(childIdx - 1) : null;
                if (leftSibling != null && leftSibling.getValueCount() > this.minValueCount) {
                    parentNode.rotateRight(childIdx, leftSibling, childNode);
                } else if (leftSibling != null) {
                    leftSibling.mergeWithRightSibling(parentNode.removeValueRight(childIdx - 1), childNode);
                } else {
                    childNode.mergeWithRightSibling(parentNode.removeValueRight(childIdx), rightSibling);
                }
                if (leftSibling != null) {
                    leftSibling.release();
                }
            }
            if (rightSibling != null) {
                rightSibling.release();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clear() throws IOException {
        this.btreeLock.writeLock().lock();
        try {
            Map<Integer, Node> map = this.nodeCache;
            synchronized (map) {
                this.nodeCache.clear();
                this.mruNodes.clear();
            }
            this.nioFile.truncate(16L);
            if (this.rootNodeID != 0) {
                this.rootNodeID = 0;
                this.writeFileHeader();
            }
            this.allocatedNodesList.clear();
        }
        finally {
            this.btreeLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Node createNewNode() throws IOException {
        int newNodeID = this.allocatedNodesList.allocateNode();
        Node node = new Node(newNodeID);
        Map<Integer, Node> map = this.nodeCache;
        synchronized (map) {
            if (this.nodeCache.size() >= 10 && this.mruNodes.size() > 4) {
                this.expelNodeFromCache();
            }
            node.use();
            this.nodeCache.put(node.getID(), node);
        }
        return node;
    }

    Node readRootNode() throws IOException {
        if (this.rootNodeID > 0) {
            return this.readNode(this.rootNodeID);
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Node readNode(int id) throws IOException {
        if (id <= 0) {
            throw new IllegalArgumentException("id must be larger than 0, is: " + id + " in " + this.getFile());
        }
        Map<Integer, Node> map = this.nodeCache;
        synchronized (map) {
            Node node = this.nodeCache.get(id);
            if (node != null) {
                int usageCount = node.use();
                if (usageCount == 1) {
                    this.mruNodes.remove(id);
                }
            } else {
                if (this.nodeCache.size() >= 10 && this.mruNodes.size() > 4) {
                    this.expelNodeFromCache();
                }
                node = new Node(id);
                node.read();
                this.nodeCache.put(id, node);
                node.use();
            }
            return node;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void releaseNode(Node node) throws IOException {
        if (node.isEmpty() && node.isLeaf()) {
            node.write();
            this.nodeCache.remove(node.getID());
            AllocatedNodesList allocatedNodesList = this.allocatedNodesList;
            synchronized (allocatedNodesList) {
                this.allocatedNodesList.freeNode(node.getID());
                int maxNodeID = this.allocatedNodesList.getMaxNodeID();
                if (node.getID() > maxNodeID) {
                    this.nioFile.truncate(this.nodeID2offset(maxNodeID) + (long)this.nodeSize);
                }
            }
        } else {
            this.mruNodes.put(node.getID(), node);
            if (this.nodeCache.size() > 10 && this.mruNodes.size() > 4) {
                this.expelNodeFromCache();
            }
        }
    }

    private void expelNodeFromCache() throws IOException {
        if (!this.mruNodes.isEmpty()) {
            Iterator<Node> iter = this.mruNodes.values().iterator();
            Node lruNode = iter.next();
            if (lruNode.dataChanged()) {
                lruNode.write();
            }
            iter.remove();
            this.nodeCache.remove(lruNode.getID());
        }
    }

    private void writeFileHeader() throws IOException {
        ByteBuffer buf = ByteBuffer.allocate(16);
        buf.put(MAGIC_NUMBER);
        buf.put((byte)1);
        buf.putInt(this.blockSize);
        buf.putInt(this.valueSize);
        buf.putInt(this.rootNodeID);
        buf.rewind();
        this.nioFile.write(buf, 0L);
    }

    private long nodeID2offset(int id) {
        return (long)this.blockSize * (long)id;
    }

    private int offset2nodeID(long offset) {
        return (int)(offset / (long)this.blockSize);
    }

    public static void main(String[] args) throws Exception {
        System.out.println("Running BTree test...");
        if (args.length > 2) {
            BTree.runPerformanceTest(args);
        } else {
            BTree.runDebugTest(args);
        }
        System.out.println("Done.");
    }

    public static void runPerformanceTest(String[] args) throws Exception {
        File dataDir = new File(args[0]);
        String filenamePrefix = args[1];
        int valueCount = Integer.parseInt(args[2]);
        DefaultRecordComparator comparator = new DefaultRecordComparator();
        BTree btree = new BTree(dataDir, filenamePrefix, 501, 13, comparator);
        Random random = new Random(0L);
        byte[] value = new byte[13];
        long startTime = System.currentTimeMillis();
        for (int i = 1; i <= valueCount; ++i) {
            random.nextBytes(value);
            btree.insert(value);
            if (i % 50000 != 0) continue;
            System.out.println("Inserted " + i + " values in " + (System.currentTimeMillis() - startTime) + " ms");
        }
        System.out.println("Iterating over all values in sequential order...");
        startTime = System.currentTimeMillis();
        RecordIterator iter = btree.iterateAll();
        value = iter.next();
        int count = 0;
        while (value != null) {
            ++count;
            value = iter.next();
        }
        iter.close();
        System.out.println("Iteration over " + count + " items finished in " + (System.currentTimeMillis() - startTime) + " ms");
    }

    public static void runDebugTest(String[] args) throws Exception {
        File dataDir = new File(args[0]);
        String filenamePrefix = args[1];
        BTree btree = new BTree(dataDir, filenamePrefix, 28, 1);
        btree.print(System.out);
    }

    public void print(PrintStream out) throws IOException {
        out.println("---contents of BTree file---");
        out.println("Stored parameters:");
        out.println("block size   = " + this.blockSize);
        out.println("value size   = " + this.valueSize);
        out.println("root node ID = " + this.rootNodeID);
        out.println();
        out.println("Derived parameters:");
        out.println("slot size       = " + this.slotSize);
        out.println("branch factor   = " + this.branchFactor);
        out.println("min value count = " + this.minValueCount);
        out.println("node size       = " + this.nodeSize);
        out.println();
        int nodeCount = 0;
        int valueCount = 0;
        ByteBuffer buf = ByteBuffer.allocate(this.nodeSize);
        for (long offset = (long)this.blockSize; offset < this.nioFile.size(); offset += (long)this.blockSize) {
            this.nioFile.read(buf, offset);
            buf.rewind();
            int nodeID = this.offset2nodeID(offset);
            int count = buf.getInt();
            ++nodeCount;
            valueCount += count;
            out.print("node " + nodeID + ": ");
            out.print("count=" + count + " ");
            byte[] value = new byte[this.valueSize];
            for (int i = 0; i < count; ++i) {
                out.print(buf.getInt());
                buf.get(value);
                out.print("[" + ByteArrayUtil.toHexString((byte[])value) + "]");
            }
            out.println(buf.getInt());
            buf.clear();
        }
        out.println("#nodes          = " + nodeCount);
        out.println("#values         = " + valueCount);
        out.println("---end of BTree file---");
    }

    private class RangeIterator
    implements RecordIterator,
    NodeListener {
        private final byte[] searchKey;
        private final byte[] searchMask;
        private final byte[] minValue;
        private final byte[] maxValue;
        private boolean started;
        private Node currentNode;
        private final AtomicBoolean revisitValue = new AtomicBoolean();
        private final LinkedList<Node> parentNodeStack = new LinkedList();
        private final LinkedList<Integer> parentIndexStack = new LinkedList();
        private int currentIdx;

        public RangeIterator(byte[] searchKey, byte[] searchMask, byte[] minValue, byte[] maxValue) {
            this.searchKey = searchKey;
            this.searchMask = searchMask;
            this.minValue = minValue;
            this.maxValue = maxValue;
            this.started = false;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public byte[] next() throws IOException {
            BTree.this.btreeLock.readLock().lock();
            try {
                if (!this.started) {
                    this.started = true;
                    this.findMinimum();
                }
                byte[] value = this.findNext(this.revisitValue.getAndSet(false));
                while (value != null) {
                    if (this.maxValue != null && BTree.this.comparator.compareBTreeValues(this.maxValue, value, 0, value.length) < 0) {
                        this.close();
                        value = null;
                        break;
                    }
                    if (this.searchKey == null || ByteArrayUtil.matchesPattern((byte[])value, (byte[])this.searchMask, (byte[])this.searchKey)) break;
                    value = this.findNext(false);
                }
                byte[] byArray = value;
                return byArray;
            }
            finally {
                BTree.this.btreeLock.readLock().unlock();
            }
        }

        private void findMinimum() throws IOException {
            this.currentNode = BTree.this.readRootNode();
            if (this.currentNode == null) {
                return;
            }
            this.currentNode.register(this);
            this.currentIdx = 0;
            while (true) {
                if (this.minValue != null) {
                    this.currentIdx = this.currentNode.search(this.minValue);
                    if (this.currentIdx >= 0) break;
                    this.currentIdx = -this.currentIdx - 1;
                }
                if (this.currentNode.isLeaf()) break;
                Node childNode = this.currentNode.getChildNode(this.currentIdx);
                this.pushStacks(childNode);
            }
        }

        private byte[] findNext(boolean returnedFromRecursion) throws IOException {
            if (this.currentNode == null) {
                return null;
            }
            if (returnedFromRecursion || this.currentNode.isLeaf()) {
                if (this.currentIdx >= this.currentNode.getValueCount()) {
                    this.popStacks();
                    return this.findNext(true);
                }
                return this.currentNode.getValue(this.currentIdx++);
            }
            Node childNode = this.currentNode.getChildNode(this.currentIdx);
            this.pushStacks(childNode);
            return this.findNext(false);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void set(byte[] value) {
            BTree.this.btreeLock.readLock().lock();
            try {
                if (this.currentNode == null || this.currentIdx > this.currentNode.getValueCount()) {
                    throw new IllegalStateException();
                }
                this.currentNode.setValue(this.currentIdx - 1, value);
            }
            finally {
                BTree.this.btreeLock.readLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() throws IOException {
            BTree.this.btreeLock.readLock().lock();
            try {
                while (this.popStacks()) {
                }
                assert (this.parentNodeStack.isEmpty());
                assert (this.parentIndexStack.isEmpty());
            }
            finally {
                BTree.this.btreeLock.readLock().unlock();
            }
        }

        private void pushStacks(Node newChildNode) {
            newChildNode.register(this);
            this.parentNodeStack.add(this.currentNode);
            this.parentIndexStack.add(this.currentIdx);
            this.currentNode = newChildNode;
            this.currentIdx = 0;
        }

        private boolean popStacks() throws IOException {
            if (this.currentNode == null) {
                return false;
            }
            this.currentNode.deregister(this);
            this.currentNode.release();
            if (!this.parentNodeStack.isEmpty()) {
                this.currentNode = this.parentNodeStack.removeLast();
                this.currentIdx = this.parentIndexStack.removeLast();
                return true;
            }
            this.currentNode = null;
            this.currentIdx = 0;
            return false;
        }

        @Override
        public boolean valueAdded(Node node, int addedIndex) {
            block3: {
                block2: {
                    assert (BTree.this.btreeLock.isWriteLockedByCurrentThread());
                    if (node != this.currentNode) break block2;
                    if (addedIndex >= this.currentIdx) break block3;
                    ++this.currentIdx;
                    break block3;
                }
                for (int i = 0; i < this.parentNodeStack.size(); ++i) {
                    if (node != this.parentNodeStack.get(i)) continue;
                    int parentIdx = this.parentIndexStack.get(i);
                    if (addedIndex >= parentIdx) break;
                    this.parentIndexStack.set(i, parentIdx + 1);
                    break;
                }
            }
            return false;
        }

        @Override
        public boolean valueRemoved(Node node, int removedIndex) {
            block3: {
                block2: {
                    assert (BTree.this.btreeLock.isWriteLockedByCurrentThread());
                    if (node != this.currentNode) break block2;
                    if (removedIndex >= this.currentIdx) break block3;
                    --this.currentIdx;
                    break block3;
                }
                for (int i = 0; i < this.parentNodeStack.size(); ++i) {
                    if (node != this.parentNodeStack.get(i)) continue;
                    int parentIdx = this.parentIndexStack.get(i);
                    if (removedIndex >= parentIdx) break;
                    this.parentIndexStack.set(i, parentIdx - 1);
                    break;
                }
            }
            return false;
        }

        @Override
        public boolean rotatedLeft(Node node, int valueIndex, Node leftChildNode, Node rightChildNode) throws IOException {
            block3: {
                block4: {
                    block2: {
                        if (this.currentNode != node) break block2;
                        if (valueIndex != this.currentIdx - 1) break block3;
                        this.currentIdx = valueIndex;
                        this.revisitValue.set(true);
                        if (!node.isLeaf()) {
                            this.pushStacks(leftChildNode);
                            leftChildNode.use();
                        }
                        break block3;
                    }
                    if (this.currentNode != rightChildNode) break block4;
                    if (this.currentIdx != 0) break block3;
                    this.popStacks();
                    this.currentIdx = valueIndex;
                    this.revisitValue.set(true);
                    break block3;
                }
                for (int i = 0; i < this.parentNodeStack.size(); ++i) {
                    Node stackNode = this.parentNodeStack.get(i);
                    if (stackNode != rightChildNode) continue;
                    int stackIdx = this.parentIndexStack.get(i);
                    if (stackIdx != 0) break;
                    rightChildNode.deregister(this);
                    rightChildNode.release();
                    leftChildNode.use();
                    leftChildNode.register(this);
                    this.parentNodeStack.set(i, leftChildNode);
                    this.parentIndexStack.set(i, leftChildNode.getValueCount());
                    break;
                }
            }
            return false;
        }

        @Override
        public boolean rotatedRight(Node node, int valueIndex, Node leftChildNode, Node rightChildNode) throws IOException {
            for (int i = 0; i < this.parentNodeStack.size(); ++i) {
                Node stackNode = this.parentNodeStack.get(i);
                if (stackNode != leftChildNode) continue;
                int stackIdx = this.parentIndexStack.get(i);
                if (stackIdx != leftChildNode.getValueCount()) break;
                leftChildNode.deregister(this);
                leftChildNode.release();
                rightChildNode.use();
                rightChildNode.register(this);
                this.parentNodeStack.set(i, rightChildNode);
                this.parentIndexStack.set(i, 0);
                break;
            }
            return false;
        }

        @Override
        public boolean nodeSplit(Node node, Node newNode, int medianIdx) throws IOException {
            boolean deregister;
            block3: {
                block2: {
                    assert (BTree.this.btreeLock.isWriteLockedByCurrentThread());
                    deregister = false;
                    if (node != this.currentNode) break block2;
                    if (this.currentIdx <= medianIdx) break block3;
                    this.currentNode.release();
                    deregister = true;
                    newNode.use();
                    newNode.register(this);
                    this.currentNode = newNode;
                    this.currentIdx -= medianIdx + 1;
                    break block3;
                }
                for (int i = 0; i < this.parentNodeStack.size(); ++i) {
                    Node parentNode = this.parentNodeStack.get(i);
                    if (node != parentNode) continue;
                    int parentIdx = this.parentIndexStack.get(i);
                    if (parentIdx <= medianIdx) break;
                    parentNode.release();
                    deregister = true;
                    newNode.use();
                    newNode.register(this);
                    this.parentNodeStack.set(i, newNode);
                    this.parentIndexStack.set(i, parentIdx - medianIdx - 1);
                    break;
                }
            }
            return deregister;
        }

        @Override
        public boolean nodeMergedWith(Node sourceNode, Node targetNode, int mergeIdx) throws IOException {
            assert (BTree.this.btreeLock.isWriteLockedByCurrentThread());
            boolean deregister = false;
            if (sourceNode == this.currentNode) {
                this.currentNode.release();
                deregister = true;
                targetNode.use();
                targetNode.register(this);
                this.currentNode = targetNode;
                this.currentIdx += mergeIdx;
            } else {
                for (int i = 0; i < this.parentNodeStack.size(); ++i) {
                    Node parentNode = this.parentNodeStack.get(i);
                    if (sourceNode != parentNode) continue;
                    parentNode.release();
                    deregister = true;
                    targetNode.use();
                    targetNode.register(this);
                    this.parentNodeStack.set(i, targetNode);
                    this.parentIndexStack.set(i, mergeIdx + this.parentIndexStack.get(i));
                    break;
                }
            }
            return deregister;
        }
    }

    private static interface NodeListener {
        public boolean valueAdded(Node var1, int var2);

        public boolean valueRemoved(Node var1, int var2);

        public boolean rotatedLeft(Node var1, int var2, Node var3, Node var4) throws IOException;

        public boolean rotatedRight(Node var1, int var2, Node var3, Node var4) throws IOException;

        public boolean nodeSplit(Node var1, Node var2, int var3) throws IOException;

        public boolean nodeMergedWith(Node var1, Node var2, int var3) throws IOException;
    }

    class Node {
        private final int id;
        private final byte[] data;
        private int valueCount;
        private int usageCount;
        private boolean dataChanged;
        private final LinkedList<NodeListener> listeners = new LinkedList();

        public Node(int id) {
            if (id <= 0) {
                throw new IllegalArgumentException("id must be larger than 0, is: " + id + " in " + BTree.this.getFile());
            }
            this.id = id;
            this.valueCount = 0;
            this.usageCount = 0;
            this.data = new byte[BTree.this.nodeSize + BTree.this.slotSize];
        }

        public int getID() {
            return this.id;
        }

        public String toString() {
            return "node " + this.getID();
        }

        public boolean isLeaf() {
            return this.getChildNodeID(0) == 0;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public int use() {
            Map map = BTree.this.nodeCache;
            synchronized (map) {
                return ++this.usageCount;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void release() throws IOException {
            Map map = BTree.this.nodeCache;
            synchronized (map) {
                assert (this.usageCount > 0) : "Releasing node while usage count is " + this.usageCount;
                --this.usageCount;
                if (this.usageCount == 0) {
                    BTree.this.releaseNode(this);
                }
            }
        }

        public int getUsageCount() {
            return this.usageCount;
        }

        public boolean dataChanged() {
            return this.dataChanged;
        }

        public int getValueCount() {
            return this.valueCount;
        }

        public int getNodeCount() {
            if (this.isLeaf()) {
                return 0;
            }
            return this.valueCount + 1;
        }

        public boolean isEmpty() {
            return this.valueCount == 0;
        }

        public boolean isFull() {
            return this.valueCount == BTree.this.branchFactor - 1;
        }

        public byte[] getValue(int valueIdx) {
            assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
            assert (valueIdx < this.valueCount) : "valueIdx out of range (" + valueIdx + " >= " + this.valueCount + ")";
            return ByteArrayUtil.get((byte[])this.data, (int)this.valueIdx2offset(valueIdx), (int)BTree.this.valueSize);
        }

        public void setValue(int valueIdx, byte[] value) {
            assert (value != null) : "value must not be null";
            assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
            assert (valueIdx < this.valueCount) : "valueIdx out of range (" + valueIdx + " >= " + this.valueCount + ")";
            ByteArrayUtil.put((byte[])value, (byte[])this.data, (int)this.valueIdx2offset(valueIdx));
            this.dataChanged = true;
        }

        public byte[] removeValueRight(int valueIdx) {
            assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
            assert (valueIdx < this.valueCount) : "valueIdx out of range (" + valueIdx + " >= " + this.valueCount + ")";
            byte[] value = this.getValue(valueIdx);
            int endOffset = this.valueIdx2offset(this.valueCount);
            if (valueIdx < this.valueCount - 1) {
                this.shiftData(this.valueIdx2offset(valueIdx + 1), endOffset, -BTree.this.slotSize);
            }
            this.clearData(endOffset - BTree.this.slotSize, endOffset);
            this.setValueCount(--this.valueCount);
            this.dataChanged = true;
            this.notifyValueRemoved(valueIdx);
            return value;
        }

        public byte[] removeValueLeft(int valueIdx) {
            assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
            assert (valueIdx < this.valueCount) : "valueIdx out of range (" + valueIdx + " >= " + this.valueCount + ")";
            byte[] value = this.getValue(valueIdx);
            int endOffset = this.valueIdx2offset(this.valueCount);
            this.shiftData(this.nodeIdx2offset(valueIdx + 1), endOffset, -BTree.this.slotSize);
            this.clearData(endOffset - BTree.this.slotSize, endOffset);
            this.setValueCount(--this.valueCount);
            this.dataChanged = true;
            this.notifyValueRemoved(valueIdx);
            return value;
        }

        public int getChildNodeID(int nodeIdx) {
            assert (nodeIdx >= 0) : "nodeIdx must be positive, is: " + nodeIdx;
            assert (nodeIdx <= this.valueCount) : "nodeIdx out of range (" + nodeIdx + " > " + this.valueCount + ")";
            return ByteArrayUtil.getInt((byte[])this.data, (int)this.nodeIdx2offset(nodeIdx));
        }

        public void setChildNodeID(int nodeIdx, int nodeID) {
            assert (nodeIdx >= 0) : "nodeIdx must not be negative, is: " + nodeIdx;
            assert (nodeIdx <= this.valueCount) : "nodeIdx out of range (" + nodeIdx + " > " + this.valueCount + ")";
            assert (nodeID >= 0) : "nodeID must not be negative, is: " + nodeID;
            ByteArrayUtil.putInt((int)nodeID, (byte[])this.data, (int)this.nodeIdx2offset(nodeIdx));
            this.dataChanged = true;
        }

        public Node getChildNode(int nodeIdx) throws IOException {
            assert (nodeIdx >= 0) : "nodeIdx must be positive, is: " + nodeIdx;
            assert (nodeIdx <= this.valueCount) : "nodeIdx out of range (" + nodeIdx + " > " + this.valueCount + ")";
            int childNodeID = this.getChildNodeID(nodeIdx);
            return BTree.this.readNode(childNodeID);
        }

        public int search(byte[] key) {
            int low = 0;
            int high = this.valueCount - 1;
            while (low <= high) {
                int mid = low + high >> 1;
                int diff = BTree.this.comparator.compareBTreeValues(key, this.data, this.valueIdx2offset(mid), BTree.this.valueSize);
                if (diff < 0) {
                    high = mid - 1;
                    continue;
                }
                if (diff > 0) {
                    low = mid + 1;
                    continue;
                }
                return mid;
            }
            return -low - 1;
        }

        public void insertValueNodeIDPair(int valueIdx, byte[] value, int nodeID) {
            assert (valueIdx >= 0) : "valueIdx must be positive, is: " + valueIdx;
            assert (valueIdx <= this.valueCount) : "valueIdx out of range (" + valueIdx + " > " + this.valueCount + ")";
            assert (value != null) : "value must not be null";
            assert (nodeID >= 0) : "nodeID must not be negative, is: " + nodeID;
            int offset = this.valueIdx2offset(valueIdx);
            if (valueIdx < this.valueCount) {
                this.shiftData(offset, this.valueIdx2offset(this.valueCount), BTree.this.slotSize);
            }
            ByteArrayUtil.put((byte[])value, (byte[])this.data, (int)offset);
            ByteArrayUtil.putInt((int)nodeID, (byte[])this.data, (int)(offset + BTree.this.valueSize));
            this.setValueCount(++this.valueCount);
            this.notifyValueAdded(valueIdx);
            this.dataChanged = true;
        }

        public void insertNodeIDValuePair(int nodeIdx, int nodeID, byte[] value) {
            assert (nodeIdx >= 0) : "nodeIdx must not be negative, is: " + nodeIdx;
            assert (nodeIdx <= this.valueCount) : "nodeIdx out of range (" + nodeIdx + " > " + this.valueCount + ")";
            assert (nodeID >= 0) : "nodeID must not be negative, is: " + nodeID;
            assert (value != null) : "value must not be null";
            int offset = this.nodeIdx2offset(nodeIdx);
            this.shiftData(offset, this.valueIdx2offset(this.valueCount), BTree.this.slotSize);
            ByteArrayUtil.putInt((int)nodeID, (byte[])this.data, (int)offset);
            ByteArrayUtil.put((byte[])value, (byte[])this.data, (int)(offset + 4));
            this.setValueCount(++this.valueCount);
            this.notifyValueAdded(nodeIdx);
            this.dataChanged = true;
        }

        public byte[] splitAndInsert(byte[] newValue, int newNodeID, int newValueIdx, Node newNode) throws IOException {
            this.insertValueNodeIDPair(newValueIdx, newValue, newNodeID);
            assert (this.valueCount == BTree.this.branchFactor) : "Node contains " + this.valueCount + " values, expected " + BTree.access$500(BTree.this);
            int medianIdx = BTree.this.branchFactor / 2;
            int medianOffset = this.valueIdx2offset(medianIdx);
            int splitOffset = medianOffset + BTree.this.valueSize;
            System.arraycopy(this.data, splitOffset, newNode.data, 4, this.data.length - splitOffset);
            byte[] medianValue = this.getValue(medianIdx);
            this.clearData(medianOffset, this.data.length);
            this.setValueCount(medianIdx);
            newNode.setValueCount(BTree.this.branchFactor - medianIdx - 1);
            newNode.dataChanged = true;
            this.notifyNodeSplit(newNode, medianIdx);
            return medianValue;
        }

        public void mergeWithRightSibling(byte[] medianValue, Node rightSibling) throws IOException {
            assert (this.valueCount + rightSibling.getValueCount() + 1 < BTree.this.branchFactor) : "Nodes contain too many values to be merged; left: " + this.valueCount + "; right: " + rightSibling.getValueCount();
            this.insertValueNodeIDPair(this.valueCount, medianValue, 0);
            int rightIdx = this.valueCount;
            System.arraycopy(rightSibling.data, 4, this.data, this.nodeIdx2offset(rightIdx), this.valueIdx2offset(rightSibling.valueCount) - 4);
            this.setValueCount(this.valueCount + rightSibling.valueCount);
            rightSibling.clearData(4, this.valueIdx2offset(rightSibling.valueCount));
            rightSibling.setValueCount(0);
            rightSibling.dataChanged = true;
            rightSibling.notifyNodeMerged(this, rightIdx);
        }

        public void rotateLeft(int valueIdx, Node leftChildNode, Node rightChildNode) throws IOException {
            leftChildNode.insertValueNodeIDPair(leftChildNode.getValueCount(), this.getValue(valueIdx), rightChildNode.getChildNodeID(0));
            this.setValue(valueIdx, rightChildNode.removeValueLeft(0));
            this.notifyRotatedLeft(valueIdx, leftChildNode, rightChildNode);
        }

        public void rotateRight(int valueIdx, Node leftChildNode, Node rightChildNode) throws IOException {
            rightChildNode.insertNodeIDValuePair(0, leftChildNode.getChildNodeID(leftChildNode.getValueCount()), this.getValue(valueIdx - 1));
            this.setValue(valueIdx - 1, leftChildNode.removeValueRight(leftChildNode.getValueCount() - 1));
            this.notifyRotatedRight(valueIdx, leftChildNode, rightChildNode);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void register(NodeListener listener) {
            LinkedList<NodeListener> linkedList = this.listeners;
            synchronized (linkedList) {
                assert (!this.listeners.contains(listener));
                this.listeners.add(listener);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void deregister(NodeListener listener) {
            LinkedList<NodeListener> linkedList = this.listeners;
            synchronized (linkedList) {
                assert (this.listeners.contains(listener));
                this.listeners.remove(listener);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void notifyValueAdded(int index) {
            LinkedList<NodeListener> linkedList = this.listeners;
            synchronized (linkedList) {
                Iterator iter = this.listeners.iterator();
                while (iter.hasNext()) {
                    if (!((NodeListener)iter.next()).valueAdded(this, index)) continue;
                    iter.remove();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void notifyValueRemoved(int index) {
            LinkedList<NodeListener> linkedList = this.listeners;
            synchronized (linkedList) {
                Iterator iter = this.listeners.iterator();
                while (iter.hasNext()) {
                    if (!((NodeListener)iter.next()).valueRemoved(this, index)) continue;
                    iter.remove();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void notifyRotatedLeft(int index, Node leftChildNode, Node rightChildNode) throws IOException {
            LinkedList<NodeListener> linkedList = this.listeners;
            synchronized (linkedList) {
                Iterator iter = this.listeners.iterator();
                while (iter.hasNext()) {
                    if (!((NodeListener)iter.next()).rotatedLeft(this, index, leftChildNode, rightChildNode)) continue;
                    iter.remove();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void notifyRotatedRight(int index, Node leftChildNode, Node rightChildNode) throws IOException {
            LinkedList<NodeListener> linkedList = this.listeners;
            synchronized (linkedList) {
                Iterator iter = this.listeners.iterator();
                while (iter.hasNext()) {
                    if (!((NodeListener)iter.next()).rotatedRight(this, index, leftChildNode, rightChildNode)) continue;
                    iter.remove();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void notifyNodeSplit(Node rightNode, int medianIdx) throws IOException {
            LinkedList<NodeListener> linkedList = this.listeners;
            synchronized (linkedList) {
                Iterator iter = this.listeners.iterator();
                while (iter.hasNext()) {
                    boolean deregister = ((NodeListener)iter.next()).nodeSplit(this, rightNode, medianIdx);
                    if (!deregister) continue;
                    iter.remove();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void notifyNodeMerged(Node targetNode, int mergeIdx) throws IOException {
            LinkedList<NodeListener> linkedList = this.listeners;
            synchronized (linkedList) {
                Iterator iter = this.listeners.iterator();
                while (iter.hasNext()) {
                    boolean deregister = ((NodeListener)iter.next()).nodeMergedWith(this, targetNode, mergeIdx);
                    if (!deregister) continue;
                    iter.remove();
                }
            }
        }

        public void read() throws IOException {
            ByteBuffer buf = ByteBuffer.wrap(this.data);
            buf.limit(BTree.this.nodeSize);
            int bytesRead = BTree.this.nioFile.read(buf, BTree.this.nodeID2offset(this.id));
            assert (bytesRead == BTree.this.nodeSize) : "Read operation didn't read the entire node (" + bytesRead + " of " + BTree.access$100(BTree.this) + " bytes)";
            this.valueCount = ByteArrayUtil.getInt((byte[])this.data, (int)0);
        }

        public void write() throws IOException {
            ByteBuffer buf = ByteBuffer.wrap(this.data);
            buf.limit(BTree.this.nodeSize);
            int bytesWritten = BTree.this.nioFile.write(buf, BTree.this.nodeID2offset(this.id));
            assert (bytesWritten == BTree.this.nodeSize) : "Write operation didn't write the entire node (" + bytesWritten + " of " + BTree.access$100(BTree.this) + " bytes)";
            this.dataChanged = false;
        }

        private void shiftData(int startOffset, int endOffset, int shift) {
            System.arraycopy(this.data, startOffset, this.data, startOffset + shift, endOffset - startOffset);
        }

        private void clearData(int startOffset, int endOffset) {
            Arrays.fill(this.data, startOffset, endOffset, (byte)0);
        }

        private void setValueCount(int valueCount) {
            this.valueCount = valueCount;
            ByteArrayUtil.putInt((int)valueCount, (byte[])this.data, (int)0);
        }

        private int valueIdx2offset(int id) {
            return 8 + id * BTree.this.slotSize;
        }

        private int nodeIdx2offset(int id) {
            return 4 + id * BTree.this.slotSize;
        }
    }

    private static class InsertResult {
        byte[] oldValue = null;
        byte[] overflowValue = null;
        int overflowNodeID = 0;

        private InsertResult() {
        }
    }

    private static class PathSegment {
        public final int valueIndex;
        public final int valueCount;

        public PathSegment(int valueIndex, int valueCount) {
            this.valueIndex = valueIndex;
            this.valueCount = valueCount;
        }

        public int getMinValueIndex() {
            if (this.valueIndex < 0) {
                return -this.valueIndex - 1;
            }
            return this.valueIndex;
        }

        public int getMaxValueIndex() {
            if (this.valueIndex < 0) {
                return -this.valueIndex - 2;
            }
            return this.valueIndex;
        }

        public String toString() {
            return this.valueIndex + ":" + this.valueCount;
        }
    }
}

