from typing import Optional, Any
import abc
import json
from urllib import parse

from .config import Config
from .utils import Utils


class Entity(abc.ABC):
    """Represents the single entity like the static page or the blog post.

    Attributes:
        template (str): Template that is used for the content generation.
        title (Optional[str]): Title (or name) of the page (meta tag).
        description (Optional[str]): Description of the page (meta tag).
        keywords (Optional[str]): Keywords of the page (meta tag).
        url_alias (Optional[str]): Alias for the main page (like my-page).
            If None, page is considered to be the main page (homepage).
        template_parameters (dict[str, Any]): All other additional
            parameters that are passed to the template engine.

        url_list (url_list: list[Optional[str]]): List of URLs that are
            generated by the entity (None refers to homepage).
    """
    # Define JSON encoder for the object
    JSON_ENCODER: json.JSONEncoder = json.JSONEncoder

    def __init__(
        self,
        template: str,
        title: Optional[str] = None,
        description: Optional[str] = None,
        keywords: Optional[str] = None,
        url_alias: Optional[str] = None,
        template_parameters: dict[str, Any] = None
    ):
        """Create the new entity.

        Args:
            template (str): Template that is used for the content generation.
            title (Optional[str]): Title (or name) of the page (meta tag).
            description (Optional[str]): Description of the page (meta tag).
            keywords (Optional[str]): Keywords of the page (meta tag).
            url_alias (Optional[str]): Alias for the main page (like my-page).
                If None, page is considered to be the main page (homepage).
            template_parameters (dict[str, Any]): All other additional
                parameters that are passed to the template engine.
        """
        self.template: str = template
        self.description: Optional[str] = description
        self.keywords: Optional[str] = keywords
        if url_alias and parse.quote(url_alias) != url_alias.lower():
            raise AttributeError("Only URL aliases without any special "
                                 "character are supported")
        self.url_alias: Optional[str] = url_alias
        self.title: Optional[str] = title
        if template_parameters:
            self.template_parameters: dict[str, object] = \
                template_parameters
        else:
            self.template_parameters: dict[str, object] = dict()

        self.url_list: list[Optional[str]] = []

    @abc.abstractmethod
    def generate_page(self,
                      url: str,
                      additional_tags: Optional[dict[str, Any]] = None) -> str:
        """Generate the content for the particular URL.

        Args:
            url (str): URL defining what content should be generated.
            additional_tags (Optional[dict[str, Any]]): Definition of tags
                that might be relevant for given scope.

        Returns:
            str: Content of the page.
        """
        raise NotImplementedError

    @property
    def additional_tags(self) -> dict[str, Any]:
        """Returns additional tags additional_tags = definition of tags
        that might be relevant for given scope.
        """
        return self._additional_tags

    @property
    def page_title(self) -> str:
        """Generate the page title in the logic:
            SITE_NAME + SEPARATOR + TITLE

        Returns:
            str: Generated page title as a meta tag.
        """
        title = self.title
        if title is None and self.url_alias is None:
            # If the page is homepage and title is not set-up
            title = Config.site_title_homepage
        elif title is None or len(title) == 0:
            # If the title is not set-up, return just root title
            return Config.site_title
        # Return string in the logic SITE_NAME + SEPARATOR + TITLE
        return "".join(
            [Config.site_title, Config.site_title_separator, title]
        )

    @property
    def page_name(self) -> Optional[str]:
        """Get page name.

        Returns:
            Optional[str]: Page name of None.
        """
        if self.title is None and self.url_alias is None:
            # If the page is homepage and title is not set-up
            return Config.site_title_homepage
        elif self.title is None or len(self.title) == 0:
            # If the title is not set-up, return just root title
            return None
        return self.title

    @property
    def url(self):
        """Generate the URL leading to the file (with file suffix)"""
        return Utils.generate_file_path(self.url_alias)

    @property
    def json(self) -> str:
        """Export entity to JSON. Uses custom defined encoder.

        Returns:
            str: JSON definition
        """
        return json.dumps(self.dictionary, cls=self.JSON_ENCODER)

    @property
    def dictionary(self) -> dict:
        """Export entity to JSON serializable dictionary.

        Returns:
            dict: dictionary definition (JSON serializable).
        """

        # Skip following variables:
        skip_variables = {'url_list'}

        json_def = {}
        for _var in vars(self):
            if _var not in skip_variables:
                json_def[_var] = getattr(self, _var)
        # Serialize the object type name
        json_def['object_type'] = type(self).__name__
        return json_def
