# -*- coding: utf-8 -*-
"""
:Author: HuangJianYi
:Date: 2020-05-12 20:11:48
@LastEditTime: 2021-07-20 09:49:21
@LastEditors: HuangJianYi
:description: 
"""
from seven_cloudapp_frame.libs.customize.seven_helper import *
from seven_cloudapp_frame.models.db_models.asset.asset_log_model import *
from seven_cloudapp_frame.models.db_models.asset.asset_only_model import *
from seven_cloudapp_frame.models.db_models.user.user_asset_model import *
from seven_cloudapp_frame.models.db_models.asset.asset_inventory_model import *
from seven_cloudapp_frame.models.db_models.asset.asset_warn_notice_model import *
from seven_cloudapp_frame.models.seven_model import *


class AssetBase():
    """
    :description: 资产管理基类
    """
    def __init__(self,context):
        self.context = context

    def _get_user_asset_id(self,act_id,user_id,asset_type,asset_object_id):
        """
        :description: 生成用户资产唯一标识
        :param act_id：活动标识
        :param user_id：用户标识
        :param asset_type：资产类型(1-次数2-积分3-价格档位)
        :param asset_object_id：对象标识
        :return: 用户资产唯一标识
        :last_editors: HuangJianYi
        """
        if not act_id or not user_id or not asset_type:
            return 0
        return CryptoHelper.md5_encrypt_int(f"{act_id}_{user_id}_{asset_type}_{asset_object_id}")

    def _get_asset_check_code(self, id, asset_value, sign_key):
        """
        :description: 生成用户资产校验码
        :param 用户资产唯一标识
        :param asset_value：当前资产值
        :param sign_key：签名key,目前使用app_id作为签名key
        :return: 用户资产校验码
        :last_editors: HuangJianYi
        """
        if not id or not asset_value:
            return ""
        return CryptoHelper.md5_encrypt(f"{id}_{asset_value}", sign_key)

    def modify_user_asset(self, app_id, act_id, module_id, user_id, open_id, user_nick, asset_type, asset_value, asset_object_id, source_type, source_object_id, source_object_name, log_title, only_id="", info_json={}, sub_table=None):
        """
        :description: 变更资产
        :param act_id：活动标识
        :param module_id：模块标识，没有填0
        :param user_id：用户标识
        :param open_id：open_id
        :param user_nick：昵称
        :param asset_type：资产类型(1-次数2-积分3-价格档位)
        :param asset_value：修改资产值
        :param asset_object_id：资产对象标识
        :param source_type：来源类型（1-购买2-任务3-手动配置4-抽奖5-回购）
        :param source_object_id：来源对象标识(比如来源类型是任务则对应任务类型)
        :param source_object_name：来源对象名称(比如来源类型是任务则对应任务名称)
        :param log_title：资产流水标题
        :param only_id:唯一标识(用于并发操作时校验避免重复操作)由业务方定义传入
        :param info_json：资产流水详情，用于存放业务方自定义字典
        :param sub_table：分表名称
        :return: 返回实体InvokeResultData
        :last_editors: HuangJianYi
        """

        invoke_result_data = InvokeResultData()

        if not act_id or not user_id or not asset_type or not asset_value:
            invoke_result_data.success = False
            invoke_result_data.error_code = "param_error"
            invoke_result_data.error_message = "参数不能为空或等于0"
            return invoke_result_data

        if int(asset_type) == 3 and not asset_object_id:
            invoke_result_data.success = False
            invoke_result_data.error_code = "param_error"
            invoke_result_data.error_message = "资产类型为价格档位,参数asset_object_id不能为空或等于0"
            return invoke_result_data

        user_asset_id = self._get_user_asset_id(act_id, user_id, asset_type, asset_object_id)
        if user_asset_id == 0:
            invoke_result_data.success = False
            invoke_result_data.error_code = "error"
            invoke_result_data.error_message = "修改失败"
            return invoke_result_data
        #如果only_id已经存在，直接在redis进行拦截,减少数据库的请求，时限1天
        redis_init = SevenHelper.redis_init()
        only_redis_key = ""
        if only_id:
            only_redis_key = "asset_only_list:" + app_id + "_" + str(SevenHelper.get_now_day_int())
            if redis_init.hexists(only_redis_key, only_id):
                invoke_result_data.success = False
                invoke_result_data.error_code = "error"
                invoke_result_data.error_message = "only_id已经存在"
                return invoke_result_data

        db_transaction = DbTransaction(db_config_dict=config.get_value("db_cloudapp"))
        user_asset_model = UserAssetModel(db_transaction=db_transaction, sub_table=sub_table, context=self.context)
        asset_log_model = AssetLogModel(db_transaction=db_transaction, sub_table=sub_table, context=self.context)
        asset_only_model = AssetOnlyModel(db_transaction=db_transaction, sub_table=sub_table, context=self.context)
        asset_inventory_model = AssetInventoryModel(db_transaction=db_transaction, sub_table=sub_table, context=self.context)

        acquire_lock_name = f"userasset_{user_asset_id}"
        acquire_lock_status, identifier = SevenHelper.reids_acquire_lock(acquire_lock_name)
        if acquire_lock_status == False:
            invoke_result_data.success = False
            invoke_result_data.error_code = "acquire_lock"
            invoke_result_data.error_message = "请求超时,请稍后再试"
            return invoke_result_data

        try:
            now_day_int = SevenHelper.get_now_day_int()
            now_datetime = SevenHelper.get_now_datetime()
            old_user_asset_id = 0
            history_asset_value = 0

            user_asset = user_asset_model.get_entity_by_id(user_asset_id)
            if user_asset:
                if user_asset.asset_value + asset_value < 0:
                    invoke_result_data.success = False
                    invoke_result_data.error_code = "no_enough"
                    invoke_result_data.error_message = "变更后的资产不能为负数"
                    return invoke_result_data

                old_user_asset_id = user_asset_id
                history_asset_value = user_asset.asset_value
            else:
                user_asset = UserAsset()
                user_asset.id = user_asset_id
                user_asset.app_id = app_id
                user_asset.act_id = act_id
                user_asset.user_id = user_id
                user_asset.open_id = open_id
                user_asset.user_nick = user_nick
                user_asset.asset_type = asset_type
                user_asset.asset_object_id = asset_object_id
                user_asset.create_date = now_datetime

            user_asset.asset_value += asset_value
            user_asset.asset_check_code = self._get_asset_check_code(user_asset_id, app_id)
            user_asset.modify_date = now_datetime

            old_asset_inventory_id = 0
            asset_inventory_id = CryptoHelper.md5_encrypt_int(f"{act_id}_{user_id}_{asset_type}_{asset_object_id}_{now_day_int}")
            asset_inventory = asset_inventory_model.get_entity_by_id(asset_inventory_id)
            asset_inventory_update_sql = f"process_count=0,now_value={user_asset.asset_value}"
            if asset_inventory:
                old_asset_inventory_id = asset_inventory_id
                if asset_value > 0:
                    asset_inventory_update_sql += f",inc_value=inc_value+{asset_value}"
                else:
                    asset_inventory_update_sql += f",dec_value=dec_value+{asset_value}"
            else:
                asset_inventory = AssetInventory()
                asset_inventory.id = asset_inventory_id
                asset_inventory.app_id = app_id
                asset_inventory.act_id = act_id
                asset_inventory.user_id = user_id
                asset_inventory.open_id = open_id
                asset_inventory.user_nick = user_nick
                asset_inventory.asset_type = asset_type
                asset_inventory.asset_object_id = asset_object_id
                if asset_value > 0:
                    asset_inventory.inc_value += asset_value
                else:
                    asset_inventory.dec_value += asset_value
                asset_inventory.history_value = history_asset_value
                asset_inventory.now_value = user_asset.asset_value
                asset_inventory.create_date = now_datetime
                asset_inventory.create_day = now_day_int


            asset_log = AssetLog()
            asset_log.id = CryptoHelper.md5_encrypt_int(f"{act_id}_{user_id}_{UUIDHelper.get_uuid()}")
            asset_log.app_id = app_id
            asset_log.act_id = act_id
            asset_log.module_id = module_id
            asset_log.user_id = user_id
            asset_log.open_id = open_id
            asset_log.user_nick = user_nick
            asset_log.log_title = log_title
            asset_log.info_json = info_json if not info_json else {}
            asset_log.asset_type = asset_type
            asset_log.asset_object_id = asset_object_id
            asset_log.source_type = source_type
            asset_log.source_object_id = source_object_id
            asset_log.source_object_name = source_object_name
            asset_log.only_id = only_id
            asset_log.operate_type = 0 if asset_value > 0 else 1
            asset_log.operate_value = asset_value
            asset_log.history_value = history_asset_value
            asset_log.now_value = user_asset.asset_value
            asset_log.create_date = now_datetime
            asset_log.create_day = now_day_int


            if only_id:
                asset_only = AssetOnly()
                asset_only.id = CryptoHelper.md5_encrypt_int(f"{act_id}_{user_id}_{only_id}")
                asset_only.app_id = app_id
                asset_only.act_id = act_id
                asset_only.user_id = user_id
                asset_only.open_id = open_id
                asset_only.only_id = only_id
                asset_only.create_date = now_datetime


            db_transaction.begin_transaction()

            if old_user_asset_id != 0 :
                user_asset_model.update_entity(user_asset,"asset_value,asset_check_code,modify_date")
            else:
                user_asset_model.add_entity(user_asset)
            if old_asset_inventory_id != 0 :
                asset_inventory_model.update_table(asset_inventory_update_sql,"id=%s",params=[old_asset_inventory_id])
            else:
                asset_inventory_model.add_entity(asset_inventory)
            if only_id:
                asset_only_model.add_entity(asset_only)

            asset_log_model.add_entity(asset_log)

            db_transaction.commit_transaction()

            if only_id:
                redis_init.hset(only_redis_key, only_id, 1)
                redis_init.expire(only_redis_key, 24*3600)

            invoke_result_data.data = user_asset.asset_value

        except Exception as ex:
            db_transaction.rollback_transaction()
            self.context.logging_link_error("【变更资产】" + str(ex))
            invoke_result_data.success = False
            invoke_result_data.error_code = "exception"
            invoke_result_data.error_message = "系统繁忙,请稍后再试"
        finally:
            SevenHelper.reids_release_lock(acquire_lock_name,identifier)

        return invoke_result_data

    def get_user_asset_list(self, app_id, act_id, user_id, sub_table=None):
        """
        :description: 获取用户资产列表
        :param app_id：应用标识
        :param act_id：活动标识
        :param user_id：用户标识
        :param sub_table：分表名称
        :return: 返回list
        :last_editors: HuangJianYi
        """
        if not act_id or not user_id:
            return []
        user_asset_model = UserAssetModel(sub_table=sub_table, context=self.context)
        user_asset_dict_list = user_asset_model.get_dict_list("act_id=%s and user_id=%s", params=[act_id, user_id])
        if len(user_asset_dict_list)>0:
            for user_asset_dict in user_asset_dict_list:
                if user_asset_dict["app_id"] != str(app_id):
                    user_asset_dict["asset_value"] = 0
                if self._get_asset_check_code(user_asset_dict["id"],user_asset_dict["asset_value"],app_id) != user_asset_dict["asset_check_code"]:
                    user_asset_dict["asset_value"] = 0
        return user_asset_dict_list

    def get_user_asset(self, app_id, act_id, user_id, asset_type, asset_object_id, sub_table=None):
        """
        :description: 获取具体的用户资产
        :param app_id：应用标识
        :param act_id：活动标识
        :param user_id：用户标识
        :param asset_type：资产类型(1-次数2-积分3-价格档位)
        :param asset_object_id：资产对象标识
        :param sub_table：分表名称
        :return: 返回list
        :last_editors: HuangJianYi
        """
        if not act_id or not user_id or not asset_type:
            return None
        user_asset_model = UserAssetModel(sub_table=sub_table, context=self.context)
        user_asset_id = self._get_user_asset_id(act_id, user_id, asset_type, asset_object_id)
        user_asset_dict = user_asset_model.get_dict_by_id(user_asset_id)
        if user_asset_dict:
            if user_asset_dict["app_id"] != str(app_id):
                user_asset_dict["asset_value"] = 0
            if self._get_asset_check_code(user_asset_dict["id"], user_asset_dict["asset_value"], app_id) != user_asset_dict["asset_check_code"]:
                user_asset_dict["asset_value"] = 0
        return user_asset_dict

    def get_asset_log_list(self, app_id, act_id, user_id, asset_type, page_size=0, page_index=10, asset_object_id="", start_date="", end_date="", user_nick="", open_id="", source_type=0, source_object_id="", field="*", sub_table=None):
        """
        :description: 获取用户资产流水记录
        :param app_id：应用标识
        :param act_id：活动标识
        :param user_id：用户标识
        :param asset_type：资产类型(1-次数2-积分3-价格档位)
        :param page_size：条数
        :param page_index：页数
        :param asset_object_id：资产对象标识
        :param start_date：开始时间
        :param end_date：结束时间
        :param user_nick：昵称
        :param open_id：open_id
        :param source_type：来源类型（1-购买2-任务3-手动配置4-抽奖5-回购）
        :param source_object_id：来源对象标识(比如来源类型是任务则对应任务类型)
        :param field：查询字段
        :param sub_table：分表名称
        :return: 返回PageInfo
        :last_editors: HuangJianYi
        """
        page_info = PageInfo(page_index, page_size, 0, [])
        if not act_id or asset_type <= 0:
            return page_info

        condition = "act_id=%s AND asset_type=%s"
        params = [act_id, asset_type]

        if app_id:
            condition += " AND app_id=%s"
            params.append(app_id)
        if user_id != 0:
            condition += " AND user_id=%s"
            params.append(user_id)
        if open_id:
            condition += " AND open_id=%s"
            params.append(open_id)
        if user_nick:
            condition += " AND user_nick=%s"
            params.append(user_nick)
        if asset_object_id:
            condition += " AND asset_object_id=%s"
            params.append(asset_object_id)
        if start_date:
            condition += " AND create_date>=%s"
            params.append(start_date)
        if end_date:
            condition += " AND create_date<=%s"
            params.append(end_date)
        if source_type != 0:
            condition += " AND source_type=%s"
            params.append(source_type)
        if source_object_id:
            condition += " AND source_object_id=%s"
            params.append(source_object_id)

        page_list, total = AssetLogModel(sub_table=sub_table, context=self.context).get_dict_page_list(field, page_index, page_size, condition, order_by="id desc", params=params)

        page_info = PageInfo(page_index, page_size, total, page_list)

    def get_asset_warn_list(self, app_id, act_id, user_id, asset_type, page_size=0, page_index=10, asset_object_id="", start_date="", end_date="", user_nick="", open_id="", field="*"):
        """
        :description: 获取资产预警记录
        :param app_id：应用标识
        :param act_id：活动标识
        :param user_id：用户标识
        :param asset_type：资产类型(1-次数2-积分3-价格档位)
        :param page_size：条数
        :param page_index：页数
        :param asset_object_id：资产对象标识
        :param start_date：开始时间
        :param end_date：结束时间
        :param user_nick：昵称
        :param open_id：open_id
        :param field：查询字段
        :return: 返回PageInfo
        :last_editors: HuangJianYi
        """
        page_info = PageInfo(page_index, page_size, 0, [])
        if not act_id:
            return page_info

        condition = "act_id=%s and asset_type=%s"
        params = [act_id, asset_type]
        if app_id:
            condition += " AND app_id=%s"
            params.append(app_id)
        if asset_type != 0:
            condition += " AND asset_type=%s"
            params.append(asset_type)
        if asset_object_id:
            condition += " AND asset_object_id=%s"
            params.append(asset_object_id)
        if user_id != 0:
            condition += " AND user_id=%s"
            params.append(user_id)
        if open_id:
            condition += " AND open_id=%s"
            params.append(open_id)
        if user_nick:
            condition += " AND user_nick=%s"
            params.append(user_nick)
        if start_date:
            condition += " AND create_date>=%s"
            params.append(start_date)
        if end_date:
            condition += " AND create_date<=%s"
            params.append(end_date)

        page_list, total = AssetWarnNoticeModel(context=self.context).get_dict_page_list(field, page_index, page_size, condition, order_by="id desc", params=params)

        page_info = PageInfo(page_index, page_size, total, page_list)
