import os
import random

import numpy as np
import pandas as pd


class EmpaticaReader:
    """
    Parses, timeshifts and writes data generated by Empatica E4.

    Attributes
    ----------
    start_times : dict
        Contains the timestamp of the first measurement for all
        measured signals (BVP, ACC etc.).

    sample_freqs : dict
        Contains the sampling frequencies of all
        measured signals in Hz.

    IBI : pandas.DataFrame
        Contains inter-beat interval data.
        The column "seconds_since_start" is the time in seconds between the start of measurements
        and the column "IBI" is the duration in seconds between consecutive beats.
        
    ACC : pandas.DataFrame
        Contains the data measured with the onboard MEMS type 3-axis
        accelerometer, indexed by time of measurement.

    BVP : pandas.DataFrame
        Contains blood volume pulse data, indexed by
        time of measurement.

    EDA : pandas.DataFrame
        Contains data captured from the electrodermal activity
        sensor, indexed by time of measurement.

    HR : pandas.DataFrame
        Contains heart rate data, indexed by time of measurement.

    TEMP : pandas.DataFrame
        Contains temperature data, indexed by time of measurement.

    data : pandas.DataFrame
        Joined dataframe of the ACC, BVP, EDA, HR and TEMP dataframes (see
        above). May contain NaN values because sampling frequencies differ
        across signals.
    """

    def __init__(self, path):
        """
        Parse the csv files located in the specified directory into dataframes.

        Parameters
        ----------
        path : str
            Path of the directory that contains the individual signal csv
            files. The files must be named ACC.csv, BVP.csv, EDA.csv, HR.csv,
            IBI.csv and TEMP.csv. If present, the file tags.csv is read too.
        """

        self.start_times = {}
        self.sample_freqs = {}

        self.ACC = self._read_signal(os.path.join(path, 'ACC.csv'), 'ACC', col_names=['X', 'Y', 'Z'])
        if self.ACC is not None:
            self.ACC['mag'] = np.linalg.norm(self.ACC.values, axis=1)
        self.BVP = self._read_signal(os.path.join(path, 'BVP.csv'), 'BVP')
        self.EDA = self._read_signal(os.path.join(path, 'EDA.csv'), 'EDA')
        self.HR = self._read_signal(os.path.join(path, 'HR.csv'), 'HR')
        self.TEMP = self._read_signal(os.path.join(path, 'TEMP.csv'), 'TEMP')
        self.IBI = self._read_ibi(os.path.join(path, 'IBI.csv'))

        self.tags = self._read_tags(os.path.join(path, 'tags.csv'))
        
        self.data = self._get_joined_dataframe()

    def write(self, dir_path):
        """
        Write the signal dataframes back to individual csv files formatted the
        same way as the read csv files.

        Parameters
        ----------
        path : str
            Path of the directory in which the csv files are created.

            If the directory exists, the csv files are written using writing mode 'w'
            ignoring other files in the directory.

            If the directory doe not exist, it will be created.
        """

        if not os.path.exists(dir_path):
            os.mkdir(dir_path)
        self._write_signal(os.path.join(dir_path, 'ACC.csv'), self.ACC.drop(columns='mag'), 'ACC')
        self._write_signal(os.path.join(dir_path, 'BVP.csv'), self.BVP, 'BVP')
        self._write_signal(os.path.join(dir_path, 'EDA.csv'), self.EDA, 'EDA')
        self._write_signal(os.path.join(dir_path, 'HR.csv'), self.HR, 'HR')
        self._write_signal(os.path.join(dir_path, 'TEMP.csv'), self.TEMP, 'TEMP')
        self._write_ibi(os.path.join(dir_path, 'IBI.csv'))
        self._write_tags(os.path.join(dir_path, 'tags.csv'))

    def _read_signal(self, path, signal_name, col_names=None):
        with open(path, 'r') as f:
            start_time_str = f.readline().split(',')[0]
            self.start_times[signal_name] = pd.Timestamp(float(start_time_str), unit='s')
            sample_freq_str = f.readline().split(',')[0]
            self.sample_freqs[signal_name] = float(sample_freq_str)
            col_names = [signal_name] if col_names is None else col_names
            df = pd.read_csv(f, header=None, names=col_names)
            df.index = pd.date_range(
                start=self.start_times[signal_name],
                freq=f"{1 / self.sample_freqs[signal_name]}S",
                periods=len(df))
            if col_names is not None:
                df.rename(dict(enumerate(col_names)), inplace=True)
            else:
                df.rename({0: signal_name}, inplace=True)
            
            return df.squeeze()

    def _write_signal(self, path, df, signal_name):
        n_cols = len(df.columns) if isinstance(df, pd.DataFrame) else 1
        meta = np.array([[self.start_times[signal_name].value / 1e9] * n_cols,
                            [self.sample_freqs[signal_name]] * n_cols])
        meta_df = pd.DataFrame(meta)
        with open(path, 'w') as f:
            meta_df.to_csv(f, index=None, header=None)
            df.to_csv(f, index=None, header=None)

    def _read_ibi(self, path):
        with open(path, 'r') as f:
            self.start_times['IBI'] = pd.Timestamp(float(f.readline().split(',')[0]), unit='s')
            df = pd.read_csv(f, names=['seconds_since_start', 'IBI'], header=None)
            return df

    def _write_ibi(self, path):
        with open(path, 'w') as f:
            f.write(f"{float(self.start_times['IBI'].value / 1e9)}, IBI\n")
            f.write(self.IBI.to_csv(index=None, header=None))

    def _read_tags(self, path):
        try:
            if os.stat(path).st_size > 0:
                return pd.read_csv(path, header=None,
                                         parse_dates=[0],
                                         date_parser=lambda x : pd.to_datetime(x, unit='s'),
                                         names=['tags'],
                                         squeeze=True)

            else:
                print(f"Not reading tags because the file {path} is empty.")
        except OSError:
            print(f"Not reading tags because the file {path} does not exist.")

        return None

    def _write_tags(self, path):
        if self.tags is not None:
            tags_write_series = self.tags.map(lambda x: x.value / 1e9)
            tags_write_series.to_csv(path, header=None, index=None)

    def timeshift(self, shift='random'):
        """
        Shift all time related columns as well as the starting_times dict.

        Parameters
        ----------
        shift : None/'random', pd.Timestamp or pd.Timedelta
            If shift is not specified, shifts the data by a random time interval
            between one month and two years to the past.

            If shift is a timdelta, adds that timedelta to all time-related attributes. 

            If shift is a timestamp, shifts the data such that the earliest entry
            is at that timestamp and the remaining values keep the same
            time distance to the first entry.
        """

        if shift == 'random':
            one_month = pd.Timedelta('- 30 days').value
            two_years = pd.Timedelta('- 730 days').value
            random_timedelta = pd.Timedelta(random.uniform(one_month, two_years))
            self.timeshift(random_timedelta)

        dfs = [self.ACC, self.BVP, self.EDA, self.HR, self.TEMP, self.data]

        if isinstance(shift, pd.Timestamp):
            min_start_time = min(self.start_times.values())
            new_start_times = dict()
            for signal_name, start_time in self.start_times.items():
                new_start_times[signal_name] = shift + (start_time - min_start_time)
            self.start_times = new_start_times
            if self.tags is not None:
                timedeltas = self.tags - self.tags.min()
                self.tags = shift + timedeltas
            for df in dfs:
                timedeltas = df.index - df.index.min()
                df.index = shift + timedeltas

        if isinstance(shift, pd.Timedelta):
            for signal_name in self.start_times:
                self.start_times[signal_name] += shift
            if self.tags is not None:
                self.tags += shift
            for df in dfs:
                df.index += shift

    def _get_joined_dataframe(self):
        dfs = [self.ACC, self.BVP, self.EDA, self.HR, self.TEMP]
        joined_idx = pd.concat([pd.Series(df.index) for df in dfs])
        joined_idx = pd.Index(joined_idx.drop_duplicates().sort_values())
        joined_columns = ['ACC_X', 'ACC_Y', 'ACC_Z', 'BVP', 'EDA', 'HR', 'TEMP']

        joined_df = pd.DataFrame(columns=joined_columns, index=joined_idx)
        joined_df.loc[self.ACC.index, 'ACC_X'] = self.ACC['X']
        joined_df.loc[self.ACC.index, 'ACC_Y'] = self.ACC['Y']
        joined_df.loc[self.ACC.index, 'ACC_Z'] = self.ACC['Z']
        joined_df.loc[self.BVP.index, 'BVP'] = self.BVP
        joined_df.loc[self.EDA.index, 'EDA'] = self.EDA
        joined_df.loc[self.HR.index, 'HR'] = self.HR
        joined_df.loc[self.TEMP.index, 'TEMP'] = self.TEMP

        return joined_df