import json
import os
import random

import numpy as np
import pandas as pd
import pyedflib as edf


class FarosReader:
    """
    Parses, timeshifts and writes data
    generated by Bittium Faros devices.

    Attributes
    ----------
    start_time : pandas.Timestamp
        start time of all measurements
    
    sample_freqs : dict
        sampling frequencies of all signals in Hz

    units : dict
        units of all signals

    ECG : pandas.Series
        ECG signal, indexed by timestamp
    
    ACC : pandas.DataFrame
        three ACC axes, indexed by timestamp

    Marker : pandas.Series
        Markers, indexed by timestamp
    
    HRV : pandas.Series
        HRV signal, indexed by timestamp

    data : DataFrame
        contains all signals (ECG, ACC, Marker, HRV) indexed by timestamp
        Because the signals have different sampling frequencies, a lot of values are nan.
    """

    def __init__(self, path):
        """
        Reads a Faros-generated EDF file or a directory created by a FarosReader.

        Parameters
        ----------
        path : str
            Can either be Faros-generated EDF file or a directory created by the FarosReader.write() method.
        """

        self.start_time = None
        self.sample_freqs = None
        self.units = None
        self._edf_metadata = None
        self.ECG = None
        self.ACC = None
        self.Marker = None
        self.HRV = None

        if os.path.isfile(path):
            self._read_from_edf_file(path)
        if os.path.isdir(path):
            self._read_from_directory(path)

        self.data = self._get_joined_df()

    def _read_from_edf_file(self, path):
        reader = edf.EdfReader(path)
        self.start_time = pd.Timestamp(reader.getStartdatetime())
        self.sample_freqs = {
            'ECG': reader.getSampleFrequency(0),
            'ACC': reader.getSampleFrequency(1),
            'Marker': reader.getSampleFrequency(4),
            'HRV': reader.getSampleFrequency(5)
        }
        self.units = {
            'ECG': reader.getSignalHeader(0)['dimension'],
            'ACC': reader.getSignalHeader(1)['dimension'],
            'HRV': reader.getSignalHeader(5)['dimension'],
        }
        self._edf_metadata = reader.getSignalHeaders()

        self._n_samples = reader.getNSamples()
        self._n_datarecords = reader.datarecords_in_file

        ecg = reader.readSignal(0)
        self.ECG = pd.Series(ecg, name='ECG',
                             index=pd.date_range(start=self.start_time,
                                                 periods=len(ecg),
                                                 freq=f"{1/self.sample_freqs['ECG']}S")
        )

        acc = np.array([reader.readSignal(i) for i in range(1, 4)]).T
        self.ACC = pd.DataFrame(acc, columns=['X', 'Y', 'Z'],
                                     index=pd.date_range(start=self.start_time,
                                                         periods=len(acc),
                                                         freq=f"{1/self.sample_freqs['ACC']}S")
        )
        self.ACC['mag'] = np.linalg.norm(self.ACC, axis=1)
        
        marker = reader.readSignal(4)
        self.Marker = pd.Series(marker, name='Marker',
                                index=pd.date_range(start=self.start_time,
                                                    periods=len(marker),
                                                    freq=f"{1/self.sample_freqs['Marker']}S")
        )

        hrv = reader.readSignal(5)
        self.HRV = pd.Series(
            hrv, name='HRV',
            index=pd.date_range(start=self.start_time,
                                periods=len(hrv),
                                freq=f"{1/self.sample_freqs['HRV']}S")
        )

        reader.close()

    def _read_from_directory(self, path):
        with open(os.path.join(path, 'meta.json'), 'r') as meta_file:
            meta = json.load(meta_file)
        self.start_time = pd.Timestamp(meta['start_time'])
        self.sample_freqs = meta['sample_freqs']
        self.units = meta['units']

        self.ECG = self._read_dir_csv(os.path.join(path, 'ECG.csv'), self.start_time, self.sample_freqs['ECG'])
        self.ACC = self._read_dir_csv(os.path.join(path, 'ACC.csv'), self.start_time, self.sample_freqs['ACC'])
        self.ACC['mag'] = np.linalg.norm(self.ACC, axis=1)
        self.Marker = self._read_dir_csv(os.path.join(path, 'Marker.csv'), self.start_time, self.sample_freqs['Marker'])
        self.HRV = self._read_dir_csv(os.path.join(path, 'HRV.csv'), self.start_time, self.sample_freqs['HRV'])

    def _read_dir_csv(self, path, start_time, sample_freq):
        df = pd.read_csv(path)
        idx = pd.date_range(start=start_time, periods=len(df), freq=f"{1/sample_freq}S")
        df.index = idx

        return df.squeeze()

    def _get_joined_df(self):
        # get index for joined dataframe
        joined_idx = pd.concat(map(pd.Series, [self.ECG.index, self.ACC.index, self.Marker.index, self.HRV.index]))
        joined_idx = pd.Index(joined_idx.drop_duplicates().sort_values())

        # create joined dataframe
        col_names = ['ECG', 'ACC_X', 'ACC_Y', 'ACC_Z', 'ACC_mag', 'Marker', 'HRV']
        joined_df = pd.DataFrame(index=joined_idx, columns=col_names)

        # set non-nan values of joined dataframe
        joined_df.loc[self.ECG.index, 'ECG'] = self.ECG
        joined_df.loc[self.ACC.index, 'ACC_X'] = self.ACC['X']
        joined_df.loc[self.ACC.index, 'ACC_Y'] = self.ACC['Y']
        joined_df.loc[self.ACC.index, 'ACC_Z'] = self.ACC['Z']
        joined_df.loc[self.ACC.index, 'ACC_mag'] = self.ACC['mag']
        joined_df.loc[self.Marker.index, 'Marker'] = self.Marker
        joined_df.loc[self.HRV.index, 'HRV'] = self.HRV

        return joined_df

    def write(self, path, format='directory'):
        """
        Writes the data either to an EDF file or to several files into a new directory.
        Because of the `special structure of EDF files <https://www.edfplus.info/specs/edf.html>`_ 
        writing to EDF is only possible for readers that have been created from an EDF file and without any changes to the ACC, ECG, Marker, HRV and sample_freqs attributes.
        Because we want you to be able to modify the signals, you can write the data back to a directory of individual files.
        Writing to a directory is the preferred method and works in all cases.

        Parameters
        ----------
        path : str
            Name of the file or directory to write the data to.
        format: {'directory', 'edf'}, default 'directory'
            Format of the written data.
        """

        if format == 'directory':
            self._write_to_directory(path)
        if format == 'edf':
            self._write_to_edf(path)

    def _write_to_edf(self, path):
        if self._edf_metadata is None:
            raise Exception("There is no EDF metadata in this reader, most likely because it was initialized from a directory. Writing to EDF file not possible.")
        
        writer = edf.EdfWriter(path, 6, 0)
        writer.setStartdatetime(self.start_time.to_pydatetime())
        writer.setSignalHeaders(self._edf_metadata)

        ecg_freq = int(self.sample_freqs['ECG'])
        acc_freq = int(self.sample_freqs['ACC'])
        marker_freq = int(self.sample_freqs['Marker'])
        hrv_freq = int(self.sample_freqs['HRV'])

        n_records = int(len(self.ECG) / ecg_freq)

        for i in range(n_records):
            writer.writePhysicalSamples(self.ECG.values[ecg_freq * i: ecg_freq * (i + 1)])
            writer.writePhysicalSamples(self.ACC['X'].values[acc_freq * i: acc_freq * (i + 1)])
            writer.writePhysicalSamples(self.ACC['Y'].values[acc_freq * i: acc_freq * (i + 1)])
            writer.writePhysicalSamples(self.ACC['Z'].values[acc_freq * i: acc_freq * (i + 1)])
            writer.writePhysicalSamples(self.Marker.values[marker_freq * i: marker_freq * (i + 1)])
            writer.writePhysicalSamples(self.HRV.values[hrv_freq * i: hrv_freq * (i + 1)])
        
        writer.close()

    def _write_to_directory(self, path):
        if not os.path.isdir(path):
            os.mkdir(path)
            
        meta = {
            'start_time': str(self.start_time),
            'sample_freqs': self.sample_freqs,
            'units': self.units
        }
        with open(os.path.join(path, 'meta.json'), 'w') as meta_file:
            json.dump(meta, meta_file)

        self.ECG.to_csv(os.path.join(path, 'ECG.csv'), index=None)
        self.ACC.drop(columns=['mag']).to_csv(os.path.join(path, 'ACC.csv'), index=None)
        self.Marker.to_csv(os.path.join(path, 'Marker.csv'), index=None)
        self.HRV.to_csv(os.path.join(path, 'HRV.csv'), index=None)

    def timeshift(self, shift='random'):
        """
        Timeshifts the data by shifting all time related values (i.e. start_time and data.index).

        Parameters
        ----------
        shift : None/'random', pd.Timestamp or pd.Timedelta
            If shift is not specified, shifts the data by a random time interval
            between one month and two years to the past.

            If shift is a timdelta, shifts the data by that timedelta.

            If shift is a timestamp, shifts the data such that the earliest entry
            is at that timestamp and the remaining values keep the same time distance to the first entry.
        """
        if shift == 'random':
            one_month = pd.Timedelta('30 days').value
            two_years = pd.Timedelta('730 days').value
            random_timedelta = - pd.Timedelta(random.uniform(one_month, two_years)).round('s')
            self.timeshift(random_timedelta)
        if isinstance(shift, pd.Timestamp):
            self.start_time = shift
            for df in [self.ECG, self.ACC, self.Marker, self.HRV, self.data]:
                timedeltas = df.index - df.index.min()
                df.index = shift + timedeltas
        if isinstance(shift, pd.Timedelta):
            for df in [self.ECG, self.ACC, self.Marker, self.HRV, self.data]:
                df.index += shift
