import random
import re

import numpy as np
import pandas as pd


class ShimmerPlusReader:
    """
    Parses, deidentifies (timeshifts) and writes data
    generated by Shimmer Consensys GSR.

    Attributes
    ----------
    data : DataFrame
        DataFrame with the read signals. The column 'Shimmer_40AC_Timestamp_Unix_CAL' contains the timestamps of measurement.

    units : pandas.Series
        a unit for each column in the dataframe
    """
    def __init__(self, path):
        """
        Reads the csv file generated by the Shimmer device and saves the
        parsed DataFrame.

        Parameters
        ----------
        path : str
            Path of the csv file.
        delimiter, optional : str
            delimiter used in the csv file
        """
        with open(path, 'r') as f:
            self.delimiter = re.findall(f'"sep=(.)"', f.readline())[0]
            col_names = list(filter(bool, f.readline().strip().split(self.delimiter)))
            units = list(filter(bool, f.readline().strip().split(self.delimiter)))
            self.units = dict(zip(col_names, units))
            self.data = pd.read_csv(f, sep=self.delimiter, header=None, index_col=False,
                                                           names=col_names,
                                                           parse_dates=['Shimmer_40AC_Timestamp_Unix_CAL'],
                                                           date_parser=lambda x: pd.to_datetime(x, unit='ms').round('ms')
                                                           )
        self.data['Shimmer_40AC_Accel_LN_mag'] = np.linalg.norm(self.data[['Shimmer_40AC_Accel_LN_X_CAL',
                                                                           'Shimmer_40AC_Accel_LN_Y_CAL',
                                                                           'Shimmer_40AC_Accel_LN_Z_CAL']],
                                                                axis=1)

    def write(self, path):
        """
        Writes the DataFrame and units to the
        writing path in the same format as it was read.

        Parameters
        ----------
        path : str
            Path to writing csv file. Writing mode: 'w'.
        """
        write_df = self.data.drop(columns=['Shimmer_40AC_Accel_LN_mag'])
        write_df['Shimmer_40AC_Timestamp_Unix_CAL'] = write_df['Shimmer_40AC_Timestamp_Unix_CAL'].map(lambda x: float(x.round('ms').value / 1e6))
        units_df = pd.DataFrame(self.units, index=[0])
        write_df = pd.concat([units_df, write_df])

        with open(path, 'w') as f:
            f.write(f'"sep={self.delimiter}"\n')
            write_df.to_csv(f, index=False, sep=self.delimiter, line_terminator=f"{self.delimiter}\n")

    def timeshift(self, shift='random'):
        """
        Timeshifts the data by shifting all time related columns.

        Parameters
        ----------
        shift : None/'random', pd.Timestamp or pd.Timedelta
            If shift is not specified, shifts the data by a random time interval
            between one month and two years to the past.

            If shift is a timdelta, shifts the data by that timedelta.

            If shift is a timestamp, shifts the data such that the earliest entry
            is at that timestamp and the remaining values keep the same time distance to the first entry.
        """
        if shift == 'random':
            one_month = pd.Timedelta('30 days').value
            two_years = pd.Timedelta('730 days').value
            random_timedelta = - pd.Timedelta(random.uniform(one_month, two_years))
            self.timeshift(random_timedelta)
        if isinstance(shift, pd.Timestamp):
            shift = shift.round('ms')
            timedeltas = self.data['Shimmer_40AC_Timestamp_Unix_CAL'] - self.data['Shimmer_40AC_Timestamp_Unix_CAL'].min()
            self.data['Shimmer_40AC_Timestamp_Unix_CAL'] = shift + timedeltas
        if isinstance(shift, pd.Timedelta):
            self.data['Shimmer_40AC_Timestamp_Unix_CAL'] += shift.round('ms')
