# -*- coding: utf-8 -*-
#
# This file is part of INGInious. See the LICENSE and the COPYRIGHTS files for
# more information about the licensing of this file.

import logging

import flask
from flask import redirect
from werkzeug.exceptions import NotFound, Forbidden
from bson.errors import InvalidId

from inginious.frontend.pages.course_admin.utils import INGIniousAdminPage


class SubmissionPage(INGIniousAdminPage):
    """ List information about a task done by a student """
    _logger = logging.getLogger("inginious.frontend.submissions")

    def fetch_submission(self, submissionid):
        try:
            submission = self.submission_manager.get_submission(submissionid, False)
            if not submission:
                raise NotFound(description=_("This submission doesn't exist."))
        except InvalidId as ex:
            self._logger.info("Invalid ObjectId : %s", submissionid)
            raise Forbidden(description=_("Invalid ObjectId."))

        courseid = submission["courseid"]
        taskid = submission["taskid"]
        course, task = self.get_course_and_check_rights(courseid, taskid)
        return course, task, submission

    def GET_AUTH(self, submissionid):  # pylint: disable=arguments-differ
        """ GET request """
        course, task, submission = self.fetch_submission(submissionid)
        return self.page(course, task, submission)

    def POST_AUTH(self, submissionid):  # pylint: disable=arguments-differ
        course, task, submission = self.fetch_submission(submissionid)
        is_admin = self.user_manager.has_admin_rights_on_course(course)

        webinput = flask.request.form
        if "replay" in webinput and is_admin:
            self.submission_manager.replay_job(task, submission)
        elif "replay-copy" in webinput:  # Authorized for tutors
            self.submission_manager.replay_job(task, submission, True)
            return redirect(self.app.get_homepath() + "/course/" + course.get_id() + "/" + task.get_id())
        elif "replay-debug" in webinput and is_admin:
            self.submission_manager.replay_job(task, submission, True, "ssh")
            return redirect(self.app.get_homepath() + "/course/" + course.get_id() + "/" + task.get_id())

        return self.page(course, task, submission)

    def page(self, course, task, submission):
        """ Get all data and display the page """
        submission = self.submission_manager.get_input_from_submission(submission)
        submission = self.submission_manager.get_feedback_from_submission(
            submission,
            show_everything=True,
            translation=self.app.l10n_manager.get_translation_obj()
        )

        to_display = {
            problem.get_id(): {
                "id": problem.get_id(),
                "name": problem.get_name(self.user_manager.session_language()),
                "defined": True
            } for problem in task.get_problems()
        }

        to_display.update({
            pid: {
                "id": pid,
                "name": pid,
                "defined": False
            } for pid in (set(submission["input"]) - set(to_display))
        })

        return self.template_helper.render("course_admin/submission.html", course=course, task=task,
                                           submission=submission, to_display=to_display.values(),
                                           pdict={problem.get_id(): problem.get_type() for problem in task.get_problems()})
