"""A registry that keeps track of all active widgets."""

from __future__ import annotations

from dataclasses import dataclass, field
from functools import lru_cache
from typing import Generic, Iterator, TypeVar

Widget = ""

T = TypeVar("T")


@dataclass(frozen=True)
class Selector:
    types: tuple[Widget]
    classes: tuple[str]
    properties: tuple[str]

    # TODO: Cache this!
    # @lru_cache(1024)
    def targets(self, item: Widget) -> bool:
        ...


@dataclass(frozen=True)
class Rule:
    declarations: list[dict[str, str]] = field(default_factory=dict)

    @classmethod
    def parse(cls, rule: str) -> Rule:
        decls = []

        for line in rule.split(";"):
            decls.append({key: value.lstrip(" ") for key, value in line.split(":")})

        return cls(decls)


@dataclass(frozen=True)
class Selection(Generic[T]):
    items: list[T]

    def __iter__(self) -> Iterator[T]:
        return iter(self.items)

    def apply_rule(self, rule: str | Rule) -> None:
        if isinstance(rule, str):
            rule = Rule.parse(rule)

        for widget in self.items:
            widget.apply_rule(rule)


class Registry:
    def __init__(self) -> None:
        self._rules: dict[Selector, Rule] = {}
        self._widgets = []

    def register(self, widget: Widget) -> None:
        self._widgets[id(widget)] = widget

    def __setitem__(
        self, selector: str | Selector, rule: str, update: bool = True
    ) -> None:
        if isinstance(selector, str):
            selector = Selector.parse(selector)

        self._rules[selector] = rule
        self.select(selector).apply_rule(rule)

        if update:
            self.update()

    def _query(self, selector: Selector) -> Iterator[Widget]:
        ...

    def select(self, selector: str | Selector) -> Selection[Widget, ...]:
        if isinstance(selector, str):
            selector = Selector.parse(selector)

        return Selection(list(self._query(selector)))

    def update(self) -> int:
        for selector, rule in self._widgets:
            self.select(selector).apply_rule(rule)


import json

import yaml

print(
    json.dumps(
        yaml.safe_load(
            """
widgets:
    FishEditor:
        classes:
            - dialogue

        contains:
            - Label:
                classes: [header]
                value: Create a fish!
            - ""

            - Container:
                width: 20
                title:
                    location: left
                    content: Preview
            - ""

            - Container:
                contains:
                    - Splitter: &choice
                        classes: [no-border]

                        contains:
                            - Field 1
                            - HorizontalChoice

                    - Splitter: *choice
                    - Splitter: *choice
"""
        ),
        indent=2,
    )
)

# registry["InputField .[highlighted]"] = "foreground_style: red; fill_style: @black"
#
# registry = Registry()
#
# my_widget = Widget()
# my_widget.classes.append("helo")
#
# [field] = registry.select("+field")
#
# for widget in registry.select(".[field]"):
#     print(widget)
