# QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
# Lean CLI v1.0. Copyright 2021 QuantConnect Corporation.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from typing import Any, Dict

import click

from lean.components.util.logger import Logger
from lean.models.brokerages.local.base import LeanConfigConfigurer, LocalBrokerage


class ZerodhaBrokerage(LocalBrokerage):
    """A LocalBrokerage implementation for the Zerodha brokerage."""

    @classmethod
    def get_name(cls) -> str:
        return "Zerodha"

    @classmethod
    def _configure_environment(cls, lean_config: Dict[str, Any], environment_name: str) -> None:
        lean_config["environments"][environment_name]["live-mode-brokerage"] = "ZerodhaBrokerage"
        lean_config["environments"][environment_name]["transaction-handler"] = \
            "QuantConnect.Lean.Engine.TransactionHandlers.BrokerageTransactionHandler"

    @classmethod
    def _configure_credentials(cls, lean_config: Dict[str, Any], logger: Logger) -> None:
        logger.info("You need API credentials for Kite Connect (https://kite.trade/) to use the Zerodha brokerage.")

        lean_config["zerodha-api-key"] = click.prompt("API key", cls._get_default(lean_config, "zerodha-api-key"))
        lean_config["zerodha-access-token"] = logger.prompt_password(
            "Access token",
            cls._get_default(lean_config, "zerodha-access-token")
        )

        logger.info(
            "The product type must be set to MIS if you are targeting intraday products, CNC if you are targeting delivery products or NRML if you are targeting carry forward products.")

        lean_config["zerodha-product-type"] = click.prompt(
            "Product type",
            cls._get_default(lean_config, "zerodha-product-type"),
            type=click.Choice(["MIS", "CNC", "NRML"], case_sensitive=False)
        )

        logger.info(
            "The trading segment must be set to EQUITY if you are trading equities on NSE or BSE, or COMMODITY if you are trading commodities on MCX.")

        lean_config["zerodha-trading-segment"] = click.prompt(
            "Trading segment",
            cls._get_default(lean_config, "zerodha-trading-segment"),
            type=click.Choice(["EQUITY", "COMMODITY"], case_sensitive=False)
        )

        cls._save_properties(lean_config, ["zerodha-api-key",
                                           "zerodha-access-token",
                                           "zerodha-product-type",
                                           "zerodha-trading-segment"])


class ZerodhaDataFeed(LeanConfigConfigurer):
    """A LeanConfigConfigurer implementation for the Zerodha data feed."""

    @classmethod
    def get_name(cls) -> str:
        return ZerodhaBrokerage.get_name()

    @classmethod
    def configure(cls, lean_config: Dict[str, Any], environment_name: str, logger: Logger) -> None:
        lean_config["environments"][environment_name]["data-queue-handler"] = "ZerodhaBrokerage"
        lean_config["environments"][environment_name]["history-provider"] = "BrokerageHistoryProvider"

        ZerodhaBrokerage.configure_credentials(lean_config, logger)

        lean_config["zerodha-history-subscription"] = click.confirm(
            "Do you have a history API subscription?",
            cls._get_default(lean_config, "zerodha-history-subscription")
        )

        cls._save_properties(lean_config, ["zerodha-history-subscription"])
