# -*- coding: UTF-8 -*-
# Python dictionnaries for MCC / MNC
# Mobile Country Code / Mobile Network Code

def _process_html(fn):
    # processing the HTML tab extracted from http://www.mcc-mnc.com/
    # take care to have a blank line at the end of the file
    L, ret = open(fn).readlines(), []
    for l in L[:-1]:
        i = l.split('<td>')
        ret.append( (i[1][:-5].strip(), i[2][:-5].strip(), i[3][:-5].strip().upper(), 
                     i[4][:-5].strip(), i[5][:-5].strip(), i[6][:-11].strip()) )
    
    # build country code dict
    MCC_dict = {}
    for l in ret:
        if l[0] not in MCC_dict:
            if not l[4]:
                cind = None
            else:
                cind = '+'+l[4]
            # number ind., country abbr., country name
            MCC_dict[ l[0] ] = (cind, l[2], l[3])
    
    # build network code dict
    MNC_dict = {}
    for l in ret:
        if not l[4]:
            cind = None
        else:
            cind = '+'+l[4]
        # number ind., country abbr., country name, MNO name
        MNC_dict[ l[0]+l[1] ] = (cind, l[2], l[3], l[5])
    
    # build reverse country code from country abbr.
    MCCr_dict = {}
    for l in ret:
        if l[2] not in MCCr_dict:
            if not l[4]:
                cind = None
            else:
                cind = '+'+l[4]
            MCCr_dict[ l[2] ] = (l[0], cind, l[3])
    
    return MCC_dict, MNC_dict, MCCr_dict


# last update : 2019/02/13

MCC_dict = {
 '202': ('+30', 'GR', 'Greece'),
 '204': ('+31', 'NL', 'Netherlands'),
 '206': ('+32', 'BE', 'Belgium'),
 '208': ('+33', 'FR', 'France'),
 '212': ('+377', 'MC', 'Monaco'),
 '213': ('+376', 'AD', 'Andorra'),
 '214': ('+34', 'ES', 'Spain'),
 '216': ('+36', 'HU', 'Hungary'),
 '218': ('+387', 'BA', 'Bosnia & Herzegov.'),
 '219': ('+385', 'HR', 'Croatia'),
 '220': ('+381', 'RS', 'Serbia'),
 '222': ('+39', 'IT', 'Italy'),
 '226': ('+40', 'RO', 'Romania'),
 '228': ('+41', 'CH', 'Switzerland'),
 '230': ('+420', 'CZ', 'Czech Rep.'),
 '231': ('+421', 'SK', 'Slovakia'),
 '232': ('+43', 'AT', 'Austria'),
 '234': ('+44', 'GB', 'United Kingdom'),
 '235': ('+44', 'GB', 'United Kingdom'),
 '238': ('+45', 'DK', 'Denmark'),
 '240': ('+46', 'SE', 'Sweden'),
 '242': ('+47', 'NO', 'Norway'),
 '244': ('+358', 'FI', 'Finland'),
 '246': ('+370', 'LT', 'Lithuania'),
 '247': ('+371', 'LV', 'Latvia'),
 '248': ('+372', 'EE', 'Estonia'),
 '250': ('+79', 'RU', 'Russian Federation'),
 '255': ('+380', 'UA', 'Ukraine'),
 '257': ('+375', 'BY', 'Belarus'),
 '259': ('+373', 'MD', 'Moldova'),
 '260': ('+48', 'PL', 'Poland'),
 '262': ('+49', 'DE', 'Germany'),
 '266': ('+350', 'GI', 'Gibraltar'),
 '268': ('+351', 'PT', 'Portugal'),
 '270': ('+352', 'LU', 'Luxembourg'),
 '272': ('+353', 'IE', 'Ireland'),
 '274': ('+354', 'IS', 'Iceland'),
 '276': ('+355', 'AL', 'Albania'),
 '278': ('+356', 'MT', 'Malta'),
 '280': ('+357', 'CY', 'Cyprus'),
 '282': ('+995', 'GE', 'Georgia'),
 '283': ('+374', 'AM', 'Armenia'),
 '284': ('+359', 'BG', 'Bulgaria'),
 '286': ('+90', 'TR', 'Turkey'),
 '288': ('+298', 'FO', 'Faroe Islands'),
 '289': ('+7', 'GE', 'Abkhazia'),
 '290': ('+299', 'GL', 'Greenland'),
 '292': ('+378', 'SM', 'San Marino'),
 '293': ('+386', 'SI', 'Slovenia'),
 '294': ('+389', 'MK', 'Macedonia'),
 '295': ('+423', 'LI', 'Liechtenstein'),
 '297': ('+382', 'ME', 'Montenegro'),
 '302': ('+1', 'CA', 'Canada'),
 '308': ('+508', 'PM', 'St. Pierre & Miquelon'),
 '310': ('+1671', 'GU', 'Guam'),
 '311': ('+1671', 'GU', 'Guam'),
 '312': ('+1', 'US', 'United States'),
 '316': ('+1', 'US', 'United States'),
 '330': ('+1', 'PR', 'Puerto Rico'),
 '334': ('+52', 'MX', 'Mexico'),
 '338': ('+1876', 'JM', 'Jamaica'),
 '340': ('+594', 'FG', 'French Guiana'),
 '342': ('+1246', 'BB', 'Barbados'),
 '344': ('+1268', 'AG', 'Antigua and Barbuda'),
 '346': ('+1345', 'KY', 'Cayman Islands'),
 '348': ('+284', 'VG', 'British Virgin Islands'),
 '350': ('+1441', 'BM', 'Bermuda'),
 '352': ('+1473', 'GD', 'Grenada'),
 '354': ('+1664', 'MS', 'Montserrat'),
 '356': ('+1869', 'KN', 'Saint Kitts and Nevis'),
 '358': ('+1758', 'LC', 'Saint Lucia'),
 '360': ('+1784', 'VC', 'St. Vincent & Gren.'),
 '362': ('+599', 'CW', 'Curacao'),
 '363': ('+297', 'AW', 'Aruba'),
 '364': ('+1242', 'BS', 'Bahamas'),
 '365': ('+1264', 'AI', 'Anguilla'),
 '366': ('+1767', 'DM', 'Dominica'),
 '368': ('+53', 'CU', 'Cuba'),
 '370': ('+1809', 'DO', 'Dominican Republic'),
 '372': ('+509', 'HT', 'Haiti'),
 '374': ('+1868', 'TT', 'Trinidad and Tobago'),
 '376': ('+1', 'TC', 'Turks and Caicos Islands'),
 '400': ('+994', 'AZ', 'Azerbaijan'),
 '401': ('+7', 'KZ', 'Kazakhstan'),
 '402': ('+975', 'BT', 'Bhutan'),
 '404': ('+91', 'IN', 'India'),
 '405': ('+91', 'IN', 'India'),
 '410': ('+92', 'PK', 'Pakistan'),
 '412': ('+93', 'AF', 'Afghanistan'),
 '413': ('+94', 'LK', 'Sri Lanka'),
 '414': ('+95', 'MM', 'Myanmar (Burma)'),
 '415': ('+961', 'LB', 'Lebanon'),
 '416': ('+962', 'JO', 'Jordan'),
 '417': ('+963', 'SY', 'Syrian Arab Republic'),
 '418': ('+964', 'IQ', 'Iraq'),
 '419': ('+965', 'KW', 'Kuwait'),
 '420': ('+966', 'SA', 'Saudi Arabia'),
 '421': ('+967', 'YE', 'Yemen'),
 '422': ('+968', 'OM', 'Oman'),
 '424': ('+971', 'AE', 'United Arab Emirates'),
 '425': ('+972', 'IL', 'Israel'),
 '426': ('+973', 'BH', 'Bahrain'),
 '427': ('+974', 'QA', 'Qatar'),
 '428': ('+976', 'MN', 'Mongolia'),
 '429': ('+977', 'NP', 'Nepal'),
 '430': ('+971', 'AE', 'United Arab Emirates'),
 '431': ('+971', 'AE', 'United Arab Emirates'),
 '432': ('+98', 'IR', 'Iran'),
 '434': ('+998', 'UZ', 'Uzbekistan'),
 '436': ('+992', 'TK', 'Tajikistan'),
 '437': ('+996', 'KG', 'Kyrgyzstan'),
 '438': ('+993', 'TM', 'Turkmenistan'),
 '440': ('+81', 'JP', 'Japan'),
 '441': ('+81', 'JP', 'Japan'),
 '450': ('+82', 'KR', 'Korea S, Republic of'),
 '452': ('+84', 'VN', 'Viet Nam'),
 '454': ('+852', 'HK', 'Hongkong, China'),
 '455': ('+853', 'MO', 'Macao, China'),
 '456': ('+855', 'KH', 'Cambodia'),
 '457': ('+856', 'LA', 'Laos P.D.R.'),
 '460': ('+86', 'CN', 'China'),
 '466': ('+886', 'TW', 'Taiwan'),
 '467': ('+850', 'KP', "Korea N., Dem. People's Rep."),
 '470': ('+880', 'BD', 'Bangladesh'),
 '472': ('+960', 'MV', 'Maldives'),
 '502': ('+60', 'MY', 'Malaysia'),
 '505': ('+61', 'AU', 'Australia'),
 '510': ('+62', 'ID', 'Indonesia'),
 '514': ('+670', 'TP', 'Timor-Leste'),
 '515': ('+63', 'PH', 'Philippines'),
 '520': ('+66', 'TH', 'Thailand'),
 '525': ('+65', 'SG', 'Singapore'),
 '528': ('+673', 'BN', 'Brunei Darussalam'),
 '530': ('+64', 'NZ', 'New Zealand'),
 '537': ('+675', 'PG', 'Papua New Guinea'),
 '539': ('+676', 'TO', 'Tonga'),
 '540': ('+677', 'SB', 'Solomon Islands'),
 '541': ('+678', 'VU', 'Vanuatu'),
 '542': ('+679', 'FJ', 'Fiji'),
 '544': ('+684', 'AS', 'American Samoa'),
 '545': ('+686', 'KI', 'Kiribati'),
 '546': ('+687', 'NC', 'New Caledonia'),
 '547': ('+689', 'PF', 'French Polynesia'),
 '548': ('+682', 'CK', 'Cook Islands'),
 '549': ('+685', 'WS', 'Samoa'),
 '550': ('+691', 'FM', 'Micronesia'),
 '552': ('+680', 'PW', 'Palau (Republic of)'),
 '553': ('+688', 'TV', 'Tuvalu'),
 '555': ('+683', 'NU', 'Niue'),
 '602': ('+20', 'EG', 'Egypt'),
 '603': ('+213', 'DZ', 'Algeria'),
 '604': ('+212', 'MA', 'Morocco'),
 '605': ('+216', 'TN', 'Tunisia'),
 '606': ('+218', 'LY', 'Libya'),
 '607': ('+220', 'GM', 'Gambia'),
 '608': ('+221', 'SN', 'Senegal'),
 '609': ('+222', 'MR', 'Mauritania'),
 '610': ('+223', 'ML', 'Mali'),
 '611': ('+224', 'GN', 'Guinea'),
 '612': ('+225', 'CI', 'Ivory Coast'),
 '613': ('+226', 'BF', 'Burkina Faso'),
 '614': ('+227', 'NE', 'Niger'),
 '615': ('+228', 'TG', 'Togo'),
 '616': ('+229', 'BJ', 'Benin'),
 '617': ('+230', 'MU', 'Mauritius'),
 '618': ('+231', 'LR', 'Liberia'),
 '619': ('+232', 'SL', 'Sierra Leone'),
 '620': ('+233', 'GH', 'Ghana'),
 '621': ('+234', 'NG', 'Nigeria'),
 '622': ('+235', 'TD', 'Chad'),
 '623': ('+236', 'CF', 'Central African Rep.'),
 '624': ('+237', 'CM', 'Cameroon'),
 '625': ('+238', 'CV', 'Cape Verde'),
 '626': ('+239', 'ST', 'Sao Tome & Principe'),
 '627': ('+240', 'GQ', 'Equatorial Guinea'),
 '628': ('+241', 'GA', 'Gabon'),
 '629': ('+242', 'CG', 'Congo, Republic'),
 '630': ('+243', 'CD', 'Congo, Dem. Rep.'),
 '631': ('+244', 'AO', 'Angola'),
 '632': ('+245', 'GW', 'Guinea-Bissau'),
 '633': ('+248', 'SC', 'Seychelles'),
 '634': ('+249', 'SD', 'Sudan'),
 '635': ('+250', 'RW', 'Rwanda'),
 '636': ('+251', 'ET', 'Ethiopia'),
 '637': ('+252', 'SO', 'Somalia'),
 '638': ('+253', 'DJ', 'Djibouti'),
 '639': ('+254', 'KE', 'Kenya'),
 '640': ('+255', 'TZ', 'Tanzania'),
 '641': ('+256', 'UG', 'Uganda'),
 '642': ('+257', 'BI', 'Burundi'),
 '643': ('+258', 'MZ', 'Mozambique'),
 '645': ('+260', 'ZM', 'Zambia'),
 '646': ('+261', 'MG', 'Madagascar'),
 '647': ('+262', 'RE', 'Reunion'),
 '648': ('+263', 'ZW', 'Zimbabwe'),
 '649': ('+264', 'NA', 'Namibia'),
 '650': ('+265', 'MW', 'Malawi'),
 '651': ('+266', 'LS', 'Lesotho'),
 '652': ('+267', 'BW', 'Botswana'),
 '653': ('+268', 'SZ', 'Swaziland'),
 '654': ('+269', 'KM', 'Comoros'),
 '655': ('+27', 'ZA', 'South Africa'),
 '657': ('+291', 'ER', 'Eritrea'),
 '659': ('+211', 'SS', 'South Sudan (Republic of)'),
 '702': ('+501', 'BZ', 'Belize'),
 '704': ('+502', 'GT', 'Guatemala'),
 '706': ('+503', 'SV', 'El Salvador'),
 '708': ('+504', 'HN', 'Honduras'),
 '710': ('+505', 'NI', 'Nicaragua'),
 '712': ('+506', 'CR', 'Costa Rica'),
 '714': ('+507', 'PA', 'Panama'),
 '716': ('+51', 'PE', 'Peru'),
 '722': ('+54', 'AR', 'Argentina Republic'),
 '724': ('+55', 'BR', 'Brazil'),
 '730': ('+56', 'CL', 'Chile'),
 '732': ('+57', 'CO', 'Colombia'),
 '734': ('+58', 'VE', 'Venezuela'),
 '736': ('+591', 'BO', 'Bolivia'),
 '738': ('+592', 'GY', 'Guyana'),
 '740': ('+593', 'EC', 'Ecuador'),
 '744': ('+595', 'PY', 'Paraguay'),
 '746': ('+597', 'SR', 'Suriname'),
 '748': ('+598', 'UY', 'Uruguay'),
 '750': ('+500', 'FK', 'Falkland Islands (Malvinas)'),
 '901': ('+882', 'Intl', 'International Networks')
 }


MNC_dict = {
 '20201': ('+30', 'GR', 'Greece', 'Cosmote'),
 '20202': ('+30', 'GR', 'Greece', 'Cosmote'),
 '20203': ('+30', 'GR', 'Greece', 'OTE Hellenic Telecommunications Organization SA'),
 '20204': ('+30', 'GR', 'Greece', 'Organismos Sidirodromon Ellados (OSE)'),
 '20205': ('+30', 'GR', 'Greece', 'Vodafone'),
 '20207': ('+30', 'GR', 'Greece', 'AMD Telecom SA'),
 '20209': ('+30', 'GR', 'Greece', 'Tim/Wind'),
 '20210': ('+30', 'GR', 'Greece', 'Tim/Wind'),
 '20214': ('+30', 'GR', 'Greece', 'CyTa Mobile'),
 '20402': ('+31', 'NL', 'Netherlands', 'Tele2'),
 '20403': ('+31', 'NL', 'Netherlands', 'Voiceworks Mobile BV'),
 '20404': ('+31', 'NL', 'Netherlands', 'Vodafone Libertel'),
 '20405': ('+31', 'NL', 'Netherlands', 'Elephant Talk Communications Premium Rate Services Netherlands BV'),
 '20406': ('+31', 'NL', 'Netherlands', 'Mundio/Vectone Mobile'),
 '20407': ('+31', 'NL', 'Netherlands', 'Teleena Holding BV'),
 '20408': ('+31', 'NL', 'Netherlands', 'KPN Telecom B.V.'),
 '20409': ('+31', 'NL', 'Netherlands', 'Lycamobile Ltd'),
 '20410': ('+31', 'NL', 'Netherlands', 'KPN Telecom B.V.'),
 '20412': ('+31', 'NL', 'Netherlands', 'KPN/Telfort'),
 '20414': ('+31', 'NL', 'Netherlands', '6GMOBILE BV'),
 '20415': ('+31', 'NL', 'Netherlands', 'Ziggo BV'),
 '20416': ('+31', 'NL', 'Netherlands', 'T-Mobile B.V.'),
 '20417': ('+31', 'NL', 'Netherlands', 'Intercity Mobile Communications BV'),
 '20418': ('+31', 'NL', 'Netherlands', 'UPC Nederland BV'),
 '20420': ('+31', 'NL', 'Netherlands', 'T-mobile/former Orange'),
 '20421': ('+31', 'NL', 'Netherlands', 'NS Railinfrabeheer B.V.'),
 '20423': ('+31', 'NL', 'Netherlands', 'Aspider Solutions'),
 '20424': ('+31', 'NL', 'Netherlands', 'Private Mobility Nederland BV'),
 '20428': ('+31', 'NL', 'Netherlands', 'Lancelot BV'),
 '20468': ('+31', 'NL', 'Netherlands', 'Unify Mobile'),
 '20469': ('+31', 'NL', 'Netherlands', 'KPN Telecom B.V.'),
 '20498': ('+31', 'NL', 'Netherlands', 'T-Mobile B.V.'),
 '20601': ('+32', 'BE', 'Belgium', 'Belgacom/Proximus'),
 '20602': ('+32', 'BE', 'Belgium', 'SNCT/NMBS'),
 '20605': ('+32', 'BE', 'Belgium', 'Telenet BidCo NV'),
 '20606': ('+32', 'BE', 'Belgium', 'Lycamobile Belgium'),
 '20610': ('+32', 'BE', 'Belgium', 'Mobistar/Orange'),
 '20620': ('+32', 'BE', 'Belgium', 'Base/KPN'),
 '20800': ('+33', 'FR', 'France', 'Tel/Tel'),
 '20801': ('+33', 'FR', 'France', 'Orange'),
 '20802': ('+33', 'FR', 'France', 'Orange'),
 '20803': ('+33', 'FR', 'France', 'MobiquiThings'),
 '20804': ('+33', 'FR', 'France', 'SISTEER'),
 '20805': ('+33', 'FR', 'France', 'GlobalStar'),
 '20806': ('+33', 'FR', 'France', 'GlobalStar'),
 '20807': ('+33', 'FR', 'France', 'GlobalStar'),
 '20809': ('+33', 'FR', 'France', 'S.F.R.'),
 '20810': ('+33', 'FR', 'France', 'S.F.R.'),
 '20811': ('+33', 'FR', 'France', 'S.F.R.'),
 '20813': ('+33', 'FR', 'France', 'S.F.R.'),
 '20814': ('+33', 'FR', 'France', 'Lliad/FREE Mobile'),
 '20815': ('+33', 'FR', 'France', 'Lliad/FREE Mobile'),
 '20816': ('+33', 'FR', 'France', 'Lliad/FREE Mobile'),
 '20817': ('+33', 'FR', 'France', 'Legos - Local Exchange Global Operation Services SA'),
 '20820': ('+33', 'FR', 'France', 'Bouygues Telecom'),
 '20821': ('+33', 'FR', 'France', 'Bouygues Telecom'),
 '20822': ('+33', 'FR', 'France', 'Transatel SA'),
 '20823': ('+33', 'FR', 'France', 'Virgin Mobile/Omer'),
 '20824': ('+33', 'FR', 'France', 'MobiquiThings'),
 '20825': ('+33', 'FR', 'France', 'Lycamobile SARL'),
 '20826': ('+33', 'FR', 'France', 'NRJ'),
 '20827': ('+33', 'FR', 'France', 'AFONE SA'),
 '20828': ('+33', 'FR', 'France', 'Astrium'),
 '20829': ('+33', 'FR', 'France', 'Orange'),
 '20831': ('+33', 'FR', 'France', 'Mundio Mobile (France) Ltd'),
 '20888': ('+33', 'FR', 'France', 'Bouygues Telecom'),
 '20889': ('+33', 'FR', 'France', 'Virgin Mobile/Omer'),
 '20891': ('+33', 'FR', 'France', 'Orange'),
 '20892': ('+33', 'FR', 'France', 'Association Plate-forme Telecom'),
 '21201': ('+377', 'MC', 'Monaco', 'Monaco Telecom'),
 '21210': ('+377', 'MC', 'Monaco', 'Monaco Telecom'),
 '21303': ('+376', 'AD', 'Andorra', 'Mobiland'),
 '21401': ('+34', 'ES', 'Spain', 'Vodafone'),
 '21403': ('+34', 'ES', 'Spain', 'Orange'),
 '21404': ('+34', 'ES', 'Spain', 'Yoigo'),
 '21405': ('+34', 'ES', 'Spain', 'Movistar'),
 '21406': ('+34', 'ES', 'Spain', 'Vodafone Enabler Espana SL'),
 '21407': ('+34', 'ES', 'Spain', 'Movistar'),
 '21408': ('+34', 'ES', 'Spain', 'Euskaltel SA'),
 '21409': ('+34', 'ES', 'Spain', 'Orange'),
 '21411': ('+34', 'ES', 'Spain', 'Orange'),
 '21415': ('+34', 'ES', 'Spain', 'BT Espana  SAU'),
 '21416': ('+34', 'ES', 'Spain', 'Telecable de Asturias SA'),
 '21417': ('+34', 'ES', 'Spain', 'R Cable y Telec. Galicia SA'),
 '21418': ('+34', 'ES', 'Spain', 'Cableuropa SAU (ONO)'),
 '21419': ('+34', 'ES', 'Spain', 'Simyo/KPN'),
 '21420': ('+34', 'ES', 'Spain', 'fonYou Wireless SL'),
 '21421': ('+34', 'ES', 'Spain', 'Jazz Telecom SAU'),
 '21422': ('+34', 'ES', 'Spain', 'Digi Spain Telecom SL'),
 '21423': ('+34', 'ES', 'Spain', 'Lycamobile SL'),
 '21425': ('+34', 'ES', 'Spain', 'Lycamobile SL'),
 '21426': ('+34', 'ES', 'Spain', 'Lleida'),
 '21427': ('+34', 'ES', 'Spain', 'Truphone'),
 '21601': ('+36', 'HU', 'Hungary', 'Pannon/Telenor'),
 '21630': ('+36', 'HU', 'Hungary', 'T-mobile/Magyar'),
 '21670': ('+36', 'HU', 'Hungary', 'Vodafone'),
 '21671': ('+36', 'HU', 'Hungary', 'UPC Magyarorszag Kft.'),
 '21803': ('+387', 'BA', 'Bosnia & Herzegov.', 'Eronet Mobile'),
 '21805': ('+387', 'BA', 'Bosnia & Herzegov.', 'M-Tel'),
 '21890': ('+387', 'BA', 'Bosnia & Herzegov.', 'BH Mobile'),
 '21901': ('+385', 'HR', 'Croatia', 'T-Mobile/Cronet'),
 '21902': ('+385', 'HR', 'Croatia', 'Tele2'),
 '21910': ('+385', 'HR', 'Croatia', 'VIPnet d.o.o.'),
 '22001': ('+381', 'RS', 'Serbia', 'Telenor/Mobtel'),
 '22002': ('+381', 'RS', 'Serbia', 'Telenor/Mobtel'),
 '22003': ('+381', 'RS', 'Serbia', 'MTS/Telekom Srbija'),
 '22005': ('+381', 'RS', 'Serbia', 'VIP Mobile'),
 '22200': ('+39', 'IT', 'Italy', 'VOIP Line'),
 '22201': ('+39', 'IT', 'Italy', 'TIM'),
 '22202': ('+39', 'IT', 'Italy', 'Elsacom'),
 '22206': ('+39', 'IT', 'Italy', 'Vodafone'),
 '22207': ('+39', 'IT', 'Italy', 'Noverca Italia Srl'),
 '22208': ('+39', 'IT', 'Italy', 'Fastweb SpA'),
 '22210': ('+39', 'IT', 'Italy', 'Vodafone'),
 '22230': ('+39', 'IT', 'Italy', 'RFI Rete Ferroviaria Italiana SpA'),
 '22233': ('+39', 'IT', 'Italy', 'PosteMobile SpA'),
 '22234': ('+39', 'IT', 'Italy', 'BT Italia SpA'),
 '22235': ('+39', 'IT', 'Italy', 'Lycamobile Srl'),
 '22243': ('+39', 'IT', 'Italy', 'Telecom Italia Mobile SpA'),
 '22244': ('+39', 'IT', 'Italy', 'WIND (Blu) -'),
 '22248': ('+39', 'IT', 'Italy', 'Telecom Italia Mobile SpA'),
 '22277': ('+39', 'IT', 'Italy', 'IPSE 2000'),
 '22288': ('+39', 'IT', 'Italy', 'WIND (Blu) -'),
 '22299': ('+39', 'IT', 'Italy', 'Hi3G'),
 '22601': ('+40', 'RO', 'Romania', 'Vodafone'),
 '22602': ('+40', 'RO', 'Romania', 'Romtelecom SA'),
 '22603': ('+40', 'RO', 'Romania', 'Cosmote'),
 '22604': ('+40', 'RO', 'Romania', 'Telemobil/Zapp'),
 '22605': ('+40', 'RO', 'Romania', 'RCS&RDS Digi Mobile'),
 '22606': ('+40', 'RO', 'Romania', 'Telemobil/Zapp'),
 '22610': ('+40', 'RO', 'Romania', 'Orange'),
 '22611': ('+40', 'RO', 'Romania', 'Enigma Systems'),
 '22801': ('+41', 'CH', 'Switzerland', 'Swisscom'),
 '22802': ('+41', 'CH', 'Switzerland', 'TDC Sunrise'),
 '22803': ('+41', 'CH', 'Switzerland', 'Salt/Orange'),
 '22805': ('+41', 'CH', 'Switzerland', 'Comfone AG'),
 '22807': ('+41', 'CH', 'Switzerland', 'TDC Sunrise'),
 '22808': ('+41', 'CH', 'Switzerland', 'TDC Sunrise'),
 '22809': ('+41', 'CH', 'Switzerland', 'Comfone AG'),
 '22812': ('+41', 'CH', 'Switzerland', 'TDC Sunrise'),
 '22851': ('+41', 'CH', 'Switzerland', 'BebbiCell AG'),
 '22852': ('+41', 'CH', 'Switzerland', 'Mundio Mobile AG'),
 '22853': ('+41', 'CH', 'Switzerland', 'upc cablecom GmbH'),
 '22854': ('+41', 'CH', 'Switzerland', 'Lycamobile AG'),
 '23001': ('+420', 'CZ', 'Czech Rep.', 'T-Mobile / RadioMobil'),
 '23002': ('+420', 'CZ', 'Czech Rep.', 'O2'),
 '23003': ('+420', 'CZ', 'Czech Rep.', 'Vodafone'),
 '23004': ('+420', 'CZ', 'Czech Rep.', 'Ufone'),
 '23005': ('+420', 'CZ', 'Czech Rep.', 'Travel Telekommunikation s.r.o.'),
 '23008': ('+420', 'CZ', 'Czech Rep.', 'Compatel s.r.o.'),
 '23099': ('+420', 'CZ', 'Czech Rep.', 'Vodafone'),
 '23101': ('+421', 'SK', 'Slovakia', 'Orange'),
 '23102': ('+421', 'SK', 'Slovakia', 'T-Mobile'),
 '23103': ('+421', 'SK', 'Slovakia', '4Ka'),
 '23104': ('+421', 'SK', 'Slovakia', 'T-Mobile'),
 '23105': ('+421', 'SK', 'Slovakia', 'Orange'),
 '23106': ('+421', 'SK', 'Slovakia', 'O2'),
 '23115': ('+421', 'SK', 'Slovakia', 'Orange'),
 '23199': ('+421', 'SK', 'Slovakia', 'Zeleznice Slovenskej republiky (ZSR)'),
 '23201': ('+43', 'AT', 'Austria', 'A1 MobilKom'),
 '23202': ('+43', 'AT', 'Austria', 'A1 MobilKom'),
 '23203': ('+43', 'AT', 'Austria', 'T-Mobile/Telering'),
 '23204': ('+43', 'AT', 'Austria', 'T-Mobile/Telering'),
 '23205': ('+43', 'AT', 'Austria', '3/Orange/One Connect'),
 '23206': ('+43', 'AT', 'Austria', '3/Orange/One Connect'),
 '23207': ('+43', 'AT', 'Austria', 'T-Mobile/Telering'),
 '23208': ('+43', 'AT', 'Austria', 'A1 MobilKom'),
 '23209': ('+43', 'AT', 'Austria', 'A1 MobilKom'),
 '23210': ('+43', 'AT', 'Austria', 'H3G'),
 '23211': ('+43', 'AT', 'Austria', 'A1 MobilKom'),
 '23212': ('+43', 'AT', 'Austria', '3/Orange/One Connect'),
 '23213': ('+43', 'AT', 'Austria', 'UPC Austria'),
 '23214': ('+43', 'AT', 'Austria', 'H3G'),
 '23215': ('+43', 'AT', 'Austria', 'T-Mobile/Telering'),
 '23217': ('+43', 'AT', 'Austria', 'Spusu/Mass Response'),
 '23219': ('+43', 'AT', 'Austria', 'Tele2'),
 '23401': ('+44', 'GB', 'United Kingdom', 'Mapesbury C. Ltd'),
 '23402': ('+44', 'GB', 'United Kingdom', 'O2 Ltd.'),
 '23403': ('+44', 'GB', 'United Kingdom', 'Airtel/Vodafone'),
 '23407': ('+44', 'GB', 'United Kingdom', 'Cable and Wireless'),
 '23408': ('+44', 'GB', 'United Kingdom', 'OnePhone'),
 '23409': ('+44', 'GB', 'United Kingdom', 'Tismi'),
 '23410': ('+44', 'GB', 'United Kingdom', 'O2 Ltd.'),
 '23411': ('+44', 'GB', 'United Kingdom', 'O2 Ltd.'),
 '23412': ('+44', 'GB', 'United Kingdom', 'Railtrack Plc'),
 '23414': ('+44', 'GB', 'United Kingdom', 'HaySystems'),
 '23415': ('+44', 'GB', 'United Kingdom', 'Vodafone'),
 '23416': ('+44', 'GB', 'United Kingdom', 'Opal Telecom'),
 '23417': ('+44', 'GB', 'United Kingdom', 'FlexTel'),
 '23418': ('+44', 'GB', 'United Kingdom', 'Cloud9/wire9 Tel.'),
 '23419': ('+44', 'GB', 'United Kingdom', 'PMN/Teleware'),
 '23420': ('+44', 'GB', 'United Kingdom', 'H3G Hutchinson'),
 '23422': ('+44', 'GB', 'United Kingdom', 'Routotelecom'),
 '23423': ('+44', 'GB', 'United Kingdom', 'Vectofone Mobile Wifi'),
 '23424': ('+44', 'GB', 'United Kingdom', 'Stour Marine'),
 '23425': ('+44', 'GB', 'United Kingdom', 'Truphone'),
 '23426': ('+44', 'GB', 'United Kingdom', 'Lycamobile'),
 '23427': ('+44', 'GB', 'United Kingdom', 'Vodafone'),
 '23428': ('+44', 'GB', 'United Kingdom', 'Marthon Telecom'),
 '23430': ('+44', 'GB', 'United Kingdom', 'Everyth. Ev.wh./T-Mobile'),
 '23431': ('+44', 'GB', 'United Kingdom', 'Everyth. Ev.wh./T-Mobile'),
 '23432': ('+44', 'GB', 'United Kingdom', 'Everyth. Ev.wh./T-Mobile'),
 '23433': ('+44', 'GB', 'United Kingdom', 'Everyth. Ev.wh./Orange'),
 '23434': ('+44', 'GB', 'United Kingdom', 'Everyth. Ev.wh./Orange'),
 '23435': ('+44', 'GB', 'United Kingdom', 'JSC Ingenicum'),
 '23436': ('+44', 'GB', 'United Kingdom', 'Cable and Wireless Isle of Man'),
 '23437': ('+44', 'GB', 'United Kingdom', 'Synectiv Ltd.'),
 '23450': ('+44', 'GB', 'United Kingdom', 'Jersey Telecom'),
 '23451': ('+44', 'GB', 'United Kingdom', 'Jersey Telecom'),
 '23455': ('+44', 'GB', 'United Kingdom', 'Guernsey Telecoms'),
 '23458': ('+44', 'GB', 'United Kingdom', 'Manx Telecom'),
 '23475': ('+44', 'GB', 'United Kingdom', 'Inquam Telecom Ltd'),
 '23476': ('+44', 'GB', 'United Kingdom', 'BT Group'),
 '23477': ('+44', 'GB', 'United Kingdom', 'BT Group'),
 '23478': ('+44', 'GB', 'United Kingdom', 'Wave Telecom Ltd'),
 '23491': ('+44', 'GB', 'United Kingdom', 'Vodafone'),
 '23492': ('+44', 'GB', 'United Kingdom', 'Cable and Wireless'),
 '23494': ('+44', 'GB', 'United Kingdom', 'H3G Hutchinson'),
 '23502': ('+44', 'GB', 'United Kingdom', 'Everyth. Ev.wh.'),
 '23801': ('+45', 'DK', 'Denmark', 'TDC Denmark'),
 '23802': ('+45', 'DK', 'Denmark', 'Telenor/Sonofon'),
 '23803': ('+45', 'DK', 'Denmark', 'Mach Connectivity ApS'),
 '23804': ('+45', 'DK', 'Denmark', 'NextGen Mobile Ltd (CardBoardFish)'),
 '23805': ('+45', 'DK', 'Denmark', 'ApS KBUS'),
 '23806': ('+45', 'DK', 'Denmark', 'H3G'),
 '23807': ('+45', 'DK', 'Denmark', 'Mundio Mobile'),
 '23810': ('+45', 'DK', 'Denmark', 'TDC Denmark'),
 '23812': ('+45', 'DK', 'Denmark', 'Lycamobile Ltd'),
 '23820': ('+45', 'DK', 'Denmark', 'Telia'),
 '23823': ('+45', 'DK', 'Denmark', 'Banedanmark'),
 '23828': ('+45', 'DK', 'Denmark', 'CoolTEL ApS'),
 '23830': ('+45', 'DK', 'Denmark', 'Telia'),
 '23877': ('+45', 'DK', 'Denmark', 'Telenor/Sonofon'),
 '24001': ('+46', 'SE', 'Sweden', 'Telia Mobile'),
 '24002': ('+46', 'SE', 'Sweden', 'H3G Access AB'),
 '24004': ('+46', 'SE', 'Sweden', 'H3G Access AB'),
 '24005': ('+46', 'SE', 'Sweden', 'Svenska UMTS-N'),
 '24006': ('+46', 'SE', 'Sweden', 'Telenor (Vodafone)'),
 '24007': ('+46', 'SE', 'Sweden', 'Tele2 Sverige AB'),
 '24008': ('+46', 'SE', 'Sweden', 'Telenor (Vodafone)'),
 '24010': ('+46', 'SE', 'Sweden', 'Spring Mobil AB'),
 '24011': ('+46', 'SE', 'Sweden', 'Lindholmen Science Park AB'),
 '24012': ('+46', 'SE', 'Sweden', 'Lycamobile Ltd'),
 '24013': ('+46', 'SE', 'Sweden', 'Ventelo Sverige AB'),
 '24014': ('+46', 'SE', 'Sweden', 'TDC Sverige AB'),
 '24015': ('+46', 'SE', 'Sweden', 'Wireless Maingate Nordic AB'),
 '24016': ('+46', 'SE', 'Sweden', '42 Telecom AB'),
 '24017': ('+46', 'SE', 'Sweden', 'Gotalandsnatet AB'),
 '24018': ('+46', 'SE', 'Sweden', 'Generic Mobile Systems Sweden AB'),
 '24019': ('+46', 'SE', 'Sweden', 'Mundio Mobile (Sweden) Ltd'),
 '24020': ('+46', 'SE', 'Sweden', 'Wireless Maingate AB'),
 '24022': ('+46', 'SE', 'Sweden', 'Eu Tel AB'),
 '24023': ('+46', 'SE', 'Sweden', 'Infobip Ltd.'),
 '24024': ('+46', 'SE', 'Sweden', 'Telenor (Vodafone)'),
 '24025': ('+46', 'SE', 'Sweden', 'Digitel Mobile Srl'),
 '24026': ('+46', 'SE', 'Sweden', 'Beepsend'),
 '24027': ('+46', 'SE', 'Sweden', 'Fogg Mobile AB'),
 '24028': ('+46', 'SE', 'Sweden', 'CoolTEL Aps'),
 '24029': ('+46', 'SE', 'Sweden', 'Mercury International Carrier Services'),
 '24030': ('+46', 'SE', 'Sweden', 'NextGen Mobile Ltd (CardBoardFish)'),
 '24035': ('+46', 'SE', 'Sweden', '42 Telecom AB'),
 '24036': ('+46', 'SE', 'Sweden', 'ID Mobile'),
 '24201': ('+47', 'NO', 'Norway', 'Telenor'),
 '242017': ('+47', 'NO', 'Norway', 'Ventelo AS'),
 '24202': ('+47', 'NO', 'Norway', 'Netcom'),
 '24203': ('+47', 'NO', 'Norway', 'Teletopia'),
 '24204': ('+47', 'NO', 'Norway', 'Tele2'),
 '24205': ('+47', 'NO', 'Norway', 'Network Norway AS'),
 '24206': ('+47', 'NO', 'Norway', 'ICE Nordisk Mobiltelefon AS'),
 '24207': ('+47', 'NO', 'Norway', 'Ventelo AS'),
 '24208': ('+47', 'NO', 'Norway', 'TDC Mobil A/S'),
 '24209': ('+47', 'NO', 'Norway', 'Com4 AS'),
 '24212': ('+47', 'NO', 'Norway', 'Telenor'),
 '24220': ('+47', 'NO', 'Norway', 'Jernbaneverket (GSM-R)'),
 '24221': ('+47', 'NO', 'Norway', 'Jernbaneverket (GSM-R)'),
 '24222': ('+47', 'NO', 'Norway', 'Network Norway AS'),
 '24223': ('+47', 'NO', 'Norway', 'Lycamobile Ltd'),
 '24403': ('+358', 'FI', 'Finland', 'DNA/Finnet'),
 '24404': ('+358', 'FI', 'Finland', 'DNA/Finnet'),
 '24405': ('+358', 'FI', 'Finland', 'Elisa/Saunalahti'),
 '24409': ('+358', 'FI', 'Finland', 'Nokia Oyj'),
 '24410': ('+358', 'FI', 'Finland', 'TDC Oy Finland'),
 '24411': ('+358', 'FI', 'Finland', 'Mundio Mobile (Finland) Ltd'),
 '24412': ('+358', 'FI', 'Finland', 'DNA/Finnet'),
 '24413': ('+358', 'FI', 'Finland', 'DNA/Finnet'),
 '24414': ('+358', 'FI', 'Finland', 'Alands'),
 '24421': ('+358', 'FI', 'Finland', 'Elisa/Saunalahti'),
 '24426': ('+358', 'FI', 'Finland', 'Compatel Ltd'),
 '24482': ('+358', 'FI', 'Finland', 'ID-Mobile'),
 '24491': ('+358', 'FI', 'Finland', 'TeliaSonera'),
 '24601': ('+370', 'LT', 'Lithuania', 'Omnitel'),
 '24602': ('+370', 'LT', 'Lithuania', 'Bite'),
 '24603': ('+370', 'LT', 'Lithuania', 'Tele2'),
 '24701': ('+371', 'LV', 'Latvia', 'Latvian Mobile Phone'),
 '24702': ('+371', 'LV', 'Latvia', 'Tele2'),
 '24703': ('+371', 'LV', 'Latvia', 'TRIATEL/Telekom Baltija'),
 '24705': ('+371', 'LV', 'Latvia', 'Bite'),
 '24706': ('+371', 'LV', 'Latvia', 'SIA Rigatta'),
 '24707': ('+371', 'LV', 'Latvia', 'SIA Master Telecom'),
 '24708': ('+371', 'LV', 'Latvia', 'SIA IZZI'),
 '24709': ('+371', 'LV', 'Latvia', 'SIA Camel Mobile'),
 '24801': ('+372', 'EE', 'Estonia', 'EMT GSM'),
 '24802': ('+372', 'EE', 'Estonia', 'Radiolinja Eesti'),
 '24803': ('+372', 'EE', 'Estonia', 'Tele2 Eesti AS'),
 '24804': ('+372', 'EE', 'Estonia', 'Top Connect OU'),
 '25001': ('+79', 'RU', 'Russian Federation', 'MTS'),
 '25002': ('+79', 'RU', 'Russian Federation', 'Megafon'),
 '25003': ('+79', 'RU', 'Russian Federation', 'NCC'),
 '25004': ('+79', 'RU', 'Russian Federation', 'Sibchallenge'),
 '25005': ('+79', 'RU', 'Russian Federation', 'Yenisey Telecom'),
 '25007': ('+79', 'RU', 'Russian Federation', 'ZAO SMARTS'),
 '25010': ('+79', 'RU', 'Russian Federation', 'DTC/Don Telecom'),
 '25011': ('+79', 'RU', 'Russian Federation', 'Orensot'),
 '25012': ('+79', 'RU', 'Russian Federation', 'Baykal Westcom'),
 '25013': ('+79', 'RU', 'Russian Federation', 'Kuban GSM'),
 '25015': ('+79', 'RU', 'Russian Federation', 'ZAO SMARTS'),
 '25016': ('+79', 'RU', 'Russian Federation', 'NTC'),
 '25017': ('+79', 'RU', 'Russian Federation', 'UralTel'),
 '25019': ('+79', 'RU', 'Russian Federation', 'OJSC Altaysvyaz'),
 '25020': ('+79', 'RU', 'Russian Federation', 'Tele2/ECC/Volgogr.'),
 '25028': ('+79', 'RU', 'Russian Federation', 'BeeLine/VimpelCom'),
 '25035': ('+79', 'RU', 'Russian Federation', 'MOTIV/LLC Ekaterinburg-2000'),
 '25039': ('+79', 'RU', 'Russian Federation', 'UralTel'),
 '25044': ('+79', 'RU', 'Russian Federation', 'StavTelesot'),
 '25092': ('+79', 'RU', 'Russian Federation', 'Printelefone'),
 '25093': ('+79', 'RU', 'Russian Federation', 'Telecom XXL'),
 '25099': ('+79', 'RU', 'Russian Federation', 'BeeLine/VimpelCom'),
 '25501': ('+380', 'UA', 'Ukraine', 'UMC/MTS'),
 '25502': ('+380', 'UA', 'Ukraine', 'Beeline'),
 '25503': ('+380', 'UA', 'Ukraine', 'KyivStar'),
 '25504': ('+380', 'UA', 'Ukraine', 'Intertelecom Ltd (IT)'),
 '25505': ('+380', 'UA', 'Ukraine', 'Golden Telecom'),
 '25506': ('+380', 'UA', 'Ukraine', 'Astelit/LIFE'),
 '25507': ('+380', 'UA', 'Ukraine', 'TriMob LLC'),
 '25521': ('+380', 'UA', 'Ukraine', 'Telesystems Of Ukraine CJSC (TSU)'),
 '25539': ('+380', 'UA', 'Ukraine', 'Golden Telecom'),
 '25550': ('+380', 'UA', 'Ukraine', 'UMC/MTS'),
 '25567': ('+380', 'UA', 'Ukraine', 'KyivStar'),
 '25568': ('+380', 'UA', 'Ukraine', 'Beeline'),
 '25701': ('+375', 'BY', 'Belarus', 'Mobile Digital Communications'),
 '25702': ('+375', 'BY', 'Belarus', 'MTS'),
 '25703': ('+375', 'BY', 'Belarus', 'BelCel JV'),
 '25704': ('+375', 'BY', 'Belarus', 'BeST'),
 '25901': ('+373', 'MD', 'Moldova', 'Orange/Voxtel'),
 '25902': ('+373', 'MD', 'Moldova', 'Moldcell'),
 '25903': ('+373', 'MD', 'Moldova', 'IDC/Unite'),
 '25904': ('+373', 'MD', 'Moldova', 'Eventis Mobile'),
 '25905': ('+373', 'MD', 'Moldova', 'IDC/Unite'),
 '25999': ('+373', 'MD', 'Moldova', 'IDC/Unite'),
 '26001': ('+48', 'PL', 'Poland', 'Polkomtel/Plus'),
 '26002': ('+48', 'PL', 'Poland', 'T-Mobile/ERA'),
 '26003': ('+48', 'PL', 'Poland', 'Orange/IDEA/Centertel'),
 '26004': ('+48', 'PL', 'Poland', 'Tele2'),
 '26005': ('+48', 'PL', 'Poland', 'Orange/IDEA/Centertel'),
 '26006': ('+48', 'PL', 'Poland', 'Play/P4'),
 '26007': ('+48', 'PL', 'Poland', 'Play/P4'),
 '26008': ('+48', 'PL', 'Poland', 'e-Telko'),
 '26009': ('+48', 'PL', 'Poland', 'Lycamobile'),
 '26010': ('+48', 'PL', 'Poland', 'Sferia'),
 '26011': ('+48', 'PL', 'Poland', 'NORDISK Polska'),
 '26012': ('+48', 'PL', 'Poland', 'Cyfrowy POLSAT S.A.'),
 '26013': ('+48', 'PL', 'Poland', 'Sferia'),
 '26014': ('+48', 'PL', 'Poland', 'Sferia'),
 '26015': ('+48', 'PL', 'Poland', 'Tele2'),
 '26016': ('+48', 'PL', 'Poland', 'Mobyland'),
 '26017': ('+48', 'PL', 'Poland', 'Aero2 SP.'),
 '26018': ('+48', 'PL', 'Poland', 'AMD Telecom.'),
 '26034': ('+48', 'PL', 'Poland', 'T-Mobile/ERA'),
 '26035': ('+48', 'PL', 'Poland', 'PKP Polskie Linie Kolejowe S.A.'),
 '26036': ('+48', 'PL', 'Poland', 'Mundio Mobile Sp. z o.o.'),
 '26038': ('+48', 'PL', 'Poland', 'CallFreedom Sp. z o.o.'),
 '26098': ('+48', 'PL', 'Poland', 'Play/P4'),
 '26201': ('+49', 'DE', 'Germany', 'T-mobile/Telekom'),
 '26202': ('+49', 'DE', 'Germany', 'Vodafone D2'),
 '26203': ('+49', 'DE', 'Germany', 'E-Plus'),
 '26204': ('+49', 'DE', 'Germany', 'Vodafone D2'),
 '26205': ('+49', 'DE', 'Germany', 'E-Plus'),
 '26206': ('+49', 'DE', 'Germany', 'T-mobile/Telekom'),
 '26207': ('+49', 'DE', 'Germany', 'O2'),
 '26208': ('+49', 'DE', 'Germany', 'O2'),
 '26209': ('+49', 'DE', 'Germany', 'Vodafone D2'),
 '26210': ('+49', 'DE', 'Germany', 'DB Netz AG'),
 '26211': ('+49', 'DE', 'Germany', 'O2'),
 '26212': ('+49', 'DE', 'Germany', 'E-Plus'),
 '26213': ('+49', 'DE', 'Germany', 'Mobilcom'),
 '26214': ('+49', 'DE', 'Germany', 'Group 3G UMTS'),
 '26216': ('+49', 'DE', 'Germany', 'Telogic/ViStream'),
 '26217': ('+49', 'DE', 'Germany', 'E-Plus'),
 '26220': ('+49', 'DE', 'Germany', 'E-Plus'),
 '26242': ('+49', 'DE', 'Germany', 'Vodafone D2'),
 '26243': ('+49', 'DE', 'Germany', 'Lycamobile'),
 '26277': ('+49', 'DE', 'Germany', 'E-Plus'),
# '262n/a': ('+49', 'DE', 'Germany', 'Talkline'),
 '26601': ('+350', 'GI', 'Gibraltar', 'Gibtel GSM'),
 '26606': ('+350', 'GI', 'Gibraltar', 'CTS Mobile'),
 '26609': ('+350', 'GI', 'Gibraltar', 'eazi telecom'),
 '26801': ('+351', 'PT', 'Portugal', 'Vodafone'),
 '26803': ('+351', 'PT', 'Portugal', 'NOS/Optimus'),
 '26804': ('+351', 'PT', 'Portugal', 'CTT - Correios de Portugal SA'),
 '26806': ('+351', 'PT', 'Portugal', 'MEO/TMN'),
 '26807': ('+351', 'PT', 'Portugal', 'NOS/Optimus'),
 '27001': ('+352', 'LU', 'Luxembourg', 'P+T/Post LUXGSM'),
 '27077': ('+352', 'LU', 'Luxembourg', 'Millicom Tango GSM'),
 '27099': ('+352', 'LU', 'Luxembourg', 'Orange/VOXmobile S.A.'),
 '27201': ('+353', 'IE', 'Ireland', 'Vodafone Eircell'),
 '27202': ('+353', 'IE', 'Ireland', 'Three/O2/Digifone'),
 '27203': ('+353', 'IE', 'Ireland', 'Meteor Mobile Ltd.'),
 '27204': ('+353', 'IE', 'Ireland', 'Access Telecom Ltd.'),
 '27205': ('+353', 'IE', 'Ireland', 'Three/H3G'),
 '27207': ('+353', 'IE', 'Ireland', 'eircom Ltd'),
 '27209': ('+353', 'IE', 'Ireland', 'Clever Communications Ltd'),
 '27211': ('+353', 'IE', 'Ireland', 'Liffey Telecom'),
 '27213': ('+353', 'IE', 'Ireland', 'Lycamobile'),
 '27401': ('+354', 'IS', 'Iceland', 'Siminn'),
 '27402': ('+354', 'IS', 'Iceland', 'Vodafone/Tal hf'),
 '27403': ('+354', 'IS', 'Iceland', 'Vodafone/Tal hf'),
 '27404': ('+354', 'IS', 'Iceland', 'VIKING/IMC'),
 '27405': ('+354', 'IS', 'Iceland', 'Vodafone/Tal hf'),
 '27407': ('+354', 'IS', 'Iceland', 'IceCell'),
 '27408': ('+354', 'IS', 'Iceland', 'Siminn'),
 '27409': ('+354', 'IS', 'Iceland', 'Amitelo'),
 '27411': ('+354', 'IS', 'Iceland', 'NOVA'),
 '27601': ('+355', 'AL', 'Albania', 'AMC/Cosmote'),
 '27602': ('+355', 'AL', 'Albania', 'Vodafone'),
 '27603': ('+355', 'AL', 'Albania', 'Eagle Mobile'),
 '27604': ('+355', 'AL', 'Albania', 'PLUS Communication Sh.a'),
 '27801': ('+356', 'MT', 'Malta', 'Vodafone'),
 '27821': ('+356', 'MT', 'Malta', 'GO Mobile'),
 '27877': ('+356', 'MT', 'Malta', 'Melita'),
 '28001': ('+357', 'CY', 'Cyprus', 'Vodafone/CyTa'),
 '28010': ('+357', 'CY', 'Cyprus', 'MTN/Areeba'),
 '28020': ('+357', 'CY', 'Cyprus', 'PrimeTel PLC'),
 '28201': ('+995', 'GE', 'Georgia', 'Geocell Ltd.'),
 '28202': ('+995', 'GE', 'Georgia', 'Magti GSM Ltd.'),
 '28203': ('+995', 'GE', 'Georgia', 'Iberiatel Ltd.'),
 '28204': ('+995', 'GE', 'Georgia', 'MobiTel/Beeline'),
 '28205': ('+995', 'GE', 'Georgia', 'Silknet'),
 '28301': ('+374', 'AM', 'Armenia', 'ArmenTel/Beeline'),
 '28304': ('+374', 'AM', 'Armenia', 'Karabakh Telecom'),
 '28305': ('+374', 'AM', 'Armenia', 'Vivacell'),
 '28310': ('+374', 'AM', 'Armenia', 'Orange'),
 '28401': ('+359', 'BG', 'Bulgaria', 'MobilTel AD'),
 '28403': ('+359', 'BG', 'Bulgaria', 'BTC Mobile EOOD (vivatel)'),
 '28405': ('+359', 'BG', 'Bulgaria', 'Telenor/Cosmo/Globul'),
 '28406': ('+359', 'BG', 'Bulgaria', 'BTC Mobile EOOD (vivatel)'),
 '28601': ('+90', 'TR', 'Turkey', 'Turkcell'),
 '28602': ('+90', 'TR', 'Turkey', 'Vodafone-Telsim'),
 '28603': ('+90', 'TR', 'Turkey', 'AVEA/Aria'),
 '28604': ('+90', 'TR', 'Turkey', 'AVEA/Aria'),
 '28801': ('+298', 'FO', 'Faroe Islands', 'Faroese Telecom'),
 '28802': ('+298', 'FO', 'Faroe Islands', 'Kall GSM'),
 '28803': ('+298', 'FO', 'Faroe Islands', 'Edge Mobile Sp/F'),
 '28967': ('+7', 'GE', 'Abkhazia', 'Aquafon'),
 '28968': ('+7', 'GE', 'Abkhazia', 'A-Mobile'),
 '28988': ('+7', 'GE', 'Abkhazia', 'A-Mobile'),
 '29001': ('+299', 'GL', 'Greenland', 'Tele Greenland'),
 '29201': ('+378', 'SM', 'San Marino', 'Prima Telecom'),
 '29310': ('+386', 'SI', 'Slovenia', 'Slovenske zeleznice d.o.o.'),
 '29340': ('+386', 'SI', 'Slovenia', 'SI.Mobil'),
 '29341': ('+386', 'SI', 'Slovenia', 'Mobitel'),
 '29364': ('+386', 'SI', 'Slovenia', 'T-2 d.o.o.'),
 '29370': ('+386', 'SI', 'Slovenia', 'Telemach/TusMobil/VEGA'),
 '29401': ('+389', 'MK', 'Macedonia', 'T-Mobile/Mobimak'),
 '29402': ('+389', 'MK', 'Macedonia', 'ONE/Cosmofone'),
 '29403': ('+389', 'MK', 'Macedonia', 'VIP Mobile'),
 '29475': ('+389', 'MK', 'Macedonia', 'ONE/Cosmofone'),
 '29501': ('+423', 'LI', 'Liechtenstein', 'Swisscom FL AG'),
 '29502': ('+423', 'LI', 'Liechtenstein', 'Orange'),
 '29505': ('+423', 'LI', 'Liechtenstein', 'Telecom FL1 AG'),
 '29506': ('+423', 'LI', 'Liechtenstein', 'CUBIC (Liechtenstein'),
 '29507': ('+423', 'LI', 'Liechtenstein', 'First Mobile AG'),
 '29577': ('+423', 'LI', 'Liechtenstein', 'Alpmobile/Tele2'),
 '29701': ('+382', 'ME', 'Montenegro', 'Telenor/Promonte GSM'),
 '29702': ('+382', 'ME', 'Montenegro', 'Monet/T-mobile'),
 '29703': ('+382', 'ME', 'Montenegro', 'Mtel'),
 '302220': ('+1', 'CA', 'Canada', 'Telus Mobility'),
 '302320': ('+1', 'CA', 'Canada', 'mobilicity'),
 '302360': ('+1', 'CA', 'Canada', 'Clearnet'),
 '302361': ('+1', 'CA', 'Canada', 'Clearnet'),
 '302370': ('+1', 'CA', 'Canada', 'FIDO (Rogers AT&T/ Microcell)'),
 '302380': ('+1', 'CA', 'Canada', 'DMTS Mobility'),
 '302490': ('+1', 'CA', 'Canada', 'WIND'),
 '302500': ('+1', 'CA', 'Canada', 'Videotron'),
 '302610': ('+1', 'CA', 'Canada', 'Bell Mobility'),
 '302630': ('+1', 'CA', 'Canada', 'Bell Aliant'),
 '302640': ('+1', 'CA', 'Canada', 'Latitude Wireless'),
 '302651': ('+1', 'CA', 'Canada', 'Bell Mobility'),
 '302652': ('+1', 'CA', 'Canada', 'BC Tel Mobility'),
 '302653': ('+1', 'CA', 'Canada', 'Telus Mobility'),
 '302654': ('+1', 'CA', 'Canada', 'Sask Tel Mobility'),
 '302655': ('+1', 'CA', 'Canada', 'MTS Mobility'),
 '302656': ('+1', 'CA', 'Canada', 'Tbay Mobility'),
 '302657': ('+1', 'CA', 'Canada', 'Quebectel Mobility'),
 '302660': ('+1', 'CA', 'Canada', 'MTS Mobility'),
 '302670': ('+1', 'CA', 'Canada', 'CityWest Mobility'),
 '302680': ('+1', 'CA', 'Canada', 'Sask Tel Mobility'),
 '302701': ('+1', 'CA', 'Canada', 'NB Tel Mobility'),
 '302702': ('+1', 'CA', 'Canada', 'MT&T Mobility'),
 '302703': ('+1', 'CA', 'Canada', 'New Tel Mobility'),
 '302710': ('+1', 'CA', 'Canada', 'Globalstar Canada'),
 '302720': ('+1', 'CA', 'Canada', 'Rogers AT&T Wireless'),
 '302760': ('+1', 'CA', 'Canada', 'Public Mobile'),
 '30801': ('+508', 'PM', 'St. Pierre & Miquelon', 'Ameris'),
 '310003': ('+1', 'US', 'United States', 'Unknown'),
 '310004': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '310010': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '310011': ('+1', 'US', 'United States', 'Northstar'),
 '310012': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '310013': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '310016': ('+1', 'US', 'United States', 'Leap Wireless International Inc.'),
 '310020': ('+1', 'US', 'United States', 'Union Telephone Co.'),
 '310030': ('+1', 'US', 'United States', ''),
 '310032': ('+1671', 'GU', 'Guam', 'IT&E OverSeas'),
 '310033': ('+1671', 'GU', 'Guam', 'Guam Teleph. Auth.'),
 '310040': ('+1', 'US', 'United States', 'Matanuska Tel. Assn. Inc.'),
 '310050': ('+1', 'US', 'United States', ''),
 '31006': ('+1', 'US', 'United States', 'Consolidated Telcom'),
 '310070': ('+1', 'US', 'United States', 'AT&T Wireless Inc.'),
 '310080': ('+1', 'US', 'United States', ''),
 '310090': ('+1', 'US', 'United States', 'Edge Wireless LLC'),
 '310100': ('+1', 'US', 'United States', 'Plateau Telecommunications Inc.'),
 '310120': ('+1', 'US', 'United States', 'Sprint Spectrum'),
 '310130': ('+1', 'US', 'United States', 'North Carolina RSA 3 Cellular Tel. Co.'),
 '31014': ('+1', 'US', 'United States', 'Testing'),
 '310140': ('+1671', 'GU', 'Guam', 'GTA Wireless'),
 '31015': ('+1', 'US', 'United States', 'Unknown'),
 '310150': ('+1', 'US', 'United States', 'AT&T Wireless Inc.'),
 '310160': ('+1', 'US', 'United States', 'T-Mobile'),
 '310170': ('+1', 'US', 'United States', 'AT&T Wireless Inc.'),
 '310180': ('+1', 'US', 'United States', 'Cingular Wireless'),
 '310190': ('+1', 'US', 'United States', 'Unknown'),
 '310200': ('+1', 'US', 'United States', 'T-Mobile'),
 '310210': ('+1', 'US', 'United States', 'T-Mobile'),
 '310220': ('+1', 'US', 'United States', 'T-Mobile'),
 '31023': ('+1', 'US', 'United States', 'Unknown'),
 '310230': ('+1', 'US', 'United States', 'T-Mobile'),
 '31024': ('+1', 'US', 'United States', 'Unknown'),
 '310240': ('+1', 'US', 'United States', 'T-Mobile'),
 '31025': ('+1', 'US', 'United States', 'Unknown'),
 '310250': ('+1', 'US', 'United States', 'T-Mobile'),
 '31026': ('+1', 'US', 'United States', 'Unknown'),
 '310260': ('+1', 'US', 'United States', 'T-Mobile'),
 '310270': ('+1', 'US', 'United States', 'T-Mobile'),
 '310280': ('+1', 'US', 'United States', 'T-Mobile'),
 '310290': ('+1', 'US', 'United States', 'NEP Cellcorp Inc.'),
 '310300': ('+1', 'US', 'United States', 'T-Mobile'),
 '31031': ('+1', 'US', 'United States', 'T-Mobile'),
 '310310': ('+1', 'US', 'United States', 'T-Mobile'),
 '310320': ('+1', 'US', 'United States', 'Smith Bagley Inc.'),
 '310330': ('+1', 'US', 'United States', 'T-Mobile'),
 '31034': ('+1', 'US', 'United States', 'Nevada Wireless LLC'),
 '310340': ('+1', 'US', 'United States', 'Westlink Communications, LLC'),
 '310350': ('+1', 'US', 'United States', 'Mohave Cellular LP'),
 '310360': ('+1', 'US', 'United States', 'Cellular Network Partnership LLC'),
 '310370': ('+1671', 'GU', 'Guam', 'Docomo'),
 '31038': ('+1', 'US', 'United States', 'USA 3650 AT&T'),
 '310380': ('+1', 'US', 'United States', 'AT&T Wireless Inc.'),
 '310390': ('+1', 'US', 'United States', 'Yorkville Telephone Cooperative'),
 '310400': ('+1', 'US', 'United States', 'Minnesota South. Wirel. Co. / Hickory'),
 '310410': ('+1', 'US', 'United States', 'AT&T Wireless Inc.'),
 '310420': ('+1', 'US', 'United States', 'Cincinnati Bell Wireless LLC'),
 '310430': ('+1', 'US', 'United States', 'GCI Communication Corp.'),
 '310440': ('+1', 'US', 'United States', 'Dobson Cellular Systems'),
 '310450': ('+1', 'US', 'United States', 'Northeast Colorado Cellular Inc.'),
 '31046': ('+1', 'US', 'United States', 'SIMMETRY'),
 '310460': ('+1', 'US', 'United States', 'TMP Corporation'),
 '310470': ('+1671', 'GU', 'Guam', 'Docomo'),
 '310480': ('+1', 'US', 'United States', 'Choice Phone LLC'),
 '310490': ('+1', 'US', 'United States', 'Triton PCS'),
 '310500': ('+1', 'US', 'United States', 'Public Service Cellular Inc.'),
 '310510': ('+1', 'US', 'United States', 'Airtel Wireless LLC'),
 '310520': ('+1', 'US', 'United States', 'VeriSign'),
 '310530': ('+1', 'US', 'United States', 'West Virginia Wireless'),
 '310540': ('+1', 'US', 'United States', ''),
 '310560': ('+1', 'US', 'United States', 'AT&T Wireless Inc.'),
 '310570': ('+1', 'US', 'United States', 'MTPCS LLC'),
 '310580': ('+1', 'US', 'United States', 'PCS ONE'),
 '310590': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '31060': ('+1', 'US', 'United States', 'Consolidated Telcom'),
 '310600': ('+1', 'US', 'United States', 'New-Cell Inc.'),
 '310610': ('+1', 'US', 'United States', 'Elkhart TelCo. / Epic Touch Co.'),
 '310620': ('+1', 'US', 'United States', 'Coleman County Telco /Trans TX'),
 '310630': ('+1', 'US', 'United States', ''),
 '310640': ('+1', 'US', 'United States', ''),
 '310650': ('+1', 'US', 'United States', 'Jasper'),
 '310660': ('+1', 'US', 'United States', 'T-Mobile'),
 '310670': ('+1', 'US', 'United States', 'Northstar'),
 '310680': ('+1', 'US', 'United States', 'AT&T Wireless Inc.'),
 '310690': ('+1', 'US', 'United States', 'Keystone Wireless LLC'),
 '310700': ('+1', 'US', 'United States', 'Cross Valliant Cellular Partnership'),
 '310710': ('+1', 'US', 'United States', 'Arctic Slope Telephone Association Cooperative Inc.'),
 '310730': ('+1', 'US', 'United States', 'United States Cellular Corp.'),
 '310740': ('+1', 'US', 'United States', 'Telemetrix Inc.'),
 '310750': ('+1', 'US', 'United States', 'East Kentucky Network LLC'),
 '310760': ('+1', 'US', 'United States', 'Panhandle Telephone Cooperative Inc.'),
 '310770': ('+1', 'US', 'United States', 'Iowa Wireless Services LLC'),
 '310780': ('+1', 'US', 'United States', 'Message Express Co. / Airlink PCS'),
 '310790': ('+1', 'US', 'United States', ''),
 '310800': ('+1', 'US', 'United States', 'T-Mobile'),
 '310830': ('+1', 'US', 'United States', 'Caprock Cellular Ltd.'),
 '310850': ('+1', 'US', 'United States', 'Aeris Comm. Inc.'),
 '310860': ('+1', 'US', 'United States', 'Texas RSA 15B2 Limited Partnership'),
 '310870': ('+1', 'US', 'United States', 'Kaplan Telephone Company Inc.'),
 '310880': ('+1', 'US', 'United States', ''),
 '310890': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '310900': ('+1', 'US', 'United States', 'Cable & Communications Corp.'),
 '310910': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '310920': ('+1', 'US', 'United States', 'Get Mobile Inc.'),
 '310930': ('+1', 'US', 'United States', ''),
 '310940': ('+1', 'US', 'United States', 'Poka Lambro Telco Ltd.'),
 '310950': ('+1', 'US', 'United States', 'Unknown'),
 '310960': ('+1', 'US', 'United States', 'Uintah Basin Electronics Telecommunications Inc.'),
 '310970': ('+1', 'US', 'United States', ''),
 '310980': ('+1', 'US', 'United States', 'AT&T Wireless Inc.'),
 '310990': ('+1', 'US', 'United States', 'E.N.M.R. Telephone Coop.'),
 '311000': ('+1', 'US', 'United States', ''),
 '311010': ('+1', 'US', 'United States', 'Missouri RSA No 5 Partnership'),
 '311020': ('+1', 'US', 'United States', 'Missouri RSA No 5 Partnership'),
 '311030': ('+1', 'US', 'United States', ''),
 '311040': ('+1', 'US', 'United States', ''),
 '311050': ('+1', 'US', 'United States', 'Thumb Cellular Limited Partnership'),
 '311070': ('+1', 'US', 'United States', 'Wisconsin RSA #7 Limited Partnership'),
 '311080': ('+1', 'US', 'United States', ''),
 '311090': ('+1', 'US', 'United States', ''),
 '311100': ('+1', 'US', 'United States', ''),
 '311110': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311120': ('+1', 'US', 'United States', 'Choice Phone LLC'),
 '311130': ('+1', 'US', 'United States', ''),
 '311140': ('+1', 'US', 'United States', 'Cross Wireless Telephone Co.'),
 '311150': ('+1', 'US', 'United States', ''),
 '311170': ('+1', 'US', 'United States', 'PetroCom'),
 '311190': ('+1', 'US', 'United States', ''),
 '311210': ('+1', 'US', 'United States', ''),
 '311220': ('+1', 'US', 'United States', 'United States Cellular Corp.'),
 '311240': ('+1', 'US', 'United States', ''),
 '311250': ('+1671', 'GU', 'Guam', 'Wave Runner LLC'),
 '311260': ('+1', 'US', 'United States', 'SLO Cellular Inc / Cellular One of San Luis'),
 '311270': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311271': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311272': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311273': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311274': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311275': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311276': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311277': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311278': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311279': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311280': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311281': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311282': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311283': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311284': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311285': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311286': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311287': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311288': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311289': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311300': ('+1', 'US', 'United States', 'Nexus Communications Inc.'),
 '311310': ('+1', 'US', 'United States', 'Lamar County Cellular'),
 '311311': ('+1', 'US', 'United States', 'Farmers'),
 '311330': ('+1', 'US', 'United States', 'Michigan Wireless LLC'),
 '311340': ('+1', 'US', 'United States', 'Illinois Valley Cellular RSA 2 Partnership'),
 '311350': ('+1', 'US', 'United States', 'Sagebrush Cellular Inc.'),
 '311370': ('+1', 'US', 'United States', 'GCI Communication Corp.'),
 '311380': ('+1', 'US', 'United States', ''),
 '311390': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311410': ('+1', 'US', 'United States', 'Iowa RSA No. 2 Limited Partnership'),
 '311420': ('+1', 'US', 'United States', 'Northwest Missouri Cellular Limited Partnership'),
 '311430': ('+1', 'US', 'United States', 'RSA 1 Limited Partnership'),
 '311440': ('+1', 'US', 'United States', 'Bluegrass Wireless LLC'),
 '311460': ('+1', 'US', 'United States', 'Fisher Wireless Services Inc.'),
 '311480': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311481': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311482': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311483': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311484': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311485': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311486': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311487': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311488': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311489': ('+1', 'US', 'United States', 'Verizon Wireless'),
 '311490': ('+1', 'US', 'United States', 'Sprint Spectrum'),
 '311500': ('+1', 'US', 'United States', 'Cambridge Telephone Company Inc.'),
 '311520': ('+1', 'US', 'United States', ''),
 '311540': ('+1', 'US', 'United States', ''),
 '311590': ('+1', 'US', 'United States', 'California RSA No. 3 Limited Partnership'),
 '311610': ('+1', 'US', 'United States', 'North Dakota Network Company'),
 '311650': ('+1', 'US', 'United States', 'United Wireless Communications Inc.'),
 '311660': ('+1', 'US', 'United States', ''),
 '311670': ('+1', 'US', 'United States', 'Pine Belt Cellular, Inc.'),
 '311710': ('+1', 'US', 'United States', 'Northeast Wireless Networks LLC'),
 '311730': ('+1', 'US', 'United States', ''),
 '311740': ('+1', 'US', 'United States', ''),
 '311800': ('+1', 'US', 'United States', 'Bluegrass Wireless LLC'),
 '311810': ('+1', 'US', 'United States', 'Bluegrass Wireless LLC'),
 '311830': ('+1', 'US', 'United States', 'Thumb Cellular Limited Partnership'),
 '311860': ('+1', 'US', 'United States', 'Uintah Basin Electronics Telecommunications Inc.'),
 '311870': ('+1', 'US', 'United States', 'Sprint Spectrum'),
 '311880': ('+1', 'US', 'United States', 'Sprint Spectrum'),
 '311910': ('+1', 'US', 'United States', ''),
 '311920': ('+1', 'US', 'United States', 'Missouri RSA No 5 Partnership'),
 '312010': ('+1', 'US', 'United States', 'Missouri RSA No 5 Partnership'),
 '312030': ('+1', 'US', 'United States', 'Cross Wireless Telephone Co.'),
 '312040': ('+1', 'US', 'United States', 'Custer Telephone Cooperative Inc.'),
 '312090': ('+1', 'US', 'United States', 'Allied Wireless Communications Corporation'),
 '312120': ('+1', 'US', 'United States', 'East Kentucky Network LLC'),
 '312130': ('+1', 'US', 'United States', 'East Kentucky Network LLC'),
 '312160': ('+1', 'US', 'United States', 'RSA 1 Limited Partnership'),
 '312170': ('+1', 'US', 'United States', 'Iowa RSA No. 2 Limited Partnership'),
 '312180': ('+1', 'US', 'United States', 'Keystone Wireless LLC'),
 '312190': ('+1', 'US', 'United States', 'Sprint Spectrum'),
 '312220': ('+1', 'US', 'United States', 'Missouri RSA No 5 Partnership'),
 '312230': ('+1', 'US', 'United States', 'North Dakota Network Company'),
 '312270': ('+1', 'US', 'United States', 'Cellular Network Partnership LLC'),
 '312280': ('+1', 'US', 'United States', 'Cellular Network Partnership LLC'),
 '312290': ('+1', 'US', 'United States', 'Uintah Basin Electronics Telecommunications Inc.'),
 '312380': ('+1', 'US', 'United States', ''),
 '312530': ('+1', 'US', 'United States', 'Sprint Spectrum'),
 '316010': ('+1', 'US', 'United States', 'Sprint Spectrum'),
 '316011': ('+1', 'US', 'United States', 'Southern Communications Services Inc.'),
 '33011': ('+1', 'PR', 'Puerto Rico', 'Puerto Rico Telephone Company Inc. (PRTC)'),
 '330110': ('+1', 'PR', 'Puerto Rico', 'Puerto Rico Telephone Company Inc. (PRTC)'),
 '33401': ('+52', 'MX', 'Mexico', 'NEXTEL'),
 '334010': ('+52', 'MX', 'Mexico', 'NEXTEL'),
 '33402': ('+52', 'MX', 'Mexico', 'TelCel/America Movil'),
 '334020': ('+52', 'MX', 'Mexico', 'TelCel/America Movil'),
 '33403': ('+52', 'MX', 'Mexico', 'Movistar/Pegaso'),
 '334030': ('+52', 'MX', 'Mexico', 'Movistar/Pegaso'),
 '33404': ('+52', 'MX', 'Mexico', 'AT&T/IUSACell'),
 '334040': ('+52', 'MX', 'Mexico', 'AT&T/IUSACell'),
 '334050': ('+52', 'MX', 'Mexico', 'AT&T/IUSACell'),
 '334060': ('+52', 'MX', 'Mexico', 'SAI PCS'),
 '334070': ('+52', 'MX', 'Mexico', 'Operadora Unefon SA de CV'),
 '334080': ('+52', 'MX', 'Mexico', 'Operadora Unefon SA de CV'),
 '33409': ('+52', 'MX', 'Mexico', 'NEXTEL'),
 '334090': ('+52', 'MX', 'Mexico', 'NEXTEL'),
 '33450': ('+52', 'MX', 'Mexico', 'AT&T/IUSACell'),
 '338020': ('+1876', 'JM', 'Jamaica', 'Cable & Wireless'),
 '338050': ('+1876', 'JM', 'Jamaica', 'DIGICEL/Mossel'),
 '338110': ('+1876', 'JM', 'Jamaica', 'Cable & Wireless'),
 '338180': ('+1876', 'JM', 'Jamaica', 'Cable & Wireless'),
 '34001': ('+594', 'FG', 'French Guiana', 'Orange Caribe'),
 '34002': ('+594', 'FG', 'French Guiana', 'Outremer Telecom'),
 '34003': ('+594', 'FG', 'French Guiana', 'TelCell GSM'),
 '34008': ('+590', 'GP', 'Guadeloupe', 'Dauphin Telecom SU (Guadeloupe Telecom)'),
 '34010': ('+590', 'GP', 'Guadeloupe', ''),
 '34011': ('+594', 'FG', 'French Guiana', 'TelCell GSM'),
 '34012': ('+596', 'MQ', 'Martinique (French Department of)', 'UTS Caraibe'),
 '34020': ('+594', 'FG', 'French Guiana', 'Bouygues/DigiCel'),
 '342050': ('+1246', 'BB', 'Barbados', 'Digicel'),
 '342600': ('+1246', 'BB', 'Barbados', 'C & W BET Ltd.'),
 '342750': ('+1246', 'BB', 'Barbados', 'Digicel'),
 '342810': ('+1246', 'BB', 'Barbados', 'Cingular Wireless'),
 '342820': ('+1246', 'BB', 'Barbados', 'Sunbeach'),
 '344030': ('+1268', 'AG', 'Antigua and Barbuda', 'APUA PCS'),
 '344920': ('+1268', 'AG', 'Antigua and Barbuda', 'C & W'),
 '344930': ('+1268', 'AG', 'Antigua and Barbuda', 'DigiCel/Cing. Wireless'),
 '346006': ('+1345', 'KY', 'Cayman Islands', 'Digicel Ltd.'),
 '346050': ('+1345', 'KY', 'Cayman Islands', 'Digicel Cayman Ltd'),
 '346140': ('+1345', 'KY', 'Cayman Islands', 'LIME / Cable & Wirel.'),
 '348170': ('+284', 'VG', 'British Virgin Islands', 'LIME'),
 '348570': ('+284', 'VG', 'British Virgin Islands', 'Caribbean Cellular'),
 '348770': ('+284', 'VG', 'British Virgin Islands', 'Digicel'),
 '350000': ('+1441', 'BM', 'Bermuda', 'Bermuda Digital Communications Ltd (BDC)'),
 '35001': ('+1441', 'BM', 'Bermuda', 'Telecommunications (Bermuda & West Indies) Ltd (Digicel Bermuda)'),
 '35002': ('+1441', 'BM', 'Bermuda', 'M3 Wireless Ltd'),
 '35010': ('+1441', 'BM', 'Bermuda', 'DigiCel / Cingular'),
 '35099': ('+1441', 'BM', 'Bermuda', 'CellOne Ltd'),
 '352030': ('+1473', 'GD', 'Grenada', 'Digicel'),
 '352050': ('+1473', 'GD', 'Grenada', 'Digicel'),
 '352110': ('+1473', 'GD', 'Grenada', 'Cable & Wireless'),
 '354860': ('+1664', 'MS', 'Montserrat', 'Cable & Wireless'),
 '356110': ('+1869', 'KN', 'Saint Kitts and Nevis', 'Cable & Wireless'),
 '35650': ('+1869', 'KN', 'Saint Kitts and Nevis', 'Digicel'),
 '35670': ('+1869', 'KN', 'Saint Kitts and Nevis', 'UTS Cariglobe'),
 '358110': ('+1758', 'LC', 'Saint Lucia', 'Cable & Wireless'),
 '35830': ('+1758', 'LC', 'Saint Lucia', 'Cingular Wireless'),
 '35850': ('+1758', 'LC', 'Saint Lucia', 'Digicel (St Lucia) Limited'),
 '360050': ('+1784', 'VC', 'St. Vincent & Gren.', 'Digicel'),
 '36010': ('+1784', 'VC', 'St. Vincent & Gren.', 'Cingular'),
 '360100': ('+1784', 'VC', 'St. Vincent & Gren.', 'Cingular'),
 '360110': ('+1784', 'VC', 'St. Vincent & Gren.', 'C & W'),
 '36070': ('+1784', 'VC', 'St. Vincent & Gren.', 'Digicel'),
 '36251': ('+599', 'AN', 'Netherlands Antilles', 'TELCELL GSM'),
 '362630': ('+599', 'AN', 'Netherlands Antilles', 'Cingular Wireless'),
 '36269': ('+599', 'CW', 'Curacao', 'Polycom N.V./ Digicel'),
 '36291': ('+599', 'AN', 'Netherlands Antilles', 'SETEL GSM'),
 '36295': ('+599', 'CW', 'Curacao', 'EOCG Wireless NV'),
 '362951': ('+599', 'AN', 'Netherlands Antilles', 'UTS Wireless'),
 '36301': ('+297', 'AW', 'Aruba', 'Setar GSM'),
 '36302': ('+297', 'AW', 'Aruba', 'Digicel'),
 '36320': ('+297', 'AW', 'Aruba', 'Digicel'),
 '36403': ('+1242', 'BS', 'Bahamas', 'Smart Communications'),
 '36430': ('+1242', 'BS', 'Bahamas', 'Bahamas Telco. Comp.'),
 '36439': ('+1242', 'BS', 'Bahamas', 'Bahamas Telco. Comp.'),
 '364390': ('+1242', 'BS', 'Bahamas', 'Bahamas Telco. Comp.'),
 '365010': ('+1264', 'AI', 'Anguilla', 'Digicell / Wireless Vent. Ltd'),
 '365840': ('+1264', 'AI', 'Anguilla', 'Cable and Wireless'),
 '366020': ('+1767', 'DM', 'Dominica', 'Cingular Wireless/Digicel'),
 '366050': ('+1767', 'DM', 'Dominica', 'Wireless Ventures (Dominica) Ltd (Digicel Dominica)'),
 '366110': ('+1767', 'DM', 'Dominica', 'C & W'),
 '36801': ('+53', 'CU', 'Cuba', 'C-COM'),
 '37001': ('+1809', 'DO', 'Dominican Republic', 'Orange'),
 '37002': ('+1809', 'DO', 'Dominican Republic', 'Claro'),
 '37003': ('+1809', 'DO', 'Dominican Republic', 'TRIcom'),
 '37004': ('+1809', 'DO', 'Dominican Republic', 'Trilogy Dominicana S. A.'),
 '37201': ('+509', 'HT', 'Haiti', 'Comcel'),
 '37202': ('+509', 'HT', 'Haiti', 'Digicel'),
 '37203': ('+509', 'HT', 'Haiti', 'National Telecom SA (NatCom)'),
 '37412': ('+1868', 'TT', 'Trinidad and Tobago', 'Bmobile/TSTT'),
 '374120': ('+1868', 'TT', 'Trinidad and Tobago', 'Bmobile/TSTT'),
 '374130': ('+1868', 'TT', 'Trinidad and Tobago', 'Digicel'),
 '374140': ('+1868', 'TT', 'Trinidad and Tobago', 'LaqTel Ltd.'),
 '376050': ('+1', 'TC', 'Turks and Caicos Islands', 'Digicel TCI Ltd'),
 '376350': ('+1', 'TC', 'Turks and Caicos Islands', 'Cable & Wireless (TCI) Ltd'),
 '376352': ('+1', 'TC', 'Turks and Caicos Islands', 'IslandCom Communications Ltd.'),
 '37650': ('+1340', 'VI', 'Virgin Islands, U.S.', 'Digicel'),
 '40001': ('+994', 'AZ', 'Azerbaijan', 'Azercell Telekom B.M.'),
 '40002': ('+994', 'AZ', 'Azerbaijan', 'J.V. Bakcell GSM 2000'),
 '40003': ('+994', 'AZ', 'Azerbaijan', 'Caspian American Telecommunications LLC (CATEL)'),
 '40004': ('+994', 'AZ', 'Azerbaijan', 'Azerfon.'),
 '40101': ('+7', 'KZ', 'Kazakhstan', 'Beeline/KaR-Tel LLP'),
 '40102': ('+7', 'KZ', 'Kazakhstan', 'K-Cell'),
 '40107': ('+7', 'KZ', 'Kazakhstan', 'Dalacom/Altel'),
 '40177': ('+7', 'KZ', 'Kazakhstan', 'Tele2/NEO/MTS'),
 '40211': ('+975', 'BT', 'Bhutan', 'B-Mobile'),
 '40217': ('+975', 'BT', 'Bhutan', 'Bhutan Telecom Ltd (BTL)'),
 '40277': ('+975', 'BT', 'Bhutan', 'TashiCell'),
 '40401': ('+91', 'IN', 'India', 'Aircel Digilink India'),
 '40404': ('+91', 'IN', 'India', 'Idea Cellular Ltd.'),
 '404045': ('+91', 'IN', 'India', 'Bharti Airtel Limited (Karnataka) (India)'),
 '40405': ('+91', 'IN', 'India', 'Fascel'),
 '40407': ('+91', 'IN', 'India', 'Idea Cellular Ltd.'),
 '40409': ('+91', 'IN', 'India', 'Reliance Telecom Private'),
 '40410': ('+91', 'IN', 'India', 'Bharti Airtel Limited (Delhi)'),
 '40411': ('+91', 'IN', 'India', 'Sterling Cellular Ltd.'),
 '40412': ('+91', 'IN', 'India', 'Escotel Mobile Communications'),
 '40413': ('+91', 'IN', 'India', 'Barakhamba Sales & Serv.'),
 '40414': ('+91', 'IN', 'India', 'Spice'),
 '40415': ('+91', 'IN', 'India', 'Aircel Digilink India'),
 '40416': ('+91', 'IN', 'India', 'Hexcom India'),
 '40417': ('+91', 'IN', 'India', 'Aircel'),
 '40418': ('+91', 'IN', 'India', 'Reliance Telecom Private'),
 '40419': ('+91', 'IN', 'India', 'Escotel Mobile Communications'),
 '40422': ('+91', 'IN', 'India', 'Idea Cellular Ltd.'),
 '40424': ('+91', 'IN', 'India', 'Idea Cellular Ltd.'),
 '40425': ('+91', 'IN', 'India', 'Aircel'),
 '40428': ('+91', 'IN', 'India', 'Aircel'),
 '40429': ('+91', 'IN', 'India', 'Aircel'),
 '40430': ('+91', 'IN', 'India', 'Usha Martin Telecom'),
 '40433': ('+91', 'IN', 'India', 'Aircel'),
 '40434': ('+91', 'IN', 'India', 'BSNL'),
 '40436': ('+91', 'IN', 'India', 'Reliance Telecom Private'),
 '40438': ('+91', 'IN', 'India', 'BSNL'),
 '40441': ('+91', 'IN', 'India', 'RPG Cellular'),
 '40442': ('+91', 'IN', 'India', 'Aircel'),
 '40444': ('+91', 'IN', 'India', 'Spice'),
 '40450': ('+91', 'IN', 'India', 'Reliance Telecom Private'),
 '40451': ('+91', 'IN', 'India', 'BSNL'),
 '40452': ('+91', 'IN', 'India', 'Reliance Telecom Private'),
 '40453': ('+91', 'IN', 'India', 'BSNL'),
 '40454': ('+91', 'IN', 'India', 'BSNL'),
 '40455': ('+91', 'IN', 'India', 'BSNL'),
 '40456': ('+91', 'IN', 'India', 'Escotel Mobile Communications'),
 '40457': ('+91', 'IN', 'India', 'BSNL'),
 '40458': ('+91', 'IN', 'India', 'BSNL'),
 '40459': ('+91', 'IN', 'India', 'BSNL'),
 '40460': ('+91', 'IN', 'India', 'Aircel Digilink India'),
 '40462': ('+91', 'IN', 'India', 'BSNL'),
 '40464': ('+91', 'IN', 'India', 'BSNL'),
 '40466': ('+91', 'IN', 'India', 'BSNL'),
 '40467': ('+91', 'IN', 'India', 'Reliance Telecom Private'),
 '40468': ('+91', 'IN', 'India', 'Mahanagar Telephone Nigam'),
 '40469': ('+91', 'IN', 'India', 'Mahanagar Telephone Nigam'),
 '40470': ('+91', 'IN', 'India', 'Hexacom India'),
 '40471': ('+91', 'IN', 'India', 'BSNL'),
 '40472': ('+91', 'IN', 'India', 'BSNL'),
 '40473': ('+91', 'IN', 'India', 'BSNL'),
 '40474': ('+91', 'IN', 'India', 'BSNL'),
 '40475': ('+91', 'IN', 'India', 'BSNL'),
 '40476': ('+91', 'IN', 'India', 'BSNL'),
 '40477': ('+91', 'IN', 'India', 'BSNL'),
 '40478': ('+91', 'IN', 'India', 'Idea Cellular Ltd.'),
 '40479': ('+91', 'IN', 'India', 'CellOne A&N'),
 '40480': ('+91', 'IN', 'India', 'BSNL'),
 '40481': ('+91', 'IN', 'India', 'BSNL'),
 '40482': ('+91', 'IN', 'India', 'Escorts Telecom Ltd.'),
 '40483': ('+91', 'IN', 'India', 'Reliable Internet Services'),
 '40485': ('+91', 'IN', 'India', 'Reliance Telecom Private'),
 '40486': ('+91', 'IN', 'India', 'Barakhamba Sales & Serv.'),
 '40487': ('+91', 'IN', 'India', 'Escorts Telecom Ltd.'),
 '40488': ('+91', 'IN', 'India', 'Escorts Telecom Ltd.'),
 '40489': ('+91', 'IN', 'India', 'Escorts Telecom Ltd.'),
 '405034': ('+91', 'IN', 'India', 'TATA / Karnataka'),
 '40505': ('+91', 'IN', 'India', 'Fascel Limited'),
 '40553': ('+91', 'IN', 'India', 'AirTel'),
 '41001': ('+92', 'PK', 'Pakistan', 'Mobilink'),
 '41003': ('+92', 'PK', 'Pakistan', 'UFONE/PAKTel'),
 '41004': ('+92', 'PK', 'Pakistan', 'ZONG/CMPak'),
 '41006': ('+92', 'PK', 'Pakistan', 'Telenor'),
 '41007': ('+92', 'PK', 'Pakistan', 'Warid Telecom'),
 '41008': ('+92', 'PK', 'Pakistan', 'Instaphone'),
 '41201': ('+93', 'AF', 'Afghanistan', 'Afghan Wireless/AWCC'),
 '41203': ('+93', 'AF', 'Afghanistan', 'WaselTelecom (WT)'),
 '41220': ('+93', 'AF', 'Afghanistan', 'Roshan/TDCA'),
 '41240': ('+93', 'AF', 'Afghanistan', 'Areeba/MTN'),
 '41250': ('+93', 'AF', 'Afghanistan', 'Etisalat'),
 '41280': ('+93', 'AF', 'Afghanistan', 'Afghan Telecom Corp. (AT)'),
 '41288': ('+93', 'AF', 'Afghanistan', 'Afghan Telecom Corp. (AT)'),
 '41301': ('+94', 'LK', 'Sri Lanka', 'Mobitel Ltd.'),
 '41302': ('+94', 'LK', 'Sri Lanka', 'MTN/Dialog'),
 '41303': ('+94', 'LK', 'Sri Lanka', 'Etisalat/Tigo'),
 '41305': ('+94', 'LK', 'Sri Lanka', 'Airtel'),
 '41308': ('+94', 'LK', 'Sri Lanka', 'H3G Hutchison'),
 '41401': ('+95', 'MM', 'Myanmar (Burma)', 'Myanmar Post & Teleco.'),
 '41405': ('+95', 'MM', 'Myanmar (Burma)', 'Oreedoo'),
 '41406': ('+95', 'MM', 'Myanmar (Burma)', 'Telenor'),
 '41501': ('+961', 'LB', 'Lebanon', 'MIC1 (Alfa)'),
 '41503': ('+961', 'LB', 'Lebanon', 'MIC2/LibanCell/MTC'),
 '41532': ('+961', 'LB', 'Lebanon', 'Cellis'),
 '41533': ('+961', 'LB', 'Lebanon', 'Cellis'),
 '41534': ('+961', 'LB', 'Lebanon', 'FTML Cellis'),
 '41535': ('+961', 'LB', 'Lebanon', 'Cellis'),
 '41536': ('+961', 'LB', 'Lebanon', 'MIC2/LibanCell/MTC'),
 '41537': ('+961', 'LB', 'Lebanon', 'MIC2/LibanCell/MTC'),
 '41538': ('+961', 'LB', 'Lebanon', 'MIC2/LibanCell/MTC'),
 '41539': ('+961', 'LB', 'Lebanon', 'MIC2/LibanCell/MTC'),
 '41601': ('+962', 'JO', 'Jordan', 'ZAIN /J.M.T.S'),
 '41602': ('+962', 'JO', 'Jordan', 'Xpress'),
 '41603': ('+962', 'JO', 'Jordan', 'Umniah Mobile Co.'),
 '41677': ('+962', 'JO', 'Jordan', 'Orange/Petra'),
 '41701': ('+963', 'SY', 'Syrian Arab Republic', 'Syriatel Holdings'),
 '41702': ('+963', 'SY', 'Syrian Arab Republic', 'MTN/Spacetel'),
 '41709': ('+963', 'SY', 'Syrian Arab Republic', 'Syriatel Holdings'),
 '41805': ('+964', 'IQ', 'Iraq', 'Asia Cell'),
 '41808': ('+964', 'IQ', 'Iraq', 'Sanatel'),
 '41820': ('+964', 'IQ', 'Iraq', 'ZAIN/Atheer/Orascom'),
 '41830': ('+964', 'IQ', 'Iraq', 'Orascom Telecom'),
 '41840': ('+964', 'IQ', 'Iraq', 'Korek'),
 '41845': ('+964', 'IQ', 'Iraq', 'Mobitel (Iraq-Kurdistan) and Moutiny'),
 '41882': ('+964', 'IQ', 'Iraq', 'Korek'),
 '41892': ('+964', 'IQ', 'Iraq', 'Itisaluna and Kalemat'),
 '41902': ('+965', 'KW', 'Kuwait', 'Zain'),
 '41903': ('+965', 'KW', 'Kuwait', 'Wataniya'),
 '41904': ('+965', 'KW', 'Kuwait', 'Viva'),
 '42001': ('+966', 'SA', 'Saudi Arabia', 'STC/Al Jawal'),
 '42003': ('+966', 'SA', 'Saudi Arabia', 'Etihad/Etisalat/Mobily'),
 '42004': ('+966', 'SA', 'Saudi Arabia', 'Zain'),
 '42005': ('+966', 'SA', 'Saudi Arabia', 'Virgin Mobile'),
 '42007': ('+966', 'SA', 'Saudi Arabia', 'Zain'),
 '42101': ('+967', 'YE', 'Yemen', 'Sabaphone'),
 '42102': ('+967', 'YE', 'Yemen', 'MTN/Spacetel'),
 '42103': ('+967', 'YE', 'Yemen', 'Yemen Mob. CDMA'),
 '42104': ('+967', 'YE', 'Yemen', 'HITS/Y Unitel'),
 '42202': ('+968', 'OM', 'Oman', 'Oman Mobile/GTO'),
 '42203': ('+968', 'OM', 'Oman', 'Nawras'),
 '42402': ('+971', 'AE', 'United Arab Emirates', 'Etisalat'),
 '42403': ('+971', 'AE', 'United Arab Emirates', 'DU'),
 '42501': ('+972', 'IL', 'Israel', 'Orange/Partner Co. Ltd.'),
 '42502': ('+972', 'IL', 'Israel', 'Cellcom ltd.'),
 '42503': ('+972', 'IL', 'Israel', 'Pelephone'),
 '42505': ('+970', 'PS', 'Palestinian Territory', 'Jawwal'),
 '42506': ('+970', 'PS', 'Palestinian Territory', 'Wataniya Mobile'),
 '42507': ('+972', 'IL', 'Israel', 'Hot Mobile/Mirs'),
 '42508': ('+972', 'IL', 'Israel', 'Golan Telekom'),
 '42514': ('+972', 'IL', 'Israel', 'Alon Cellular Ltd'),
 '42515': ('+972', 'IL', 'Israel', 'Home Cellular Ltd'),
 '42516': ('+972', 'IL', 'Israel', 'Rami Levy Hashikma Marketing Communications Ltd'),
 '42519': ('+972', 'IL', 'Israel', 'Telzar/AZI'),
 '42577': ('+972', 'IL', 'Israel', 'Hot Mobile/Mirs'),
 '42601': ('+973', 'BH', 'Bahrain', 'Batelco'),
 '42602': ('+973', 'BH', 'Bahrain', 'ZAIN/Vodafone'),
 '42604': ('+973', 'BH', 'Bahrain', 'VIVA'),
 '42701': ('+974', 'QA', 'Qatar', 'Ooredoo/Qtel'),
 '42702': ('+974', 'QA', 'Qatar', 'Vodafone'),
 '42800': ('+976', 'MN', 'Mongolia', 'Skytel Co. Ltd'),
 '42888': ('+976', 'MN', 'Mongolia', 'Unitel'),
 '42891': ('+976', 'MN', 'Mongolia', 'Skytel Co. Ltd'),
 '42898': ('+976', 'MN', 'Mongolia', 'G-Mobile Corporation Ltd'),
 '42899': ('+976', 'MN', 'Mongolia', 'Mobicom'),
 '42901': ('+977', 'NP', 'Nepal', 'NT Mobile / Namaste'),
 '42902': ('+977', 'NP', 'Nepal', 'Ncell'),
 '42904': ('+977', 'NP', 'Nepal', 'Smart Cell'),
 '43002': ('+971', 'AE', 'United Arab Emirates', 'Etisalat'),
 '43102': ('+971', 'AE', 'United Arab Emirates', 'Etisalat'),
 '43211': ('+98', 'IR', 'Iran', 'MCI/TCI'),
 '43214': ('+98', 'IR', 'Iran', 'TKC/KFZO'),
 '43219': ('+98', 'IR', 'Iran', 'Mobile Telecommunications Company of Esfahan JV-PJS (MTCE)'),
 '43220': ('+98', 'IR', 'Iran', 'Rightel'),
 '43232': ('+98', 'IR', 'Iran', 'Taliya'),
 '43235': ('+98', 'IR', 'Iran', 'MTN/IranCell'),
 '43270': ('+98', 'IR', 'Iran', 'MTCE'),
 '43401': ('+998', 'UZ', 'Uzbekistan', 'Buztel'),
 '43402': ('+998', 'UZ', 'Uzbekistan', 'Uzmacom'),
 '43404': ('+998', 'UZ', 'Uzbekistan', 'Bee Line/Unitel'),
 '43405': ('+998', 'UZ', 'Uzbekistan', 'Ucell/Coscom'),
 '43407': ('+998', 'UZ', 'Uzbekistan', 'MTS/Uzdunrobita'),
 '43601': ('+992', 'TK', 'Tajikistan', 'Tcell/JC Somoncom'),
 '43602': ('+992', 'TK', 'Tajikistan', 'CJSC Indigo Tajikistan'),
 '43603': ('+992', 'TK', 'Tajikistan', 'MLT/TT mobile'),
 '43604': ('+992', 'TK', 'Tajikistan', 'Babilon-M'),
 '43605': ('+992', 'TK', 'Tajikistan', 'Bee Line'),
 '43612': ('+992', 'TK', 'Tajikistan', 'Tcell/JC Somoncom'),
 '43701': ('+996', 'KG', 'Kyrgyzstan', 'Beeline/Bitel'),
 '43703': ('+996', 'KG', 'Kyrgyzstan', 'AkTel LLC'),
 '43705': ('+996', 'KG', 'Kyrgyzstan', 'MEGACOM'),
 '43709': ('+996', 'KG', 'Kyrgyzstan', 'O!/NUR Telecom'),
 '43801': ('+993', 'TM', 'Turkmenistan', 'MTS/Barash Communication'),
 '43802': ('+993', 'TM', 'Turkmenistan', 'Altyn Asyr/TM-Cell'),
 '44000': ('+81', 'JP', 'Japan', 'Y-Mobile'),
 '44001': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44002': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44003': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44004': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44006': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44007': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44008': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44009': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44010': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44011': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44012': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44013': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44014': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44015': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44016': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44017': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44018': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44019': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44020': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44021': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44022': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44023': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44024': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44025': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44026': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44027': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44028': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44029': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44030': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44031': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44032': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44033': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44034': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44035': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44036': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44037': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44038': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44039': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44040': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44041': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44042': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44043': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44044': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44045': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44046': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44047': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44048': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44049': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44050': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44051': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44052': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44053': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44054': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44055': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44056': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44058': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44060': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44061': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44062': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44063': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44064': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44065': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44066': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44067': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44068': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44069': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44070': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44071': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44072': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44073': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44074': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44075': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44076': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44077': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44078': ('+81', 'JP', 'Japan', 'Okinawa Cellular Telephone'),
 '44079': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44080': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44081': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44082': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44083': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44084': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44085': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44086': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44087': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44088': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44089': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44090': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44092': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44093': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44094': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44095': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44096': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44097': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44098': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44099': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44140': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44141': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44142': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44143': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44144': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44145': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44161': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44162': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44163': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44164': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44165': ('+81', 'JP', 'Japan', 'SoftBank Mobile Corp'),
 '44170': ('+81', 'JP', 'Japan', 'KDDI Corporation'),
 '44190': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44191': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44192': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44193': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44194': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44198': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '44199': ('+81', 'JP', 'Japan', 'NTT Docomo'),
 '45002': ('+82', 'KR', 'Korea S, Republic of', 'KT Freetel Co. Ltd.'),
 '45003': ('+82', 'KR', 'Korea S, Republic of', 'SK Telecom'),
 '45004': ('+82', 'KR', 'Korea S, Republic of', 'KT Freetel Co. Ltd.'),
 '45005': ('+82', 'KR', 'Korea S, Republic of', 'SK Telecom Co. Ltd'),
 '45006': ('+82', 'KR', 'Korea S, Republic of', 'LG Telecom'),
 '45008': ('+82', 'KR', 'Korea S, Republic of', 'KT Freetel Co. Ltd.'),
 '45201': ('+84', 'VN', 'Viet Nam', 'Mobifone'),
 '45202': ('+84', 'VN', 'Viet Nam', 'Vinaphone'),
 '45203': ('+84', 'VN', 'Viet Nam', 'S-Fone/Telecom'),
 '45204': ('+84', 'VN', 'Viet Nam', 'Viettel Mobile'),
 '45205': ('+84', 'VN', 'Viet Nam', 'VietnaMobile'),
 '45206': ('+84', 'VN', 'Viet Nam', 'Viettel Mobile'),
 '45207': ('+84', 'VN', 'Viet Nam', 'Beeline'),
 '45208': ('+84', 'VN', 'Viet Nam', 'Viettel Mobile'),
 '45400': ('+852', 'HK', 'Hongkong, China', 'CSL Ltd.'),
 '45401': ('+852', 'HK', 'Hongkong, China', 'Citic Telecom Ltd.'),
 '45402': ('+852', 'HK', 'Hongkong, China', 'CSL Ltd.'),
 '45403': ('+852', 'HK', 'Hongkong, China', 'H3G/Hutchinson'),
 '45404': ('+852', 'HK', 'Hongkong, China', 'H3G/Hutchinson'),
 '45405': ('+852', 'HK', 'Hongkong, China', 'H3G/Hutchinson'),
 '45406': ('+852', 'HK', 'Hongkong, China', 'Vodafone/SmarTone'),
 '45407': ('+852', 'HK', 'Hongkong, China', 'China Unicom Ltd'),
 '45408': ('+852', 'HK', 'Hongkong, China', 'Truephone'),
 '45409': ('+852', 'HK', 'Hongkong, China', 'China Motion'),
 '45410': ('+852', 'HK', 'Hongkong, China', 'CSL/New World PCS Ltd.'),
 '45411': ('+852', 'HK', 'Hongkong, China', 'China-HongKong Telecom Ltd (CHKTL)'),
 '45412': ('+852', 'HK', 'Hongkong, China', 'China Mobile/Peoples'),
 '45413': ('+852', 'HK', 'Hongkong, China', 'China Mobile/Peoples'),
 '45414': ('+852', 'HK', 'Hongkong, China', 'H3G/Hutchinson'),
 '45415': ('+852', 'HK', 'Hongkong, China', 'Vodafone/SmarTone'),
 '45416': ('+852', 'HK', 'Hongkong, China', 'HKT/PCCW'),
 '45417': ('+852', 'HK', 'Hongkong, China', 'Vodafone/SmarTone'),
 '45418': ('+852', 'HK', 'Hongkong, China', 'CSL Ltd.'),
 '45419': ('+852', 'HK', 'Hongkong, China', 'HKT/PCCW'),
 '45420': ('+852', 'HK', 'Hongkong, China', 'HKT/PCCW'),
 '45428': ('+852', 'HK', 'Hongkong, China', 'China Mobile/Peoples'),
 '45429': ('+852', 'HK', 'Hongkong, China', 'HKT/PCCW'),
 '45440': ('+852', 'HK', 'Hongkong, China', 'shared by private TETRA systems'),
 '45447': ('+852', 'HK', 'Hongkong, China', 'shared by private TETRA systems'),
 '45500': ('+853', 'MO', 'Macao, China', 'Smartone Mobile'),
 '45501': ('+853', 'MO', 'Macao, China', 'C.T.M. TELEMOVEL+'),
 '45502': ('+853', 'MO', 'Macao, China', 'China Telecom'),
 '45503': ('+853', 'MO', 'Macao, China', 'Hutchison Telephone Co. Ltd'),
 '45504': ('+853', 'MO', 'Macao, China', 'C.T.M. TELEMOVEL+'),
 '45505': ('+853', 'MO', 'Macao, China', 'Hutchison Telephone Co. Ltd'),
 '45506': ('+853', 'MO', 'Macao, China', 'Smartone Mobile'),
 '45601': ('+855', 'KH', 'Cambodia', 'Mobitel/Cam GSM'),
 '45602': ('+855', 'KH', 'Cambodia', 'Smart Mobile'),
 '45603': ('+855', 'KH', 'Cambodia', 'QB/Cambodia Adv. Comms.'),
 '45604': ('+855', 'KH', 'Cambodia', 'Cambodia Advance Communications Co. Ltd (CADCOMMS)'),
 '45605': ('+855', 'KH', 'Cambodia', 'Smart Mobile'),
 '45606': ('+855', 'KH', 'Cambodia', 'Smart Mobile'),
 '45608': ('+855', 'KH', 'Cambodia', 'Metfone'),
 '45609': ('+855', 'KH', 'Cambodia', 'Sotelco/Beeline'),
 '45618': ('+855', 'KH', 'Cambodia', 'MFone/Camshin/Cellcard'),
 '45701': ('+856', 'LA', 'Laos P.D.R.', 'Lao Tel'),
 '45702': ('+856', 'LA', 'Laos P.D.R.', 'ETL Mobile'),
 '45703': ('+856', 'LA', 'Laos P.D.R.', 'UNITEL/LAT'),
 '45708': ('+856', 'LA', 'Laos P.D.R.', 'Beeline/Tigo/Millicom'),
 '46000': ('+86', 'CN', 'China', 'China Mobile GSM'),
 '46001': ('+86', 'CN', 'China', 'China Unicom'),
 '46002': ('+86', 'CN', 'China', 'China Mobile GSM'),
 '46003': ('+86', 'CN', 'China', 'China Telecom'),
 '46004': ('+86', 'CN', 'China', 'China Space Mobile Satellite Telecommunications Co. Ltd (China Spacecom)'),
 '46005': ('+86', 'CN', 'China', 'China Telecom'),
 '46006': ('+86', 'CN', 'China', 'China Unicom'),
 '46007': ('+86', 'CN', 'China', 'China Mobile GSM'),
 '46601': ('+886', 'TW', 'Taiwan', 'Far EasTone'),
 '46602': ('+886', 'TW', 'Taiwan', 'Far EasTone'),
 '46603': ('+886', 'TW', 'Taiwan', 'Far EasTone'),
 '46605': ('+886', 'TW', 'Taiwan', 'Asia Pacific Telecom Co. Ltd (APT)'),
 '46606': ('+886', 'TW', 'Taiwan', 'Far EasTone'),
 '46607': ('+886', 'TW', 'Taiwan', 'Far EasTone'),
 '46609': ('+886', 'TW', 'Taiwan', 'VMAX Telecom Co. Ltd'),
 '46610': ('+886', 'TW', 'Taiwan', 'Global Mobile Corp.'),
 '46611': ('+886', 'TW', 'Taiwan', 'Chunghwa Telecom LDM'),
 '46656': ('+886', 'TW', 'Taiwan', 'International Telecom Co. Ltd (FITEL)'),
 '46668': ('+886', 'TW', 'Taiwan', 'ACeS Taiwan - ACeS Taiwan Telecommunications Co Ltd'),
 '46688': ('+886', 'TW', 'Taiwan', 'KG Telecom'),
 '46689': ('+886', 'TW', 'Taiwan', 'T-Star/VIBO'),
 '46692': ('+886', 'TW', 'Taiwan', 'Chunghwa Telecom LDM'),
 '46693': ('+886', 'TW', 'Taiwan', 'Mobitai'),
 '46697': ('+886', 'TW', 'Taiwan', 'Taiwan Cellular'),
 '46699': ('+886', 'TW', 'Taiwan', 'TransAsia'),
 '467193': ('+850', 'KP', "Korea N., Dem. People's Rep.", 'Sun Net'),
 '47001': ('+880', 'BD', 'Bangladesh', 'GrameenPhone'),
 '47002': ('+880', 'BD', 'Bangladesh', 'Robi/Aktel'),
 '47003': ('+880', 'BD', 'Bangladesh', 'Orascom/Banglalink'),
 '47004': ('+880', 'BD', 'Bangladesh', 'TeleTalk'),
 '47005': ('+880', 'BD', 'Bangladesh', 'Citycell'),
 '47006': ('+880', 'BD', 'Bangladesh', 'Citycell'),
 '47007': ('+880', 'BD', 'Bangladesh', 'Airtel/Warid'),
 '47201': ('+960', 'MV', 'Maldives', 'Dhiraagu/C&W'),
 '47202': ('+960', 'MV', 'Maldives', 'Ooredo/Wataniya'),
 '50201': ('+60', 'MY', 'Malaysia', 'Art900'),
 '50210': ('+60', 'MY', 'Malaysia', 'Digi Telecommunications'),
 '50211': ('+60', 'MY', 'Malaysia', 'MTX Utara'),
 '50212': ('+60', 'MY', 'Malaysia', 'Maxis'),
 '50213': ('+60', 'MY', 'Malaysia', 'CelCom'),
 '502151': ('+60', 'MY', 'Malaysia', 'Baraka Telecom Sdn Bhd'),
 '502152': ('+60', 'MY', 'Malaysia', 'YES'),
 '502153': ('+60', 'MY', 'Malaysia', 'Webe/Packet One Networks (Malaysia) Sdn Bhd'),
 '502154': ('+60', 'MY', 'Malaysia', 'Tron/Talk Focus Sdn Bhd'),
 '502155': ('+60', 'MY', 'Malaysia', 'Samata Communications Sdn Bhd'),
 '50216': ('+60', 'MY', 'Malaysia', 'Digi Telecommunications'),
 '50217': ('+60', 'MY', 'Malaysia', 'Maxis'),
 '50218': ('+60', 'MY', 'Malaysia', 'U Mobile'),
 '50219': ('+60', 'MY', 'Malaysia', 'CelCom'),
 '502195': ('+60', 'MY', 'Malaysia', 'XOX Com Sdn Bhd'),
 '502198': ('+60', 'MY', 'Malaysia', 'CelCom'),
 '50220': ('+60', 'MY', 'Malaysia', 'Electcoms Wireless Sdn Bhd'),
 '50501': ('+61', 'AU', 'Australia', 'Telstra Corp. Ltd.'),
 '50502': ('+61', 'AU', 'Australia', 'Singtel Optus'),
 '50503': ('+61', 'AU', 'Australia', 'Vodafone'),
 '50504': ('+61', 'AU', 'Australia', 'Department of Defense'),
 '50505': ('+61', 'AU', 'Australia', 'The Ozitel Network Pty.'),
 '50506': ('+61', 'AU', 'Australia', 'H3G Ltd.'),
 '50507': ('+61', 'AU', 'Australia', 'Vodafone'),
 '50508': ('+61', 'AU', 'Australia', 'Railcorp/Vodafone'),
 '50509': ('+61', 'AU', 'Australia', 'Airnet Commercial Australia Ltd..'),
 '50511': ('+61', 'AU', 'Australia', 'Telstra Corp. Ltd.'),
 '50512': ('+61', 'AU', 'Australia', 'H3G Ltd.'),
 '50513': ('+61', 'AU', 'Australia', 'Railcorp/Vodafone'),
 '50514': ('+61', 'AU', 'Australia', 'AAPT Ltd.'),
 '50516': ('+61', 'AU', 'Australia', 'Victorian Rail Track Corp. (VicTrack)'),
 '50519': ('+61', 'AU', 'Australia', 'Lycamobile Pty Ltd'),
 '50524': ('+61', 'AU', 'Australia', 'Advanced Comm Tech Pty.'),
 '50526': ('+61', 'AU', 'Australia', 'Dialogue Communications Pty Ltd'),
 '50571': ('+61', 'AU', 'Australia', 'Telstra Corp. Ltd.'),
 '50572': ('+61', 'AU', 'Australia', 'Telstra Corp. Ltd.'),
 '50588': ('+61', 'AU', 'Australia', 'Localstar Holding Pty. Ltd'),
 '50590': ('+61', 'AU', 'Australia', 'Singtel Optus'),
 '50599': ('+61', 'AU', 'Australia', 'Railcorp/Vodafone'),
 '51000': ('+62', 'ID', 'Indonesia', 'PT Pasifik Satelit Nusantara (PSN)'),
 '51001': ('+62', 'ID', 'Indonesia', 'Indosat/Satelindo/M3'),
 '51007': ('+62', 'ID', 'Indonesia', 'Flexi (PT Telkom) (CDMA)'),
 '51008': ('+62', 'ID', 'Indonesia', 'Axis/Natrindo'),
 '51009': ('+62', 'ID', 'Indonesia', 'PT Smartfren Telecom Tbk'),
 '51010': ('+62', 'ID', 'Indonesia', 'Telkomsel'),
 '51011': ('+62', 'ID', 'Indonesia', 'PT. Excelcom'),
 '51021': ('+62', 'ID', 'Indonesia', 'Indosat/Satelindo/M3'),
 '51027': ('+62', 'ID', 'Indonesia', 'PT Sampoerna Telekomunikasi Indonesia (STI)'),
 '51028': ('+62', 'ID', 'Indonesia', 'PT Smartfren Telecom Tbk'),
 '51089': ('+62', 'ID', 'Indonesia', 'H3G CP'),
 '51099': ('+62', 'ID', 'Indonesia', 'Esia (PT Bakrie Telecom) (CDMA)'),
 '51401': ('+670', 'TP', 'Timor-Leste', 'Telin/ Telkomcel'),
 '51402': ('+670', 'TP', 'Timor-Leste', 'Timor Telecom'),
 '51500': ('+63', 'PH', 'Philippines', 'Fix Line'),
 '51501': ('+63', 'PH', 'Philippines', 'Globe Telecom'),
 '51502': ('+63', 'PH', 'Philippines', 'Globe Telecom'),
 '51503': ('+63', 'PH', 'Philippines', 'Smart'),
 '51505': ('+63', 'PH', 'Philippines', 'SUN/Digitel'),
 '51518': ('+63', 'PH', 'Philippines', 'RED Mobile/Cure'),
 '51588': ('+63', 'PH', 'Philippines', 'Next Mobile'),
 '52000': ('+66', 'TH', 'Thailand', 'Hutch/CAT CDMA'),
 '52001': ('+66', 'TH', 'Thailand', 'AIS/Advanced Info Service'),
 '52003': ('+66', 'TH', 'Thailand', 'Advanced Wireless Networks/AWN'),
 '52004': ('+66', 'TH', 'Thailand', 'True Move/Orange'),
 '52005': ('+66', 'TH', 'Thailand', 'Total Access (DTAC)'),
 '52015': ('+66', 'TH', 'Thailand', 'ACT Mobile'),
 '52018': ('+66', 'TH', 'Thailand', 'Total Access (DTAC)'),
 '52020': ('+66', 'TH', 'Thailand', 'ACeS Thailand - ACeS Regional Services Co Ltd'),
 '52023': ('+66', 'TH', 'Thailand', 'Digital Phone Co.'),
 '52099': ('+66', 'TH', 'Thailand', 'True Move/Orange'),
 '52501': ('+65', 'SG', 'Singapore', 'Singtel'),
 '52502': ('+65', 'SG', 'Singapore', 'Singtel'),
 '52503': ('+65', 'SG', 'Singapore', 'MobileOne Ltd'),
 '52505': ('+65', 'SG', 'Singapore', 'Starhub'),
 '52506': ('+65', 'SG', 'Singapore', 'Starhub'),
 '52507': ('+65', 'SG', 'Singapore', 'Singtel'),
 '52512': ('+65', 'SG', 'Singapore', 'GRID Communications Pte Ltd'),
 '52801': ('+673', 'BN', 'Brunei Darussalam', 'Telekom Brunei Bhd (TelBru)'),
 '52802': ('+673', 'BN', 'Brunei Darussalam', 'b-mobile'),
 '52811': ('+673', 'BN', 'Brunei Darussalam', 'Datastream (DTSCom)'),
 '53001': ('+64', 'NZ', 'New Zealand', 'Vodafone'),
 '53002': ('+64', 'NZ', 'New Zealand', 'Spark/NZ Telecom'),
 '53003': ('+64', 'NZ', 'New Zealand', 'Walker Wireless Ltd.'),
 '53004': ('+64', 'NZ', 'New Zealand', 'Telstra'),
 '53005': ('+64', 'NZ', 'New Zealand', 'Spark/NZ Telecom'),
 '53024': ('+64', 'NZ', 'New Zealand', 'Two Degrees Mobile Ltd'),
 '53028': ('+64', 'NZ', 'New Zealand', '2degrees'),
 '53701': ('+675', 'PG', 'Papua New Guinea', 'Pacific Mobile'),
 '53702': ('+675', 'PG', 'Papua New Guinea', 'GreenCom PNG Ltd'),
 '53703': ('+675', 'PG', 'Papua New Guinea', 'Digicel'),
 '53901': ('+676', 'TO', 'Tonga', 'Tonga Communications'),
 '53943': ('+676', 'TO', 'Tonga', 'Shoreline Communication'),
 '54001': ('+677', 'SB', 'Solomon Islands', 'BREEZE'),
 '54002': ('+677', 'SB', 'Solomon Islands', 'bemobile'),
 '54010': ('+677', 'SB', 'Solomon Islands', 'BREEZE'),
 '54101': ('+678', 'VU', 'Vanuatu', 'SMILE'),
 '54105': ('+678', 'VU', 'Vanuatu', 'DigiCel'),
 '54201': ('+679', 'FJ', 'Fiji', 'Vodafone'),
 '54202': ('+679', 'FJ', 'Fiji', 'DigiCell'),
 '54411': ('+684', 'AS', 'American Samoa', 'Blue Sky Communications'),
 '54509': ('+686', 'KI', 'Kiribati', 'Kiribati Frigate'),
 '54601': ('+687', 'NC', 'New Caledonia', 'OPT Mobilis'),
 '54715': ('+689', 'PF', 'French Polynesia', 'Pacific Mobile Telecom (PMT)'),
 '54720': ('+689', 'PF', 'French Polynesia', 'Vini/Tikiphone'),
 '54801': ('+682', 'CK', 'Cook Islands', 'Telecom Cook Islands'),
 '54901': ('+685', 'WS', 'Samoa', 'Telecom Samoa Cellular Ltd.'),
 '54927': ('+685', 'WS', 'Samoa', 'Samoatel Mobile'),
 '55001': ('+691', 'FM', 'Micronesia', 'FSM Telecom'),
 '55201': ('+680', 'PW', 'Palau (Republic of)', 'Palau National Communications Corp. (PNCC) (Palau'),
 '55280': ('+680', 'PW', 'Palau (Republic of)', 'Palau Mobile Corp. (PMC) (Palau'),
 '55301': ('+688', 'TV', 'Tuvalu', 'Tuvalu Telecommunication Corporation (TTC)'),
 '55501': ('+683', 'NU', 'Niue', 'Niue Telecom'),
 '60201': ('+20', 'EG', 'Egypt', 'Orange/Mobinil'),
 '60202': ('+20', 'EG', 'Egypt', 'Vodafone/Mirsfone'),
 '60203': ('+20', 'EG', 'Egypt', 'ETISALAT'),
 '60301': ('+213', 'DZ', 'Algeria', 'ATM Mobils'),
 '60302': ('+213', 'DZ', 'Algeria', 'Orascom / DJEZZY'),
 '60303': ('+213', 'DZ', 'Algeria', 'Oreedo/Wataniya / Nedjma'),
 '60400': ('+212', 'MA', 'Morocco', 'Medi Telecom'),
 '60401': ('+212', 'MA', 'Morocco', 'IAM/Itissallat'),
 '60402': ('+212', 'MA', 'Morocco', 'INWI/WANA'),
 '60501': ('+216', 'TN', 'Tunisia', 'Orange'),
 '60502': ('+216', 'TN', 'Tunisia', 'TuniCell/Tunisia Telecom'),
 '60503': ('+216', 'TN', 'Tunisia', 'Oreedo/Orascom'),
 '60600': ('+218', 'LY', 'Libya', 'Libyana'),
 '60601': ('+218', 'LY', 'Libya', 'Al-Madar'),
 '60602': ('+218', 'LY', 'Libya', 'Al-Madar'),
 '60603': ('+218', 'LY', 'Libya', 'Libyana'),
 '60606': ('+218', 'LY', 'Libya', 'Hatef'),
 '60701': ('+220', 'GM', 'Gambia', 'Gamcel'),
 '60702': ('+220', 'GM', 'Gambia', 'Africel'),
 '60703': ('+220', 'GM', 'Gambia', 'Comium'),
 '60704': ('+220', 'GM', 'Gambia', 'Q-Cell'),
 '60801': ('+221', 'SN', 'Senegal', 'Orange/Sonatel'),
 '60802': ('+221', 'SN', 'Senegal', 'TIGO/Sentel GSM'),
 '60803': ('+221', 'SN', 'Senegal', 'Expresso/Sudatel'),
 '60901': ('+222', 'MR', 'Mauritania', 'Mattel'),
 '60902': ('+222', 'MR', 'Mauritania', 'Chinguitel SA'),
 '60910': ('+222', 'MR', 'Mauritania', 'Mauritel'),
 '61001': ('+223', 'ML', 'Mali', 'Malitel'),
 '61002': ('+223', 'ML', 'Mali', 'Orange/IKATEL'),
 '61101': ('+224', 'GN', 'Guinea', 'Orange/Sonatel/Spacetel'),
 '61102': ('+224', 'GN', 'Guinea', 'SotelGui'),
 '61103': ('+224', 'GN', 'Guinea', 'Intercel'),
 '61104': ('+224', 'GN', 'Guinea', 'MTN/Areeba'),
 '61105': ('+224', 'GN', 'Guinea', 'Celcom'),
 '61201': ('+225', 'CI', 'Ivory Coast', 'Comstar'),
 '61202': ('+225', 'CI', 'Ivory Coast', 'Atlantik Tel./Moov'),
 '61203': ('+225', 'CI', 'Ivory Coast', 'Orange'),
 '61204': ('+225', 'CI', 'Ivory Coast', 'Comium'),
 '61205': ('+225', 'CI', 'Ivory Coast', 'MTN'),
 '61206': ('+225', 'CI', 'Ivory Coast', 'OriCell'),
 '61207': ('+225', 'CI', 'Ivory Coast', 'Aircomm SA'),
 '61301': ('+226', 'BF', 'Burkina Faso', 'TeleMob-OnaTel'),
 '61302': ('+226', 'BF', 'Burkina Faso', 'Airtel/ZAIN/CelTel'),
 '61303': ('+226', 'BF', 'Burkina Faso', 'TeleCel'),
 '61401': ('+227', 'NE', 'Niger', 'Orange/Sahelc.'),
 '61402': ('+227', 'NE', 'Niger', 'Airtel/Zain/CelTel'),
 '61403': ('+227', 'NE', 'Niger', 'MOOV/TeleCel'),
 '61404': ('+227', 'NE', 'Niger', 'Orange/Sahelc.'),
 '61501': ('+228', 'TG', 'Togo', 'Togo Telecom/TogoCELL'),
 '61502': ('+228', 'TG', 'Togo', 'Telecel/MOOV'),
 '61503': ('+228', 'TG', 'Togo', 'Telecel/MOOV'),
 '61601': ('+229', 'BJ', 'Benin', 'Libercom'),
 '61602': ('+229', 'BJ', 'Benin', 'Etisalat/MOOV'),
 '61603': ('+229', 'BJ', 'Benin', 'MTN/Spacetel'),
 '61604': ('+229', 'BJ', 'Benin', 'Bell Benin/BBCOM'),
 '61605': ('+229', 'BJ', 'Benin', 'GloMobile'),
 '61701': ('+230', 'MU', 'Mauritius', 'Orange/Cellplus'),
 '61702': ('+230', 'MU', 'Mauritius', 'Mahanagar Telephone'),
 '61703': ('+230', 'MU', 'Mauritius', 'Mahanagar Telephone'),
 '61710': ('+230', 'MU', 'Mauritius', 'Emtel Ltd'),
 '61801': ('+231', 'LR', 'Liberia', 'Lonestar'),
 '61802': ('+231', 'LR', 'Liberia', 'Libercell'),
 '61804': ('+231', 'LR', 'Liberia', 'Comium BVI'),
 '61807': ('+231', 'LR', 'Liberia', 'CELLCOM'),
 '61820': ('+231', 'LR', 'Liberia', 'LibTelco'),
 '61901': ('+232', 'SL', 'Sierra Leone', 'Airtel/Zain/Celtel'),
 '61902': ('+232', 'SL', 'Sierra Leone', 'Tigo/Millicom'),
 '61903': ('+232', 'SL', 'Sierra Leone', 'Africel'),
 '61904': ('+232', 'SL', 'Sierra Leone', 'Comium'),
 '61905': ('+232', 'SL', 'Sierra Leone', 'Africel'),
 '61925': ('+232', 'SL', 'Sierra Leone', 'Mobitel'),
 '62001': ('+233', 'GH', 'Ghana', 'MTN'),
 '62002': ('+233', 'GH', 'Ghana', 'Vodafone'),
 '62003': ('+233', 'GH', 'Ghana', 'Milicom/Tigo'),
 '62004': ('+233', 'GH', 'Ghana', 'Expresso Ghana Ltd'),
 '62006': ('+233', 'GH', 'Ghana', 'Airtel/ZAIN'),
 '62007': ('+233', 'GH', 'Ghana', 'GloMobile'),
 '62101': ('+234', 'NG', 'Nigeria', 'Visafone'),
 '62120': ('+234', 'NG', 'Nigeria', 'Airtel/ZAIN/Econet'),
 '62125': ('+234', 'NG', 'Nigeria', 'Visafone'),
 '62130': ('+234', 'NG', 'Nigeria', 'MTN'),
 '62140': ('+234', 'NG', 'Nigeria', 'M-Tel/Nigeria Telecom. Ltd.'),
 '62150': ('+234', 'NG', 'Nigeria', 'Glo Mobile'),
 '62160': ('+234', 'NG', 'Nigeria', 'ETISALAT'),
 '62199': ('+234', 'NG', 'Nigeria', 'Starcomms'),
 '62201': ('+235', 'TD', 'Chad', 'Airtel/ZAIN/Celtel'),
 '62202': ('+235', 'TD', 'Chad', 'Tchad Mobile'),
 '62203': ('+235', 'TD', 'Chad', 'Tigo/Milicom/Tchad Mobile'),
 '62204': ('+235', 'TD', 'Chad', 'Salam/Sotel'),
 '62301': ('+236', 'CF', 'Central African Rep.', 'Centrafr. Telecom+'),
 '62302': ('+236', 'CF', 'Central African Rep.', 'Telecel Centraf.'),
 '62303': ('+236', 'CF', 'Central African Rep.', 'Orange/Celca'),
 '62304': ('+236', 'CF', 'Central African Rep.', 'Nationlink'),
 '62401': ('+237', 'CM', 'Cameroon', 'MTN'),
 '62402': ('+237', 'CM', 'Cameroon', 'Orange'),
 '62404': ('+237', 'CM', 'Cameroon', 'Nextel'),
 '62501': ('+238', 'CV', 'Cape Verde', 'CV Movel'),
 '62502': ('+238', 'CV', 'Cape Verde', 'T+ Telecom'),
 '62601': ('+239', 'ST', 'Sao Tome & Principe', 'CSTmovel'),
 '62701': ('+240', 'GQ', 'Equatorial Guinea', 'ORANGE/GETESA'),
 '62703': ('+240', 'GQ', 'Equatorial Guinea', 'HiTs-GE'),
 '62801': ('+241', 'GA', 'Gabon', 'Libertis S.A.'),
 '62802': ('+241', 'GA', 'Gabon', 'MOOV/Telecel'),
 '62803': ('+241', 'GA', 'Gabon', 'Airtel/ZAIN/Celtel Gabon S.A.'),
 '62804': ('+241', 'GA', 'Gabon', 'Azur/Usan S.A.'),
 '62901': ('+242', 'CG', 'Congo, Republic', 'Airtel SA'),
 '62902': ('+242', 'CG', 'Congo, Republic', 'Azur SA (ETC)'),
 '62907': ('+242', 'CG', 'Congo, Republic', 'Warid'),
 '62910': ('+242', 'CG', 'Congo, Republic', 'MTN/Libertis'),
 '63001': ('+243', 'CD', 'Congo, Dem. Rep.', 'Vodacom'),
 '63002': ('+243', 'CD', 'Congo, Dem. Rep.', 'Airtel/ZAIN'),
 '63005': ('+243', 'CD', 'Congo, Dem. Rep.', 'SuperCell'),
 '63086': ('+243', 'CD', 'Congo, Dem. Rep.', 'Orange RDC sarl'),
 '63088': ('+243', 'CD', 'Congo, Dem. Rep.', 'Yozma Timeturns sprl (YTT)'),
 '63089': ('+243', 'CD', 'Congo, Dem. Rep.', 'TIGO/Oasis'),
 '63090': ('+243', 'CD', 'Congo, Dem. Rep.', 'Africell'),
 '63102': ('+244', 'AO', 'Angola', 'Unitel'),
 '63104': ('+244', 'AO', 'Angola', 'MoviCel'),
 '63201': ('+245', 'GW', 'Guinea-Bissau', 'GuineTel'),
 '63202': ('+245', 'GW', 'Guinea-Bissau', 'SpaceTel'),
 '63203': ('+245', 'GW', 'Guinea-Bissau', 'Orange'),
 '63301': ('+248', 'SC', 'Seychelles', 'C&W'),
 '63302': ('+248', 'SC', 'Seychelles', 'Smartcom'),
 '63310': ('+248', 'SC', 'Seychelles', 'Airtel'),
 '63400': ('+249', 'SD', 'Sudan', 'Canar Telecom'),
 '63401': ('+249', 'SD', 'Sudan', 'ZAIN/Mobitel'),
 '63402': ('+249', 'SD', 'Sudan', 'MTN'),
 '63405': ('+249', 'SD', 'Sudan', 'Vivacell'),
 '63406': ('+249', 'SD', 'Sudan', 'ZAIN/Mobitel'),
 '63407': ('+249', 'SD', 'Sudan', 'Sudani One'),
 '63408': ('+249', 'SD', 'Sudan', 'Vivacell'),
 '63415': ('+249', 'SD', 'Sudan', 'Sudani One'),
 '63422': ('+249', 'SD', 'Sudan', 'MTN'),
 '63510': ('+250', 'RW', 'Rwanda', 'MTN/Rwandacell'),
 '63513': ('+250', 'RW', 'Rwanda', 'TIGO'),
 '63514': ('+250', 'RW', 'Rwanda', 'Airtel'),
 '63601': ('+251', 'ET', 'Ethiopia', 'ETH/MTN'),
 '63701': ('+252', 'SO', 'Somalia', 'Telesom'),
 '63704': ('+252', 'SO', 'Somalia', 'Somafone'),
 '63710': ('+252', 'SO', 'Somalia', 'Nationlink'),
 '63719': ('+252', 'SO', 'Somalia', 'HorTel'),
 '63730': ('+252', 'SO', 'Somalia', 'Golis'),
 '63760': ('+252', 'SO', 'Somalia', 'Nationlink'),
 '63782': ('+252', 'SO', 'Somalia', 'Telcom Mobile Somalia'),
 '63801': ('+253', 'DJ', 'Djibouti', 'Djibouti Telecom SA (Evatis)'),
 '63902': ('+254', 'KE', 'Kenya', 'Safaricom Ltd.'),
 '63903': ('+254', 'KE', 'Kenya', 'Airtel/Zain/Celtel Ltd.'),
 '63905': ('+254', 'KE', 'Kenya', 'Econet Wireless'),
 '63907': ('+254', 'KE', 'Kenya', 'Orange'),
 '64001': ('+255', 'TZ', 'Tanzania', 'Tri Telecomm. Ltd.'),
 '64002': ('+255', 'TZ', 'Tanzania', 'TIGO/MIC'),
 '64003': ('+255', 'TZ', 'Tanzania', 'Zantel/Zanzibar Telecom'),
 '64004': ('+255', 'TZ', 'Tanzania', 'Vodacom Ltd'),
 '64005': ('+255', 'TZ', 'Tanzania', 'Airtel/ZAIN/Celtel'),
 '64006': ('+255', 'TZ', 'Tanzania', 'Dovetel (T) Ltd'),
 '64007': ('+255', 'TZ', 'Tanzania', 'Tanzania Telecommunications Company Ltd (TTCL)'),
 '64008': ('+255', 'TZ', 'Tanzania', 'Benson Informatics Ltd'),
 '64009': ('+255', 'TZ', 'Tanzania', 'ExcellentCom (T) Ltd'),
 '64011': ('+255', 'TZ', 'Tanzania', 'Smile Communications Tanzania Ltd'),
 '64101': ('+256', 'UG', 'Uganda', 'Airtel/Celtel'),
 '64110': ('+256', 'UG', 'Uganda', 'MTN Ltd.'),
 '64111': ('+256', 'UG', 'Uganda', 'Uganda Telecom Ltd.'),
 '64114': ('+256', 'UG', 'Uganda', 'Orange'),
 '64118': ('+256', 'UG', 'Uganda', 'Suretelecom Uganda Ltd'),
 '64122': ('+256', 'UG', 'Uganda', 'Airtel/Warid'),
 '64130': ('+256', 'UG', 'Uganda', 'K2 Telecom Ltd'),
 '64133': ('+256', 'UG', 'Uganda', 'Smile Communications Uganda Ltd'),
 '64166': ('+256', 'UG', 'Uganda', 'i-Tel Ltd'),
 '64201': ('+257', 'BI', 'Burundi', 'Spacetel / Econet / Leo'),
 '64202': ('+257', 'BI', 'Burundi', 'Africel / Safaris'),
 '64203': ('+257', 'BI', 'Burundi', 'Onatel / Telecel'),
 '64207': ('+257', 'BI', 'Burundi', 'Smart Mobile / LACELL'),
 '64208': ('+257', 'BI', 'Burundi', 'Lumitel/Viettel'),
 '64282': ('+257', 'BI', 'Burundi', 'Spacetel / Econet / Leo'),
 '64301': ('+258', 'MZ', 'Mozambique', 'mCel'),
 '64303': ('+258', 'MZ', 'Mozambique', 'Movitel'),
 '64304': ('+258', 'MZ', 'Mozambique', 'Vodacom'),
 '64501': ('+260', 'ZM', 'Zambia', 'Airtel/Zain/Celtel'),
 '64502': ('+260', 'ZM', 'Zambia', 'MTN/Telecel'),
 '64503': ('+260', 'ZM', 'Zambia', 'Zamtel/Cell Z/MTS'),
 '64601': ('+261', 'MG', 'Madagascar', 'Airtel/MADACOM'),
 '64602': ('+261', 'MG', 'Madagascar', 'Orange/Soci'),
 '64603': ('+261', 'MG', 'Madagascar', 'Sacel'),
 '64604': ('+261', 'MG', 'Madagascar', 'Telma'),
 '64700': ('+262', 'RE', 'Reunion', 'Orange'),
 '64702': ('+262', 'RE', 'Reunion', 'Outremer Telecom'),
 '64710': ('+262', 'RE', 'Reunion', 'SFR'),
 '64801': ('+263', 'ZW', 'Zimbabwe', 'Net One'),
 '64803': ('+263', 'ZW', 'Zimbabwe', 'Telecel'),
 '64804': ('+263', 'ZW', 'Zimbabwe', 'Econet'),
 '64901': ('+264', 'NA', 'Namibia', 'MTC'),
 '64902': ('+264', 'NA', 'Namibia', 'Switch/Nam. Telec.'),
 '64903': ('+264', 'NA', 'Namibia', 'Leo / Orascom'),
 '65001': ('+265', 'MW', 'Malawi', 'TNM/Telekom Network Ltd.'),
 '65010': ('+265', 'MW', 'Malawi', 'Airtel/Zain/Celtel ltd.'),
 '65101': ('+266', 'LS', 'Lesotho', 'Vodacom Lesotho'),
 '65102': ('+266', 'LS', 'Lesotho', 'Econet/Ezi-cel'),
 '65201': ('+267', 'BW', 'Botswana', 'Mascom Wireless (Pty) Ltd.'),
 '65202': ('+267', 'BW', 'Botswana', 'Orange'),
 '65204': ('+267', 'BW', 'Botswana', 'BeMOBILE'),
 '65301': ('+268', 'SZ', 'Swaziland', 'SwaziTelecom'),
 '65310': ('+268', 'SZ', 'Swaziland', 'Swazi MTN'),
 '65401': ('+269', 'KM', 'Comoros', 'HURI - SNPT'),
 '65501': ('+27', 'ZA', 'South Africa', 'Vodacom'),
 '65502': ('+27', 'ZA', 'South Africa', '8.ta'),
 '65506': ('+27', 'ZA', 'South Africa', 'Sentech'),
 '65507': ('+27', 'ZA', 'South Africa', 'Cell C'),
 '65510': ('+27', 'ZA', 'South Africa', 'MTN'),
 '65512': ('+27', 'ZA', 'South Africa', 'MTN'),
 '65519': ('+27', 'ZA', 'South Africa', 'Wireless Business Solutions (Pty) Ltd'),
 '65521': ('+27', 'ZA', 'South Africa', 'Cape Town Metropolitan'),
 '65701': ('+291', 'ER', 'Eritrea', 'Eritel'),
 '65902': ('+211', 'SS', 'South Sudan (Republic of)', 'MTN South Sudan (South Sudan'),
 '65903': ('+211', 'SS', 'South Sudan (Republic of)', 'Gemtel Ltd (South Sudan'),
 '65904': ('+211', 'SS', 'South Sudan (Republic of)', 'Network of The World Ltd (NOW) (South Sudan'),
 '65906': ('+211', 'SS', 'South Sudan (Republic of)', 'Zain South Sudan (South Sudan'),
 '70267': ('+501', 'BZ', 'Belize', 'DigiCell'),
 '70268': ('+501', 'BZ', 'Belize', 'International Telco (INTELCO)'),
 '70401': ('+502', 'GT', 'Guatemala', 'SERCOM'),
 '70402': ('+502', 'GT', 'Guatemala', 'TIGO/COMCEL'),
 '70403': ('+502', 'GT', 'Guatemala', 'Telefonica'),
 '70601': ('+503', 'SV', 'El Salvador', 'CLARO/CTE'),
 '70602': ('+503', 'SV', 'El Salvador', 'Digicel'),
 '70603': ('+503', 'SV', 'El Salvador', 'Telemovil'),
 '70604': ('+503', 'SV', 'El Salvador', 'Telefonica'),
 '70605': ('+503', 'SV', 'El Salvador', 'INTELFON SA de CV'),
 '708001': ('+504', 'HN', 'Honduras', 'SERCOM/CLARO'),
 '708002': ('+504', 'HN', 'Honduras', 'Telefonica/CELTEL'),
 '708030': ('+504', 'HN', 'Honduras', 'HonduTel'),
 '708040': ('+504', 'HN', 'Honduras', 'Digicel'),
 '71021': ('+505', 'NI', 'Nicaragua', 'Empresa Nicaraguense de Telecomunicaciones SA (ENITEL)'),
 '71030': ('+505', 'NI', 'Nicaragua', 'Movistar'),
 '71073': ('+505', 'NI', 'Nicaragua', 'Claro'),
 '71201': ('+506', 'CR', 'Costa Rica', 'ICE'),
 '71202': ('+506', 'CR', 'Costa Rica', 'ICE'),
 '71203': ('+506', 'CR', 'Costa Rica', 'Claro'),
 '71204': ('+506', 'CR', 'Costa Rica', 'Movistar'),
 '71220': ('+506', 'CR', 'Costa Rica', 'Virtualis'),
 '71401': ('+507', 'PA', 'Panama', 'Cable & W./Mas Movil'),
 '71402': ('+507', 'PA', 'Panama', 'Movistar'),
 '714020': ('+507', 'PA', 'Panama', 'Movistar'),
 '71403': ('+507', 'PA', 'Panama', 'Claro'),
 '71404': ('+507', 'PA', 'Panama', 'Digicel'),
 '71601': ('+51', 'PE', 'Peru', 'GlobalStar'),
 '71602': ('+51', 'PE', 'Peru', 'GlobalStar'),
 '71606': ('+51', 'PE', 'Peru', 'Movistar'),
 '71607': ('+51', 'PE', 'Peru', 'Nextel'),
 '71610': ('+51', 'PE', 'Peru', 'Claro /Amer.Mov./TIM'),
 '71615': ('+51', 'PE', 'Peru', 'Viettel Mobile'),
 '71617': ('+51', 'PE', 'Peru', 'Nextel'),
 '71620': ('+51', 'PE', 'Peru', 'Claro /Amer.Mov./TIM'),
 '722010': ('+54', 'AR', 'Argentina Republic', 'Compania De Radiocomunicaciones Moviles SA'),
 '722020': ('+54', 'AR', 'Argentina Republic', 'Nextel'),
 '722070': ('+54', 'AR', 'Argentina Republic', 'Movistar/Telefonica'),
 '722310': ('+54', 'AR', 'Argentina Republic', 'Claro/ CTI/AMX'),
 '722320': ('+54', 'AR', 'Argentina Republic', 'Claro/ CTI/AMX'),
 '722330': ('+54', 'AR', 'Argentina Republic', 'Claro/ CTI/AMX'),
 '722341': ('+54', 'AR', 'Argentina Republic', 'Telecom Personal S.A.'),
 '72400': ('+55', 'BR', 'Brazil', 'Nextel (Telet)'),
 '72401': ('+55', 'BR', 'Brazil', 'Vivo S.A./Telemig'),
 '72402': ('+55', 'BR', 'Brazil', 'TIM'),
 '72403': ('+55', 'BR', 'Brazil', 'TIM'),
 '72404': ('+55', 'BR', 'Brazil', 'TIM'),
 '72405': ('+55', 'BR', 'Brazil', 'Claro/Albra/America Movil'),
 '72406': ('+55', 'BR', 'Brazil', 'Vivo S.A./Telemig'),
 '72407': ('+55', 'BR', 'Brazil', 'CTBC/Triangulo'),
 '72408': ('+55', 'BR', 'Brazil', 'TIM'),
 '72410': ('+55', 'BR', 'Brazil', 'Vivo S.A./Telemig'),
 '72411': ('+55', 'BR', 'Brazil', 'Vivo S.A./Telemig'),
 '72412': ('+55', 'BR', 'Brazil', 'Claro/Albra/America Movil'),
 '72415': ('+55', 'BR', 'Brazil', 'Sercontel Cel'),
 '72416': ('+55', 'BR', 'Brazil', 'Brazil Telcom'),
 '72419': ('+55', 'BR', 'Brazil', 'Vivo S.A./Telemig'),
 '72423': ('+55', 'BR', 'Brazil', 'Vivo S.A./Telemig'),
 '72424': ('+55', 'BR', 'Brazil', 'Amazonia Celular S/A'),
 '72430': ('+55', 'BR', 'Brazil', 'Oi (TNL PCS / Oi)'),
 '72431': ('+55', 'BR', 'Brazil', 'Oi (TNL PCS / Oi)'),
 '72432': ('+55', 'BR', 'Brazil', 'CTBC Celular SA (CTBC)'),
 '72433': ('+55', 'BR', 'Brazil', 'CTBC Celular SA (CTBC)'),
 '72434': ('+55', 'BR', 'Brazil', 'CTBC Celular SA (CTBC)'),
 '72437': ('+55', 'BR', 'Brazil', 'Unicel do Brasil Telecomunicacoes Ltda'),
 '72438': ('+55', 'BR', 'Brazil', 'Claro/Albra/America Movil'),
 '72439': ('+55', 'BR', 'Brazil', 'Nextel (Telet)'),
 '72454': ('+55', 'BR', 'Brazil', 'PORTO SEGURO TELECOMUNICACOES'),
 '73000': ('+56', 'CL', 'Chile', 'TESAM SA'),
 '73001': ('+56', 'CL', 'Chile', 'Entel Telefonia Mov'),
 '73002': ('+56', 'CL', 'Chile', 'TELEFONICA'),
 '73003': ('+56', 'CL', 'Chile', 'Claro'),
 '73004': ('+56', 'CL', 'Chile', 'Nextel SA'),
 '73005': ('+56', 'CL', 'Chile', 'Nextel SA'),
 '73006': ('+56', 'CL', 'Chile', 'Blue Two Chile SA'),
 '73007': ('+56', 'CL', 'Chile', 'TELEFONICA'),
 '73008': ('+56', 'CL', 'Chile', 'VTR Banda Ancha SA'),
 '73009': ('+56', 'CL', 'Chile', 'Nextel SA'),
 '73010': ('+56', 'CL', 'Chile', 'Entel Telefonia'),
 '73011': ('+56', 'CL', 'Chile', 'Celupago SA'),
 '73012': ('+56', 'CL', 'Chile', 'Telestar Movil SA'),
 '73013': ('+56', 'CL', 'Chile', 'Tribe Mobile SPA'),
 '73014': ('+56', 'CL', 'Chile', 'Netline Telefonica Movil Ltda'),
 '73015': ('+56', 'CL', 'Chile', 'Cibeles Telecom SA'),
 '732001': ('+57', 'CO', 'Colombia', 'TIGO/Colombia Movil'),
 '732002': ('+57', 'CO', 'Colombia', 'Edatel S.A.'),
 '732020': ('+57', 'CO', 'Colombia', 'UNE EPM Telecomunicaciones SA ESP'),
 '732101': ('+57', 'CO', 'Colombia', 'Comcel S.A. Occel S.A./Celcaribe'),
 '732102': ('+57', 'CO', 'Colombia', 'Movistar'),
 '732103': ('+57', 'CO', 'Colombia', 'TIGO/Colombia Movil'),
 '732111': ('+57', 'CO', 'Colombia', 'TIGO/Colombia Movil'),
 '732123': ('+57', 'CO', 'Colombia', 'Movistar'),
 '732130': ('+57', 'CO', 'Colombia', 'Avantel SAS'),
 '732142': ('+57', 'CO', 'Colombia', 'UNE EPM Telecomunicaciones SA ESP'),
 '732154': ('+57', 'CO', 'Colombia', 'Virgin Mobile Colombia SAS'),
 '73401': ('+58', 'VE', 'Venezuela', 'DigiTel C.A.'),
 '73402': ('+58', 'VE', 'Venezuela', 'DigiTel C.A.'),
 '73403': ('+58', 'VE', 'Venezuela', 'DigiTel C.A.'),
 '73404': ('+58', 'VE', 'Venezuela', 'Movistar/TelCel'),
 '73406': ('+58', 'VE', 'Venezuela', 'Movilnet C.A.'),
 '73601': ('+591', 'BO', 'Bolivia', 'Nuevatel'),
 '73602': ('+591', 'BO', 'Bolivia', 'Entel Pcs'),
 '73603': ('+591', 'BO', 'Bolivia', 'TELECEL BOLIVIA'),
 '73801': ('+592', 'GY', 'Guyana', 'DigiCel'),
 '73802': ('+592', 'GY', 'Guyana', 'Cellink Plus'),
 '74000': ('+593', 'EC', 'Ecuador', 'MOVISTAR/OteCel'),
 '74001': ('+593', 'EC', 'Ecuador', 'Porta/Conecel'),
 '74002': ('+593', 'EC', 'Ecuador', 'Alegro/Telcsa'),
 '74401': ('+595', 'PY', 'Paraguay', 'Hola/VOX'),
 '74402': ('+595', 'PY', 'Paraguay', 'Claro/Hutchison'),
 '74403': ('+595', 'PY', 'Paraguay', 'Compa'),
 '74404': ('+595', 'PY', 'Paraguay', 'Tigo/Telecel'),
 '74405': ('+595', 'PY', 'Paraguay', 'TIM/Nucleo/Personal'),
 '74601': ('+597', 'SR', 'Suriname', 'Telesur'),
 '74602': ('+597', 'SR', 'Suriname', 'Telecommunicatiebedrijf Suriname (TELESUR)'),
 '74603': ('+597', 'SR', 'Suriname', 'Digicel'),
 '74604': ('+597', 'SR', 'Suriname', 'UNIQA'),
 '74801': ('+598', 'UY', 'Uruguay', 'Ancel/Antel'),
 '74803': ('+598', 'UY', 'Uruguay', 'Ancel/Antel'),
 '74807': ('+598', 'UY', 'Uruguay', 'MOVISTAR'),
 '74810': ('+598', 'UY', 'Uruguay', 'Claro/AM Wireless'),
 '750001': ('+500', 'FK', 'Falkland Islands (Malvinas)', 'Cable and Wireless South Atlantic Ltd (Falkland Islands'),
 '90105': ('+870', 'Intl', 'Satellite Networks', 'Thuraya Satellite'),
 '90111': ('+870', 'Intl', 'Satellite Networks', 'InMarSAT'),
 '90112': ('+870', 'Intl', 'Satellite Networks', 'Maritime Communications Partner AS'),
 '90113': ('+882', 'Intl', 'International Networks', 'Antarctica'),
 '90114': ('+870', 'Intl', 'Satellite Networks', 'AeroMobile')
}

MCCr_dict = {
 'AD': ('213', '+376', 'Andorra'),
 'AE': ('424', '+971', 'United Arab Emirates'),
 'AF': ('412', '+93', 'Afghanistan'),
 'AG': ('344', '+1268', 'Antigua and Barbuda'),
 'AI': ('365', '+1264', 'Anguilla'),
 'AL': ('276', '+355', 'Albania'),
 'AM': ('283', '+374', 'Armenia'),
 'AN': ('362', '+599', 'Netherlands Antilles'),
 'AO': ('631', '+244', 'Angola'),
 'AR': ('722', '+54', 'Argentina Republic'),
 'AS': ('544', '+684', 'American Samoa'),
 'AT': ('232', '+43', 'Austria'),
 'AU': ('505', '+61', 'Australia'),
 'AW': ('363', '+297', 'Aruba'),
 'AZ': ('400', '+994', 'Azerbaijan'),
 'BA': ('218', '+387', 'Bosnia & Herzegov.'),
 'BB': ('342', '+1246', 'Barbados'),
 'BD': ('470', '+880', 'Bangladesh'),
 'BE': ('206', '+32', 'Belgium'),
 'BF': ('613', '+226', 'Burkina Faso'),
 'BG': ('284', '+359', 'Bulgaria'),
 'BH': ('426', '+973', 'Bahrain'),
 'BI': ('642', '+257', 'Burundi'),
 'BJ': ('616', '+229', 'Benin'),
 'BM': ('350', '+1441', 'Bermuda'),
 'BN': ('528', '+673', 'Brunei Darussalam'),
 'BO': ('736', '+591', 'Bolivia'),
 'BR': ('724', '+55', 'Brazil'),
 'BS': ('364', '+1242', 'Bahamas'),
 'BT': ('402', '+975', 'Bhutan'),
 'BW': ('652', '+267', 'Botswana'),
 'BY': ('257', '+375', 'Belarus'),
 'BZ': ('702', '+501', 'Belize'),
 'CA': ('302', '+1', 'Canada'),
 'CD': ('630', '+243', 'Congo, Dem. Rep.'),
 'CF': ('623', '+236', 'Central African Rep.'),
 'CG': ('629', '+242', 'Congo, Republic'),
 'CH': ('228', '+41', 'Switzerland'),
 'CI': ('612', '+225', 'Ivory Coast'),
 'CK': ('548', '+682', 'Cook Islands'),
 'CL': ('730', '+56', 'Chile'),
 'CM': ('624', '+237', 'Cameroon'),
 'CN': ('460', '+86', 'China'),
 'CO': ('732', '+57', 'Colombia'),
 'CR': ('712', '+506', 'Costa Rica'),
 'CU': ('368', '+53', 'Cuba'),
 'CV': ('625', '+238', 'Cape Verde'),
 'CW': ('362', '+599', 'Curacao'),
 'CY': ('280', '+357', 'Cyprus'),
 'CZ': ('230', '+420', 'Czech Rep.'),
 'DE': ('262', '+49', 'Germany'),
 'DJ': ('638', '+253', 'Djibouti'),
 'DK': ('238', '+45', 'Denmark'),
 'DM': ('366', '+1767', 'Dominica'),
 'DO': ('370', '+1809', 'Dominican Republic'),
 'DZ': ('603', '+213', 'Algeria'),
 'EC': ('740', '+593', 'Ecuador'),
 'EE': ('248', '+372', 'Estonia'),
 'EG': ('602', '+20', 'Egypt'),
 'ER': ('657', '+291', 'Eritrea'),
 'ES': ('214', '+34', 'Spain'),
 'ET': ('636', '+251', 'Ethiopia'),
 'FG': ('340', '+594', 'French Guiana'),
 'FI': ('244', '+358', 'Finland'),
 'FJ': ('542', '+679', 'Fiji'),
 'FK': ('750', '+500', 'Falkland Islands (Malvinas)'),
 'FM': ('550', '+691', 'Micronesia'),
 'FO': ('288', '+298', 'Faroe Islands'),
 'FR': ('208', '+33', 'France'),
 'GA': ('628', '+241', 'Gabon'),
 'GB': ('234', '+44', 'United Kingdom'),
 'GD': ('352', '+1473', 'Grenada'),
 'GE': ('289', '+7', 'Abkhazia'),
 'GH': ('620', '+233', 'Ghana'),
 'GI': ('266', '+350', 'Gibraltar'),
 'GL': ('290', '+299', 'Greenland'),
 'GM': ('607', '+220', 'Gambia'),
 'GN': ('611', '+224', 'Guinea'),
 'GP': ('340', '+590', 'Guadeloupe'),
 'GQ': ('627', '+240', 'Equatorial Guinea'),
 'GR': ('202', '+30', 'Greece'),
 'GT': ('704', '+502', 'Guatemala'),
 'GU': ('310', '+1671', 'Guam'),
 'GW': ('632', '+245', 'Guinea-Bissau'),
 'GY': ('738', '+592', 'Guyana'),
 'HK': ('454', '+852', 'Hongkong, China'),
 'HN': ('708', '+504', 'Honduras'),
 'HR': ('219', '+385', 'Croatia'),
 'HT': ('372', '+509', 'Haiti'),
 'HU': ('216', '+36', 'Hungary'),
 'ID': ('510', '+62', 'Indonesia'),
 'IE': ('272', '+353', 'Ireland'),
 'IL': ('425', '+972', 'Israel'),
 'IN': ('404', '+91', 'India'),
 'IQ': ('418', '+964', 'Iraq'),
 'IR': ('432', '+98', 'Iran'),
 'IS': ('274', '+354', 'Iceland'),
 'IT': ('222', '+39', 'Italy'),
 'JM': ('338', '+1876', 'Jamaica'),
 'JO': ('416', '+962', 'Jordan'),
 'JP': ('440', '+81', 'Japan'),
 'KE': ('639', '+254', 'Kenya'),
 'KG': ('437', '+996', 'Kyrgyzstan'),
 'KH': ('456', '+855', 'Cambodia'),
 'KI': ('545', '+686', 'Kiribati'),
 'KM': ('654', '+269', 'Comoros'),
 'KN': ('356', '+1869', 'Saint Kitts and Nevis'),
 'KP': ('467', '+850', "Korea N., Dem. People's Rep."),
 'KR': ('450', '+82', 'Korea S, Republic of'),
 'KW': ('419', '+965', 'Kuwait'),
 'KY': ('346', '+1345', 'Cayman Islands'),
 'KZ': ('401', '+7', 'Kazakhstan'),
 'LA': ('457', '+856', 'Laos P.D.R.'),
 'LB': ('415', '+961', 'Lebanon'),
 'LC': ('358', '+1758', 'Saint Lucia'),
 'LI': ('295', '+423', 'Liechtenstein'),
 'LK': ('413', '+94', 'Sri Lanka'),
 'LR': ('618', '+231', 'Liberia'),
 'LS': ('651', '+266', 'Lesotho'),
 'LT': ('246', '+370', 'Lithuania'),
 'LU': ('270', '+352', 'Luxembourg'),
 'LV': ('247', '+371', 'Latvia'),
 'LY': ('606', '+218', 'Libya'),
 'MA': ('604', '+212', 'Morocco'),
 'MC': ('212', '+377', 'Monaco'),
 'MD': ('259', '+373', 'Moldova'),
 'ME': ('297', '+382', 'Montenegro'),
 'MG': ('646', '+261', 'Madagascar'),
 'MK': ('294', '+389', 'Macedonia'),
 'ML': ('610', '+223', 'Mali'),
 'MM': ('414', '+95', 'Myanmar (Burma)'),
 'MN': ('428', '+976', 'Mongolia'),
 'MO': ('455', '+853', 'Macao, China'),
 'MQ': ('340', '+596', 'Martinique (French Department of)'),
 'MR': ('609', '+222', 'Mauritania'),
 'MS': ('354', '+1664', 'Montserrat'),
 'MT': ('278', '+356', 'Malta'),
 'MU': ('617', '+230', 'Mauritius'),
 'MV': ('472', '+960', 'Maldives'),
 'MW': ('650', '+265', 'Malawi'),
 'MX': ('334', '+52', 'Mexico'),
 'MY': ('502', '+60', 'Malaysia'),
 'MZ': ('643', '+258', 'Mozambique'),
 'Intl': ('901', '+882', 'International Networks'),
 'NA': ('649', '+264', 'Namibia'),
 'NC': ('546', '+687', 'New Caledonia'),
 'NE': ('614', '+227', 'Niger'),
 'NG': ('621', '+234', 'Nigeria'),
 'NI': ('710', '+505', 'Nicaragua'),
 'NL': ('204', '+31', 'Netherlands'),
 'NO': ('242', '+47', 'Norway'),
 'NP': ('429', '+977', 'Nepal'),
 'NU': ('555', '+683', 'Niue'),
 'NZ': ('530', '+64', 'New Zealand'),
 'OM': ('422', '+968', 'Oman'),
 'PA': ('714', '+507', 'Panama'),
 'PE': ('716', '+51', 'Peru'),
 'PF': ('547', '+689', 'French Polynesia'),
 'PG': ('537', '+675', 'Papua New Guinea'),
 'PH': ('515', '+63', 'Philippines'),
 'PK': ('410', '+92', 'Pakistan'),
 'PL': ('260', '+48', 'Poland'),
 'PM': ('308', '+508', 'St. Pierre & Miquelon'),
 'PR': ('330', '+1', 'Puerto Rico'),
 'PS': ('425', '+970', 'Palestinian Territory'),
 'PT': ('268', '+351', 'Portugal'),
 'PW': ('552', '+680', 'Palau (Republic of)'),
 'PY': ('744', '+595', 'Paraguay'),
 'QA': ('427', '+974', 'Qatar'),
 'RE': ('647', '+262', 'Reunion'),
 'RO': ('226', '+40', 'Romania'),
 'RS': ('220', '+381', 'Serbia'),
 'RU': ('250', '+79', 'Russian Federation'),
 'RW': ('635', '+250', 'Rwanda'),
 'SA': ('420', '+966', 'Saudi Arabia'),
 'SB': ('540', '+677', 'Solomon Islands'),
 'SC': ('633', '+248', 'Seychelles'),
 'SD': ('634', '+249', 'Sudan'),
 'SE': ('240', '+46', 'Sweden'),
 'SG': ('525', '+65', 'Singapore'),
 'SI': ('293', '+386', 'Slovenia'),
 'SK': ('231', '+421', 'Slovakia'),
 'SL': ('619', '+232', 'Sierra Leone'),
 'SM': ('292', '+378', 'San Marino'),
 'SN': ('608', '+221', 'Senegal'),
 'SO': ('637', '+252', 'Somalia'),
 'SR': ('746', '+597', 'Suriname'),
 'SS': ('659', '+211', 'South Sudan (Republic of)'),
 'ST': ('626', '+239', 'Sao Tome & Principe'),
 'SV': ('706', '+503', 'El Salvador'),
 'SY': ('417', '+963', 'Syrian Arab Republic'),
 'SZ': ('653', '+268', 'Swaziland'),
 'TC': ('376', '+1', 'Turks and Caicos Islands'),
 'TD': ('622', '+235', 'Chad'),
 'TG': ('615', '+228', 'Togo'),
 'TH': ('520', '+66', 'Thailand'),
 'TK': ('436', '+992', 'Tajikistan'),
 'TM': ('438', '+993', 'Turkmenistan'),
 'TN': ('605', '+216', 'Tunisia'),
 'TO': ('539', '+676', 'Tonga'),
 'TP': ('514', '+670', 'Timor-Leste'),
 'TR': ('286', '+90', 'Turkey'),
 'TT': ('374', '+1868', 'Trinidad and Tobago'),
 'TV': ('553', '+688', 'Tuvalu'),
 'TW': ('466', '+886', 'Taiwan'),
 'TZ': ('640', '+255', 'Tanzania'),
 'UA': ('255', '+380', 'Ukraine'),
 'UG': ('641', '+256', 'Uganda'),
 'US': ('310', '+1', 'United States'),
 'UY': ('748', '+598', 'Uruguay'),
 'UZ': ('434', '+998', 'Uzbekistan'),
 'VC': ('360', '+1784', 'St. Vincent & Gren.'),
 'VE': ('734', '+58', 'Venezuela'),
 'VG': ('348', '+284', 'British Virgin Islands'),
 'VI': ('376', '+1340', 'Virgin Islands, U.S.'),
 'VN': ('452', '+84', 'Viet Nam'),
 'VU': ('541', '+678', 'Vanuatu'),
 'WS': ('549', '+685', 'Samoa'),
 'YE': ('421', '+967', 'Yemen'),
 'ZA': ('655', '+27', 'South Africa'),
 'ZM': ('645', '+260', 'Zambia'),
 'ZW': ('648', '+263', 'Zimbabwe')
}


CN_dict = {}
for cc, (mcc, cn, country) in MCCr_dict.items():
    cn = cn[1:]
    if cn in CN_dict:
        CN_dict[cn].add(cc)
    else:
        CN_dict[cn] = {cc}


# provide a function that returns the numbering prefix from an msisdn
def build_pref_sets():
    pref_sets = [set(), set(), set(), set()]
    for mcc, country_info in MCC_dict.items():
        pref = country_info[0]
        assert( isinstance(pref, str) )
        assert( pref[0] == '+' )
        pref_sets[len(pref[1:])-1].add(pref[1:])
    return pref_sets

_PREF_SETS = build_pref_sets()


def identify_prefix(msisdn):
    """returns the number prefix of the MSISDN,
    or None if the prefix is not recognized
    """
    if msisdn[:4] in _PREF_SETS[3]:
        return msisdn[:4]
    elif msisdn[:3] in _PREF_SETS[2]:
        return msisdn[:3]
    elif msisdn[:2] in _PREF_SETS[1]:
        return msisdn[:2]
    elif msisdn[:1] in _PREF_SETS[0]:
        return msisdn[:1]
    else:
        return None


def identify_country(msisdn):
    """returns the set of 2-character country codes associated with the msisdn,
    or the empty set if the prefix is not recognized
    """
    return CN_dict.get(identify_prefix(msisdn), set())


# table of ISO 3166 country codes
# https://www.iso.org/iso-3166-country-codes.html
# 
# last update : 2019/06/12

CC_table = [
 ('AF', 'AFG', '004', 'Afghanistan'),
 ('AX', 'ALA', '248', 'Åland Islands'),
 ('AL', 'ALB', '008', 'Albania'),
 ('DZ', 'DZA', '012', 'Algeria'),
 ('AS', 'ASM', '016', 'American Samoa'),
 ('AD', 'AND', '020', 'Andorra'),
 ('AO', 'AGO', '024', 'Angola'),
 ('AI', 'AIA', '660', 'Anguilla'),
 ('AQ', 'ATA', '010', 'Antarctica'),
 ('AG', 'ATG', '028', 'Antigua and Barbuda'),
 ('AR', 'ARG', '032', 'Argentina'),
 ('AM', 'ARM', '051', 'Armenia'),
 ('AW', 'ABW', '533', 'Aruba'),
 ('AU', 'AUS', '036', 'Australia'),
 ('AT', 'AUT', '040', 'Austria'),
 ('AZ', 'AZE', '031', 'Azerbaijan'),
 ('BS', 'BHS', '044', 'Bahamas (the)'),
 ('BH', 'BHR', '048', 'Bahrain'),
 ('BD', 'BGD', '050', 'Bangladesh'),
 ('BB', 'BRB', '052', 'Barbados'),
 ('BY', 'BLR', '112', 'Belarus'),
 ('BE', 'BEL', '056', 'Belgium'),
 ('BZ', 'BLZ', '084', 'Belize'),
 ('BJ', 'BEN', '204', 'Benin'),
 ('BM', 'BMU', '060', 'Bermuda'),
 ('BT', 'BTN', '064', 'Bhutan'),
 ('BO', 'BOL', '068', 'Bolivia (Plurinational State of)'),
 ('BQ', 'BES', '535', 'Bonaire, Sint Eustatius and Saba'),
 ('BA', 'BIH', '070', 'Bosnia and Herzegovina'),
 ('BW', 'BWA', '072', 'Botswana'),
 ('BV', 'BVT', '074', 'Bouvet Island'),
 ('BR', 'BRA', '076', 'Brazil'),
 ('IO', 'IOT', '086', 'British Indian Ocean Territory (the)'),
 ('BN', 'BRN', '096', 'Brunei Darussalam'),
 ('BG', 'BGR', '100', 'Bulgaria'),
 ('BF', 'BFA', '854', 'Burkina Faso'),
 ('BI', 'BDI', '108', 'Burundi'),
 ('CV', 'CPV', '132', 'Cabo Verde'),
 ('KH', 'KHM', '116', 'Cambodia'),
 ('CM', 'CMR', '120', 'Cameroon'),
 ('CA', 'CAN', '124', 'Canada'),
 ('KY', 'CYM', '136', 'Cayman Islands (the)'),
 ('CF', 'CAF', '140', 'Central African Republic (the)'),
 ('TD', 'TCD', '148', 'Chad'),
 ('CL', 'CHL', '152', 'Chile'),
 ('CN', 'CHN', '156', 'China'),
 ('CX', 'CXR', '162', 'Christmas Island'),
 ('CC', 'CCK', '166', 'Cocos (Keeling) Islands (the)'),
 ('CO', 'COL', '170', 'Colombia'),
 ('KM', 'COM', '174', 'Comoros (the)'),
 ('CD', 'COD', '180', 'Congo (the Democratic Republic of the)'),
 ('CG', 'COG', '178', 'Congo (the)'),
 ('CK', 'COK', '184', 'Cook Islands (the)'),
 ('CR', 'CRI', '188', 'Costa Rica'),
 ('CI', 'CIV', '384', "Côte d'Ivoire"),
 ('HR', 'HRV', '191', 'Croatia'),
 ('CU', 'CUB', '192', 'Cuba'),
 ('CW', 'CUW', '531', 'Curaçao'),
 ('CY', 'CYP', '196', 'Cyprus'),
 ('CZ', 'CZE', '203', 'Czechia'),
 ('DK', 'DNK', '208', 'Denmark'),
 ('DJ', 'DJI', '262', 'Djibouti'),
 ('DM', 'DMA', '212', 'Dominica'),
 ('DO', 'DOM', '214', 'Dominican Republic (the)'),
 ('EC', 'ECU', '218', 'Ecuador'),
 ('EG', 'EGY', '818', 'Egypt'),
 ('SV', 'SLV', '222', 'El Salvador'),
 ('GQ', 'GNQ', '226', 'Equatorial Guinea'),
 ('ER', 'ERI', '232', 'Eritrea'),
 ('EE', 'EST', '233', 'Estonia'),
 ('SZ', 'SWZ', '748', 'Eswatini'),
 ('ET', 'ETH', '231', 'Ethiopia'),
 ('FK', 'FLK', '238', 'Falkland Islands (the) [Malvinas]'),
 ('FO', 'FRO', '234', 'Faroe Islands (the)'),
 ('FJ', 'FJI', '242', 'Fiji'),
 ('FI', 'FIN', '246', 'Finland'),
 ('FR', 'FRA', '250', 'France'),
 ('GF', 'GUF', '254', 'French Guiana'),
 ('PF', 'PYF', '258', 'French Polynesia'),
 ('TF', 'ATF', '260', 'French Southern Territories (the)'),
 ('GA', 'GAB', '266', 'Gabon'),
 ('GM', 'GMB', '270', 'Gambia (the)'),
 ('GE', 'GEO', '268', 'Georgia'),
 ('DE', 'DEU', '276', 'Germany'),
 ('GH', 'GHA', '288', 'Ghana'),
 ('GI', 'GIB', '292', 'Gibraltar'),
 ('GR', 'GRC', '300', 'Greece'),
 ('GL', 'GRL', '304', 'Greenland'),
 ('GD', 'GRD', '308', 'Grenada'),
 ('GP', 'GLP', '312', 'Guadeloupe'),
 ('GU', 'GUM', '316', 'Guam'),
 ('GT', 'GTM', '320', 'Guatemala'),
 ('GG', 'GGY', '831', 'Guernsey'),
 ('GN', 'GIN', '324', 'Guinea'),
 ('GW', 'GNB', '624', 'Guinea-Bissau'),
 ('GY', 'GUY', '328', 'Guyana'),
 ('HT', 'HTI', '332', 'Haiti'),
 ('HM', 'HMD', '334', 'Heard Island and McDonald Islands'),
 ('VA', 'VAT', '336', 'Holy See (the)'),
 ('HN', 'HND', '340', 'Honduras'),
 ('HK', 'HKG', '344', 'Hong Kong'),
 ('HU', 'HUN', '348', 'Hungary'),
 ('IS', 'ISL', '352', 'Iceland'),
 ('IN', 'IND', '356', 'India'),
 ('ID', 'IDN', '360', 'Indonesia'),
 ('IR', 'IRN', '364', 'Iran (Islamic Republic of)'),
 ('IQ', 'IRQ', '368', 'Iraq'),
 ('IE', 'IRL', '372', 'Ireland'),
 ('IM', 'IMN', '833', 'Isle of Man'),
 ('IL', 'ISR', '376', 'Israel'),
 ('IT', 'ITA', '380', 'Italy'),
 ('JM', 'JAM', '388', 'Jamaica'),
 ('JP', 'JPN', '392', 'Japan'),
 ('JE', 'JEY', '832', 'Jersey'),
 ('JO', 'JOR', '400', 'Jordan'),
 ('KZ', 'KAZ', '398', 'Kazakhstan'),
 ('KE', 'KEN', '404', 'Kenya'),
 ('KI', 'KIR', '296', 'Kiribati'),
 ('KP', 'PRK', '408', "Korea (the Democratic People's Republic of)"),
 ('KR', 'KOR', '410', 'Korea (the Republic of)'),
 ('KW', 'KWT', '414', 'Kuwait'),
 ('KG', 'KGZ', '417', 'Kyrgyzstan'),
 ('LA', 'LAO', '418', "Lao People's Democratic Republic (the)"),
 ('LV', 'LVA', '428', 'Latvia'),
 ('LB', 'LBN', '422', 'Lebanon'),
 ('LS', 'LSO', '426', 'Lesotho'),
 ('LR', 'LBR', '430', 'Liberia'),
 ('LY', 'LBY', '434', 'Libya'),
 ('LI', 'LIE', '438', 'Liechtenstein'),
 ('LT', 'LTU', '440', 'Lithuania'),
 ('LU', 'LUX', '442', 'Luxembourg'),
 ('MO', 'MAC', '446', 'Macao'),
 ('MG', 'MDG', '450', 'Madagascar'),
 ('MW', 'MWI', '454', 'Malawi'),
 ('MY', 'MYS', '458', 'Malaysia'),
 ('MV', 'MDV', '462', 'Maldives'),
 ('ML', 'MLI', '466', 'Mali'),
 ('MT', 'MLT', '470', 'Malta'),
 ('MH', 'MHL', '584', 'Marshall Islands (the)'),
 ('MQ', 'MTQ', '474', 'Martinique'),
 ('MR', 'MRT', '478', 'Mauritania'),
 ('MU', 'MUS', '480', 'Mauritius'),
 ('YT', 'MYT', '175', 'Mayotte'),
 ('MX', 'MEX', '484', 'Mexico'),
 ('FM', 'FSM', '583', 'Micronesia (Federated States of)'),
 ('MD', 'MDA', '498', 'Moldova (the Republic of)'),
 ('MC', 'MCO', '492', 'Monaco'),
 ('MN', 'MNG', '496', 'Mongolia'),
 ('ME', 'MNE', '499', 'Montenegro'),
 ('MS', 'MSR', '500', 'Montserrat'),
 ('MA', 'MAR', '504', 'Morocco'),
 ('MZ', 'MOZ', '508', 'Mozambique'),
 ('MM', 'MMR', '104', 'Myanmar'),
 ('NA', 'NAM', '516', 'Namibia'),
 ('NR', 'NRU', '520', 'Nauru'),
 ('NP', 'NPL', '524', 'Nepal'),
 ('NL', 'NLD', '528', 'Netherlands (the)'),
 ('NC', 'NCL', '540', 'New Caledonia'),
 ('NZ', 'NZL', '554', 'New Zealand'),
 ('NI', 'NIC', '558', 'Nicaragua'),
 ('NE', 'NER', '562', 'Niger (the)'),
 ('NG', 'NGA', '566', 'Nigeria'),
 ('NU', 'NIU', '570', 'Niue'),
 ('NF', 'NFK', '574', 'Norfolk Island'),
 ('MK', 'MKD', '807', 'North Macedonia'),
 ('MP', 'MNP', '580', 'Northern Mariana Islands (the)'),
 ('NO', 'NOR', '578', 'Norway'),
 ('OM', 'OMN', '512', 'Oman'),
 ('PK', 'PAK', '586', 'Pakistan'),
 ('PW', 'PLW', '585', 'Palau'),
 ('PS', 'PSE', '275', 'Palestine, State of'),
 ('PA', 'PAN', '591', 'Panama'),
 ('PG', 'PNG', '598', 'Papua New Guinea'),
 ('PY', 'PRY', '600', 'Paraguay'),
 ('PE', 'PER', '604', 'Peru'),
 ('PH', 'PHL', '608', 'Philippines (the)'),
 ('PN', 'PCN', '612', 'Pitcairn'),
 ('PL', 'POL', '616', 'Poland'),
 ('PT', 'PRT', '620', 'Portugal'),
 ('PR', 'PRI', '630', 'Puerto Rico'),
 ('QA', 'QAT', '634', 'Qatar'),
 ('RE', 'REU', '638', 'Réunion'),
 ('RO', 'ROU', '642', 'Romania'),
 ('RU', 'RUS', '643', 'Russian Federation (the)'),
 ('RW', 'RWA', '646', 'Rwanda'),
 ('BL', 'BLM', '652', 'Saint Barthélemy'),
 ('SH', 'SHN', '654', 'Saint Helena, Ascension and Tristan da Cunha'),
 ('KN', 'KNA', '659', 'Saint Kitts and Nevis'),
 ('LC', 'LCA', '662', 'Saint Lucia'),
 ('MF', 'MAF', '663', 'Saint Martin (French part)'),
 ('PM', 'SPM', '666', 'Saint Pierre and Miquelon'),
 ('VC', 'VCT', '670', 'Saint Vincent and the Grenadines'),
 ('WS', 'WSM', '882', 'Samoa'),
 ('SM', 'SMR', '674', 'San Marino'),
 ('ST', 'STP', '678', 'Sao Tome and Principe'),
 ('SA', 'SAU', '682', 'Saudi Arabia'),
 ('SN', 'SEN', '686', 'Senegal'),
 ('RS', 'SRB', '688', 'Serbia'),
 ('SC', 'SYC', '690', 'Seychelles'),
 ('SL', 'SLE', '694', 'Sierra Leone'),
 ('SG', 'SGP', '702', 'Singapore'),
 ('SX', 'SXM', '534', 'Sint Maarten (Dutch part)'),
 ('SK', 'SVK', '703', 'Slovakia'),
 ('SI', 'SVN', '705', 'Slovenia'),
 ('SB', 'SLB', '090', 'Solomon Islands'),
 ('SO', 'SOM', '706', 'Somalia'),
 ('ZA', 'ZAF', '710', 'South Africa'),
 ('GS', 'SGS', '239', 'South Georgia and the South Sandwich Islands'),
 ('SS', 'SSD', '728', 'South Sudan'),
 ('ES', 'ESP', '724', 'Spain'),
 ('LK', 'LKA', '144', 'Sri Lanka'),
 ('SD', 'SDN', '729', 'Sudan (the)'),
 ('SR', 'SUR', '740', 'Suriname'),
 ('SJ', 'SJM', '744', 'Svalbard and Jan Mayen'),
 ('SE', 'SWE', '752', 'Sweden'),
 ('CH', 'CHE', '756', 'Switzerland'),
 ('SY', 'SYR', '760', 'Syrian Arab Republic (the)'),
 ('TW', 'TWN', '158', 'Taiwan (Province of China)'),
 ('TJ', 'TJK', '762', 'Tajikistan'),
 ('TZ', 'TZA', '834', 'Tanzania, the United Republic of'),
 ('TH', 'THA', '764', 'Thailand'),
 ('TL', 'TLS', '626', 'Timor-Leste'),
 ('TG', 'TGO', '768', 'Togo'),
 ('TK', 'TKL', '772', 'Tokelau'),
 ('TO', 'TON', '776', 'Tonga'),
 ('TT', 'TTO', '780', 'Trinidad and Tobago'),
 ('TN', 'TUN', '788', 'Tunisia'),
 ('TR', 'TUR', '792', 'Turkey'),
 ('TM', 'TKM', '795', 'Turkmenistan'),
 ('TC', 'TCA', '796', 'Turks and Caicos Islands (the)'),
 ('TV', 'TUV', '798', 'Tuvalu'),
 ('UG', 'UGA', '800', 'Uganda'),
 ('UA', 'UKR', '804', 'Ukraine'),
 ('AE', 'ARE', '784', 'United Arab Emirates (the)'),
 ('GB', 'GBR', '826', 'United Kingdom of Great Britain and Northern Ireland (the)'),
 ('UM', 'UMI', '581', 'United States Minor Outlying Islands (the)'),
 ('US', 'USA', '840', 'United States of America (the)'),
 ('UY', 'URY', '858', 'Uruguay'),
 ('UZ', 'UZB', '860', 'Uzbekistan'),
 ('VU', 'VUT', '548', 'Vanuatu'),
 ('VE', 'VEN', '862', 'Venezuela (Bolivarian Republic of)'),
 ('VN', 'VNM', '704', 'Viet Nam'),
 ('VG', 'VGB', '092', 'Virgin Islands (British)'),
 ('VI', 'VIR', '850', 'Virgin Islands (U.S.)'),
 ('WF', 'WLF', '876', 'Wallis and Futuna'),
 ('EH', 'ESH', '732', 'Western Sahara*'),
 ('YE', 'YEM', '887', 'Yemen'),
 ('ZM', 'ZMB', '894', 'Zambia'),
 ('ZW', 'ZWE', '716', 'Zimbabwe')]


CC2_CC3_dict = {i[0]: i[1] for i in CC_table}
CC3_CC2_dict = {i[1]: i[0] for i in CC_table}

