# envappconfig

envappconfig is intended to provide simple configuration via environment variables, in the same spirit as argparse.

Features:
* Autogenerates usage output if an environment variable is missing
* Default settings for missing environment variables
* Functions that transform the environment variable string to the value you need
* Environment variable prefixes
* Optional dictionary of the config

## Basic example

```python
from envappconfig import EnvAppConfig

config = EnvAppConfig(description='Amazing app')
config.add_env('port', default=1234, transform=int, help='The listen port')
config.add_env('mirror', help='The URL to mirror')
config.configure()

# Returns PORT from os.environ transformed to an int,
# or 1234 if PORT does not exist.
config.port

# Returns MIRROR from os.environ,
# or displays usage if MIRROR does not exist, then exits.
config.mirror
```

## Adding a prefix

If all the environment variables for the app have the same prefix, it can be specified with the `prefix` parameter.

```python
from envappconfig import EnvAppConfig

config = EnvAppConfig(prefix='MYAPP', description='Amazing app')
config.add_env('port', default=1234, transform=int, help='The listen port')
config.add_env('mirror', help='The URL to mirror')
config.configure()

# Returns MYAPP_PORT from os.environ transformed to an int,
# or 1234 if MYAPP_PORT does not exist.
config.port

# Returns MYAPP_MIRROR from os.environ,
# or displays usage if MYAPP_MIRROR does not exist, then exits.
config.mirror
```

## Custom transforms

The `transform` parameter can be used to specify normal transforms, like `int` or `float` (the default is `str`), but it can also take custom transform functions.  The transform function must take a single parameter, which will be filled in with the string value from the environment variable.

```python
config = EnvAppConfig(prefix='MYAPP', description='Amazing app')

# Double the timeout specified in the TIMEOUT environment variable,
# or default to 60.
config.add_env('timeout', default=60, transform=lambda x: int(x) * 2, help='Timeout in seconds')
...
```

## Adding more config values

Additional config values can be added to an existing EnvAppConfig, which can be helpful when there's a config value that needs to be calculated based on other config values.

```python
from envappconfig import EnvAppConfig

config = EnvAppConfig(description='Amazing app')
config.add_env('bind', help='IP address to bind to')
config.add_env('port', default=1234, transform=int, help='The listen port')
config.configure()
config.add_conf('listen', f'{config.bind}:{config.port}')

# Returns the combined bind:port string.
config.listen
```

## Getting a config dictionary

The EnvAppConfig instance is also available as a dictionary.

```python
from envappconfig import EnvAppConfig

config = EnvAppConfig(description='Amazing app')
config.add_env('bind', help='IP address to bind to')
config.add_env('port', default=1234, transform=int, help='The listen port')
config.configure()

# Returns a dictionary containing {'bind': '1.2.3.4', 'port': 1234}
config.asdict()
```
