from typing import overload
import abc
import datetime
import typing

import QuantConnect
import QuantConnect.Data
import QuantConnect.Data.Market
import QuantConnect.Data.UniverseSelection
import QuantConnect.DataSource
import QuantConnect.Orders
import QuantConnect.Util
import System
import System.Collections.Generic

IsoDateTimeConverter = typing.Any
QuantConnect_DataSource_BrainStockRanking10Day = typing.Any
QuantConnect_DataSource_BrainStockRanking2Day = typing.Any
QuantConnect_DataSource_BrainStockRanking30Day = typing.Any
QuantConnect_DataSource_BrainCompanyFilingLanguageMetrics10K = typing.Any
QuantConnect_DataSource_BrainCompanyFilingLanguageMetricsAll = typing.Any
QuantConnect_DataSource_BrainStockRanking5Day = typing.Any
QuantConnect_DataSource_BrainSentimentIndicator7Day = typing.Any
QuantConnect_DataSource_BrainStockRanking3Day = typing.Any
QuantConnect_DataSource_BrainSentimentIndicator30Day = typing.Any

QuantConnect_DataSource_BrainStockRankingBase_T = typing.TypeVar("QuantConnect_DataSource_BrainStockRankingBase_T")
QuantConnect_DataSource_BrainSentimentIndicatorBase_T = typing.TypeVar("QuantConnect_DataSource_BrainSentimentIndicatorBase_T")
QuantConnect_DataSource_BrainCompanyFilingLanguageMetricsBase_T = typing.TypeVar("QuantConnect_DataSource_BrainCompanyFilingLanguageMetricsBase_T")


class SmartInsiderEventType(System.Enum):
    """Describes what will or has taken place in an execution"""

    Authorization = 0
    """Notification that the board has gained the authority to repurchase"""

    Intention = 1
    """Notification of the board that shares will be repurchased."""

    Transaction = 2
    """Repurchase transactions that have been actioned."""

    UpwardsRevision = 3
    """Increase in the scope of the existing plan (extended date, increased value, etc.)"""

    DownwardsRevision = 4
    """Decrease in the scope of the existing plan (shortened date, reduced value, etc.)"""

    RevisedDetails = 5
    """General change of details of the plan (max/min price alteration, etc.)"""

    Cancellation = 6
    """Total cancellation of the plan"""

    SeekAuthorization = 7
    """Announcement by a company that the board of directors or management will be seeking to obtain authorisation for a repurchase plan."""

    PlanSuspension = 8
    """Announcement by a company that a plan of repurchase has been suspended. Further details of the suspension are included in the note."""

    PlanReStarted = 9
    """Announcement by a company that a suspended plan has been re-started. Further details of the suspension are included in the note."""

    NotSpecified = 10
    """Announcement by a company not specified and/or not documented in the other categories. Further details are included in the note."""


class SmartInsiderEvent(QuantConnect.Data.BaseData, metaclass=abc.ABCMeta):
    """
    SmartInsider Intention and Transaction events. These are fields
    that are shared between intentions and transactions.
    """

    @property
    def TransactionID(self) -> str:
        """Proprietary unique field. Not nullable"""
        ...

    @TransactionID.setter
    def TransactionID(self, value: str):
        """Proprietary unique field. Not nullable"""
        ...

    @property
    def EventType(self) -> typing.Optional[QuantConnect.DataSource.SmartInsiderEventType]:
        """Description of what has or will take place in an execution"""
        ...

    @EventType.setter
    def EventType(self, value: typing.Optional[QuantConnect.DataSource.SmartInsiderEventType]):
        """Description of what has or will take place in an execution"""
        ...

    @property
    def LastUpdate(self) -> datetime.datetime:
        """The date when a transaction is updated after it has been reported. Not nullable"""
        ...

    @LastUpdate.setter
    def LastUpdate(self, value: datetime.datetime):
        """The date when a transaction is updated after it has been reported. Not nullable"""
        ...

    @property
    def LastIDsUpdate(self) -> typing.Optional[datetime.datetime]:
        ...

    @LastIDsUpdate.setter
    def LastIDsUpdate(self, value: typing.Optional[datetime.datetime]):
        ...

    @property
    def ISIN(self) -> str:
        """Industry classification number"""
        ...

    @ISIN.setter
    def ISIN(self, value: str):
        """Industry classification number"""
        ...

    @property
    def USDMarketCap(self) -> typing.Optional[float]:
        """The market capitalization at the time of the transaction stated in US Dollars"""
        ...

    @USDMarketCap.setter
    def USDMarketCap(self, value: typing.Optional[float]):
        """The market capitalization at the time of the transaction stated in US Dollars"""
        ...

    @property
    def CompanyID(self) -> typing.Optional[int]:
        """Smart Insider proprietary identifier for the company"""
        ...

    @CompanyID.setter
    def CompanyID(self, value: typing.Optional[int]):
        """Smart Insider proprietary identifier for the company"""
        ...

    @property
    def ICBIndustry(self) -> str:
        """FTSE Russell Sector Classification"""
        ...

    @ICBIndustry.setter
    def ICBIndustry(self, value: str):
        """FTSE Russell Sector Classification"""
        ...

    @property
    def ICBSuperSector(self) -> str:
        """FTSE Russell Sector Classification"""
        ...

    @ICBSuperSector.setter
    def ICBSuperSector(self, value: str):
        """FTSE Russell Sector Classification"""
        ...

    @property
    def ICBSector(self) -> str:
        """FTSE Russell Sector Classification"""
        ...

    @ICBSector.setter
    def ICBSector(self, value: str):
        """FTSE Russell Sector Classification"""
        ...

    @property
    def ICBSubSector(self) -> str:
        """FTSE Russell Sector Classification"""
        ...

    @ICBSubSector.setter
    def ICBSubSector(self, value: str):
        """FTSE Russell Sector Classification"""
        ...

    @property
    def ICBCode(self) -> typing.Optional[int]:
        """Numeric code that is the most granular level in ICB classification"""
        ...

    @ICBCode.setter
    def ICBCode(self, value: typing.Optional[int]):
        """Numeric code that is the most granular level in ICB classification"""
        ...

    @property
    def CompanyName(self) -> str:
        """Company name. PLC is always excluded"""
        ...

    @CompanyName.setter
    def CompanyName(self, value: str):
        """Company name. PLC is always excluded"""
        ...

    @property
    def PreviousResultsAnnouncementDate(self) -> typing.Optional[datetime.datetime]:
        """Announcement date of last results, this will be the end date of the last "Close Period\""""
        ...

    @PreviousResultsAnnouncementDate.setter
    def PreviousResultsAnnouncementDate(self, value: typing.Optional[datetime.datetime]):
        """Announcement date of last results, this will be the end date of the last "Close Period\""""
        ...

    @property
    def NextResultsAnnouncementsDate(self) -> typing.Optional[datetime.datetime]:
        """Announcement date of next results, this will be the end date of the next "Close Period\""""
        ...

    @NextResultsAnnouncementsDate.setter
    def NextResultsAnnouncementsDate(self, value: typing.Optional[datetime.datetime]):
        """Announcement date of next results, this will be the end date of the next "Close Period\""""
        ...

    @property
    def NextCloseBegin(self) -> typing.Optional[datetime.datetime]:
        """Start date of next trading embargo ahead of scheduled results announcment"""
        ...

    @NextCloseBegin.setter
    def NextCloseBegin(self, value: typing.Optional[datetime.datetime]):
        """Start date of next trading embargo ahead of scheduled results announcment"""
        ...

    @property
    def LastCloseEnded(self) -> typing.Optional[datetime.datetime]:
        """Date trading embargo (Close Period) is lifted as results are made public"""
        ...

    @LastCloseEnded.setter
    def LastCloseEnded(self, value: typing.Optional[datetime.datetime]):
        """Date trading embargo (Close Period) is lifted as results are made public"""
        ...

    @property
    def SecurityDescription(self) -> str:
        """Type of security. Does not contain nominal value"""
        ...

    @SecurityDescription.setter
    def SecurityDescription(self, value: str):
        """Type of security. Does not contain nominal value"""
        ...

    @property
    def TickerCountry(self) -> str:
        """Country of local identifier, denoting where the trade took place"""
        ...

    @TickerCountry.setter
    def TickerCountry(self, value: str):
        """Country of local identifier, denoting where the trade took place"""
        ...

    @property
    def TickerSymbol(self) -> str:
        """Local market identifier"""
        ...

    @TickerSymbol.setter
    def TickerSymbol(self, value: str):
        """Local market identifier"""
        ...

    @property
    def AnnouncementDate(self) -> typing.Optional[datetime.datetime]:
        """Date Transaction was entered onto our system. Where a transaction is after the London market close (usually 4.30pm) this will be stated as the next day"""
        ...

    @AnnouncementDate.setter
    def AnnouncementDate(self, value: typing.Optional[datetime.datetime]):
        """Date Transaction was entered onto our system. Where a transaction is after the London market close (usually 4.30pm) this will be stated as the next day"""
        ...

    @property
    def TimeReleased(self) -> typing.Optional[datetime.datetime]:
        """Time the announcement first appeared on a Regulatory News Service or other disclosure system and became available to the market, time stated is local market time"""
        ...

    @TimeReleased.setter
    def TimeReleased(self, value: typing.Optional[datetime.datetime]):
        """Time the announcement first appeared on a Regulatory News Service or other disclosure system and became available to the market, time stated is local market time"""
        ...

    @property
    def TimeProcessed(self) -> typing.Optional[datetime.datetime]:
        """Time the transaction was entered into Smart Insider systems and appeared on their website, time stated is local to London, UK"""
        ...

    @TimeProcessed.setter
    def TimeProcessed(self, value: typing.Optional[datetime.datetime]):
        """Time the transaction was entered into Smart Insider systems and appeared on their website, time stated is local to London, UK"""
        ...

    @property
    def TimeReleasedUtc(self) -> typing.Optional[datetime.datetime]:
        """Time the announcement first appeared on a Regulatory News Service or other disclosure system and became available to the market. Time stated is GMT standard"""
        ...

    @TimeReleasedUtc.setter
    def TimeReleasedUtc(self, value: typing.Optional[datetime.datetime]):
        """Time the announcement first appeared on a Regulatory News Service or other disclosure system and became available to the market. Time stated is GMT standard"""
        ...

    @property
    def TimeProcessedUtc(self) -> typing.Optional[datetime.datetime]:
        """Time the transaction was entered onto our systems and appeared on our website. Time stated is GMT standard"""
        ...

    @TimeProcessedUtc.setter
    def TimeProcessedUtc(self, value: typing.Optional[datetime.datetime]):
        """Time the transaction was entered onto our systems and appeared on our website. Time stated is GMT standard"""
        ...

    @property
    def AnnouncedIn(self) -> str:
        """Market in which the transaction was announced, this can reference more than one country"""
        ...

    @AnnouncedIn.setter
    def AnnouncedIn(self, value: str):
        """Market in which the transaction was announced, this can reference more than one country"""
        ...

    @overload
    def __init__(self) -> None:
        ...

    @overload
    def __init__(self, tsvLine: str) -> None:
        """
        Parses a line of TSV (tab delimited) from Smart Insider data
        
        :param tsvLine: Tab delimited line of data
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the timezone of this data source
        
        :returns: Timezone.
        """
        ...

    def FromRawData(self, line: str) -> None:
        """
        Derived class instances populate their fields from raw TSV
        
        :param line: Line of raw TSV (raw with fields 46, 36, 14, 7 removed in descending order)
        """
        ...

    @staticmethod
    def ParseDate(date: str) -> datetime.datetime:
        """
        Attempts to normalize and parse SmartInsider dates that include a time component.
        
        :param date: Date string to parse
        :returns: DateTime object.
        """
        ...

    def ToLine(self) -> str:
        """
        Converts data to TSV
        
        :returns: String of TSV.
        """
        ...


class SmartInsiderExecution(System.Enum):
    """Describes how the transaction was executed"""

    Market = 0
    """Took place via the open market"""

    TenderOffer = 1
    """Via a companywide tender offer to all shareholders"""

    OffMarket = 2
    """Under a specific agreement between the issuer and shareholder"""

    Error = 3
    """Field is not in this enum"""


class SmartInsiderExecutionEntity(System.Enum):
    """Entity that intends to or executed the transaction"""

    Issuer = 0
    """Issuer of the stock"""

    Subsidiary = 1
    """Subsidiary of the issuer"""

    Broker = 2
    """
    Brokers are commonly used to repurchase shares under mandate to avoid insider
    information rules and to allow repurchases to carry on through close periods
    """

    EmployerBenefitTrust = 3
    """Unknown - Transaction"""

    EmployeeBenefitTrust = 4
    """To cater for shares which will need to be transferred to employees as part of remunerative plans"""

    ThirdParty = 5
    """Undisclosed independent third party. Likely to be a broker."""

    Error = 6
    """The field was not found in this enum"""


class SmartInsiderExecutionHolding(System.Enum):
    """Details regarding the way holdings will be or were processed in a buyback execution"""

    Treasury = 0
    """Held in treasury until they are sold back to the market"""

    Cancellation = 1
    """Immediately cancelled"""

    Trust = 2
    """Held in trust, generally to cover employee renumerative plans"""

    SatisfyEmployeeTax = 3
    """Shares will be used to satisfy employee tax liabilities"""

    NotReported = 4
    """Not disclosed by the issuer in the announcements"""

    SatisfyStockVesting = 5
    """Shares will be used to satisfy vesting of employee stock"""

    Error = 6
    """The field was not found in the enum, or is representative of a SatisfyStockVesting entry."""


class SmartInsiderTransaction(QuantConnect.DataSource.SmartInsiderEvent):
    """Smart Insider Transaction - Execution of a stock buyback and details about the event occurred"""

    DataSourceId: int
    """Data source ID"""

    @property
    def BuybackDate(self) -> typing.Optional[datetime.datetime]:
        """Date traded through the market"""
        ...

    @BuybackDate.setter
    def BuybackDate(self, value: typing.Optional[datetime.datetime]):
        """Date traded through the market"""
        ...

    @property
    def Execution(self) -> typing.Optional[QuantConnect.DataSource.SmartInsiderExecution]:
        """Describes how transaction was executed"""
        ...

    @Execution.setter
    def Execution(self, value: typing.Optional[QuantConnect.DataSource.SmartInsiderExecution]):
        """Describes how transaction was executed"""
        ...

    @property
    def ExecutionEntity(self) -> typing.Optional[QuantConnect.DataSource.SmartInsiderExecutionEntity]:
        """Describes which entity carried out the transaction"""
        ...

    @ExecutionEntity.setter
    def ExecutionEntity(self, value: typing.Optional[QuantConnect.DataSource.SmartInsiderExecutionEntity]):
        """Describes which entity carried out the transaction"""
        ...

    @property
    def ExecutionHolding(self) -> typing.Optional[QuantConnect.DataSource.SmartInsiderExecutionHolding]:
        """Describes what will be done with those shares following repurchase"""
        ...

    @ExecutionHolding.setter
    def ExecutionHolding(self, value: typing.Optional[QuantConnect.DataSource.SmartInsiderExecutionHolding]):
        """Describes what will be done with those shares following repurchase"""
        ...

    @property
    def Currency(self) -> str:
        """Currency of transation (ISO Code)"""
        ...

    @Currency.setter
    def Currency(self, value: str):
        """Currency of transation (ISO Code)"""
        ...

    @property
    def ExecutionPrice(self) -> typing.Optional[float]:
        """Denominated in Currency of Transaction"""
        ...

    @ExecutionPrice.setter
    def ExecutionPrice(self, value: typing.Optional[float]):
        """Denominated in Currency of Transaction"""
        ...

    @property
    def Amount(self) -> typing.Optional[float]:
        """Number of shares traded"""
        ...

    @Amount.setter
    def Amount(self, value: typing.Optional[float]):
        """Number of shares traded"""
        ...

    @property
    def GBPValue(self) -> typing.Optional[float]:
        """Currency conversion rates are updated daily and values are calculated at rate prevailing on the trade date"""
        ...

    @GBPValue.setter
    def GBPValue(self, value: typing.Optional[float]):
        """Currency conversion rates are updated daily and values are calculated at rate prevailing on the trade date"""
        ...

    @property
    def EURValue(self) -> typing.Optional[float]:
        """Currency conversion rates are updated daily and values are calculated at rate prevailing on the trade date"""
        ...

    @EURValue.setter
    def EURValue(self, value: typing.Optional[float]):
        """Currency conversion rates are updated daily and values are calculated at rate prevailing on the trade date"""
        ...

    @property
    def USDValue(self) -> typing.Optional[float]:
        """Currency conversion rates are updated daily and values are calculated at rate prevailing on the trade date"""
        ...

    @USDValue.setter
    def USDValue(self, value: typing.Optional[float]):
        """Currency conversion rates are updated daily and values are calculated at rate prevailing on the trade date"""
        ...

    @property
    def NoteText(self) -> str:
        """Free text which expains futher details about the trade"""
        ...

    @NoteText.setter
    def NoteText(self, value: str):
        """Free text which expains futher details about the trade"""
        ...

    @property
    def BuybackPercentage(self) -> typing.Optional[float]:
        """Percentage of value of the trade as part of the issuers total Market Cap"""
        ...

    @BuybackPercentage.setter
    def BuybackPercentage(self, value: typing.Optional[float]):
        """Percentage of value of the trade as part of the issuers total Market Cap"""
        ...

    @property
    def VolumePercentage(self) -> typing.Optional[float]:
        """Percentage of the volume traded on the day of the buyback."""
        ...

    @VolumePercentage.setter
    def VolumePercentage(self, value: typing.Optional[float]):
        """Percentage of the volume traded on the day of the buyback."""
        ...

    @property
    def ConversionRate(self) -> typing.Optional[float]:
        """Rate used to calculate 'Value (GBP)' from 'Price' multiplied by 'Amount'. Will be 1 where Currency is also 'GBP'"""
        ...

    @ConversionRate.setter
    def ConversionRate(self, value: typing.Optional[float]):
        """Rate used to calculate 'Value (GBP)' from 'Price' multiplied by 'Amount'. Will be 1 where Currency is also 'GBP'"""
        ...

    @property
    def AmountAdjustedFactor(self) -> typing.Optional[float]:
        """Multiplier which can be applied to 'Amount' field to account for subsequent corporate action"""
        ...

    @AmountAdjustedFactor.setter
    def AmountAdjustedFactor(self, value: typing.Optional[float]):
        """Multiplier which can be applied to 'Amount' field to account for subsequent corporate action"""
        ...

    @property
    def PriceAdjustedFactor(self) -> typing.Optional[float]:
        """Multiplier which can be applied to 'Price' and 'LastClose' fields to account for subsequent corporate actions"""
        ...

    @PriceAdjustedFactor.setter
    def PriceAdjustedFactor(self, value: typing.Optional[float]):
        """Multiplier which can be applied to 'Price' and 'LastClose' fields to account for subsequent corporate actions"""
        ...

    @property
    def TreasuryHolding(self) -> typing.Optional[int]:
        """Post trade holding of the Treasury or Trust in the security traded"""
        ...

    @TreasuryHolding.setter
    def TreasuryHolding(self, value: typing.Optional[int]):
        """Post trade holding of the Treasury or Trust in the security traded"""
        ...

    @overload
    def __init__(self) -> None:
        """Empty contsructor required for Slice.Get{T}()"""
        ...

    @overload
    def __init__(self, line: str) -> None:
        """
        Creates an instance of the object by taking a formatted TSV line
        
        :param line: Line of formatted TSV
        """
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the object to a new instance. This method
        is required for custom data sources that make use
        of properties with more complex types since otherwise
        the values will default to null using the default clone method
        
        :returns: A new cloned instance of this object.
        """
        ...

    def FromRawData(self, line: str) -> None:
        """
        Creates an instance of the object by taking a formatted TSV line
        
        :param line: Line of formatted TSV
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Specifies the location of the data and directs LEAN where to load the data from
        
        :param config: Subscription configuration
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: Subscription data source object pointing LEAN to the data location.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reads the data into LEAN for use in algorithms
        
        :param config: Subscription configuration
        :param line: Line of TSV
        :param date: Algorithm date
        :param isLiveMode: Is live mode
        :returns: Instance of the object.
        """
        ...

    def ToLine(self) -> str:
        """
        Converts the data to TSV
        
        :returns: String of TSV.
        """
        ...


class SmartInsiderIntention(QuantConnect.DataSource.SmartInsiderEvent):
    """Smart Insider Intentions - Intention to execute a stock buyback and details about the future event"""

    DataSourceId: int
    """Data source ID"""

    @property
    def Execution(self) -> typing.Optional[QuantConnect.DataSource.SmartInsiderExecution]:
        """Describes how the transaction was executed"""
        ...

    @Execution.setter
    def Execution(self, value: typing.Optional[QuantConnect.DataSource.SmartInsiderExecution]):
        """Describes how the transaction was executed"""
        ...

    @property
    def ExecutionEntity(self) -> typing.Optional[QuantConnect.DataSource.SmartInsiderExecutionEntity]:
        """Describes which entity intends to execute the transaction"""
        ...

    @ExecutionEntity.setter
    def ExecutionEntity(self, value: typing.Optional[QuantConnect.DataSource.SmartInsiderExecutionEntity]):
        """Describes which entity intends to execute the transaction"""
        ...

    @property
    def ExecutionHolding(self) -> typing.Optional[QuantConnect.DataSource.SmartInsiderExecutionHolding]:
        """Describes what will be done with those shares following repurchase"""
        ...

    @ExecutionHolding.setter
    def ExecutionHolding(self, value: typing.Optional[QuantConnect.DataSource.SmartInsiderExecutionHolding]):
        """Describes what will be done with those shares following repurchase"""
        ...

    @property
    def Amount(self) -> typing.Optional[int]:
        """Number of shares to be or authorised to be traded"""
        ...

    @Amount.setter
    def Amount(self, value: typing.Optional[int]):
        """Number of shares to be or authorised to be traded"""
        ...

    @property
    def ValueCurrency(self) -> str:
        """Currency of the value of shares to be/Authorised to be traded (ISO Code)"""
        ...

    @ValueCurrency.setter
    def ValueCurrency(self, value: str):
        """Currency of the value of shares to be/Authorised to be traded (ISO Code)"""
        ...

    @property
    def AmountValue(self) -> typing.Optional[int]:
        """Value of shares to be authorised to be traded"""
        ...

    @AmountValue.setter
    def AmountValue(self, value: typing.Optional[int]):
        """Value of shares to be authorised to be traded"""
        ...

    @property
    def Percentage(self) -> typing.Optional[float]:
        """Percentage of oustanding shares to be authorised to be traded"""
        ...

    @Percentage.setter
    def Percentage(self, value: typing.Optional[float]):
        """Percentage of oustanding shares to be authorised to be traded"""
        ...

    @property
    def AuthorizationStartDate(self) -> typing.Optional[datetime.datetime]:
        """start of the period the intention/authorisation applies to"""
        ...

    @AuthorizationStartDate.setter
    def AuthorizationStartDate(self, value: typing.Optional[datetime.datetime]):
        """start of the period the intention/authorisation applies to"""
        ...

    @property
    def AuthorizationEndDate(self) -> typing.Optional[datetime.datetime]:
        """End of the period the intention/authorisation applies to"""
        ...

    @AuthorizationEndDate.setter
    def AuthorizationEndDate(self, value: typing.Optional[datetime.datetime]):
        """End of the period the intention/authorisation applies to"""
        ...

    @property
    def PriceCurrency(self) -> str:
        """Currency of min/max prices (ISO Code)"""
        ...

    @PriceCurrency.setter
    def PriceCurrency(self, value: str):
        """Currency of min/max prices (ISO Code)"""
        ...

    @property
    def MinimumPrice(self) -> typing.Optional[float]:
        """Minimum price shares will or may be purchased at"""
        ...

    @MinimumPrice.setter
    def MinimumPrice(self, value: typing.Optional[float]):
        """Minimum price shares will or may be purchased at"""
        ...

    @property
    def MaximumPrice(self) -> typing.Optional[float]:
        """Maximum price shares will or may be purchased at"""
        ...

    @MaximumPrice.setter
    def MaximumPrice(self, value: typing.Optional[float]):
        """Maximum price shares will or may be purchased at"""
        ...

    @property
    def NoteText(self) -> str:
        """Free text which explains further details about the trade"""
        ...

    @NoteText.setter
    def NoteText(self, value: str):
        """Free text which explains further details about the trade"""
        ...

    @overload
    def __init__(self) -> None:
        """Empty constructor required for Slice.Get{T}()"""
        ...

    @overload
    def __init__(self, line: str) -> None:
        """
        Constructs instance of this via a *formatted* TSV line (tab delimited)
        
        :param line: Line of formatted TSV data
        """
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the object to a new instance. This method
        is required for custom data sources that make use
        of properties with more complex types since otherwise
        the values will default to null using the default clone method
        
        :returns: A new cloned instance of this object.
        """
        ...

    def FromRawData(self, line: str) -> None:
        """
        Constructs a new instance from unformatted TSV data
        
        :param line: Line of raw TSV (raw with fields 46, 36, 14, 7 removed in descending order)
        :returns: Instance of the object.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Specifies the location of the data and directs LEAN where to load the data from
        
        :param config: Subscription configuration
        :param date: Algorithm date
        :param isLiveMode: Is live mode
        :returns: Subscription data source object pointing LEAN to the data location.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Loads and reads the data to be used in LEAN
        
        :param config: Subscription configuration
        :param line: TSV line
        :param date: Algorithm date
        :param isLiveMode: Is live mode
        :returns: Instance of the object.
        """
        ...

    def ToLine(self) -> str:
        """
        Converts the data to TSV
        
        :returns: String of TSV.
        """
        ...


class QuandlBitfinexDownloaderProgram(System.Object):
    """This class has no documentation."""

    @staticmethod
    def QuandlBitfinexDownloader(fromDate: typing.Union[datetime.datetime, datetime.date], apiKey: str) -> None:
        """Quandl Bitfinex Toolbox Project For LEAN Algorithmic Trading Engine."""
        ...


class NasdaqBitfinexDownloaderProgram(System.Object):
    """This class has no documentation."""

    @staticmethod
    def NasdaqBitfinexDownloader(fromDate: typing.Union[datetime.datetime, datetime.date], apiKey: str) -> None:
        """Nasdaq Bitfinex Toolbox Project For LEAN Algorithmic Trading Engine."""
        ...


class NasdaqBitfinexDownloader(System.Object, QuantConnect.IDataDownloader):
    """Nasdaq Bitfinex Data Downloader class"""

    @property
    def _apiKey(self) -> str:
        """This field is protected."""
        ...

    def __init__(self, apiKey: str) -> None:
        """
        Initializes a new instance of the NasdaqBitfinexDownloader class
        
        :param apiKey: The nasdaq api key
        """
        ...

    def Get(self, dataDownloaderGetParameters: QuantConnect.DataDownloaderGetParameters) -> System.Collections.Generic.IEnumerable[QuantConnect.Data.BaseData]:
        """
        Get historical data enumerable for Bitfinex from Nasdaq
        
        :param dataDownloaderGetParameters: model class for passing in parameters for historical data
        :returns: Enumerable of base data for this symbol.
        """
        ...


class QuandlBitfinexDownloader(QuantConnect.DataSource.NasdaqBitfinexDownloader):
    """Quandl Bitfinex Data Downloader class"""

    def __init__(self, apiKey: str) -> None:
        """
        Initializes a new instance of the QuandlBitfinexDownloader class
        
        :param apiKey: The quandl api key
        """
        ...


class NasdaqDataLink(QuantConnect.Data.DynamicData):
    """Nasdaq Data Link dataset"""

    @property
    def ValueColumnName(self) -> str:
        """
        Name of the column is going to be used for the field Value
        
        This property is protected.
        """
        ...

    @ValueColumnName.setter
    def ValueColumnName(self, value: str):
        """
        Name of the column is going to be used for the field Value
        
        This property is protected.
        """
        ...

    IsAuthCodeSet: bool
    """Flag indicating whether or not the Nasdaq Data Link auth code has been set yet"""

    @property
    def EndTime(self) -> datetime.datetime:
        """
        The end time of this data. Some data covers spans (trade bars) and as such we want
        to know the entire time span covered
        """
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """
        The end time of this data. Some data covers spans (trade bars) and as such we want
        to know the entire time span covered
        """
        ...

    @property
    def Period(self) -> datetime.timedelta:
        """Gets a time span of one day"""
        ...

    @overload
    def __init__(self) -> None:
        """Default NasdaqDataLink constructor uses Close as its value column"""
        ...

    @overload
    def __init__(self, valueColumnName: str) -> None:
        """
        Constructor for creating customized NasdaqDataLink instance which doesn't use close, price, settle or value as its value item.
        
        This method is protected.
        
        :param valueColumnName: The name of the column we want to use as reference, the Value property
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The NodaTime.DateTimeZone of this data type.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Using the Nasdaq Data Link V3 API automatically set the URL for the dataset.
        
        :param config: Subscription configuration object
        :param date: Date of the data file we're looking for
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: STRING API Url for Nasdaq Data Link.
        """
        ...

    def IsSparseData(self) -> bool:
        """
        Indicates whether the data is sparse.
        If true, we disable logging for missing files
        
        :returns: true.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: CSV line of data from the souce
        :param date: Date of the requested line
        :param isLiveMode: Is live mode
        :returns: New instance.
        """
        ...

    @staticmethod
    def SetAuthCode(authCode: str) -> None:
        """Set the auth code for the Nasdaq Data Link set to the QuantConnect auth code."""
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...


class Fred(QuantConnect.Data.BaseData):
    """Federal Reserve Economic Data"""

    class CBOE(System.Object):
        """Chicago Board Options Exchange"""

        VIXOnGoogle: str = "VXGOGCLS"
        """CBOE Equity VIX on Google (in Index)"""

        VXD: str = "VXDCLS"
        """CBOE DJIA Volatility Index (in Index)"""

        VIXOnGoldmanSachs: str = "VXGSCLS"
        """CBOE Equity VIX on Goldman Sachs (in Index)"""

        VIXOnIBM: str = "VXIBMCLS"
        """CBOE Equity VIX on IBM (in Index)"""

        VIXOnAmazon: str = "VXAZNCLS"
        """CBOE Equity VIX on Amazon (in Index)"""

        VXO: str = "VXOCLS"
        """CBOE S&P 100 Volatility Index: VXO (in Index)"""

        VXN: str = "VXNCLS"
        """CBOE NASDAQ 100 Volatility Index (in Index)"""

        TenYearTreasuryNoteVolatilityFutures: str = "VXTYN"
        """CBOE 10-Year Treasury Note Volatility Futures (in Index)"""

        RVX: str = "RVXCLS"
        """CBOE Russell 2000 Volatility Index (in Index)"""

        SP500ThreeMonthVolatilityIndex: str = "VXVCLS"
        """CBOE S&P 500 3-Month Volatility Index (in Index)"""

        VIXOnApple: str = "VXAPLCLS"
        """CBOE Equity VIX on Apple (in Index)"""

        GoldMinersETFVolatilityIndex: str = "VXGDXCLS"
        """CBOE Gold Miners ETF Volatility Index (in Index)"""

        ChinaETFVolatilityIndex: str = "VXFXICLS"
        """CBOE China ETF Volatility Index (in Index)"""

        BrazilETFVolatilityIndex: str = "VXEWZCLS"
        """CBOE Brazil ETF Volatility Index (in Index)"""

        EmergingMarketsETFVolatilityIndex: str = "VXEEMCLS"
        """CBOE Emerging Markets ETF Volatility Index (in Index)"""

        EuroCurrencyETFVolatilityIndex: str = "EVZCLS"
        """CBOE EuroCurrency ETF Volatility Index (in Index)"""

        GoldETFVolatilityIndex: str = "GVZCLS"
        """CBOE Gold ETF Volatility Index (in Index)"""

        CrudeOilETFVolatilityIndex: str = "OVXCLS"
        """CBOE Crude Oil ETF Volatility Index (in Index)"""

        SilverETFVolatilityIndex: str = "VXSLVCLS"
        """CBOE Silver ETF Volatility Index (in Index)"""

        EnergySectorETFVolatilityIndex: str = "VXXLECLS"
        """CBOE Energy Sector ETF Volatility Index (in Index)"""

        VIX: str = "VIXCLS"
        """CBOE Volatility Index: VIX (in Index)"""

    class TradeWeightedIndexes(System.Object):
        """Trade Weight Indexes"""

        MajorCurrenciesGoods: str = "DTWEXM"
        """Trade Weighted U.S. Dollar Index: Major Currencies, Goods (in Index Mar 1973=100)"""

        OtherImportantTradingPartnersGoods: str = "DTWEXO"
        """Trade Weighted U.S. Dollar Index: Other Important Trading Partners, Goods (in Index Jan 1997=100)"""

        BroadGoods: str = "DTWEXB"
        """Trade Weighted U.S. Dollar Index: Broad, Goods (in Index Jan 1997=100)"""

        AdvancedForeignEconomiesGoodsAndServices: str = "DTWEXAFEGS"
        """Trade Weighted U.S. Dollar Index: Advanced Foreign Economies, Goods and Services (in Index Jan 2006=100)"""

        BroadGoodsAndServices: str = "DTWEXBGS"
        """Trade Weighted U.S. Dollar Index: Broad, Goods and Services (in Index Jan 2006=100)"""

        EmergingMarketsEconomiesGoodsAndServices: str = "DTWEXEMEGS"
        """Trade Weighted U.S. Dollar Index: Emerging Markets Economies, Goods and Services (in Index Jan 2006=100)"""

    class Wilshire(System.Object):
        """Wilshire Indexes help clients, investment professionals and researchers accurately measure and better understand the market. The Wilshire Index family leverages more than 40 years of Wilshire performance measurement expertise and employs unbiased construction rules."""

        USSmallCapValuePrice: str = "WILLSMLCAPVALPR"
        """Wilshire US Small-Cap Value Price Index (in Index)"""

        Price2500: str = "WILL2500PR"
        """Wilshire 2500 Price Index (in Index)"""

        Price4500: str = "WILL4500PR"
        """Wilshire 4500 Price Index (in Index)"""

        ValuePrice2500: str = "WILL2500PRVAL"
        """Wilshire 2500 Value Price Index (in Index)"""

        GrowthPrice2500: str = "WILL2500PRGR"
        """Wilshire 2500 Growth Price Index (in Index)"""

        USSmallCapPrice: str = "WILLSMLCAPPR"
        """Wilshire US Small-Cap Price Index (in Index)"""

        Price5000: str = "WILL5000PR"
        """Wilshire 5000 Price Index (in Index)"""

        USSmallCapGrowthPrice: str = "WILLSMLCAPGRPR"
        """Wilshire US Small-Cap Growth Price Index (in Index)"""

        USMidCapValuePrice: str = "WILLMIDCAPVALPR"
        """Wilshire US Mid-Cap Value Price Index (in Index)"""

        USRealEstateSecuritiesPrice: str = "WILLRESIPR"
        """Wilshire US Real Estate Securities Price Index (Wilshire US RESI) (in Index)"""

        USLargeCapPrice: str = "WILLLRGCAPPR"
        """Wilshire US Large-Cap Price Index (in Index)"""

        USMidCapPrice: str = "WILLMIDCAPPR"
        """Wilshire US Mid-Cap Price Index (in Index)"""

        USMidCapGrowthPrice: str = "WILLMIDCAPGRPR"
        """Wilshire US Mid-Cap Growth Price Index (in Index)"""

        USMicroCapPrice: str = "WILLMICROCAPPR"
        """Wilshire US Micro-Cap Price Index (in Index)"""

        USRealEstateInvestmentTrustPrice: str = "WILLREITPR"
        """Wilshire US Real Estate Investment Trust Price Index (Wilshire US REIT) (in Index)"""

        USLargeCapValuePrice: str = "WILLLRGCAPVALPR"
        """Wilshire US Large-Cap Value Price Index (in Index)"""

        USLargeCapGrowthPrice: str = "WILLLRGCAPGRPR"
        """Wilshire US Large-Cap Growth Price Index (in Index)"""

        FullCapPrice5000: str = "WILL5000PRFC"
        """Wilshire 5000 Full Cap Price Index (in Index)"""

        USMidCapValue: str = "WILLMIDCAPVAL"
        """Wilshire US Mid-Cap Value Total Market Index (in Index)"""

        USMidCapGrowth: str = "WILLMIDCAPGR"
        """Wilshire US Mid-Cap Growth Total Market Index (in Index)"""

        USMidCap: str = "WILLMIDCAP"
        """Wilshire US Mid-Cap Total Market Index (in Index)"""

        USRealEstateSecurities: str = "WILLRESIND"
        """Wilshire US Real Estate Securities Total Market Index (Wilshire US RESI) (in Index)"""

        Index4500: str = "WILL4500IND"
        """Wilshire 4500 Total Market Index (in Index)"""

        Index5000: str = "WILL5000IND"
        """Wilshire 5000 Total Market Index (in Index)"""

        USLargeCapGrowth: str = "WILLLRGCAPGR"
        """Wilshire US Large-Cap Growth Total Market Index (in Index)"""

        USMicroCap: str = "WILLMICROCAP"
        """Wilshire US Micro-Cap Total Market Index (in Index)"""

        Value2500: str = "WILL2500INDVAL"
        """Wilshire 2500 Value Total Market Index (in Index)"""

        USSmallCapGrowth: str = "WILLSMLCAPGR"
        """Wilshire US Small-Cap Growth Total Market Index (in Index)"""

        USSmallCapValue: str = "WILLSMLCAPVAL"
        """Wilshire US Small-Cap Value Total Market Index (in Index)"""

        USLargeCapValue: str = "WILLLRGCAPVAL"
        """Wilshire US Large-Cap Value Total Market Index (in Index)"""

        USRealEstateInvestmentTrust: str = "WILLREITIND"
        """Wilshire US Real Estate Investment Trust Total Market Index (Wilshire US REIT) (in Index)"""

        Index2500: str = "WILL2500IND"
        """Wilshire 2500 Total Market Index (in Index)"""

        USSmallCap: str = "WILLSMLCAP"
        """Wilshire US Small-Cap Total Market Index (in Index)"""

        USLargeCap: str = "WILLLRGCAP"
        """Wilshire US Large-Cap Total Market Index (in Index)"""

        Growth2500: str = "WILL2500INDGR"
        """Wilshire 2500 Growth Total Market Index (in Index)"""

        TotalMarketFullCap5000: str = "WILL5000INDFC"
        """Wilshire 5000 Total Market Full Cap Index (in Index)"""

    class ICEBofAML(System.Object):
        """ICE BofAML"""

        AAAAEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEM1BRRAAA2ACRPITRIV"
        """ICE BofAML AAA-A Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        AAAAUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEM1RAAA2ALCRPIUSTRIV"
        """ICE BofAML AAA-A US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        AsiaEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMRACRPIASIATRIV"
        """ICE BofAML Asia Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        AsiaUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMALLCRPIASIAUSTRIV"
        """ICE BofAML Asia US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        BandLowerEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEM4BRRBLCRPITRIV"
        """ICE BofAML B and Lower Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        BandLowerUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEM4RBLLCRPIUSTRIV"
        """ICE BofAML B and Lower US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        BBEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEM3BRRBBCRPITRIV"
        """ICE BofAML BB Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        BBUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEM3RBBLCRPIUSTRIV"
        """ICE BofAML BB US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        BBBEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEM2BRRBBBCRPITRIV"
        """ICE BofAML BBB Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        BBBUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEM2RBBBLCRPIUSTRIV"
        """ICE BofAML BBB US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        CrossoverEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEM5BCOCRPITRIV"
        """ICE BofAML Crossover Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        CrossoverUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMXOCOLCRPIUSTRIV"
        """ICE BofAML Crossover US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        EmergingMarketsCorporatePlusIndexTotalReturnIndexValue: str = "BAMLEMCBPITRIV"
        """ICE BofAML Emerging Markets Corporate Plus Index Total Return Index Value (in Index)"""

        EuroEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMEBCRPIETRIV"
        """ICE BofAML Euro Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        EMEAEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMRECRPIEMEATRIV"
        """ICE BofAML Europe, the Middle East, and Africa (EMEA) Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        EMEAUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMELLCRPIEMEAUSTRIV"
        """ICE BofAML Europe, the Middle East, and Africa (EMEA) US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        FinancialEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMFSFCRPITRIV"
        """ICE BofAML Financial Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        FinancialUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMFLFLCRPIUSTRIV"
        """ICE BofAML Financial US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        HighGradeEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMIBHGCRPITRIV"
        """ICE BofAML High Grade Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        HighGradeUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMHGHGLCRPIUSTRIV"
        """ICE BofAML High Grade US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        HighYieldEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMHBHYCRPITRIV"
        """ICE BofAML High Yield Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        HighYieldUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMHYHYLCRPIUSTRIV"
        """ICE BofAML High Yield US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        LatinAmericaEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMRLCRPILATRIV"
        """ICE BofAML Latin America Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        LatinAmericaUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMLLLCRPILAUSTRIV"
        """ICE BofAML Latin America US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        NonFinancialEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMNSNFCRPITRIV"
        """ICE BofAML Non-Financial Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        NonFinancialUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMNFNFLCRPIUSTRIV"
        """ICE BofAML Non-Financial US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        USCorporateMasterOptionAdjustedSpread: str = "BAMLC0A0CM"
        """ICE BofAML US Corporate Master Option-Adjusted Spread (in Percent)"""

        USHighYieldMasterIIOptionAdjustedSpread: str = "BAMLH0A0HYM2"
        """ICE BofAML US High Yield Master II Option-Adjusted Spread (in Percent)"""

        USCorporate1To3YearOptionAdjustedSpread: str = "BAMLC1A0C13Y"
        """ICE BofAML US Corporate 1-3 Year Option-Adjusted Spread (in Percent)"""

        USCorporate10To15YearOptionAdjustedSpread: str = "BAMLC7A0C1015Y"
        """ICE BofAML US Corporate 10-15 Year Option-Adjusted Spread (in Percent)"""

        USCorporateMoreThan15YearOptionAdjustedSpread: str = "BAMLC8A0C15PY"
        """ICE BofAML US Corporate 15+ Year Option-Adjusted Spread (in Percent)"""

        USCorporate3To5YearOptionAdjustedSpread: str = "BAMLC2A0C35Y"
        """ICE BofAML US Corporate 3-5 Year Option-Adjusted Spread (in Percent)"""

        USCorporate5To7YearOptionAdjustedSpread: str = "BAMLC3A0C57Y"
        """ICE BofAML US Corporate 5-7 Year Option-Adjusted Spread (in Percent)"""

        USCorporate7To10YearOptionAdjustedSpread: str = "BAMLC4A0C710Y"
        """ICE BofAML US Corporate 7-10 Year Option-Adjusted Spread (in Percent)"""

        PublicSectorIssuersUSEmergingMarketsLiquidCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMPUPUBSLCRPIUSTRIV"
        """ICE BofAML Public Sector Issuers US Emerging Markets Liquid Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        USEmergingMarketsCorporatePlusSubIndexTotalReturnIndexValue: str = "BAMLEMUBCRPIUSTRIV"
        """ICE BofAML US Emerging Markets Corporate Plus Sub-Index Total Return Index Value (in Index)"""

        USEmergingMarketsLiquidCorporatePlusIndexTotalReturnIndexValue: str = "BAMLEMCLLCRPIUSTRIV"
        """ICE BofAML US Emerging Markets Liquid Corporate Plus Index Total Return Index Value (in Index)"""

        EuroHighYieldIndexTotalReturnIndexValue: str = "BAMLHE00EHYITRIV"
        """ICE BofAML Euro High Yield Index Total Return Index Value (in Index)"""

        USCorp1To3YearsTotalReturnIndexValue: str = "BAMLCC1A013YTRIV"
        """ICE BofAML US Corp 1-3yr Total Return Index Value (in Index)"""

        USCorp10To15TotalReturnIndexValue: str = "BAMLCC7A01015YTRIV"
        """ICE BofAML US Corp 10-15yr Total Return Index Value (in Index)"""

        USCorpMoreThan15YearsTotalReturnIndexValue: str = "BAMLCC8A015PYTRIV"
        """ICE BofAML US Corp 15+yr Total Return Index Value (in Index)"""

        USCorpeTo5YearsTotalReturnIndexValue: str = "BAMLCC2A035YTRIV"
        """ICE BofAML US Corp 3-5yr Total Return Index Value (in Index)"""

        USCorp5To7YearsTotalReturnIndexValue: str = "BAMLCC3A057YTRIV"
        """ICE BofAML US Corp 5-7yr Total Return Index Value (in Index)"""

        USCorporate7To10YearsTotalReturnIndexValue: str = "BAMLCC4A0710YTRIV"
        """ICE BofAML US Corporate 7-10yr Total Return Index Value (in Index)"""

        USCorpATotalReturnIndexValue: str = "BAMLCC0A3ATRIV"
        """ICE BofAML US Corp A Total Return Index Value (in Index)"""

        USCorpAATotalReturnIndexValue: str = "BAMLCC0A2AATRIV"
        """ICE BofAML US Corp AA Total Return Index Value (in Index)"""

        USCorpAAATotalReturnIndexValue: str = "BAMLCC0A1AAATRIV"
        """ICE BofAML US Corp AAA Total Return Index Value (in Index)"""

        USHighYieldBTotalReturnIndexValue: str = "BAMLHYH0A2BTRIV"
        """ICE BofAML US High Yield B Total Return Index Value (in Index)"""

        USHighYieldBBTotalReturnIndexValue: str = "BAMLHYH0A1BBTRIV"
        """ICE BofAML US High Yield BB Total Return Index Value (in Index)"""

        USCorpBBBTotalReturnIndexValue: str = "BAMLCC0A4BBBTRIV"
        """ICE BofAML US Corp BBB Total Return Index Value (in Index)"""

        USHighYieldCCCorBelowTotalReturnIndexValue: str = "BAMLHYH0A3CMTRIV"
        """ICE BofAML US High Yield CCC or Below Total Return Index Value (in Index)"""

        USCorpMasterTotalReturnIndexValue: str = "BAMLCC0A0CMTRIV"
        """ICE BofAML US Corp Master Total Return Index Value (in Index)"""

        USHighYieldMasterIITotalReturnIndexValue: str = "BAMLHYH0A0HYM2TRIV"
        """ICE BofAML US High Yield Master II Total Return Index Value (in Index)"""

        AAAAEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEM1BRRAAA2ACRPIOAS"
        """ICE BofAML AAA-A Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        AAAAUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEM1RAAA2ALCRPIUSOAS"
        """ICE BofAML AAA-A US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        AsiaEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMRACRPIASIAOAS"
        """ICE BofAML Asia Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        AsiaUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMALLCRPIASIAUSOAS"
        """ICE BofAML Asia US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        BandLowerEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEM4BRRBLCRPIOAS"
        """ICE BofAML B and Lower Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        BandLowerUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEM4RBLLCRPIUSOAS"
        """ICE BofAML B and Lower US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        BBEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEM3BRRBBCRPIOAS"
        """ICE BofAML BB Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        BBUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEM3RBBLCRPIUSOAS"
        """ICE BofAML BB US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        BBBEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEM2BRRBBBCRPIOAS"
        """ICE BofAML BBB Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        BBBUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEM2RBBBLCRPIUSOAS"
        """ICE BofAML BBB US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        CrossoverEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEM5BCOCRPIOAS"
        """ICE BofAML Crossover Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        CrossoverUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMXOCOLCRPIUSOAS"
        """ICE BofAML Crossover US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        EmergingMarketsCorporatePlusIndexOptionAdjustedSpread: str = "BAMLEMCBPIOAS"
        """ICE BofAML Emerging Markets Corporate Plus Index Option-Adjusted Spread (in Percent)"""

        EuroEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMEBCRPIEOAS"
        """ICE BofAML Euro Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        EMEAEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMRECRPIEMEAOAS"
        """ICE BofAML Europe, the Middle East, and Africa (EMEA) Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        EMEAUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMELLCRPIEMEAUSOAS"
        """ICE BofAML Europe, the Middle East, and Africa (EMEA) US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        FinancialEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMFSFCRPIOAS"
        """ICE BofAML Financial Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        FinancialUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMFLFLCRPIUSOAS"
        """ICE BofAML Financial US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        HighGradeEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMIBHGCRPIOAS"
        """ICE BofAML High Grade Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        HighGradeUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMHGHGLCRPIUSOAS"
        """ICE BofAML High Grade US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        HighYieldEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMHBHYCRPIOAS"
        """ICE BofAML High Yield Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        HighYieldUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMHYHYLCRPIUSOAS"
        """ICE BofAML High Yield US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        LatinAmericaEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMRLCRPILAOAS"
        """ICE BofAML Latin America Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        LatinAmericaUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMLLLCRPILAUSOAS"
        """ICE BofAML Latin America US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        NonFinancialEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMNSNFCRPIOAS"
        """ICE BofAML Non-Financial Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        NonFinancialUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMNFNFLCRPIUSOAS"
        """ICE BofAML Non-Financial US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        PublicSectorIssuersUSEmergingMarketsLiquidCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMPUPUBSLCRPIUSOAS"
        """ICE BofAML Public Sector Issuers US Emerging Markets Liquid Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        USEmergingMarketsCorporatePlusSubIndexOptionAdjustedSpread: str = "BAMLEMUBCRPIUSOAS"
        """ICE BofAML US Emerging Markets Corporate Plus Sub-Index Option-Adjusted Spread (in Percent)"""

        USEmergingMarketsLiquidCorporatePlusIndexOptionAdjustedSpread: str = "BAMLEMCLLCRPIUSOAS"
        """ICE BofAML US Emerging Markets Liquid Corporate Plus Index Option-Adjusted Spread (in Percent)"""

        EuroHighYieldIndexOptionAdjustedSpread: str = "BAMLHE00EHYIOAS"
        """ICE BofAML Euro High Yield Index Option-Adjusted Spread (in Percent)"""

        USCorporateAOptionAdjustedSpread: str = "BAMLC0A3CA"
        """ICE BofAML US Corporate A Option-Adjusted Spread (in Percent)"""

        USCorporateAAOptionAdjustedSpread: str = "BAMLC0A2CAA"
        """ICE BofAML US Corporate AA Option-Adjusted Spread (in Percent)"""

        USCorporateAAAOptionAdjustedSpread: str = "BAMLC0A1CAAA"
        """ICE BofAML US Corporate AAA Option-Adjusted Spread (in Percent)"""

        USHighYieldBOptionAdjustedSpread: str = "BAMLH0A2HYB"
        """ICE BofAML US High Yield B Option-Adjusted Spread (in Percent)"""

        USHighYieldBBOptionAdjustedSpread: str = "BAMLH0A1HYBB"
        """ICE BofAML US High Yield BB Option-Adjusted Spread (in Percent)"""

        USCorporateBBBOptionAdjustedSpread: str = "BAMLC0A4CBBB"
        """ICE BofAML US Corporate BBB Option-Adjusted Spread (in Percent)"""

        USHighYieldCCCorBelowOptionAdjustedSpread: str = "BAMLH0A3HYC"
        """ICE BofAML US High Yield CCC or Below Option-Adjusted Spread (in Percent)"""

        AAAAEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEM1BRRAAA2ACRPIEY"
        """ICE BofAML AAA-A Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        AAAAUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEM1RAAA2ALCRPIUSEY"
        """ICE BofAML AAA-A US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        AsiaEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEMRACRPIASIAEY"
        """ICE BofAML Asia Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        AsiaUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEMALLCRPIASIAUSEY"
        """ICE BofAML Asia US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        BandLowerEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEM4BRRBLCRPIEY"
        """ICE BofAML B and Lower Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        BandLowerUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEM4RBLLCRPIUSEY"
        """ICE BofAML B and Lower US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        BBEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEM3BRRBBCRPIEY"
        """ICE BofAML BB Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        BBUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEM3RBBLCRPIUSEY"
        """ICE BofAML BB US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        BBBEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEM2BRRBBBCRPIEY"
        """ICE BofAML BBB Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        BBBUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEM2RBBBLCRPIUSEY"
        """ICE BofAML BBB US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        CrossoverEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEM5BCOCRPIEY"
        """ICE BofAML Crossover Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        CrossoverUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEMXOCOLCRPIUSEY"
        """ICE BofAML Crossover US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        EmergingMarketsCorporatePlusIndexEffectiveYield: str = "BAMLEMCBPIEY"
        """ICE BofAML Emerging Markets Corporate Plus Index Effective Yield (in Percent)"""

        EuroEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEMEBCRPIEEY"
        """ICE BofAML Euro Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        EuroHighYieldIndexEffectiveYield: str = "BAMLHE00EHYIEY"
        """ICE BofAML Euro High Yield Index Effective Yield (in Percent)"""

        EMEAEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEMRECRPIEMEAEY"
        """ICE BofAML Europe, the Middle East, and Africa (EMEA) Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        EMEAUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEMELLCRPIEMEAUSEY"
        """ICE BofAML Europe, the Middle East, and Africa (EMEA) US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        FinancialEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEMFSFCRPIEY"
        """ICE BofAML Financial Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        FinancialUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEMFLFLCRPIUSEY"
        """ICE BofAML Financial US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        HighGradeEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEMIBHGCRPIEY"
        """ICE BofAML High Grade Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        HighGradeUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEMHGHGLCRPIUSEY"
        """ICE BofAML High Grade US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        HighYieldEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEMHBHYCRPIEY"
        """ICE BofAML High Yield Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        HighYieldUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEMHYHYLCRPIUSEY"
        """ICE BofAML High Yield US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        LatinAmericaEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEMRLCRPILAEY"
        """ICE BofAML Latin America Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        LatinAmericaUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEMLLLCRPILAUSEY"
        """ICE BofAML Latin America US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        NonFinancialEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEMNSNFCRPIEY"
        """ICE BofAML Non-Financial Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        NonFinancialUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEMNFNFLCRPIUSEY"
        """ICE BofAML Non-Financial US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        PublicSectorIssuersUSEmergingMarketsLiquidCorporatePlusSubIndexEffectiveYield: str = "BAMLEMPUPUBSLCRPIUSEY"
        """ICE BofAML Public Sector Issuers US Emerging Markets Liquid Corporate Plus Sub-Index Effective Yield (in Percent)"""

        USCorporate1ThreeYearEffectiveYield: str = "BAMLC1A0C13YEY"
        """ICE BofAML US Corporate 1-3 Year Effective Yield (in Percent)"""

        USCorporate10To15YearEffectiveYield: str = "BAMLC7A0C1015YEY"
        """ICE BofAML US Corporate 10-15 Year Effective Yield (in Percent)"""

        USCorporateMoreThan15YearEffectiveYield: str = "BAMLC8A0C15PYEY"
        """ICE BofAML US Corporate 15+ Year Effective Yield (in Percent)"""

        USCorporate3To5YearEffectiveYield: str = "BAMLC2A0C35YEY"
        """ICE BofAML US Corporate 3-5 Year Effective Yield (in Percent)"""

        USCorporate5To7YearEffectiveYield: str = "BAMLC3A0C57YEY"
        """ICE BofAML US Corporate 5-7 Year Effective Yield (in Percent)"""

        USCorporate7To10YearEffectiveYield: str = "BAMLC4A0C710YEY"
        """ICE BofAML US Corporate 7-10 Year Effective Yield (in Percent)"""

        USCorporateAEffectiveYield: str = "BAMLC0A3CAEY"
        """ICE BofAML US Corporate A Effective Yield (in Percent)"""

        USCorporateAAEffectiveYield: str = "BAMLC0A2CAAEY"
        """ICE BofAML US Corporate AA Effective Yield (in Percent)"""

        USCorporateAAAEffectiveYield: str = "BAMLC0A1CAAAEY"
        """ICE BofAML US Corporate AAA Effective Yield (in Percent)"""

        USHighYieldBEffectiveYield: str = "BAMLH0A2HYBEY"
        """ICE BofAML US High Yield B Effective Yield (in Percent)"""

        USHighYieldBBEffectiveYield: str = "BAMLH0A1HYBBEY"
        """ICE BofAML US High Yield BB Effective Yield (in Percent)"""

        USCorporateBBBEffectiveYield: str = "BAMLC0A4CBBBEY"
        """ICE BofAML US Corporate BBB Effective Yield (in Percent)"""

        USHighYieldCCCorBelowEffectiveYield: str = "BAMLH0A3HYCEY"
        """ICE BofAML US High Yield CCC or Below Effective Yield (in Percent)"""

        USCorporateMasterEffectiveYield: str = "BAMLC0A0CMEY"
        """ICE BofAML US Corporate Master Effective Yield (in Percent)"""

        USEmergingMarketsCorporatePlusSubIndexEffectiveYield: str = "BAMLEMUBCRPIUSEY"
        """ICE BofAML US Emerging Markets Corporate Plus Sub-Index Effective Yield (in Percent)"""

        USEmergingMarketsLiquidCorporatePlusIndexEffectiveYield: str = "BAMLEMCLLCRPIUSEY"
        """ICE BofAML US Emerging Markets Liquid Corporate Plus Index Effective Yield (in Percent)"""

        USHighYieldMasterIIEffectiveYield: str = "BAMLH0A0HYM2EY"
        """ICE BofAML US High Yield Master II Effective Yield (in Percent)"""

        AAAAEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEM1BRRAAA2ACRPISYTW"
        """ICE BofAML AAA-A Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        AAAAUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEM1RAAA2ALCRPIUSSYTW"
        """ICE BofAML AAA-A US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        AsiaEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMRACRPIASIASYTW"
        """ICE BofAML Asia Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        AsiaUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMALLCRPIASIAUSSYTW"
        """ICE BofAML Asia US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        BandLowerEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEM4BRRBLCRPISYTW"
        """ICE BofAML B and Lower Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        BandLowerUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEM4RBLLCRPIUSSYTW"
        """ICE BofAML B and Lower US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        BBEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEM3BRRBBCRPISYTW"
        """ICE BofAML BB Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        BBUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEM3RBBLCRPIUSSYTW"
        """ICE BofAML BB US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        BBBEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEM2BRRBBBCRPISYTW"
        """ICE BofAML BBB Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        BBBUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEM2RBBBLCRPIUSSYTW"
        """ICE BofAML BBB US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        CrossoverEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEM5BCOCRPISYTW"
        """ICE BofAML Crossover Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        CrossoverUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMXOCOLCRPIUSSYTW"
        """ICE BofAML Crossover US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        EmergingMarketsCorporatePlusIndexSemiAnnualYieldtoWorst: str = "BAMLEMCBPISYTW"
        """ICE BofAML Emerging Markets Corporate Plus Index Semi-Annual Yield to Worst (in Percent)"""

        EuroEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMEBCRPIESYTW"
        """ICE BofAML Euro Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        EuroHighYieldIndexSemiAnnualYieldtoWorst: str = "BAMLHE00EHYISYTW"
        """ICE BofAML Euro High Yield Index Semi-Annual Yield to Worst (in Percent)"""

        EMEAEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMRECRPIEMEASYTW"
        """ICE BofAML Europe, the Middle East, and Africa (EMEA) Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        EMEAUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMELLCRPIEMEAUSSYTW"
        """ICE BofAML Europe, the Middle East, and Africa (EMEA) US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        FinancialEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMFSFCRPISYTW"
        """ICE BofAML Financial Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        FinancialUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMFLFLCRPIUSSYTW"
        """ICE BofAML Financial US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        HighGradeEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMIBHGCRPISYTW"
        """ICE BofAML High Grade Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        HighGradeUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMHGHGLCRPIUSSYTW"
        """ICE BofAML High Grade US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        HighYieldEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMHBHYCRPISYTW"
        """ICE BofAML High Yield Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        HighYieldUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMHYHYLCRPIUSSYTW"
        """ICE BofAML High Yield US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        LatinAmericaEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMRLCRPILASYTW"
        """ICE BofAML Latin America Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        LatinAmericaUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMLLLCRPILAUSSYTW"
        """ICE BofAML Latin America US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        NonFinancialEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMNSNFCRPISYTW"
        """ICE BofAML Non-Financial Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        NonFinancialUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMNFNFLCRPIUSSYTW"
        """ICE BofAML Non-Financial US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        PrivateSectorIssuersEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMPTPRVICRPISYTW"
        """ICE BofAML Private Sector Issuers Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        PrivateSectorIssuersUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMPVPRIVSLCRPIUSSYTW"
        """ICE BofAML Private Sector Issuers US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        PublicSectorIssuersEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMPBPUBSICRPISYTW"
        """ICE BofAML Public Sector Issuers Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        PublicSectorIssuersUSEmergingMarketsLiquidCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMPUPUBSLCRPIUSSYTW"
        """ICE BofAML Public Sector Issuers US Emerging Markets Liquid Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        USCorporate1To3YearSemiAnnualYieldtoWorst: str = "BAMLC1A0C13YSYTW"
        """ICE BofAML US Corporate 1-3 Year Semi-Annual Yield to Worst (in Percent)"""

        USCorporate10To15YearSemiAnnualYieldtoWorst: str = "BAMLC7A0C1015YSYTW"
        """ICE BofAML US Corporate 10-15 Year Semi-Annual Yield to Worst (in Percent)"""

        USCorporateMoreThan15YearSemiAnnualYieldtoWorst: str = "BAMLC8A0C15PYSYTW"
        """ICE BofAML US Corporate 15+ Year Semi-Annual Yield to Worst (in Percent)"""

        USCorporate3To5YearSemiAnnualYieldtoWorst: str = "BAMLC2A0C35YSYTW"
        """ICE BofAML US Corporate 3-5 Year Semi-Annual Yield to Worst (in Percent)"""

        USCorporate5To7YearSemiAnnualYieldtoWorst: str = "BAMLC3A0C57YSYTW"
        """ICE BofAML US Corporate 5-7 Year Semi-Annual Yield to Worst (in Percent)"""

        USCorporate7To10YearSemiAnnualYieldtoWorst: str = "BAMLC4A0C710YSYTW"
        """ICE BofAML US Corporate 7-10 Year Semi-Annual Yield to Worst (in Percent)"""

        USCorporateASemiAnnualYieldtoWorst: str = "BAMLC0A3CASYTW"
        """ICE BofAML US Corporate A Semi-Annual Yield to Worst (in Percent)"""

        USCorporateAASemiAnnualYieldtoWorst: str = "BAMLC0A2CAASYTW"
        """ICE BofAML US Corporate AA Semi-Annual Yield to Worst (in Percent)"""

        USCorporateAAASemiAnnualYieldtoWorst: str = "BAMLC0A1CAAASYTW"
        """ICE BofAML US Corporate AAA Semi-Annual Yield to Worst (in Percent)"""

        USHighYieldBSemiAnnualYieldtoWorst: str = "BAMLH0A2HYBSYTW"
        """ICE BofAML US High Yield B Semi-Annual Yield to Worst (in Percent)"""

        USHighYieldBBSemiAnnualYieldtoWorst: str = "BAMLH0A1HYBBSYTW"
        """ICE BofAML US High Yield BB Semi-Annual Yield to Worst (in Percent)"""

        USCorporateBBBSemiAnnualYieldtoWorst: str = "BAMLC0A4CBBBSYTW"
        """ICE BofAML US Corporate BBB Semi-Annual Yield to Worst (in Percent)"""

        USHighYieldCCCorBelowSemiAnnualYieldtoWorst: str = "BAMLH0A3HYCSYTW"
        """ICE BofAML US High Yield CCC or Below Semi-Annual Yield to Worst (in Percent)"""

        USCorporateMasterSemiAnnualYieldtoWorst: str = "BAMLC0A0CMSYTW"
        """ICE BofAML US Corporate Master Semi-Annual Yield to Worst (in Percent)"""

        USEmergingMarketsCorporatePlusSubIndexSemiAnnualYieldtoWorst: str = "BAMLEMUBCRPIUSSYTW"
        """ICE BofAML US Emerging Markets Corporate Plus Sub-Index Semi-Annual Yield to Worst (in Percent)"""

        USEmergingMarketsLiquidCorporatePlusIndexSemiAnnualYieldtoWorst: str = "BAMLEMCLLCRPIUSSYTW"
        """ICE BofAML US Emerging Markets Liquid Corporate Plus Index Semi-Annual Yield to Worst (in Percent)"""

        USHighYieldMasterIISemiAnnualYieldtoWorst: str = "BAMLH0A0HYM2SYTW"
        """ICE BofAML US High Yield Master II Semi-Annual Yield to Worst (in Percent)"""

    class CommercialPaper(System.Object):
        """
        Commercial paper (CP) consists of short-term, promissory notes issued primarily by corporations. Maturities range up to 270 days but average about 30 days. Many companies use CP to raise cash needed for current transactions, and many find it to be a lower-cost alternative to bank loans.
        The Federal Reserve Board disseminates information on CP primarily through its World Wide Web site. In addition, the Board publishes one-, two-, and three-month rates on AA nonfinancial and AA financial CP weekly in its H.15 Statistical Release.
        The Federal Reserve Board's CP release is derived from data supplied by The Depository Trust & Clearing Corporation (DTCC), a national clearinghouse for the settlement of securities trades and a custodian for securities. DTCC performs these functions for almost all activity in the domestic CP market. The Federal Reserve Board only considers maturities of 270 days or less. CP is exempt from SEC registration if its maturity does not exceed 270 days.
        Data on CP issuance rates and volumes typically are updated daily and typically posted with a one-day lag. Data on CP outstanding usually are available as of the close of business each Wednesday and as of the last business day of the month; these data are also posted with a one-day lag. The daily CP release will usually be available at 9:45 a.m. EST. However, the Federal Reserve Board makes no guarantee regarding the timing of the daily CP release. This policy is subject to change at any time without notice.
        """

        ThreeMonthAANonfinancialCommercialPaperRate: str = "DCPN3M"
        """3-Month AA Nonfinancial Commercial Paper Rate (in Percent)"""

        OneMonthAANonfinancialCommercialPaperRate: str = "DCPN30"
        """1-Month AA Nonfinancial Commercial Paper Rate (in Percent)"""

        TwoMonthAANonfinancialCommercialPaperRate: str = "DCPN2M"
        """2-Month AA Nonfinancial Commercial Paper Rate (in Percent)"""

        ThreeMonthAAFinancialCommercialPaperRate: str = "DCPF3M"
        """3-Month AA Financial Commercial Paper Rate (in Percent)"""

        TwoMonthAAFinancialCommercialPaperRate: str = "DCPF2M"
        """2-Month AA Financial Commercial Paper Rate (in Percent)"""

        OneMonthAAFinancialCommercialPaperRate: str = "DCPF1M"
        """1-Month AA Financial Commercial Paper Rate (in Percent)"""

        NumberOfIssuesWithMaturityBetween1and4DaysUsedForA2P2Nonfinancial: str = "NONFIN14A2P2VOL"
        """Number of Issues, with a Maturity Between 1 and 4 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Number)"""

        NumberOfIssuesWithMaturityBetween5and9DaysUsedForA2P2Nonfinancial: str = "NONFIN59A2P2VOL"
        """Number of Issues, with a Maturity Between 5 and 9 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityBetween5and9DaysUsedForA2P2Nonfinancial: str = "NONFIN59A2P2AMT"
        """Total Value of Issues, with a Maturity Between 5 and 9 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween41and80DaysUsedForAANonfinancial: str = "NONFIN4180AAVOL"
        """Number of Issues, with a Maturity Between 41 and 80 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityGreaterThan80DaysUsedForAAAssetBacked: str = "ABGT80AAAMT"
        """Total Value of Issues, with a Maturity Greater Than 80 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Millions of Dollars)"""

        TotalValueOfIssuesWithMaturityBetween41and80DaysUsedForAANonfinancial: str = "NONFIN4180AAAMT"
        """Total Value of Issues, with a Maturity Between 41 and 80 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween41and80DaysUsedForA2P2Nonfinancial: str = "NONFIN4180A2P2VOL"
        """Number of Issues, with a Maturity Between 41 and 80 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityBetween41and80DaysUsedForA2P2Nonfinancial: str = "NONFIN4180A2P2AMT"
        """Total Value of Issues, with a Maturity Between 41 and 80 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween21and40DaysUsedForAANonfinancial: str = "NONFIN2140AAVOL"
        """Number of Issues, with a Maturity Between 21 and 40 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityBetween21and40DaysUsedForAANonfinancial: str = "NONFIN2140AAAMT"
        """Total Value of Issues, with a Maturity Between 21 and 40 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween21and40DaysUsedForA2P2Nonfinancial: str = "NONFIN2140A2P2VOL"
        """Number of Issues, with a Maturity Between 21 and 40 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityBetween21and40DaysUsedForA2P2Nonfinancial: str = "NONFIN2140A2P2AMT"
        """Total Value of Issues, with a Maturity Between 21 and 40 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween1and4DaysUsedForAANonfinancial: str = "NONFIN14AAVOL"
        """Number of Issues, with a Maturity Between 1 and 4 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Number)"""

        NumberOfIssuesWithMaturityBetween10And20DaysUsedForA2P2Nonfinancial: str = "NONFIN1020A2P2VOL"
        """Number of Issues, with a Maturity Between 10 and 20 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityBetween10And20DaysUsedForAANonfinancial: str = "NONFIN1020AAAMT"
        """Total Value of Issues, with a Maturity Between 10 and 20 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        TotalValueOfIssuesWithMaturityBetween21and40DaysUsedForAAAssetBacked: str = "AB2140AAAMT"
        """Total Value of Issues, with a Maturity Between 21 and 40 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween10And20DaysUsedForAANonfinancial: str = "NONFIN1020AAVOL"
        """Number of Issues, with a Maturity Between 10 and 20 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityBetween1and4DaysUsedForA2P2Nonfinancial: str = "NONFIN14A2P2AMT"
        """Total Value of Issues, with a Maturity Between 1 and 4 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        TotalValueOfIssuesWithMaturityBetween1and4DaysUsedForAANonfinancial: str = "NONFIN14AAAMT"
        """Total Value of Issues, with a Maturity Between 1 and 4 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        TotalValueofCommercialPaperIssueswithaMaturityBetween1and4Days: str = "MKT14MKTAMT"
        """Total Value of Commercial Paper Issues with a Maturity Between 1 and 4 Days (in Millions of Dollars)"""

        TotalValueOfIssuesWithMaturityBetween10And20DaysUsedForA2P2Nonfinancial: str = "NONFIN1020A2P2AMT"
        """Total Value of Issues, with a Maturity Between 10 and 20 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityGreaterThan80DaysUsedForAAFinancial: str = "FINGT80AAVOL"
        """Number of Issues, with a Maturity Greater Than 80 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Number)"""

        NumberOfIssuesWithMaturityBetween10And20DaysUsedForAAFinancial: str = "FIN1020AAVOL"
        """Number of Issues, with a Maturity Between 10 and 20 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityBetween1and4DaysUsedForAAFinancial: str = "FIN14AAAMT"
        """Total Value of Issues, with a Maturity Between 1 and 4 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween1and4DaysUsedForAAFinancial: str = "FIN14AAVOL"
        """Number of Issues, with a Maturity Between 1 and 4 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Number)"""

        TotalValueofCommercialPaperIssueswithaMaturityBetween10And20Days: str = "MKT1020MKTAMT"
        """Total Value of Commercial Paper Issues with a Maturity Between 10 and 20 Days (in Millions of Dollars)"""

        NumberofCommercialPaperIssueswithaMaturityBetween10And20Days: str = "MKT1020MKTVOL"
        """Number of Commercial Paper Issues with a Maturity Between 10 and 20 Days (in Number)"""

        TotalValueOfIssuesWithMaturityBetween21and40DaysUsedForAAFinancial: str = "FIN2140AAAMT"
        """Total Value of Issues, with a Maturity Between 21 and 40 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Millions of Dollars)"""

        NumberofCommercialPaperIssueswithaMaturityBetween1and4Days: str = "MKT14MKTVOL"
        """Number of Commercial Paper Issues with a Maturity Between 1 and 4 Days (in Number)"""

        TotalValueofIssuersofCommercialPaperwithaMaturityBetween21and40Days: str = "MKT2140MKTAMT"
        """Total Value of Issuers of Commercial Paper with a Maturity Between 21 and 40 Days (in Millions of Dollars)"""

        NumberofCommercialPaperIssueswithaMaturityBetween21and40Days: str = "MKT2140MKTVOL"
        """Number of Commercial Paper Issues with a Maturity Between 21 and 40 Days (in Number)"""

        NumberOfIssuesWithMaturityBetween21and40DaysUsedForAAFinancial: str = "FIN2140AAVOL"
        """Number of Issues, with a Maturity Between 21 and 40 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Number)"""

        TotalValueofIssuersofCommercialPaperwithaMaturityBetween41and80Days: str = "MKT4180MKTAMT"
        """Total Value of Issuers of Commercial Paper with a Maturity Between 41 and 80 Days (in Millions of Dollars)"""

        TotalValueOfIssuesWithMaturityBetween5and9DaysUsedForAANonfinancial: str = "NONFIN59AAAMT"
        """Total Value of Issues, with a Maturity Between 5 and 9 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        NumberofCommercialPaperIssueswithaMaturityBetween41and80Days: str = "MKT4180MKTVOL"
        """Number of Commercial Paper Issues with a Maturity Between 41 and 80 Days (in Number)"""

        NumberofCommercialPaperIssueswithaMaturityBetween5and9Days: str = "MKT59MKTVOL"
        """Number of Commercial Paper Issues with a Maturity Between 5 and 9 Days (in Number)"""

        TotalValueofIssuersofCommercialPaperwithaMaturityGreaterThan80Days: str = "MKTGT80MKTAMT"
        """Total Value of Issuers of Commercial Paper with a Maturity Greater Than 80 Days (in Millions of Dollars)"""

        NumberofCommercialPaperIssueswithaMaturityGreaterThan80Days: str = "MKTGT80MKTVOL"
        """Number of Commercial Paper Issues with a Maturity Greater Than 80 Days (in Number)"""

        TotalValueOfIssuesWithMaturityBetween41and80DaysUsedForAAFinancial: str = "FIN4180AAAMT"
        """Total Value of Issues, with a Maturity Between 41 and 80 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween41and80DaysUsedForAAFinancial: str = "FIN4180AAVOL"
        """Number of Issues, with a Maturity Between 41 and 80 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityBetween41and80DaysUsedForAAAssetBacked: str = "AB4180AAAMT"
        """Total Value of Issues, with a Maturity Between 41 and 80 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Millions of Dollars)"""

        TotalValueOfIssuesWithMaturityBetween5and9DaysUsedForAAFinancial: str = "FIN59AAAMT"
        """Total Value of Issues, with a Maturity Between 5 and 9 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween5and9DaysUsedForAAFinancial: str = "FIN59AAVOL"
        """Number of Issues, with a Maturity Between 5 and 9 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityGreaterThan80DaysUsedForAAFinancial: str = "FINGT80AAAMT"
        """Total Value of Issues, with a Maturity Greater Than 80 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Millions of Dollars)"""

        TotalValueOfIssuesWithMaturityBetween10And20DaysUsedForAAFinancial: str = "FIN1020AAAMT"
        """Total Value of Issues, with a Maturity Between 10 and 20 Days, Used in Calculating the AA Financial Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween21and40DaysUsedForAAAssetBacked: str = "AB2140AAVOL"
        """Number of Issues, with a Maturity Between 21 and 40 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Number)"""

        TotalValueofIssuersofCommercialPaperwithaMaturityBetween5and9Days: str = "MKT59MKTAMT"
        """Total Value of Issuers of Commercial Paper with a Maturity Between 5 and 9 Days (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityGreaterThan80DaysUsedForAAAssetBacked: str = "ABGT80AAVOL"
        """Number of Issues, with a Maturity Greater Than 80 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Number)"""

        NumberOfIssuesWithMaturityBetween5and9DaysUsedForAANonfinancial: str = "NONFIN59AAVOL"
        """Number of Issues, with a Maturity Between 5 and 9 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Number)"""

        FifteenDayAAAssetbackedCommercialPaperInterestRate: str = "RIFSPPAAAD15NB"
        """15-Day AA Asset-backed Commercial Paper Interest Rate (in Percent)"""

        TotalValueOfIssuesWithMaturityBetween5and9DaysUsedForAAAssetBacked: str = "AB59AAAMT"
        """Total Value of Issues, with a Maturity Between 5 and 9 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Millions of Dollars)"""

        NumberOfIssuesWithMaturityBetween41and80DaysUsedForAAAssetBacked: str = "AB4180AAVOL"
        """Number of Issues, with a Maturity Between 41 and 80 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Number)"""

        FifteenDayA2P2NonfinancialCommercialPaperInterestRate: str = "RIFSPPNA2P2D15NB"
        """15-Day A2/P2 Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        SevenDayA2P2NonfinancialCommercialPaperInterestRate: str = "RIFSPPNA2P2D07NB"
        """7-Day A2/P2 Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        OvernightA2P2NonfinancialCommercialPaperInterestRate: str = "RIFSPPNA2P2D01NB"
        """Overnight A2/P2 Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        NinetyDayAAFinancialCommercialPaperInterestRate: str = "RIFSPPFAAD90NB"
        """90-Day AA Financial Commercial Paper Interest Rate (in Percent)"""

        OvernightAAAssetbackedCommercialPaperInterestRate: str = "RIFSPPAAAD01NB"
        """Overnight AA Asset-backed Commercial Paper Interest Rate (in Percent)"""

        Three0DayA2P2NonfinancialCommercialPaperInterestRate: str = "RIFSPPNA2P2D30NB"
        """30-Day A2/P2 Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        SixtyDayAAFinancialCommercialPaperInterestRate: str = "RIFSPPFAAD60NB"
        """60-Day AA Financial Commercial Paper Interest Rate (in Percent)"""

        Three0DayAAFinancialCommercialPaperInterestRate: str = "RIFSPPFAAD30NB"
        """30-Day AA Financial Commercial Paper Interest Rate (in Percent)"""

        TotalValueOfIssuesWithMaturityGreaterThan80DaysUsedForA2P2Nonfinancial: str = "NONFINGT80A2P2AMT"
        """Total Value of Issues, with a Maturity Greater Than 80 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        Three0DayAAAssetbackedCommercialPaperInterestRate: str = "RIFSPPAAAD30NB"
        """30-Day AA Asset-backed Commercial Paper Interest Rate (in Percent)"""

        SixtyDayAAAssetbackedCommercialPaperInterestRate: str = "RIFSPPAAAD60NB"
        """60-Day AA Asset-backed Commercial Paper Interest Rate (in Percent)"""

        NinetyDayAAAssetbackedCommercialPaperInterestRate: str = "RIFSPPAAAD90NB"
        """90-Day AA Asset-backed Commercial Paper Interest Rate (in Percent)"""

        FifteenDayAAFinancialCommercialPaperInterestRate: str = "RIFSPPFAAD15NB"
        """15-Day AA Financial Commercial Paper Interest Rate (in Percent)"""

        SevenDayAAFinancialCommercialPaperInterestRate: str = "RIFSPPFAAD07NB"
        """7-Day AA Financial Commercial Paper Interest Rate (in Percent)"""

        SevenDayAAAssetbackedCommercialPaperInterestRate: str = "RIFSPPAAAD07NB"
        """7-Day AA Asset-backed Commercial Paper Interest Rate (in Percent)"""

        OvernightAAFinancialCommercialPaperInterestRate: str = "RIFSPPFAAD01NB"
        """Overnight AA Financial Commercial Paper Interest Rate (in Percent)"""

        SixtyDayA2P2NonfinancialCommercialPaperInterestRate: str = "RIFSPPNA2P2D60NB"
        """60-Day A2/P2 Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        NumberOfIssuesWithMaturityBetween5and9DaysUsedForAAAssetBacked: str = "AB59AAVOL"
        """Number of Issues, with a Maturity Between 5 and 9 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Number)"""

        NumberOfIssuesWithMaturityBetween1and4DaysUsedForAAAssetBacked: str = "AB14AAVOL"
        """Number of Issues, with a Maturity Between 1 and 4 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Number)"""

        NumberOfIssuesWithMaturityGreaterThan80DaysUsedForA2P2Nonfinancial: str = "NONFINGT80A2P2VOL"
        """Number of Issues, with a Maturity Greater Than 80 Days, Used in Calculating the A2/P2 Nonfinancial Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityBetween1and4DaysUsedForAAAssetBacked: str = "AB14AAAMT"
        """Total Value of Issues, with a Maturity Between 1 and 4 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Millions of Dollars)"""

        NinetyDayA2P2NonfinancialCommercialPaperInterestRate: str = "RIFSPPNA2P2D90NB"
        """90-Day A2/P2 Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        NumberOfIssuesWithMaturityBetween10And20DaysUsedForAAAssetBacked: str = "AB1020AAVOL"
        """Number of Issues, with a Maturity Between 10 and 20 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Number)"""

        TotalValueOfIssuesWithMaturityGreaterThan80DaysUsedForAANonfinancial: str = "NONFINGT80AAAMT"
        """Total Value of Issues, with a Maturity Greater Than 80 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Millions of Dollars)"""

        OvernightAANonfinancialCommercialPaperInterestRate: str = "RIFSPPNAAD01NB"
        """Overnight AA Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        TotalValueOfIssuesWithMaturityBetween10And20DaysUsedForAAAssetBacked: str = "AB1020AAAMT"
        """Total Value of Issues, with a Maturity Between 10 and 20 Days, Used in Calculating the AA Asset-Backed Commercial Paper Rates (in Millions of Dollars)"""

        SevenDayAANonfinancialCommercialPaperInterestRate: str = "RIFSPPNAAD07NB"
        """7-Day AA Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        NinetyDayAANonfinancialCommercialPaperInterestRate: str = "RIFSPPNAAD90NB"
        """90-Day AA Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        FifteenDayAANonfinancialCommercialPaperInterestRate: str = "RIFSPPNAAD15NB"
        """15-Day AA Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        Three0DayAANonfinancialCommercialPaperInterestRate: str = "RIFSPPNAAD30NB"
        """30-Day AA Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        SixtyDayAANonfinancialCommercialPaperInterestRate: str = "RIFSPPNAAD60NB"
        """60-Day AA Nonfinancial Commercial Paper Interest Rate (in Percent)"""

        NumberOfIssuesWithMaturityGreaterThan80DaysUsedForAANonfinancial: str = "NONFINGT80AAVOL"
        """Number of Issues, with a Maturity Greater Than 80 Days, Used in Calculating the AA Nonfinancial Commercial Paper Rates (in Number)"""

        ThreeMonthCommercialPaperMinusFederalFundsRate: str = "CPFF"
        """3-Month Commercial Paper Minus Federal Funds Rate (in Percent)"""

    class CentralBankInterventions(System.Object):
        """Central Bank Interventions"""

        JapaneseBankPurchasesOfDmEuroAgainstJpy: str = "JPINTDDMEJPY"
        """Japan Intervention: Japanese Bank purchases of DM/Euro against JPY (in 100 Million Yen)"""

        JapaneseBankPurchasesOfUsdAgainstDm: str = "JPINTDEXR"
        """Japan Intervention: Japanese Bank purchases of USD against DM (in 100 Million Yen)"""

        JapaneseBankPurchasesOfUsdAgainstRupiah: str = "JPINTDUSDRP"
        """Japan Intervention: Japanese Bank purchases of USD against Rupiah (in 100 Million Yen)"""

        USInterventionInMarketTransactionsInTheJpyUsd: str = "USINTDMRKTJPY"
        """U.S. Intervention: in Market Transactions in the JPY/USD (Millions of USD) (in Millions of USD)"""

        USInterventionWithCustomerTransactionsInOtherCurrencies: str = "USINTDCSOTH"
        """U.S. Intervention: With-Customer Transactions in Other Currencies (Millions of USD) (in Millions of USD)"""

        USInterventionWithCustomerTransactionsInTheJpyUsd: str = "USINTDCSJPY"
        """U.S. Intervention: With-Customer Transactions in the JPY/USD (Millions of USD) (in Millions of USD)"""

        USInterventionWithCustomerTransactionsInTheDemUsdEuro: str = "USINTDCSDM"
        """U.S. Intervention: With-Customer Transactions in the DEM/USD (Euro since 1999) (Millions of USD) (in Millions of USD)"""

        USInterventionInMarketTransactionsInOtherCurrencies: str = "USINTDMRKTOTH"
        """U.S. Intervention: in Market Transactions in Other Currencies (Millions of USD) (in Millions of USD)"""

        CentralBankOfTurkeyPurchasesOfUsd: str = "TRINTDEXR"
        """Turkish Intervention: Central Bank of Turkey Purchases of USD (Millions of USD) (in Millions of USD)"""

        JapaneseBankPurchasesOfUsdAgainstJpy: str = "JPINTDUSDJPY"
        """Japan Intervention: Japanese Bank purchases of USD against JPY (in 100 Million Yen)"""

        USInterventionInMarketTransactionsInTheDemUsdEuro: str = "USINTDMRKTDM"
        """U.S. Intervention: in Market Transactions in the DEM/USD (Euro since 1999) (Millions of USD) (in Millions of USD)"""

        SwissNationalBankPurchasesOfDemAgainstChfMillionsOfDem: str = "CHINTDCHFDM"
        """Swiss Intervention: Swiss National Bank Purchases of DEM against CHF (Millions of DEM) (in Millions of DEM)"""

        SwissNationalBankPurchasesOfUsdAgainstDem: str = "CHINTDUSDDM"
        """Swiss Intervention: Swiss National Bank Purchases of USD against DEM (Millions of USD) (in Millions of USD)"""

        SwissNationalBankPurchasesOfUsdAgainstJpy: str = "CHINTDUSDJPY"
        """Swiss Intervention: Swiss National Bank Purchases of USD against JPY (Millions of USD) (in Millions of USD)"""

        SwissNationalBankPurchasesOfUsdAgainstChf: str = "CHINTDCHFUSD"
        """Swiss Intervention: Swiss National Bank Purchases of USD against CHF (Millions of USD) (in Millions of USD)"""

        BancoDeMexicoPurchaseOnTheUsd: str = "MEXINTDUSD"
        """Mexican Intervention: Banco de Mexico Purchase on the USD (in Millions of USD)"""

    class LIBOR(System.Object):
        """London InterBank Offered Rate"""

        SpotNextBasedOnSwissFranc: str = "CHFONTD156N"
        """Spot Next London Interbank Offered Rate (LIBOR), based on Swiss Franc (in Percent)"""

        SpotNextBasedOnJapaneseYen: str = "JPYONTD156N"
        """Spot Next London Interbank Offered Rate (LIBOR), based on Japanese Yen (in Percent)"""

        SixMonthBasedOnJapaneseYen: str = "JPY6MTD156N"
        """6-Month London Interbank Offered Rate (LIBOR), based on Japanese Yen (in Percent)"""

        ThreeMonthBasedOnJapaneseYen: str = "JPY3MTD156N"
        """3-Month London Interbank Offered Rate (LIBOR), based on Japanese Yen (in Percent)"""

        SixMonthBasedOnUSD: str = "USD6MTD156N"
        """6-Month London Interbank Offered Rate (LIBOR), based on U.S. Dollar (in Percent)"""

        OneMonthBasedOnJapaneseYen: str = "JPY1MTD156N"
        """1-Month London Interbank Offered Rate (LIBOR), based on Japanese Yen (in Percent)"""

        TwelveMonthBasedOnJapaneseYen: str = "JPY12MD156N"
        """12-Month London Interbank Offered Rate (LIBOR), based on Japanese Yen (in Percent)"""

        TwelveMonthBasedOnBritishPound: str = "GBP12MD156N"
        """12-Month London Interbank Offered Rate (LIBOR), based on British Pound (in Percent)"""

        OneMonthBasedOnBritishPound: str = "GBP1MTD156N"
        """1-Month London Interbank Offered Rate (LIBOR), based on British Pound (in Percent)"""

        OneWeekBasedOnBritishPound: str = "GBP1WKD156N"
        """1-Week London Interbank Offered Rate (LIBOR), based on British Pound (in Percent)"""

        TwoMonthBasedOnBritishPound: str = "GBP2MTD156N"
        """2-Month London Interbank Offered Rate (LIBOR), based on British Pound (in Percent)"""

        ThreeMonthBasedOnBritishPound: str = "GBP3MTD156N"
        """3-Month London Interbank Offered Rate (LIBOR), based on British Pound (in Percent)"""

        OneWeekBasedOnJapaneseYen: str = "JPY1WKD156N"
        """1-Week London Interbank Offered Rate (LIBOR), based on Japanese Yen (in Percent)"""

        TwoMonthBasedOnJapaneseYen: str = "JPY2MTD156N"
        """2-Month London Interbank Offered Rate (LIBOR), based on Japanese Yen (in Percent)"""

        SixMonthBasedOnSwissFranc: str = "CHF6MTD156N"
        """6-Month London Interbank Offered Rate (LIBOR), based on Swiss Franc (in Percent)"""

        ThreeMonthBasedOnSwissFranc: str = "CHF3MTD156N"
        """3-Month London Interbank Offered Rate (LIBOR), based on Swiss Franc (in Percent)"""

        OneMonthBasedOnUSD: str = "USD1MTD156N"
        """1-Month London Interbank Offered Rate (LIBOR), based on U.S. Dollar (in Percent)"""

        TwelveMonthBasedOnSwissFranc: str = "CHF12MD156N"
        """12-Month London Interbank Offered Rate (LIBOR), based on Swiss Franc (in Percent)"""

        TwelveMonthBasedOnUSD: str = "USD12MD156N"
        """12-Month London Interbank Offered Rate (LIBOR), based on U.S. Dollar (in Percent)"""

        OneMonthBasedOnSwissFranc: str = "CHF1MTD156N"
        """1-Month London Interbank Offered Rate (LIBOR), based on Swiss Franc (in Percent)"""

        OneWeekBasedOnSwissFranc: str = "CHF1WKD156N"
        """1-Week London Interbank Offered Rate (LIBOR), based on Swiss Franc (in Percent)"""

        TwoMonthBasedOnSwissFranc: str = "CHF2MTD156N"
        """2-Month London Interbank Offered Rate (LIBOR), based on Swiss Franc (in Percent)"""

        TwelveMonthBasedOnEuro: str = "EUR12MD156N"
        """12-Month London Interbank Offered Rate (LIBOR), based on Euro (in Percent)"""

        SixMonthBasedOnBritishPound: str = "GBP6MTD156N"
        """6-Month London Interbank Offered Rate (LIBOR), based on British Pound (in Percent)"""

        OneMonthBasedOnEuro: str = "EUR1MTD156N"
        """1-Month London Interbank Offered Rate (LIBOR), based on Euro (in Percent)"""

        TwoMonthBasedOnEuro: str = "EUR2MTD156N"
        """2-Month London Interbank Offered Rate (LIBOR), based on Euro (in Percent)"""

        ThreeMonthBasedOnEuro: str = "EUR3MTD156N"
        """3-Month London Interbank Offered Rate (LIBOR), based on Euro (in Percent)"""

        SixMonthBasedOnEuro: str = "EUR6MTD156N"
        """6-Month London Interbank Offered Rate (LIBOR), based on Euro (in Percent)"""

        OvernightBasedOnEuro: str = "EURONTD156N"
        """Overnight London Interbank Offered Rate (LIBOR), based on Euro (in Percent)"""

        OneWeekBasedOnUSD: str = "USD1WKD156N"
        """1-Week London Interbank Offered Rate (LIBOR), based on U.S. Dollar (in Percent)"""

        TwoMonthBasedOnUSD: str = "USD2MTD156N"
        """2-Month London Interbank Offered Rate (LIBOR), based on U.S. Dollar (in Percent)"""

        ThreeMonthBasedOnUSD: str = "USD3MTD156N"
        """3-Month London Interbank Offered Rate (LIBOR), based on U.S. Dollar (in Percent)"""

        OvernightBasedOnUSD: str = "USDONTD156N"
        """Overnight London Interbank Offered Rate (LIBOR), based on U.S. Dollar (in Percent)"""

        OneWeekBasedOnEuro: str = "EUR1WKD156N"
        """1-Week London Interbank Offered Rate (LIBOR), based on Euro (in Percent)"""

        OvernightBasedOnBritishPound: str = "GBPONTD156N"
        """Overnight London Interbank Offered Rate (LIBOR), based on British Pound (in Percent)"""

    class OECDRecessionIndicators(System.Object):
        """
        These time series is an interpretation of Organisation of Economic Development (OECD) Composite Leading Indicators: Reference Turning Points and Component Series data, which can be found at http://www.oecd.org/std/leading-indicators/oecdcompositeleadingindicatorsreferenceturningpointsandcomponentseries.htm. The OECD identifies months of turning points without designating a date within the month that turning points occurred. The dummy variable adopts an arbitrary convention that the turning point occurred at a specific date within the month. The arbitrary convention does not reflect any judgment on this issue by the OECD. Our time series is composed of dummy variables that represent periods of expansion and recession. A value of 1 is a recessionary period, while a value of 0 is an expansionary period. For this time series, the recession begins on the 15th day of the month of the peak and ends on the 15th day of the month of the trough. This time series is a disaggregation of the monthly series. For more options on recession shading, see the note and links below.
        The recession shading data that we provide initially comes from the source as a list of dates that are either an economic peak or trough. We interpret dates into recession shading data using one of three arbitrary methods. All of our recession shading data is available using all three interpretations. The period between a peak and trough is always shaded as a recession. The peak and trough are collectively extrema. Depending on the application, the extrema, both individually and collectively, may be included in the recession period in whole or in part. In situations where a portion of a period is included in the recession, the whole period is deemed to be included in the recession period.
        The first interpretation, known as the midpoint method, is to show a recession from the midpoint of the peak through the midpoint of the trough for monthly and quarterly data. For daily data, the recession begins on the 15th of the month of the peak and ends on the 15th of the month of the trough. Daily data is a disaggregation of monthly data. For monthly and quarterly data, the entire peak and trough periods are included in the recession shading. This method shows the maximum number of periods as a recession for monthly and quarterly data. The Federal Reserve Bank of St. Louis uses this method in its own publications. The midpoint method is used for this series.
        The second interpretation, known as the trough method, is to show a recession from the period following the peak through the trough (i.e. the peak is not included in the recession shading, but the trough is). For daily data, the recession begins on the first day of the first month following the peak and ends on the last day of the month of the trough. Daily data is a disaggregation of monthly data. The trough method is used when displaying data on FRED graphs. A version of this time series represented using the trough method can be found at:
        The third interpretation, known as the peak method, is to show a recession from the period of the peak to the trough (i.e. the peak is included in the recession shading, but the trough is not). For daily data, the recession begins on the first day of the month of the peak and ends on the last day of the month preceding the trough. Daily data is a disaggregation of monthly data. A version of this time series represented using the peak method can be found at:
        The OECD CLI system is based on the "growth cycle" approach, where business cycles and turning points are measured and identified in the deviation-from-trend series. The main reference series used in the OECD CLI system for the majority of countries is industrial production (IIP) covering all industry sectors excluding construction. This series is used because of its cyclical sensitivity and monthly availability, while the broad based Gross Domestic Product (GDP) is used to supplement the IIP series for identification of the final reference turning points in the growth cycle.
        Zones aggregates of the CLIs and the reference series are calculated as weighted averages of the corresponding zone member series (i.e. CLIs and IIPs).
        Up to December 2008 the turning points chronologies shown for regional/zone area aggregates or individual countries are determined by the rules established by the National Bureau of Economic Research (NBER) in the United States, which have been formalized and incorporated in a computer routine (Bry and Boschan) and included in the Phase-Average Trend (PAT) de-trending procedure. Starting from December 2008 the turning point detection algorithm is decoupled from the de-trending procedure, and is a simplified version of the original Bry and Boschan routine. (The routine parses local minima and maxima in the cycle series and applies censor rules to guarantee alternating peaks and troughs, as well as phase and cycle length constraints.)
        The components of the CLI are time series which exhibit leading relationship with the reference series (IIP) at turning points. Country CLIs are compiled by combining de-trended smoothed and normalized components. The component series for each country are selected based on various criteria such as economic significance; cyclical behavior; data quality; timeliness and availability.
        OECD data should be cited as follows: OECD Composite Leading Indicators, "Composite Leading Indicators: Reference Turning Points and Component Series", http://www.oecd.org/std/leading-indicators/oecdcompositeleadingindicatorsreferenceturningpointsandcomponentseries.htm
        """

        FourBigEuropeanCountriesFromPeakThroughTheTrough: str = "4BIGEURORECDM"
        """OECD based Recession Indicators for Four Big European Countries from the Peak through the Trough (in +1 or 0)"""

        AustraliaFromPeakThroughTheTrough: str = "AUSRECDM"
        """OECD based Recession Indicators for Australia from the Peak through the Trough (in +1 or 0)"""

        AustriaFromPeakThroughTheTrough: str = "AUTRECDM"
        """OECD based Recession Indicators for Austria from the Peak through the Trough (in +1 or 0)"""

        BelgiumFromPeakThroughTheTrough: str = "BELRECDM"
        """OECD based Recession Indicators for Belgium from the Peak through the Trough (in +1 or 0)"""

        BrazilFromPeakThroughTheTrough: str = "BRARECDM"
        """OECD based Recession Indicators for Brazil from the Peak through the Trough (in +1 or 0)"""

        CanadaFromPeakThroughTheTrough: str = "CANRECDM"
        """OECD based Recession Indicators for Canada from the Peak through the Trough (in +1 or 0)"""

        SwitzerlandFromPeakThroughTheTrough: str = "CHERECDM"
        """OECD based Recession Indicators for Switzerland from the Peak through the Trough (in +1 or 0)"""

        ChileFromPeakThroughTheTrough: str = "CHLRECDM"
        """OECD based Recession Indicators for Chile from the Peak through the Trough (in +1 or 0)"""

        ChinaFromPeakThroughTheTrough: str = "CHNRECDM"
        """OECD based Recession Indicators for China from the Peak through the Trough (in +1 or 0)"""

        CzechRepublicFromPeakThroughTheTrough: str = "CZERECDM"
        """OECD based Recession Indicators for the Czech Republic from the Peak through the Trough (in +1 or 0)"""

        GermanyFromPeakThroughTheTrough: str = "DEURECDM"
        """OECD based Recession Indicators for Germany from the Peak through the Trough (in +1 or 0)"""

        DenmarkFromPeakThroughTheTrough: str = "DNKRECDM"
        """OECD based Recession Indicators for Denmark from the Peak through the Trough (in +1 or 0)"""

        SpainFromPeakThroughTheTrough: str = "ESPRECDM"
        """OECD based Recession Indicators for Spain from the Peak through the Trough (in +1 or 0)"""

        EstoniaFromPeakThroughTheTrough: str = "ESTRECDM"
        """OECD based Recession Indicators for Estonia from the Peak through the Trough (in +1 or 0)"""

        EuroAreaFromPeakThroughTheTrough: str = "EURORECDM"

        FinlandFromPeakThroughTheTrough: str = "FINRECDM"
        """OECD based Recession Indicators for Finland from the Peak through the Trough (in +1 or 0)"""

        FranceFromPeakThroughTheTrough: str = "FRARECDM"
        """OECD based Recession Indicators for France from the Peak through the Trough (in +1 or 0)"""

        UnitedKingdomFromPeakThroughTheTrough: str = "GBRRECDM"
        """OECD based Recession Indicators for the United Kingdom from the Peak through the Trough (in +1 or 0)"""

        GreeceFromPeakThroughTheTrough: str = "GRCRECDM"
        """OECD based Recession Indicators for Greece from the Peak through the Trough (in +1 or 0)"""

        HungaryFromPeakThroughTheTrough: str = "HUNRECDM"
        """OECD based Recession Indicators for Hungary from the Peak through the Trough (in +1 or 0)"""

        IndonesiaFromPeakThroughTheTrough: str = "IDNRECDM"
        """OECD based Recession Indicators for Indonesia from the Peak through the Trough (in +1 or 0)"""

        IndiaFromPeakThroughTheTrough: str = "INDRECDM"
        """OECD based Recession Indicators for India from the Peak through the Trough (in +1 or 0)"""

        IrelandFromPeakThroughTheTrough: str = "IRLRECDM"
        """OECD based Recession Indicators for Ireland from the Peak through the Trough (in +1 or 0)"""

        IsraelFromPeakThroughTheTrough: str = "ISRRECDM"
        """OECD based Recession Indicators for Israel from the Peak through the Trough (in +1 or 0)"""

        ItalyFromPeakThroughTheTrough: str = "ITARECDM"
        """OECD based Recession Indicators for Italy from the Peak through the Trough (in +1 or 0)"""

        JapanFromPeakThroughTheTrough: str = "JPNRECDM"
        """OECD based Recession Indicators for Japan from the Peak through the Trough (in +1 or 0)"""

        KoreaFromPeakThroughTheTrough: str = "KORRECDM"
        """OECD based Recession Indicators for Korea from the Peak through the Trough (in +1 or 0)"""

        LuxembourgFromPeakThroughTheTrough: str = "LUXRECDM"
        """OECD based Recession Indicators for Luxembourg from the Peak through the Trough (in +1 or 0)"""

        MajorFiveAsiaFromPeakThroughTheTrough: str = "MAJOR5ASIARECDM"
        """OECD based Recession Indicators for Major 5 Asia from the Peak through the Trough (in +1 or 0)"""

        MexicoFromPeakThroughTheTrough: str = "MEXRECDM"
        """OECD based Recession Indicators for Mexico from the Peak through the Trough (in +1 or 0)"""

        MajorSevenCountriesFromPeakThroughTheTrough: str = "MSCRECDM"
        """OECD based Recession Indicators for Major Seven Countries from the Peak through the Trough (in +1 or 0)"""

        NAFTAAreaFromPeakThroughTheTrough: str = "NAFTARECDM"
        """OECD based Recession Indicators for NAFTA Area from the Peak through the Trough (in +1 or 0)"""

        NetherlandsFromPeakThroughTheTrough: str = "NDLRECDM"
        """OECD based Recession Indicators for Netherlands from the Peak through the Trough (in +1 or 0)"""

        NorwayFromPeakThroughTheTrough: str = "NORRECDM"
        """OECD based Recession Indicators for Norway from the Peak through the Trough (in +1 or 0)"""

        NewZealandFromPeakThroughTheTrough: str = "NZLRECDM"
        """OECD based Recession Indicators for New Zealand from the Peak through the Trough (in +1 or 0)"""

        OECDEuropeFromPeakThroughTheTrough: str = "OECDEUROPERECDM"
        """OECD based Recession Indicators for OECD Europe from the Peak through the Trough (in +1 or 0)"""

        OECDAndNonmemberEconomiesFromPeakThroughTheTrough: str = "OECDNMERECDM"
        """OECD based Recession Indicators for OECD and Non-member Economies from the Peak through the Trough (in +1 or 0)"""

        OECDTotalAreaFromPeakThroughTheTrough: str = "OECDRECDM"
        """OECD based Recession Indicators for the OECD Total Area from the Peak through the Trough (in +1 or 0)"""

        PolandFromPeakThroughTheTrough: str = "POLRECDM"
        """OECD based Recession Indicators for Poland from the Peak through the Trough (in +1 or 0)"""

        PortugalFromPeakThroughTheTrough: str = "PRTRECDM"
        """OECD based Recession Indicators for Portugal from the Peak through the Trough (in +1 or 0)"""

        RussianFederationFromPeakThroughTheTrough: str = "RUSRECDM"
        """OECD based Recession Indicators for Russian Federation from the Peak through the Trough (in +1 or 0)"""

        SlovakRepublicFromPeakThroughTheTrough: str = "SVKRECDM"
        """OECD based Recession Indicators for the Slovak Republic from the Peak through the Trough (in +1 or 0)"""

        SloveniaFromPeakThroughTheTrough: str = "SVNRECDM"
        """OECD based Recession Indicators for Slovenia from the Peak through the Trough (in +1 or 0)"""

        SwedenFromPeakThroughTheTrough: str = "SWERECDM"
        """OECD based Recession Indicators for Sweden from the Peak through the Trough (in +1 or 0)"""

        TurkeyFromPeakThroughTheTrough: str = "TURRECDM"
        """OECD based Recession Indicators for Turkey from the Peak through the Trough (in +1 or 0)"""

        UnitedStatesFromPeakThroughTheTrough: str = "USARECDM"
        """OECD based Recession Indicators for the United States from the Peak through the Trough (in +1 or 0)"""

        SouthAfricaFromPeakThroughTheTrough: str = "ZAFRECDM"
        """OECD based Recession Indicators for South Africa from the Peak through the Trough (in +1 or 0)"""

        FourBigEuropeanCountriesFromPeriodFollowingPeakThroughTheTrough: str = "4BIGEURORECD"
        """OECD based Recession Indicators for Four Big European Countries from the Period following the Peak through the Trough (in +1 or 0)"""

        AustraliaFromPeriodFollowingPeakThroughTheTrough: str = "AUSRECD"
        """OECD based Recession Indicators for Australia from the Period following the Peak through the Trough (in +1 or 0)"""

        AustriaFromPeriodFollowingPeakThroughTheTrough: str = "AUTRECD"
        """OECD based Recession Indicators for Austria from the Period following the Peak through the Trough (in +1 or 0)"""

        BelgiumFromPeriodFollowingPeakThroughTheTrough: str = "BELRECD"
        """OECD based Recession Indicators for Belgium from the Period following the Peak through the Trough (in +1 or 0)"""

        BrazilFromPeriodFollowingPeakThroughTheTrough: str = "BRARECD"
        """OECD based Recession Indicators for Brazil from the Period following the Peak through the Trough (in +1 or 0)"""

        CanadaFromPeriodFollowingPeakThroughTheTrough: str = "CANRECD"
        """OECD based Recession Indicators for Canada from the Period following the Peak through the Trough (in +1 or 0)"""

        SwitzerlandFromPeriodFollowingPeakThroughTheTrough: str = "CHERECD"
        """OECD based Recession Indicators for Switzerland from the Period following the Peak through the Trough (in +1 or 0)"""

        ChileFromPeriodFollowingPeakThroughTheTrough: str = "CHLRECD"
        """OECD based Recession Indicators for Chile from the Period following the Peak through the Trough (in +1 or 0)"""

        ChinaFromPeriodFollowingPeakThroughTheTrough: str = "CHNRECD"
        """OECD based Recession Indicators for China from the Period following the Peak through the Trough (in +1 or 0)"""

        CzechRepublicFromPeriodFollowingPeakThroughTheTrough: str = "CZERECD"
        """OECD based Recession Indicators for the Czech Republic from the Period following the Peak through the Trough (in +1 or 0)"""

        GermanyFromPeriodFollowingPeakThroughTheTrough: str = "DEURECD"
        """OECD based Recession Indicators for Germany from the Period following the Peak through the Trough (in +1 or 0)"""

        DenmarkFromPeriodFollowingPeakThroughTheTrough: str = "DNKRECD"
        """OECD based Recession Indicators for Denmark from the Period following the Peak through the Trough (in +1 or 0)"""

        SpainFromPeriodFollowingPeakThroughTheTrough: str = "ESPRECD"
        """OECD based Recession Indicators for Spain from the Period following the Peak through the Trough (in +1 or 0)"""

        EstoniaFromPeriodFollowingPeakThroughTheTrough: str = "ESTRECD"
        """OECD based Recession Indicators for Estonia from the Period following the Peak through the Trough (in +1 or 0)"""

        EuroAreaFromPeriodFollowingPeakThroughTheTrough: str = "EURORECD"
        """OECD based Recession Indicators for Euro Area from the Period following the Peak through the Trough (in +1 or 0)"""

        FinlandFromPeriodFollowingPeakThroughTheTrough: str = "FINRECD"
        """OECD based Recession Indicators for Finland from the Period following the Peak through the Trough (in +1 or 0)"""

        FranceFromPeriodFollowingPeakThroughTheTrough: str = "FRARECD"
        """OECD based Recession Indicators for France from the Period following the Peak through the Trough (in +1 or 0)"""

        UnitedKingdomFromPeriodFollowingPeakThroughTheTrough: str = "GBRRECD"
        """OECD based Recession Indicators for the United Kingdom from the Period following the Peak through the Trough (in +1 or 0)"""

        GreeceFromPeriodFollowingPeakThroughTheTrough: str = "GRCRECD"
        """OECD based Recession Indicators for Greece from the Period following the Peak through the Trough (in +1 or 0)"""

        HungaryFromPeriodFollowingPeakThroughTheTrough: str = "HUNRECD"
        """OECD based Recession Indicators for Hungary from the Period following the Peak through the Trough (in +1 or 0)"""

        IndonesiaFromPeriodFollowingPeakThroughTheTrough: str = "IDNRECD"
        """OECD based Recession Indicators for Indonesia from the Period following the Peak through the Trough (in +1 or 0)"""

        IndiaFromPeriodFollowingPeakThroughTheTrough: str = "INDRECD"
        """OECD based Recession Indicators for India from the Period following the Peak through the Trough (in +1 or 0)"""

        IrelandFromPeriodFollowingPeakThroughTheTrough: str = "IRLRECD"
        """OECD based Recession Indicators for Ireland from the Period following the Peak through the Trough (in +1 or 0)"""

        IsraelFromPeriodFollowingPeakThroughTheTrough: str = "ISRRECD"
        """OECD based Recession Indicators for Israel from the Period following the Peak through the Trough (in +1 or 0)"""

        ItalyFromPeriodFollowingPeakThroughTheTrough: str = "ITARECD"
        """OECD based Recession Indicators for Italy from the Period following the Peak through the Trough (in +1 or 0)"""

        JapanFromPeriodFollowingPeakThroughTheTrough: str = "JPNRECD"
        """OECD based Recession Indicators for Japan from the Period following the Peak through the Trough (in +1 or 0)"""

        KoreaFromPeriodFollowingPeakThroughTheTrough: str = "KORRECD"
        """OECD based Recession Indicators for Korea from the Period following the Peak through the Trough (in +1 or 0)"""

        LuxembourgFromPeriodFollowingPeakThroughTheTrough: str = "LUXRECD"
        """OECD based Recession Indicators for Luxembourg from the Period following the Peak through the Trough (in +1 or 0)"""

        MajorFiveAsiaFromPeriodFollowingPeakThroughTheTrough: str = "MAJOR5ASIARECD"
        """OECD based Recession Indicators for Major 5 Asia from the Period following the Peak through the Trough (in +1 or 0)"""

        MexicoFromPeriodFollowingPeakThroughTheTrough: str = "MEXRECD"
        """OECD based Recession Indicators for Mexico from the Period following the Peak through the Trough (in +1 or 0)"""

        MajorSevenCountriesFromPeriodFollowingPeakThroughTheTrough: str = "MSCRECD"
        """OECD based Recession Indicators for Major Seven Countries from the Period following the Peak through the Trough (in +1 or 0)"""

        NAFTAAreaFromPeriodFollowingPeakThroughTheTrough: str = "NAFTARECD"
        """OECD based Recession Indicators for NAFTA Area from the Period following the Peak through the Trough (in +1 or 0)"""

        NetherlandsFromPeriodFollowingPeakThroughTheTrough: str = "NDLRECD"
        """OECD based Recession Indicators for Netherlands from the Period following the Peak through the Trough (in +1 or 0)"""

        NorwayFromPeriodFollowingPeakThroughTheTrough: str = "NORRECD"
        """OECD based Recession Indicators for Norway from the Period following the Peak through the Trough (in +1 or 0)"""

        NewZealandFromPeriodFollowingPeakThroughTheTrough: str = "NZLRECD"
        """OECD based Recession Indicators for New Zealand from the Period following the Peak through the Trough (in +1 or 0)"""

        OECDEuropeFromPeriodFollowingPeakThroughTheTrough: str = "OECDEUROPERECD"
        """OECD based Recession Indicators for OECD Europe from the Period following the Peak through the Trough (in +1 or 0)"""

        OECDandNonmemberEconomiesFromPeriodFollowingPeakThroughTheTrough: str = "OECDNMERECD"
        """OECD based Recession Indicators for OECD and Non-member Economies from the Period following the Peak through the Trough (in +1 or 0)"""

        OECDTotalAreaFromPeriodFollowingPeakThroughTheTrough: str = "OECDRECD"
        """OECD based Recession Indicators for the OECD Total Area from the Period following the Peak through the Trough (in +1 or 0)"""

        PolandFromPeriodFollowingPeakThroughTheTrough: str = "POLRECD"
        """OECD based Recession Indicators for Poland from the Period following the Peak through the Trough (in +1 or 0)"""

        PortugalFromPeriodFollowingPeakThroughTheTrough: str = "PRTRECD"

        RussianFederationFromPeriodFollowingPeakThroughTheTrough: str = "RUSRECD"
        """OECD based Recession Indicators for Russian Federation from the Period following the Peak through the Trough (in +1 or 0)"""

        SlovakRepublicFromPeriodFollowingPeakThroughTheTrough: str = "SVKRECD"
        """OECD based Recession Indicators for the Slovak Republic from the Period following the Peak through the Trough (in +1 or 0)"""

        SloveniaFromPeriodFollowingPeakThroughTheTrough: str = "SVNRECD"
        """OECD based Recession Indicators for Slovenia from the Period following the Peak through the Trough (in +1 or 0)"""

        SwedenFromPeriodFollowingPeakThroughTheTrough: str = "SWERECD"
        """OECD based Recession Indicators for Sweden from the Period following the Peak through the Trough (in +1 or 0)"""

        TurkeyFromPeriodFollowingPeakThroughTheTrough: str = "TURRECD"
        """OECD based Recession Indicators for Turkey from the Period following the Peak through the Trough (in +1 or 0)"""

        UnitedStatesFromPeriodFollowingPeakThroughTheTrough: str = "USARECD"
        """OECD based Recession Indicators for the United States from the Period following the Peak through the Trough (in +1 or 0)"""

        SouthAfricaFromPeriodFollowingPeakThroughTheTrough: str = "ZAFRECD"
        """OECD based Recession Indicators for South Africa from the Period following the Peak through the Trough (in +1 or 0)"""

        FourBigEuropeanCountriesFromPeakThroughThePeriodPrecedingtheTrough: str = "4BIGEURORECDP"
        """OECD based Recession Indicators for Four Big European Countries from the Peak through the Period preceding the Trough (in +1 or 0)"""

        AustraliaFromPeakThroughThePeriodPrecedingtheTrough: str = "AUSRECDP"
        """OECD based Recession Indicators for Australia from the Peak through the Period preceding the Trough (in +1 or 0)"""

        AustriaFromPeakThroughThePeriodPrecedingtheTrough: str = "AUTRECDP"
        """OECD based Recession Indicators for Austria from the Peak through the Period preceding the Trough (in +1 or 0)"""

        BelgiumFromPeakThroughThePeriodPrecedingtheTrough: str = "BELRECDP"
        """OECD based Recession Indicators for Belgium from the Peak through the Period preceding the Trough (in +1 or 0)"""

        BrazilFromPeakThroughThePeriodPrecedingtheTrough: str = "BRARECDP"
        """OECD based Recession Indicators for Brazil from the Peak through the Period preceding the Trough (in +1 or 0)"""

        CanadaFromPeakThroughThePeriodPrecedingtheTrough: str = "CANRECDP"
        """OECD based Recession Indicators for Canada from the Peak through the Period preceding the Trough (in +1 or 0)"""

        SwitzerlandFromPeakThroughThePeriodPrecedingtheTrough: str = "CHERECDP"
        """OECD based Recession Indicators for Switzerland from the Peak through the Period preceding the Trough (in +1 or 0)"""

        ChileFromPeakThroughThePeriodPrecedingtheTrough: str = "CHLRECDP"
        """OECD based Recession Indicators for Chile from the Peak through the Period preceding the Trough (in +1 or 0)"""

        ChinaFromPeakThroughThePeriodPrecedingtheTrough: str = "CHNRECDP"
        """OECD based Recession Indicators for China from the Peak through the Period preceding the Trough (in +1 or 0)"""

        CzechRepublicFromPeakThroughThePeriodPrecedingtheTrough: str = "CZERECDP"
        """OECD based Recession Indicators for the Czech Republic from the Peak through the Period preceding the Trough (in +1 or 0)"""

        GermanyFromPeakThroughThePeriodPrecedingtheTrough: str = "DEURECDP"
        """OECD based Recession Indicators for Germany from the Peak through the Period preceding the Trough (in +1 or 0)"""

        DenmarkFromPeakThroughThePeriodPrecedingtheTrough: str = "DNKRECDP"
        """OECD based Recession Indicators for Denmark from the Peak through the Period preceding the Trough (in +1 or 0)"""

        SpainFromPeakThroughThePeriodPrecedingtheTrough: str = "ESPRECDP"
        """OECD based Recession Indicators for Spain from the Peak through the Period preceding the Trough (in +1 or 0)"""

        EstoniaFromPeakThroughThePeriodPrecedingtheTrough: str = "ESTRECDP"
        """OECD based Recession Indicators for Estonia from the Peak through the Period preceding the Trough (in +1 or 0)"""

        EuroAreaFromPeakThroughThePeriodPrecedingtheTrough: str = "EURORECDP"
        """OECD based Recession Indicators for Euro Area from the Peak through the Period preceding the Trough (in +1 or 0)"""

        FinlandFromPeakThroughThePeriodPrecedingtheTrough: str = "FINRECDP"
        """OECD based Recession Indicators for Finland from the Peak through the Period preceding the Trough (in +1 or 0)"""

        FranceFromPeakThroughThePeriodPrecedingtheTrough: str = "FRARECDP"
        """OECD based Recession Indicators for France from the Peak through the Period preceding the Trough (in +1 or 0)"""

        UnitedKingdomFromPeakThroughThePeriodPrecedingtheTrough: str = "GBRRECDP"
        """OECD based Recession Indicators for the United Kingdom from the Peak through the Period preceding the Trough (in +1 or 0)"""

        GreeceFromPeakThroughThePeriodPrecedingtheTrough: str = "GRCRECDP"
        """OECD based Recession Indicators for Greece from the Peak through the Period preceding the Trough (in +1 or 0)"""

        HungaryFromPeakThroughThePeriodPrecedingtheTrough: str = "HUNRECDP"
        """OECD based Recession Indicators for Hungary from the Peak through the Period preceding the Trough (in +1 or 0)"""

        IndonesiaFromPeakThroughThePeriodPrecedingtheTrough: str = "IDNRECDP"
        """OECD based Recession Indicators for Indonesia from the Peak through the Period preceding the Trough (in +1 or 0)"""

        IndiaFromPeakThroughThePeriodPrecedingtheTrough: str = "INDRECDP"
        """OECD based Recession Indicators for India from the Peak through the Period preceding the Trough (in +1 or 0)"""

        IrelandFromPeakThroughThePeriodPrecedingtheTrough: str = "IRLRECDP"
        """OECD based Recession Indicators for Ireland from the Peak through the Period preceding the Trough (in +1 or 0)"""

        IsraelFromPeakThroughThePeriodPrecedingtheTrough: str = "ISRRECDP"
        """OECD based Recession Indicators for Israel from the Peak through the Period preceding the Trough (in +1 or 0)"""

        ItalyFromPeakThroughThePeriodPrecedingtheTrough: str = "ITARECDP"
        """OECD based Recession Indicators for Italy from the Peak through the Period preceding the Trough (in +1 or 0)"""

        JapanFromPeakThroughThePeriodPrecedingtheTrough: str = "JPNRECDP"
        """OECD based Recession Indicators for Japan from the Peak through the Period preceding the Trough (in +1 or 0)"""

        KoreaFromPeakThroughThePeriodPrecedingtheTrough: str = "KORRECDP"
        """OECD based Recession Indicators for Korea from the Peak through the Period preceding the Trough (in +1 or 0)"""

        LuxembourgFromPeakThroughThePeriodPrecedingtheTrough: str = "LUXRECDP"
        """OECD based Recession Indicators for Luxembourg from the Peak through the Period preceding the Trough (in +1 or 0)"""

        MajorFiveAsiaFromPeakThroughThePeriodPrecedingtheTrough: str = "MAJOR5ASIARECDP"
        """OECD based Recession Indicators for Major 5 Asia from the Peak through the Period preceding the Trough (in +1 or 0)"""

        MexicoFromPeakThroughThePeriodPrecedingtheTrough: str = "MEXRECDP"
        """OECD based Recession Indicators for Mexico from the Peak through the Period preceding the Trough (in +1 or 0)"""

        MajorSevenCountriesFromPeakThroughThePeriodPrecedingtheTrough: str = "MSCRECDP"
        """OECD based Recession Indicators for Major Seven Countries from the Peak through the Period preceding the Trough (in +1 or 0)"""

        NAFTAAreaFromPeakThroughThePeriodPrecedingtheTrough: str = "NAFTARECDP"
        """OECD based Recession Indicators for NAFTA Area from the Peak through the Period preceding the Trough (in +1 or 0)"""

        NetherlandsFromPeakThroughThePeriodPrecedingtheTrough: str = "NDLRECDP"
        """OECD based Recession Indicators for Netherlands from the Peak through the Period preceding the Trough (in +1 or 0)"""

        NorwayFromPeakThroughThePeriodPrecedingtheTrough: str = "NORRECDP"
        """OECD based Recession Indicators for Norway from the Peak through the Period preceding the Trough (in +1 or 0)"""

        NewZealandFromPeakThroughThePeriodPrecedingtheTrough: str = "NZLRECDP"

        OECDEuropeFromPeakThroughThePeriodPrecedingtheTrough: str = "OECDEUROPERECDP"
        """OECD based Recession Indicators for OECD Europe from the Peak through the Period preceding the Trough (in +1 or 0)"""

        OECDandNonmemberEconomiesFromPeakThroughThePeriodPrecedingtheTrough: str = "OECDNMERECDP"
        """OECD based Recession Indicators for OECD and Non-member Economies from the Peak through the Period preceding the Trough (in +1 or 0)"""

        OECDTotalAreaFromPeakThroughThePeriodPrecedingtheTrough: str = "OECDRECDP"
        """OECD based Recession Indicators for the OECD Total Area from the Peak through the Period preceding the Trough (in +1 or 0)"""

        PolandFromPeakThroughThePeriodPrecedingtheTrough: str = "POLRECDP"
        """OECD based Recession Indicators for Poland from the Peak through the Period preceding the Trough (in +1 or 0)"""

        PortugalFromPeakThroughThePeriodPrecedingtheTrough: str = "PRTRECDP"
        """OECD based Recession Indicators for Portugal from the Peak through the Period preceding the Trough (in +1 or 0)"""

        RussianFederationFromPeakThroughThePeriodPrecedingtheTrough: str = "RUSRECDP"
        """OECD based Recession Indicators for Russian Federation from the Peak through the Period preceding the Trough (in +1 or 0)"""

        SlovakRepublicFromPeakThroughThePeriodPrecedingtheTrough: str = "SVKRECDP"
        """OECD based Recession Indicators for the Slovak Republic from the Peak through the Period preceding the Trough (in +1 or 0)"""

        SloveniaFromPeakThroughThePeriodPrecedingtheTrough: str = "SVNRECDP"
        """OECD based Recession Indicators for Slovenia from the Peak through the Period preceding the Trough (in +1 or 0)"""

        SwedenFromPeakThroughThePeriodPrecedingtheTrough: str = "SWERECDP"
        """OECD based Recession Indicators for Sweden from the Peak through the Period preceding the Trough (in +1 or 0)"""

        TurkeyFromPeakThroughThePeriodPrecedingtheTrough: str = "TURRECDP"
        """OECD based Recession Indicators for Turkey from the Peak through the Period preceding the Trough (in +1 or 0)"""

        UnitedStatesFromPeakThroughThePeriodPrecedingtheTrough: str = "USARECDP"
        """OECD based Recession Indicators for the United States from the Peak through the Period preceding the Trough (in +1 or 0)"""

        SouthAfricaFromPeakThroughThePeriodPrecedingtheTrough: str = "ZAFRECDP"
        """OECD based Recession Indicators for South Africa from the Peak through the Period preceding the Trough (in +1 or 0)"""

    DataSourceId: int
    """Data source ID"""

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the data
        
        :returns: A clone of the object.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def IsSparseData(self) -> bool:
        """
        Indicates whether the data is sparse.
        If true, we disable logging for missing files
        
        :returns: true.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: Line of data
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: New instance of USEnergy.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates whether the data source is tied
        to an underlying symbol and requires that corporate
        events be applied to it as well, such as renames and delistings
        
        :returns: false.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...

    def ToString(self) -> str:
        """Converts the instance to string"""
        ...


class FredApi(QuantConnect.Data.BaseData):
    """This class has no documentation."""

    class Observation(System.Object):
        """This class has no documentation."""

        @property
        def RealtimeStart(self) -> str:
            ...

        @RealtimeStart.setter
        def RealtimeStart(self, value: str):
            ...

        @property
        def RealtimeEnd(self) -> str:
            ...

        @RealtimeEnd.setter
        def RealtimeEnd(self, value: str):
            ...

        @property
        def Date(self) -> datetime.datetime:
            ...

        @Date.setter
        def Date(self, value: datetime.datetime):
            ...

        @property
        def Value(self) -> str:
            ...

        @Value.setter
        def Value(self, value: str):
            ...

    DataSourceId: int
    """Data source ID"""

    @property
    def RealtimeStart(self) -> str:
        ...

    @RealtimeStart.setter
    def RealtimeStart(self, value: str):
        ...

    @property
    def RealtimeEnd(self) -> str:
        ...

    @RealtimeEnd.setter
    def RealtimeEnd(self, value: str):
        ...

    @property
    def ObservationStart(self) -> str:
        ...

    @ObservationStart.setter
    def ObservationStart(self, value: str):
        ...

    @property
    def ObservationEnd(self) -> str:
        ...

    @ObservationEnd.setter
    def ObservationEnd(self, value: str):
        ...

    @property
    def Units(self) -> str:
        ...

    @Units.setter
    def Units(self, value: str):
        ...

    @property
    def OutputType(self) -> int:
        ...

    @OutputType.setter
    def OutputType(self, value: int):
        ...

    @property
    def FileType(self) -> str:
        ...

    @FileType.setter
    def FileType(self, value: str):
        ...

    @property
    def OrderBy(self) -> str:
        ...

    @OrderBy.setter
    def OrderBy(self, value: str):
        ...

    @property
    def SortOrder(self) -> str:
        ...

    @SortOrder.setter
    def SortOrder(self, value: str):
        ...

    @property
    def Count(self) -> int:
        ...

    @Count.setter
    def Count(self, value: int):
        ...

    @property
    def Offset(self) -> int:
        ...

    @Offset.setter
    def Offset(self, value: int):
        ...

    @property
    def Limit(self) -> int:
        ...

    @Limit.setter
    def Limit(self, value: int):
        ...

    @property
    def Observations(self) -> System.Collections.Generic.IList[QuantConnect.DataSource.FredApi.Observation]:
        ...

    @Observations.setter
    def Observations(self, value: System.Collections.Generic.IList[QuantConnect.DataSource.FredApi.Observation]):
        ...

    AuthCode: str

    IsAuthCodeSet: bool
    """Returns true if the FRED API token has been set."""

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, content: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Readers the specified configuration.
        
        :param config: The configuration.
        :param content: The content.
        :param date: The date.
        :param isLiveMode: if set to true [is live mode].
        """
        ...

    @staticmethod
    def SetAuthCode(authCode: str) -> None:
        """
        Sets the EIA API token.
        
        :param authCode: The EIA API token
        """
        ...


class TransactionDirectionJsonConverter(QuantConnect.Util.TypeChangeJsonConverter[QuantConnect.Orders.OrderDirection, str]):
    """Converts Quiver Quantitative TransactionDirection to OrderDirection"""

    @overload
    def Convert(self, value: QuantConnect.Orders.OrderDirection) -> str:
        """
        Convert OrderDirection to string
        
        This method is protected.
        
        :param value: OrderDirection to convert
        :returns: Resulting string.
        """
        ...

    @overload
    def Convert(self, value: str) -> int:
        """
        Convert string to OrderDirection
        
        This method is protected.
        
        :param value: string to convert
        :returns: Resulting OrderDirection. This method returns the int value of a member of the QuantConnect.Orders.OrderDirection enum.
        """
        ...


class Congress(System.Enum):
    """United States of America Legislative Branch House of Congress"""

    Senate = 0
    """The United States Senate"""

    Representatives = 1
    """The United States House of Representatives"""


class QuiverCongress(QuantConnect.Data.BaseData):
    """Personal stock transactions by U.S. Representatives"""

    DataSourceId: int
    """Data source ID"""

    @property
    def ReportDate(self) -> typing.Optional[datetime.datetime]:
        """The date the transaction was reported. Value will always exist."""
        ...

    @ReportDate.setter
    def ReportDate(self, value: typing.Optional[datetime.datetime]):
        """The date the transaction was reported. Value will always exist."""
        ...

    @property
    def TransactionDate(self) -> datetime.datetime:
        """The date the transaction took place"""
        ...

    @TransactionDate.setter
    def TransactionDate(self, value: datetime.datetime):
        """The date the transaction took place"""
        ...

    @property
    def Representative(self) -> str:
        """The Representative making the transaction"""
        ...

    @Representative.setter
    def Representative(self, value: str):
        """The Representative making the transaction"""
        ...

    @property
    def Transaction(self) -> int:
        """
        The type of transaction
        
        This property contains the int value of a member of the QuantConnect.Orders.OrderDirection enum.
        """
        ...

    @Transaction.setter
    def Transaction(self, value: int):
        """
        The type of transaction
        
        This property contains the int value of a member of the QuantConnect.Orders.OrderDirection enum.
        """
        ...

    @property
    def Amount(self) -> typing.Optional[float]:
        """The amount of the transaction (in USD)"""
        ...

    @Amount.setter
    def Amount(self, value: typing.Optional[float]):
        """The amount of the transaction (in USD)"""
        ...

    @property
    def House(self) -> int:
        """
        The House of Congress that the trader belongs to
        
        This property contains the int value of a member of the QuantConnect.DataSource.Congress enum.
        """
        ...

    @House.setter
    def House(self, value: int):
        """
        The House of Congress that the trader belongs to
        
        This property contains the int value of a member of the QuantConnect.DataSource.Congress enum.
        """
        ...

    @property
    def Period(self) -> datetime.timedelta:
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @Period.setter
    def Period(self, value: datetime.timedelta):
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @overload
    def __init__(self) -> None:
        """Required for successful Json.NET deserialization"""
        ...

    @overload
    def __init__(self, csvLine: str) -> None:
        """
        Creates a new instance of QuiverCongress from a CSV line
        
        :param csvLine: CSV line
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The DateTimeZone of this data type.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the Subscription Data Source gained from the URL
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Subscription Data Source.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reader converts each line of the data source into BaseData objects.
        
        :param config: Subscription data config setup object
        :param line: Content of the source document
        :param date: Date of the requested data
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Quiver Congress object.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates if there is support for mapping
        
        :returns: True indicates mapping should be used.
        """
        ...

    def ToString(self) -> str:
        """Formats a string with the Quiver Congress information."""
        ...


class QuiverQuantCongressUniverse(QuantConnect.Data.BaseData):
    """Universe Selection helper class for QuiverQuant Congress dataset"""

    @property
    def ReportDate(self) -> datetime.datetime:
        """ReportDate"""
        ...

    @property
    def TransactionDate(self) -> datetime.datetime:
        """TransactionDate"""
        ...

    @TransactionDate.setter
    def TransactionDate(self, value: datetime.datetime):
        """TransactionDate"""
        ...

    @property
    def Representative(self) -> str:
        """Representative"""
        ...

    @Representative.setter
    def Representative(self, value: str):
        """Representative"""
        ...

    @property
    def Transaction(self) -> int:
        """
        Month-over-month change in company's follower count
        
        This property contains the int value of a member of the QuantConnect.Orders.OrderDirection enum.
        """
        ...

    @Transaction.setter
    def Transaction(self, value: int):
        """
        Month-over-month change in company's follower count
        
        This property contains the int value of a member of the QuantConnect.Orders.OrderDirection enum.
        """
        ...

    @property
    def Amount(self) -> typing.Optional[float]:
        """The amount of the transaction (in USD)"""
        ...

    @Amount.setter
    def Amount(self, value: typing.Optional[float]):
        """The amount of the transaction (in USD)"""
        ...

    @property
    def House(self) -> int:
        """
        The House of Congress that the trader belongs to
        
        This property contains the int value of a member of the QuantConnect.DataSource.Congress enum.
        """
        ...

    @House.setter
    def House(self, value: int):
        """
        The House of Congress that the trader belongs to
        
        This property contains the int value of a member of the QuantConnect.DataSource.Congress enum.
        """
        ...

    @property
    def Period(self) -> datetime.timedelta:
        """Time passed between the date of the data and the time the data became available to us"""
        ...

    @Period.setter
    def Period(self, value: datetime.timedelta):
        """Time passed between the date of the data and the time the data became available to us"""
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        """Time the data became available"""
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """Time the data became available"""
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The NodaTime.DateTimeZone of this data type.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: Line of data
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: New instance.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...

    def ToString(self) -> str:
        """Converts the instance to string"""
        ...


class VIXCentralContango(QuantConnect.Data.BaseData):
    """VIXCentral Contango"""

    DataSourceId: int
    """Data source ID"""

    @property
    def FrontMonth(self) -> int:
        """The month of the front month contract (possible values: 1 - 12)"""
        ...

    @FrontMonth.setter
    def FrontMonth(self, value: int):
        """The month of the front month contract (possible values: 1 - 12)"""
        ...

    @property
    def F1(self) -> float:
        """Front month contract"""
        ...

    @F1.setter
    def F1(self, value: float):
        """Front month contract"""
        ...

    @property
    def F2(self) -> float:
        """Contract 1 month away from the front month contract"""
        ...

    @F2.setter
    def F2(self, value: float):
        """Contract 1 month away from the front month contract"""
        ...

    @property
    def F3(self) -> float:
        """Contract 2 months away from the front month contract"""
        ...

    @F3.setter
    def F3(self, value: float):
        """Contract 2 months away from the front month contract"""
        ...

    @property
    def F4(self) -> float:
        """Contract 3 months away from the front month contract"""
        ...

    @F4.setter
    def F4(self, value: float):
        """Contract 3 months away from the front month contract"""
        ...

    @property
    def F5(self) -> float:
        """Contract 4 months away from the front month contract"""
        ...

    @F5.setter
    def F5(self, value: float):
        """Contract 4 months away from the front month contract"""
        ...

    @property
    def F6(self) -> float:
        """Contract 5 months away from the front month contract"""
        ...

    @F6.setter
    def F6(self, value: float):
        """Contract 5 months away from the front month contract"""
        ...

    @property
    def F7(self) -> float:
        """Contract 6 months away from the front month contract"""
        ...

    @F7.setter
    def F7(self, value: float):
        """Contract 6 months away from the front month contract"""
        ...

    @property
    def F8(self) -> float:
        """Contract 7 months away from the front month contract"""
        ...

    @F8.setter
    def F8(self, value: float):
        """Contract 7 months away from the front month contract"""
        ...

    @property
    def F9(self) -> typing.Optional[float]:
        """Contract 8 months away from the front month contract"""
        ...

    @F9.setter
    def F9(self, value: typing.Optional[float]):
        """Contract 8 months away from the front month contract"""
        ...

    @property
    def F10(self) -> typing.Optional[float]:
        """Contract 9 months away from the front month contract"""
        ...

    @F10.setter
    def F10(self, value: typing.Optional[float]):
        """Contract 9 months away from the front month contract"""
        ...

    @property
    def F11(self) -> typing.Optional[float]:
        """Contract 10 months away from the front month contract"""
        ...

    @F11.setter
    def F11(self, value: typing.Optional[float]):
        """Contract 10 months away from the front month contract"""
        ...

    @property
    def F12(self) -> typing.Optional[float]:
        """Contract 11 months away from the front month contract"""
        ...

    @F12.setter
    def F12(self, value: typing.Optional[float]):
        """Contract 11 months away from the front month contract"""
        ...

    @property
    def Contango_F2_Minus_F1(self) -> float:
        """Percentage change between contract F2 and F1, calculated as: (F2 - F1) / F1"""
        ...

    @Contango_F2_Minus_F1.setter
    def Contango_F2_Minus_F1(self, value: float):
        """Percentage change between contract F2 and F1, calculated as: (F2 - F1) / F1"""
        ...

    @property
    def Contango_F7_Minus_F4(self) -> float:
        """Percentage change between contract F7 and F4, calculated as: (F7 - F4) / F4"""
        ...

    @Contango_F7_Minus_F4.setter
    def Contango_F7_Minus_F4(self, value: float):
        """Percentage change between contract F7 and F4, calculated as: (F7 - F4) / F4"""
        ...

    @property
    def Contango_F7_Minus_F4_Div_3(self) -> float:
        """Percentage change between contract F7 and F4 divided by 3, calculated as: ((F7 - F4) / F4) / 3"""
        ...

    @Contango_F7_Minus_F4_Div_3.setter
    def Contango_F7_Minus_F4_Div_3(self, value: float):
        """Percentage change between contract F7 and F4 divided by 3, calculated as: ((F7 - F4) / F4) / 3"""
        ...

    @property
    def Period(self) -> datetime.timedelta:
        """The timespan that each data point covers"""
        ...

    @Period.setter
    def Period(self, value: datetime.timedelta):
        """The timespan that each data point covers"""
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        """The ending time of the data point"""
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """The ending time of the data point"""
        ...

    def __init__(self) -> None:
        """Creates a new instance of the object"""
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """Gets the source location of the VIXCentral data"""
        ...

    def IsSparseData(self) -> bool:
        """
        Determines if data source is sparse
        
        :returns: false.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reads the data from the source and creates a BaseData instance
        
        :param config: Configuration
        :param line: Line of data
        :param date: Date we're requesting data for
        :param isLiveMode: Is live mode
        :returns: New BaseData instance to be used in the algorithm.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Determines whether the data source requires mapping
        
        :returns: false.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...

    def ToString(self) -> str:
        """
        Converts the instance to a string
        
        :returns: String containing open, high, low, close.
        """
        ...


class RegalyticsRegulatoryArticle(QuantConnect.Data.BaseData):
    """Regalytics Regulatory articles"""

    DataSourceId: int
    """Data source ID"""

    @property
    def Id(self) -> int:
        ...

    @Id.setter
    def Id(self, value: int):
        ...

    @property
    def Title(self) -> str:
        ...

    @Title.setter
    def Title(self, value: str):
        ...

    @property
    def Summary(self) -> str:
        ...

    @Summary.setter
    def Summary(self, value: str):
        ...

    @property
    def Status(self) -> str:
        ...

    @Status.setter
    def Status(self, value: str):
        ...

    @property
    def Classification(self) -> str:
        ...

    @Classification.setter
    def Classification(self, value: str):
        ...

    @property
    def FilingType(self) -> str:
        ...

    @FilingType.setter
    def FilingType(self, value: str):
        ...

    @property
    def InFederalRegister(self) -> bool:
        ...

    @InFederalRegister.setter
    def InFederalRegister(self, value: bool):
        ...

    @property
    def FederalRegisterNumber(self) -> str:
        ...

    @FederalRegisterNumber.setter
    def FederalRegisterNumber(self, value: str):
        ...

    @property
    def ProposedCommentsDueDate(self) -> typing.Optional[datetime.datetime]:
        ...

    @ProposedCommentsDueDate.setter
    def ProposedCommentsDueDate(self, value: typing.Optional[datetime.datetime]):
        ...

    @property
    def OriginalPublicationDate(self) -> typing.Optional[datetime.datetime]:
        ...

    @OriginalPublicationDate.setter
    def OriginalPublicationDate(self, value: typing.Optional[datetime.datetime]):
        ...

    @property
    def FederalRegisterPublicationDate(self) -> typing.Optional[datetime.datetime]:
        ...

    @FederalRegisterPublicationDate.setter
    def FederalRegisterPublicationDate(self, value: typing.Optional[datetime.datetime]):
        ...

    @property
    def RuleEffectiveDate(self) -> typing.Optional[datetime.datetime]:
        ...

    @RuleEffectiveDate.setter
    def RuleEffectiveDate(self, value: typing.Optional[datetime.datetime]):
        ...

    @property
    def LatestUpdate(self) -> datetime.datetime:
        ...

    @LatestUpdate.setter
    def LatestUpdate(self, value: datetime.datetime):
        ...

    @property
    def AlertType(self) -> str:
        ...

    @AlertType.setter
    def AlertType(self, value: str):
        ...

    @property
    def States(self) -> System.Collections.Generic.Dictionary[str, System.Collections.Generic.List[str]]:
        ...

    @States.setter
    def States(self, value: System.Collections.Generic.Dictionary[str, System.Collections.Generic.List[str]]):
        ...

    @property
    def Agencies(self) -> System.Collections.Generic.List[str]:
        ...

    @Agencies.setter
    def Agencies(self, value: System.Collections.Generic.List[str]):
        ...

    @property
    def AnnouncementUrl(self) -> str:
        ...

    @AnnouncementUrl.setter
    def AnnouncementUrl(self, value: str):
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the data
        
        :returns: A clone of the object.
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The NodaTime.DateTimeZone of this data type.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def IsSparseData(self) -> bool:
        """
        Indicates whether the data is sparse.
        If true, we disable logging for missing files
        
        :returns: true.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: Line of data
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: New instance.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates whether the data source is tied to an underlying symbol and requires that corporate events be applied to it as well, such as renames and delistings
        
        :returns: false.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...

    def ToString(self) -> str:
        """Converts the instance to string"""
        ...


class SECReportCompanyData(System.Object):
    """Company Information reported to SEC"""

    @property
    def ConformedName(self) -> str:
        """Current company name"""
        ...

    @ConformedName.setter
    def ConformedName(self, value: str):
        """Current company name"""
        ...

    @property
    def Cik(self) -> str:
        """Company's Central Index Key. Used to uniquely identify company filings in SEC's EDGAR system"""
        ...

    @Cik.setter
    def Cik(self, value: str):
        """Company's Central Index Key. Used to uniquely identify company filings in SEC's EDGAR system"""
        ...

    @property
    def AssignedSic(self) -> str:
        """Standard Industrial Classification"""
        ...

    @AssignedSic.setter
    def AssignedSic(self, value: str):
        """Standard Industrial Classification"""
        ...

    @property
    def IrsNumber(self) -> str:
        """Employer Identification Number"""
        ...

    @IrsNumber.setter
    def IrsNumber(self, value: str):
        """Employer Identification Number"""
        ...

    @property
    def StateOfIncorporation(self) -> str:
        """State of incorporation"""
        ...

    @StateOfIncorporation.setter
    def StateOfIncorporation(self, value: str):
        """State of incorporation"""
        ...

    @property
    def FiscalYearEnd(self) -> str:
        """Day fiscal year ends for given company. Formatted as MMdd"""
        ...

    @FiscalYearEnd.setter
    def FiscalYearEnd(self, value: str):
        """Day fiscal year ends for given company. Formatted as MMdd"""
        ...


class SECReportFilingValues(System.Object):
    """SEC Report Filing information"""

    @property
    def FormType(self) -> str:
        """SEC Form Type (e.g. 10-Q, 8-K, S-1, etc.)"""
        ...

    @FormType.setter
    def FormType(self, value: str):
        """SEC Form Type (e.g. 10-Q, 8-K, S-1, etc.)"""
        ...

    @property
    def Act(self) -> str:
        """Identification of the act(s) under which certain IM filings are made. The form type may be filed under more than one act. Required in each filing values tag nest."""
        ...

    @Act.setter
    def Act(self, value: str):
        """Identification of the act(s) under which certain IM filings are made. The form type may be filed under more than one act. Required in each filing values tag nest."""
        ...

    @property
    def FileNumber(self) -> str:
        """SEC filing number"""
        ...

    @FileNumber.setter
    def FileNumber(self, value: str):
        """SEC filing number"""
        ...

    @property
    def FilmNumber(self) -> str:
        """Used to access documents in the SEC's Virtual Private Reference Room (VPRR)"""
        ...

    @FilmNumber.setter
    def FilmNumber(self, value: str):
        """Used to access documents in the SEC's Virtual Private Reference Room (VPRR)"""
        ...


class SECReportBusinessAddress(System.Object):
    """Business Address reported to the SEC"""

    @property
    def StreetOne(self) -> str:
        """Street Address 1"""
        ...

    @StreetOne.setter
    def StreetOne(self, value: str):
        """Street Address 1"""
        ...

    @property
    def StreetTwo(self) -> str:
        """Street Address 2"""
        ...

    @StreetTwo.setter
    def StreetTwo(self, value: str):
        """Street Address 2"""
        ...

    @property
    def City(self) -> str:
        """City"""
        ...

    @City.setter
    def City(self, value: str):
        """City"""
        ...

    @property
    def State(self) -> str:
        """US State"""
        ...

    @State.setter
    def State(self, value: str):
        """US State"""
        ...

    @property
    def Zip(self) -> str:
        """ZIP Code"""
        ...

    @Zip.setter
    def Zip(self, value: str):
        """ZIP Code"""
        ...

    @property
    def Phone(self) -> str:
        """Business phone number"""
        ...

    @Phone.setter
    def Phone(self, value: str):
        """Business phone number"""
        ...


class SECReportMailAddress(System.Object):
    """SEC Report Mail Address"""

    @property
    def StreetOne(self) -> str:
        """Mailing street address"""
        ...

    @StreetOne.setter
    def StreetOne(self, value: str):
        """Mailing street address"""
        ...

    @property
    def StreetTwo(self) -> str:
        """Mailing street address 2"""
        ...

    @StreetTwo.setter
    def StreetTwo(self, value: str):
        """Mailing street address 2"""
        ...

    @property
    def City(self) -> str:
        """City"""
        ...

    @City.setter
    def City(self, value: str):
        """City"""
        ...

    @property
    def State(self) -> str:
        """US State"""
        ...

    @State.setter
    def State(self, value: str):
        """US State"""
        ...

    @property
    def Zip(self) -> str:
        """ZIP code. Not an integer because ZIP codes with dashes and letters exist"""
        ...

    @Zip.setter
    def Zip(self, value: str):
        """ZIP code. Not an integer because ZIP codes with dashes and letters exist"""
        ...


class SECReportFormerCompany(System.Object):
    """SEC Name Changing Report"""

    @property
    def FormerConformedName(self) -> str:
        """Previous company name"""
        ...

    @FormerConformedName.setter
    def FormerConformedName(self, value: str):
        """Previous company name"""
        ...

    @property
    def Changed(self) -> datetime.datetime:
        """Date the company name was changed to a new name"""
        ...

    @Changed.setter
    def Changed(self, value: datetime.datetime):
        """Date the company name was changed to a new name"""
        ...


class SECReportFiler(System.Object):
    """SEC Filing"""

    @property
    def CompanyData(self) -> QuantConnect.DataSource.SECReportCompanyData:
        """SEC data containing company data such as company name, cik, etc."""
        ...

    @CompanyData.setter
    def CompanyData(self, value: QuantConnect.DataSource.SECReportCompanyData):
        """SEC data containing company data such as company name, cik, etc."""
        ...

    @property
    def Values(self) -> System.Collections.Generic.List[QuantConnect.DataSource.SECReportFilingValues]:
        """Information regarding the filing itself"""
        ...

    @Values.setter
    def Values(self, value: System.Collections.Generic.List[QuantConnect.DataSource.SECReportFilingValues]):
        """Information regarding the filing itself"""
        ...

    @property
    def BusinessAddress(self) -> System.Collections.Generic.List[QuantConnect.DataSource.SECReportBusinessAddress]:
        """Information related to the business' address"""
        ...

    @BusinessAddress.setter
    def BusinessAddress(self, value: System.Collections.Generic.List[QuantConnect.DataSource.SECReportBusinessAddress]):
        """Information related to the business' address"""
        ...

    @property
    def MailingAddress(self) -> System.Collections.Generic.List[QuantConnect.DataSource.SECReportMailAddress]:
        """Company mailing address information"""
        ...

    @MailingAddress.setter
    def MailingAddress(self, value: System.Collections.Generic.List[QuantConnect.DataSource.SECReportMailAddress]):
        """Company mailing address information"""
        ...

    @property
    def FormerCompanies(self) -> System.Collections.Generic.List[QuantConnect.DataSource.SECReportFormerCompany]:
        """
        Former company names. Default to empty list in order to not have null values
        in the case that the company has never had a former name
        """
        ...

    @FormerCompanies.setter
    def FormerCompanies(self, value: System.Collections.Generic.List[QuantConnect.DataSource.SECReportFormerCompany]):
        """
        Former company names. Default to empty list in order to not have null values
        in the case that the company has never had a former name
        """
        ...


class SECReportDocument(System.Object):
    """Business documents reported to the SEC"""

    @property
    def FormType(self) -> str:
        """Report document type, e.g. 10-Q, 8-K, S-1"""
        ...

    @FormType.setter
    def FormType(self, value: str):
        """Report document type, e.g. 10-Q, 8-K, S-1"""
        ...

    @property
    def Sequence(self) -> int:
        """Nth attachment to the form filed"""
        ...

    @Sequence.setter
    def Sequence(self, value: int):
        """Nth attachment to the form filed"""
        ...

    @property
    def Filename(self) -> str:
        """File name that the file had when it was uploaded"""
        ...

    @Filename.setter
    def Filename(self, value: str):
        """File name that the file had when it was uploaded"""
        ...

    @property
    def Description(self) -> str:
        """Attachment content(s) description"""
        ...

    @Description.setter
    def Description(self, value: str):
        """Attachment content(s) description"""
        ...

    @property
    def Text(self) -> str:
        """
        Content of the attachment. This is the field that will most likely contain
        information related to financial reports. Sometimes, XML will
        be present in the data. If the first line starts with "<XML>", then
        XML data will be present in the contents of the document
        """
        ...

    @Text.setter
    def Text(self, value: str):
        """
        Content of the attachment. This is the field that will most likely contain
        information related to financial reports. Sometimes, XML will
        be present in the data. If the first line starts with "<XML>", then
        XML data will be present in the contents of the document
        """
        ...


class SECReportSubmission(System.Object):
    """SEC Report Submission"""

    @property
    def AccessionNumber(self) -> str:
        """Number used to access document filings on the SEC website"""
        ...

    @AccessionNumber.setter
    def AccessionNumber(self, value: str):
        """Number used to access document filings on the SEC website"""
        ...

    @property
    def FormType(self) -> str:
        """SEC form type"""
        ...

    @FormType.setter
    def FormType(self, value: str):
        """SEC form type"""
        ...

    @property
    def PublicDocumentCount(self) -> str:
        """Number of documents made public by the SEC"""
        ...

    @PublicDocumentCount.setter
    def PublicDocumentCount(self, value: str):
        """Number of documents made public by the SEC"""
        ...

    @property
    def Period(self) -> datetime.datetime:
        """End date of reporting period of filing. Optional."""
        ...

    @Period.setter
    def Period(self, value: datetime.datetime):
        """End date of reporting period of filing. Optional."""
        ...

    @property
    def Items(self) -> System.Collections.Generic.List[str]:
        """Identifies 1 or more items declared in 8-K filings. Optional & Repeatable."""
        ...

    @Items.setter
    def Items(self, value: System.Collections.Generic.List[str]):
        """Identifies 1 or more items declared in 8-K filings. Optional & Repeatable."""
        ...

    @property
    def FilingDate(self) -> datetime.datetime:
        """Date report was filed with the SEC"""
        ...

    @FilingDate.setter
    def FilingDate(self, value: datetime.datetime):
        """Date report was filed with the SEC"""
        ...

    @property
    def FilingDateChange(self) -> datetime.datetime:
        """Date when the last Post Acceptance occurred. Optional."""
        ...

    @FilingDateChange.setter
    def FilingDateChange(self, value: datetime.datetime):
        """Date when the last Post Acceptance occurred. Optional."""
        ...

    @property
    def MadeAvailableAt(self) -> datetime.datetime:
        """
        Exact time the report was filed with the SEC and made available to the public (plus 10 minute delay).
        This field is NOT included with the raw SEC report, and should be added during post processing of the data
        """
        ...

    @MadeAvailableAt.setter
    def MadeAvailableAt(self, value: datetime.datetime):
        """
        Exact time the report was filed with the SEC and made available to the public (plus 10 minute delay).
        This field is NOT included with the raw SEC report, and should be added during post processing of the data
        """
        ...

    @property
    def Filers(self) -> System.Collections.Generic.List[QuantConnect.DataSource.SECReportFiler]:
        """Contains information regarding who the filer of the report is."""
        ...

    @Filers.setter
    def Filers(self, value: System.Collections.Generic.List[QuantConnect.DataSource.SECReportFiler]):
        """Contains information regarding who the filer of the report is."""
        ...

    @property
    def Documents(self) -> System.Collections.Generic.List[QuantConnect.DataSource.SECReportDocument]:
        """Attachments/content associated with the report"""
        ...

    @Documents.setter
    def Documents(self, value: System.Collections.Generic.List[QuantConnect.DataSource.SECReportDocument]):
        """Attachments/content associated with the report"""
        ...


class ISECReport(QuantConnect.Data.IBaseData, metaclass=abc.ABCMeta):
    """
    Base interface for all SEC report types.
    Using an interface, we can retrieve all report types with a single
    call to Slice.Get{T}()
    """

    @property
    @abc.abstractmethod
    def Report(self) -> QuantConnect.DataSource.SECReportSubmission:
        """Contents of the actual SEC report"""
        ...


class SECReport10Q(QuantConnect.Data.BaseData, QuantConnect.DataSource.ISECReport):
    """
    SEC 10-Q report (quarterly earnings) BaseData implementation.
    Using this class, you can retrieve SEC report data for a security if it exists.
    If the ticker you want no longer trades, you can also use the CIK of the company
    you want data for as well except for currently traded stocks. This may change in the future.
    """

    DataSourceId: int
    """Data source ID"""

    @property
    def Report(self) -> QuantConnect.DataSource.SECReportSubmission:
        """Contents of the actual SEC report"""
        ...

    @overload
    def __init__(self) -> None:
        """Empty constructor required for Slice.Get{T}()"""
        ...

    @overload
    def __init__(self, report: QuantConnect.DataSource.SECReportSubmission) -> None:
        """
        Constructor used to initialize instance with the given report
        
        :param report: SEC report submission
        """
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the current object into a new object
        
        :returns: BaseData clone of the current object.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Returns a subscription data source pointing towards SEC 10-Q report data
        
        :param config: User configuration
        :param date: Date data has been requested for
        :param isLiveMode: Is livetrading
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data into BaseData
        
        :param config: User subscription config
        :param line: Line of source file to parse
        :param date: Date data was requested for
        :param isLiveMode: Is livetrading mode
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates if there is support for mapping
        
        :returns: True indicates mapping should be used.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...


class SECReport8K(QuantConnect.Data.BaseData, QuantConnect.DataSource.ISECReport):
    """
    SEC 8-K report (important investor notices) BaseData implementation.
    Using this class, you can retrieve SEC report data for a security if it exists.
    If the ticker you want no longer trades, you can also use the CIK of the company
    you want data for as well except for currently traded stocks. This may change in the future.
    """

    DataSourceId: int
    """Data source ID"""

    @property
    def Report(self) -> QuantConnect.DataSource.SECReportSubmission:
        """Contents of the actual SEC report"""
        ...

    @overload
    def __init__(self) -> None:
        """Empty constructor required for Slice.Get{T}()"""
        ...

    @overload
    def __init__(self, report: QuantConnect.DataSource.SECReportSubmission) -> None:
        """
        Constructor used to initialize instance with the given report
        
        :param report: SEC report submission
        """
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the current object into a new object
        
        :returns: BaseData clone of the current object.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Returns a subscription data source pointing towards SEC 8-K report data
        
        :param config: User configuration
        :param date: Date data has been requested for
        :param isLiveMode: Is livetrading
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data into instance of BaseData
        
        :param config: User subscription config
        :param line: Line of source file to parse
        :param date: Date data was requested for
        :param isLiveMode: Is live trading mode
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates if there is support for mapping
        
        :returns: True indicates mapping should be used.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...


class SECReportFactory(System.Object):
    """SEC Report Factory for deserializing XML formatted data"""

    def CreateSECReport(self, xmlText: str) -> QuantConnect.DataSource.ISECReport:
        """
        Factory method creates SEC report by deserializing XML formatted SEC data to SECReportSubmission
        
        :param xmlText: XML text containing SEC data
        """
        ...


class SECReportDateTimeConverter(IsoDateTimeConverter):
    """Specifies format for parsing DateTime values from SEC data"""

    def __init__(self) -> None:
        """Constructor, initializes DateTime format"""
        ...


class SECReportIndexItem(System.Object):
    """SEC Report Index Item"""

    @property
    def LastModified(self) -> datetime.datetime:
        """Date the SEC submission was published"""
        ...

    @LastModified.setter
    def LastModified(self, value: datetime.datetime):
        """Date the SEC submission was published"""
        ...

    @property
    def Name(self) -> str:
        """Name of folder/file. Usually accession number"""
        ...

    @Name.setter
    def Name(self, value: str):
        """Name of folder/file. Usually accession number"""
        ...

    @property
    def FileType(self) -> str:
        """Specifies what kind of file the entry is"""
        ...

    @FileType.setter
    def FileType(self, value: str):
        """Specifies what kind of file the entry is"""
        ...

    @property
    def Size(self) -> str:
        """Size of the file. Empty if directory"""
        ...

    @Size.setter
    def Size(self, value: str):
        """Size of the file. Empty if directory"""
        ...


class SECReportIndexDirectory(System.Object):
    """SEC Report Index Directory, contains a list of Items"""

    @property
    def Items(self) -> System.Collections.Generic.List[QuantConnect.DataSource.SECReportIndexItem]:
        """Contains additional metadata regarding files present on the server"""
        ...

    @Items.setter
    def Items(self, value: System.Collections.Generic.List[QuantConnect.DataSource.SECReportIndexItem]):
        """Contains additional metadata regarding files present on the server"""
        ...

    @property
    def Name(self) -> str:
        """Path directory"""
        ...

    @Name.setter
    def Name(self, value: str):
        """Path directory"""
        ...

    @property
    def ParentDirectory(self) -> str:
        """Parent directory (if one exists)"""
        ...

    @ParentDirectory.setter
    def ParentDirectory(self, value: str):
        """Parent directory (if one exists)"""
        ...


class SECReportIndexFile(System.Object):
    """SEC Report Index"""

    @property
    def Directory(self) -> QuantConnect.DataSource.SECReportIndexDirectory:
        """First and only root entry of SEC index.json"""
        ...

    @Directory.setter
    def Directory(self, value: QuantConnect.DataSource.SECReportIndexDirectory):
        """First and only root entry of SEC index.json"""
        ...


class SECReport10K(QuantConnect.Data.BaseData, QuantConnect.DataSource.ISECReport):
    """
    SEC 10-K report (annual earnings) BaseData implementation.
    Using this class, you can retrieve SEC report data for a security if it exists.
    If the ticker you want no longer trades, you can also use the CIK of the company
    you want data for as well except for currently traded stocks. This may change in the future.
    """

    DataSourceId: int
    """Data source ID"""

    @property
    def Report(self) -> QuantConnect.DataSource.SECReportSubmission:
        """Contents of the actual SEC report"""
        ...

    @overload
    def __init__(self) -> None:
        """Empty constructor required for Slice.Get{T}()"""
        ...

    @overload
    def __init__(self, report: QuantConnect.DataSource.SECReportSubmission) -> None:
        """
        Constructor used to initialize instance with the given report
        
        :param report: SEC report submission
        """
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the current object into a new object
        
        :returns: BaseData clone of the current object.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Returns a subscription data source pointing towards SEC 10-K report data
        
        :param config: User configuration
        :param date: Date data has been requested for
        :param isLiveMode: Is livetrading
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data into BaseData
        
        :param config: User subscription config
        :param line: Line of source file to parse
        :param date: Date data was requested for
        :param isLiveMode: Is livetrading mode
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates if there is support for mapping
        
        :returns: True indicates mapping should be used.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...


class QuiverWallStreetBetsUniverse(QuantConnect.Data.BaseData):
    """Universe Selection helper class for QuiverWallStreetBets dataset"""

    @property
    def Symbol(self) -> QuantConnect.Symbol:
        """Symbol of data"""
        ...

    @Symbol.setter
    def Symbol(self, value: QuantConnect.Symbol):
        """Symbol of data"""
        ...

    @property
    def Mentions(self) -> int:
        """The number of mentions on the given date"""
        ...

    @Mentions.setter
    def Mentions(self, value: int):
        """The number of mentions on the given date"""
        ...

    @property
    def Rank(self) -> int:
        """This ticker's rank on the given date (as determined by total number of mentions)"""
        ...

    @Rank.setter
    def Rank(self, value: int):
        """This ticker's rank on the given date (as determined by total number of mentions)"""
        ...

    @property
    def Sentiment(self) -> float:
        """
        Average sentiment of all comments containing the given ticker on this date. Sentiment is calculated using VADER sentiment analysis.
        The value can range between -1 and +1. Negative values imply negative sentiment, whereas positive values imply positive sentiment.
        """
        ...

    @Sentiment.setter
    def Sentiment(self, value: float):
        """
        Average sentiment of all comments containing the given ticker on this date. Sentiment is calculated using VADER sentiment analysis.
        The value can range between -1 and +1. Negative values imply negative sentiment, whereas positive values imply positive sentiment.
        """
        ...

    @property
    def Period(self) -> datetime.timedelta:
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @Period.setter
    def Period(self, value: datetime.timedelta):
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: Line of data
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: New instance.
        """
        ...


class QuiverWallStreetBets(QuantConnect.Data.BaseData):
    """Mentions of the given company's ticker in the WallStreetBets daily discussion thread"""

    DataSourceId: int
    """Data source ID"""

    @property
    def Date(self) -> datetime.datetime:
        """Date of the daily discussion thread"""
        ...

    @Date.setter
    def Date(self, value: datetime.datetime):
        """Date of the daily discussion thread"""
        ...

    @property
    def Mentions(self) -> int:
        """The number of mentions on the given date"""
        ...

    @Mentions.setter
    def Mentions(self, value: int):
        """The number of mentions on the given date"""
        ...

    @property
    def Rank(self) -> int:
        """This ticker's rank on the given date (as determined by total number of mentions)"""
        ...

    @Rank.setter
    def Rank(self, value: int):
        """This ticker's rank on the given date (as determined by total number of mentions)"""
        ...

    @property
    def Sentiment(self) -> float:
        """
        Average sentiment of all comments containing the given ticker on this date. Sentiment is calculated using VADER sentiment analysis.
        The value can range between -1 and +1. Negative values imply negative sentiment, whereas positive values imply positive sentiment.
        """
        ...

    @Sentiment.setter
    def Sentiment(self, value: float):
        """
        Average sentiment of all comments containing the given ticker on this date. Sentiment is calculated using VADER sentiment analysis.
        The value can range between -1 and +1. Negative values imply negative sentiment, whereas positive values imply positive sentiment.
        """
        ...

    @property
    def Period(self) -> datetime.timedelta:
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @Period.setter
    def Period(self, value: datetime.timedelta):
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @overload
    def __init__(self) -> None:
        """Required for successful Json.NET deserialization"""
        ...

    @overload
    def __init__(self, csvLine: str) -> None:
        """
        Creates a new instance of QuiverWallStreetBets from a CSV line
        
        :param csvLine: CSV line
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The DateTimeZone of this data type.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the Subscription Data Source gained from the URL
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Subscription Data Source.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reader converts each line of the data source into BaseData objects.
        
        :param config: Subscription data config setup object
        :param line: Content of the source document
        :param date: Date of the requested data
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Quiver WallStreetBets object.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates if there is support for mapping
        
        :returns: True indicates mapping should be used.
        """
        ...

    def ToString(self) -> str:
        """Formats a string with the Quiver WallStreetBets information."""
        ...


class USTreasuryYieldCurveRate(QuantConnect.Data.BaseData):
    """U.S. Treasury yield curve data"""

    DataSourceId: int
    """Data source ID"""

    @property
    def OneMonth(self) -> typing.Optional[float]:
        """One month yield curve"""
        ...

    @OneMonth.setter
    def OneMonth(self, value: typing.Optional[float]):
        """One month yield curve"""
        ...

    @property
    def TwoMonth(self) -> typing.Optional[float]:
        """Two month yield curve"""
        ...

    @TwoMonth.setter
    def TwoMonth(self, value: typing.Optional[float]):
        """Two month yield curve"""
        ...

    @property
    def ThreeMonth(self) -> typing.Optional[float]:
        """Three month yield curve"""
        ...

    @ThreeMonth.setter
    def ThreeMonth(self, value: typing.Optional[float]):
        """Three month yield curve"""
        ...

    @property
    def SixMonth(self) -> typing.Optional[float]:
        """Six month yield curve"""
        ...

    @SixMonth.setter
    def SixMonth(self, value: typing.Optional[float]):
        """Six month yield curve"""
        ...

    @property
    def OneYear(self) -> typing.Optional[float]:
        """One year yield curve"""
        ...

    @OneYear.setter
    def OneYear(self, value: typing.Optional[float]):
        """One year yield curve"""
        ...

    @property
    def TwoYear(self) -> typing.Optional[float]:
        """Two year yield curve"""
        ...

    @TwoYear.setter
    def TwoYear(self, value: typing.Optional[float]):
        """Two year yield curve"""
        ...

    @property
    def ThreeYear(self) -> typing.Optional[float]:
        """Three year yield curve"""
        ...

    @ThreeYear.setter
    def ThreeYear(self, value: typing.Optional[float]):
        """Three year yield curve"""
        ...

    @property
    def FiveYear(self) -> typing.Optional[float]:
        """Five year yield curve"""
        ...

    @FiveYear.setter
    def FiveYear(self, value: typing.Optional[float]):
        """Five year yield curve"""
        ...

    @property
    def SevenYear(self) -> typing.Optional[float]:
        """Seven year yield curve"""
        ...

    @SevenYear.setter
    def SevenYear(self, value: typing.Optional[float]):
        """Seven year yield curve"""
        ...

    @property
    def TenYear(self) -> typing.Optional[float]:
        """Ten year yield curve"""
        ...

    @TenYear.setter
    def TenYear(self, value: typing.Optional[float]):
        """Ten year yield curve"""
        ...

    @property
    def TwentyYear(self) -> typing.Optional[float]:
        """Twenty year yield curve"""
        ...

    @TwentyYear.setter
    def TwentyYear(self, value: typing.Optional[float]):
        """Twenty year yield curve"""
        ...

    @property
    def ThirtyYear(self) -> typing.Optional[float]:
        """Thirty year yield curve"""
        ...

    @ThirtyYear.setter
    def ThirtyYear(self, value: typing.Optional[float]):
        """Thirty year yield curve"""
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the object. This method implementation is required
        so that we don't have any null values for our properties
        when the user attempts to use it in backtesting/live trading
        
        :returns: Cloned instance.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Specifies the location of the data and directs LEAN where to load the data from
        
        :param config: Subscription configuration
        :param date: Algorithm date
        :param isLiveMode: Is live mode
        :returns: Subscription data source object pointing LEAN to the data location.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reads and parses yield curve data from a csv file
        
        :param config: Subscription configuration
        :param line: CSV line containing yield curve data
        :param date: Date request was made for
        :param isLiveMode: Is live mode
        :returns: YieldCurve instance.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...


class ExtractAlphaFiscalPeriod(System.Object):
    """Fiscal period that the ExtractAlphaTrueBeat instance has forecasts for."""

    @property
    def FiscalYear(self) -> int:
        """Fiscal year (i.e. the year that the financial report applies to in 10-Q and/or 10-K SEC filings)"""
        ...

    @FiscalYear.setter
    def FiscalYear(self, value: int):
        """Fiscal year (i.e. the year that the financial report applies to in 10-Q and/or 10-K SEC filings)"""
        ...

    @property
    def FiscalQuarter(self) -> typing.Optional[int]:
        """
        Fiscal quarter (i.e. the quarter that the financial report applies to in 10-Q filings).
        If this is null, then the fiscal period being reported is for the full year of the FiscalYear
        """
        ...

    @FiscalQuarter.setter
    def FiscalQuarter(self, value: typing.Optional[int]):
        """
        Fiscal quarter (i.e. the quarter that the financial report applies to in 10-Q filings).
        If this is null, then the fiscal period being reported is for the full year of the FiscalYear
        """
        ...

    @property
    def End(self) -> typing.Optional[datetime.datetime]:
        """The date that the fiscal quarter ends"""
        ...

    @End.setter
    def End(self, value: typing.Optional[datetime.datetime]):
        """The date that the fiscal quarter ends"""
        ...

    @property
    def ExpectedReportDate(self) -> typing.Optional[datetime.datetime]:
        """The date that the SEC report for the fiscal period is expected to be released publicly"""
        ...

    @ExpectedReportDate.setter
    def ExpectedReportDate(self, value: typing.Optional[datetime.datetime]):
        """The date that the SEC report for the fiscal period is expected to be released publicly"""
        ...

    @property
    def Annual(self) -> bool:
        """Returns true if the fiscal period is for the whole fiscal year (all quarters)"""
        ...

    @property
    def Quarterly(self) -> bool:
        """Returns true if the fiscal period is for a single quarter only"""
        ...


class ExtractAlphaTrueBeatEarningsMetric(System.Enum):
    """The earnings metric/type being forecasted"""

    EPS = 0
    """Earnings per share"""

    Revenue = 1
    """Revenue"""


class ExtractAlphaTrueBeat(QuantConnect.Data.BaseData):
    """
    EPS/Revenue earnings surprise forecasting for upcoming financial reports released
    by regulatory agencies (e.g. United States SEC)
    """

    @property
    def FiscalPeriod(self) -> QuantConnect.DataSource.ExtractAlphaFiscalPeriod:
        """The fiscal period that is being forecasted"""
        ...

    @FiscalPeriod.setter
    def FiscalPeriod(self, value: QuantConnect.DataSource.ExtractAlphaFiscalPeriod):
        """The fiscal period that is being forecasted"""
        ...

    @property
    def EarningsMetric(self) -> int:
        """
        The earnings metric being forecasted (e.g. EPS, revenue)
        
        This property contains the int value of a member of the QuantConnect.DataSource.ExtractAlphaTrueBeatEarningsMetric enum.
        """
        ...

    @EarningsMetric.setter
    def EarningsMetric(self, value: int):
        """
        The earnings metric being forecasted (e.g. EPS, revenue)
        
        This property contains the int value of a member of the QuantConnect.DataSource.ExtractAlphaTrueBeatEarningsMetric enum.
        """
        ...

    @property
    def AnalystEstimatesCount(self) -> int:
        """The number of analyst estimates that the TrueBeat used in its calculation"""
        ...

    @AnalystEstimatesCount.setter
    def AnalystEstimatesCount(self, value: int):
        """The number of analyst estimates that the TrueBeat used in its calculation"""
        ...

    @property
    def TrueBeat(self) -> float:
        """
        The forecasted earnings surprise percentage, relative to consensus estimates.
        TrueBeat is calculated as the total sum of the ExpertBeat, TrendBeat, and ManagementBeat metrics.
        """
        ...

    @TrueBeat.setter
    def TrueBeat(self, value: float):
        """
        The forecasted earnings surprise percentage, relative to consensus estimates.
        TrueBeat is calculated as the total sum of the ExpertBeat, TrendBeat, and ManagementBeat metrics.
        """
        ...

    @property
    def ExpertBeat(self) -> typing.Optional[float]:
        """The component of TrueBeat that is derived from top analyst estimates"""
        ...

    @ExpertBeat.setter
    def ExpertBeat(self, value: typing.Optional[float]):
        """The component of TrueBeat that is derived from top analyst estimates"""
        ...

    @property
    def TrendBeat(self) -> typing.Optional[float]:
        """The component of TrueBeat that is derived from trends in stock and peer surprises"""
        ...

    @TrendBeat.setter
    def TrendBeat(self, value: typing.Optional[float]):
        """The component of TrueBeat that is derived from trends in stock and peer surprises"""
        ...

    @property
    def ManagementBeat(self) -> typing.Optional[float]:
        """The component of TrueBeat that is derived from management activity (e.g. guidance)"""
        ...

    @ManagementBeat.setter
    def ManagementBeat(self, value: typing.Optional[float]):
        """The component of TrueBeat that is derived from management activity (e.g. guidance)"""
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        """The time that the data became available to the algorithm"""
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """The time that the data became available to the algorithm"""
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Return a new instance clone of this object, used in fill forward
        
        :returns: A clone of the current object.
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The NodaTime.DateTimeZone of this data type.
        """
        ...

    def IsSparseData(self) -> bool:
        """
        Indicates that the data set is expected to be sparse
        
        :returns: True if the data set represented by this type is expected to be sparse.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reader converts each line of the data source into BaseData objects. Each data type creates its own factory method, and returns a new instance of the object
        each time it is called. The returned object is assumed to be time stamped in the config.ExchangeTimeZone.
        
        :param config: Subscription data config setup object
        :param line: Line of the source document
        :param date: Date of the requested data
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Instance of the T:BaseData object generated by this line of the CSV.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates if there is support for mapping
        
        :returns: True indicates mapping should be used.
        """
        ...

    def ToString(self) -> str:
        """
        Formats a string with TrueBeat data
        
        :returns: string containing TrueBeat information.
        """
        ...


class ExtractAlphaTrueBeats(QuantConnect.Data.UniverseSelection.BaseDataCollection):
    """A collection of Extra Alpha True Beats for a Symbol and date"""

    def Add(self, newDataPoint: QuantConnect.Data.BaseData) -> None:
        """
        Adds a new data point to this collection
        
        :param newDataPoint: The new data point to add
        """
        ...

    def AddRange(self, newDataPoints: System.Collections.Generic.IEnumerable[QuantConnect.Data.BaseData]) -> None:
        """
        Adds a new data points to this collection
        
        :param newDataPoints: The new data points to add
        """
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Return a new instance clone of this object, used in fill forward
        
        :returns: A clone of the current object.
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The NodaTime.DateTimeZone of this data type.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def IsSparseData(self) -> bool:
        """
        Indicates that the data set is expected to be sparse
        
        :returns: True if the data set represented by this type is expected to be sparse.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reader converts each line of the data source into BaseData objects. Each data type creates its own factory method, and returns a new instance of the object
        each time it is called. The returned object is assumed to be time stamped in the config.ExchangeTimeZone.
        
        :param config: Subscription data config setup object
        :param line: Line of the source document
        :param date: Date of the requested data
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Instance of the T:BaseData object generated by this line of the CSV.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates if there is support for mapping
        
        :returns: True indicates mapping should be used.
        """
        ...

    def ToString(self) -> str:
        """
        Formats a string with TrueBeat data
        
        :returns: string containing TrueBeat information.
        """
        ...


class CBOE(QuantConnect.Data.Market.TradeBar):
    """CBOE data source"""

    DataSourceId: int
    """Data source ID"""

    def __init__(self) -> None:
        """Creates a new instance of the object"""
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """Gets the source location of the CBOE file"""
        ...

    def IsSparseData(self) -> bool:
        """
        Determines if data source is sparse
        
        :returns: false.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reads the data from the source and creates a BaseData instance
        
        :param config: Configuration
        :param line: Line of data
        :param date: Date we're requesting data for
        :param isLiveMode: Is live mode
        :returns: New BaseData instance to be used in the algorithm.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Determines whether the data source requires mapping
        
        :returns: false.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...

    def ToString(self) -> str:
        """
        Converts the instance to a string
        
        :returns: String containing open, high, low, close.
        """
        ...


class BrainCompanyFilingLanguageMetricsSimilarityDifference(System.Object):
    """This class has no documentation."""

    @property
    def All(self) -> typing.Optional[float]:
        ...

    @All.setter
    def All(self, value: typing.Optional[float]):
        ...

    @property
    def Positive(self) -> typing.Optional[float]:
        ...

    @Positive.setter
    def Positive(self, value: typing.Optional[float]):
        ...

    @property
    def Negative(self) -> typing.Optional[float]:
        ...

    @Negative.setter
    def Negative(self, value: typing.Optional[float]):
        ...

    @property
    def Uncertainty(self) -> typing.Optional[float]:
        ...

    @Uncertainty.setter
    def Uncertainty(self, value: typing.Optional[float]):
        ...

    @property
    def Litigious(self) -> typing.Optional[float]:
        ...

    @Litigious.setter
    def Litigious(self, value: typing.Optional[float]):
        ...

    @property
    def Constraining(self) -> typing.Optional[float]:
        ...

    @Constraining.setter
    def Constraining(self, value: typing.Optional[float]):
        ...

    @property
    def Interesting(self) -> typing.Optional[float]:
        ...

    @Interesting.setter
    def Interesting(self, value: typing.Optional[float]):
        ...

    @staticmethod
    def Parse(similarityValues: System.Collections.Generic.List[str]) -> QuantConnect.DataSource.BrainCompanyFilingLanguageMetricsSimilarityDifference:
        ...


class BrainStockRankingBase(typing.Generic[QuantConnect_DataSource_BrainStockRankingBase_T], QuantConnect.Data.BaseData):
    """Brain sentiment on 10-K/10-Q SEC reports"""

    @property
    def Rank(self) -> float:
        ...

    @Rank.setter
    def Rank(self, value: float):
        ...

    @property
    def LookbackDays(self) -> int:
        """This property is protected."""
        ...

    @LookbackDays.setter
    def LookbackDays(self, value: int):
        """This property is protected."""
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the data
        
        :returns: A clone of the object.
        """
        ...

    def CloneData(self) -> QuantConnect_DataSource_BrainStockRankingBase_T:
        """This method is protected."""
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The NodaTime.DateTimeZone of this data type.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def IsSparseData(self) -> bool:
        """
        Indicates whether the data is sparse.
        If true, we disable logging for missing files
        
        :returns: true.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: Line of data
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: New instance.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates whether the data source is tied to an underlying symbol and requires that corporate events be applied to it as well, such as renames and delistings
        
        :returns: false.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...

    def ToString(self) -> str:
        """Converts the instance to string"""
        ...


class BrainStockRanking10Day(QuantConnect.DataSource.BrainStockRankingBase[QuantConnect_DataSource_BrainStockRanking10Day]):
    """Brain universe stock rankings on expected returns in the next 10 days"""

    DataSourceId: int
    """Data source ID"""

    @property
    def LookbackDays(self) -> int:
        """This property is protected."""
        ...

    @LookbackDays.setter
    def LookbackDays(self, value: int):
        """This property is protected."""
        ...


class BrainStockRanking2Day(QuantConnect.DataSource.BrainStockRankingBase[QuantConnect_DataSource_BrainStockRanking2Day]):
    """Brain universe stock rankings on expected returns in the next 2 days"""

    DataSourceId: int
    """Data source ID"""

    @property
    def LookbackDays(self) -> int:
        """This property is protected."""
        ...

    @LookbackDays.setter
    def LookbackDays(self, value: int):
        """This property is protected."""
        ...


class BrainStockRanking30Day(QuantConnect.DataSource.BrainStockRankingBase[QuantConnect_DataSource_BrainStockRanking30Day]):
    """Brain universe stock rankings on expected returns in the next 30 days"""

    DataSourceId: int
    """Data source ID"""

    @property
    def LookbackDays(self) -> int:
        """This property is protected."""
        ...

    @LookbackDays.setter
    def LookbackDays(self, value: int):
        """This property is protected."""
        ...


class BrainCompanyFilingLanguageMetrics(System.Object):
    """This class has no documentation."""

    @property
    def SentenceCount(self) -> typing.Optional[int]:
        ...

    @SentenceCount.setter
    def SentenceCount(self, value: typing.Optional[int]):
        ...

    @property
    def MeanSentenceLength(self) -> typing.Optional[float]:
        ...

    @MeanSentenceLength.setter
    def MeanSentenceLength(self, value: typing.Optional[float]):
        ...

    @property
    def Sentiment(self) -> typing.Optional[float]:
        ...

    @Sentiment.setter
    def Sentiment(self, value: typing.Optional[float]):
        ...

    @property
    def Uncertainty(self) -> typing.Optional[float]:
        ...

    @Uncertainty.setter
    def Uncertainty(self, value: typing.Optional[float]):
        ...

    @property
    def Litigious(self) -> typing.Optional[float]:
        ...

    @Litigious.setter
    def Litigious(self, value: typing.Optional[float]):
        ...

    @property
    def Constraining(self) -> typing.Optional[float]:
        ...

    @Constraining.setter
    def Constraining(self, value: typing.Optional[float]):
        ...

    @property
    def Interesting(self) -> typing.Optional[float]:
        ...

    @Interesting.setter
    def Interesting(self, value: typing.Optional[float]):
        ...

    @property
    def Readability(self) -> typing.Optional[float]:
        ...

    @Readability.setter
    def Readability(self, value: typing.Optional[float]):
        ...

    @property
    def LexicalRichness(self) -> typing.Optional[float]:
        ...

    @LexicalRichness.setter
    def LexicalRichness(self, value: typing.Optional[float]):
        ...

    @property
    def LexicalDensity(self) -> typing.Optional[float]:
        ...

    @LexicalDensity.setter
    def LexicalDensity(self, value: typing.Optional[float]):
        ...

    @property
    def SpecificDensity(self) -> typing.Optional[float]:
        ...

    @SpecificDensity.setter
    def SpecificDensity(self, value: typing.Optional[float]):
        ...

    @property
    def Similarity(self) -> QuantConnect.DataSource.BrainCompanyFilingLanguageMetricsSimilarityDifference:
        ...

    @Similarity.setter
    def Similarity(self, value: QuantConnect.DataSource.BrainCompanyFilingLanguageMetricsSimilarityDifference):
        ...

    @staticmethod
    def Parse(metrics: System.Collections.Generic.List[str], similarity: System.Collections.Generic.List[str]) -> QuantConnect.DataSource.BrainCompanyFilingLanguageMetrics:
        ...


class BrainCompanyFilingLanguageMetricsBase(typing.Generic[QuantConnect_DataSource_BrainCompanyFilingLanguageMetricsBase_T], QuantConnect.Data.BaseData):
    """Brain sentiment on 10-K/10-Q SEC reports"""

    @property
    def ReportDate(self) -> datetime.datetime:
        ...

    @ReportDate.setter
    def ReportDate(self, value: datetime.datetime):
        ...

    @property
    def ReportCategory(self) -> str:
        ...

    @ReportCategory.setter
    def ReportCategory(self, value: str):
        ...

    @property
    def ReportPeriod(self) -> typing.Optional[int]:
        ...

    @ReportPeriod.setter
    def ReportPeriod(self, value: typing.Optional[int]):
        ...

    @property
    def PreviousReportDate(self) -> typing.Optional[datetime.datetime]:
        ...

    @PreviousReportDate.setter
    def PreviousReportDate(self, value: typing.Optional[datetime.datetime]):
        ...

    @property
    def PreviousReportCategory(self) -> str:
        ...

    @PreviousReportCategory.setter
    def PreviousReportCategory(self, value: str):
        ...

    @property
    def PreviousReportPeriod(self) -> typing.Optional[int]:
        ...

    @PreviousReportPeriod.setter
    def PreviousReportPeriod(self, value: typing.Optional[int]):
        ...

    @property
    def ReportSentiment(self) -> QuantConnect.DataSource.BrainCompanyFilingLanguageMetrics:
        ...

    @ReportSentiment.setter
    def ReportSentiment(self, value: QuantConnect.DataSource.BrainCompanyFilingLanguageMetrics):
        ...

    @property
    def RiskFactorsStatementSentiment(self) -> QuantConnect.DataSource.BrainCompanyFilingLanguageMetrics:
        ...

    @RiskFactorsStatementSentiment.setter
    def RiskFactorsStatementSentiment(self, value: QuantConnect.DataSource.BrainCompanyFilingLanguageMetrics):
        ...

    @property
    def ManagementDiscussionAnalyasisOfFinancialConditionAndResultsOfOperations(self) -> QuantConnect.DataSource.BrainCompanyFilingLanguageMetrics:
        ...

    @ManagementDiscussionAnalyasisOfFinancialConditionAndResultsOfOperations.setter
    def ManagementDiscussionAnalyasisOfFinancialConditionAndResultsOfOperations(self, value: QuantConnect.DataSource.BrainCompanyFilingLanguageMetrics):
        ...

    @property
    def ReportType(self) -> str:
        """This property is protected."""
        ...

    @ReportType.setter
    def ReportType(self, value: str):
        """This property is protected."""
        ...

    def Clone(self) -> QuantConnect.Data.BaseData:
        ...

    def CloneData(self) -> QuantConnect_DataSource_BrainCompanyFilingLanguageMetricsBase_T:
        """
        Clones the data
        
        This method is protected.
        
        :returns: A clone of the object.
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The NodaTime.DateTimeZone of this data type.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def IsSparseData(self) -> bool:
        """
        Indicates whether the data is sparse.
        If true, we disable logging for missing files
        
        :returns: true.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: Line of data
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: New instance.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates whether the data source is tied to an underlying symbol and requires that corporate events be applied to it as well, such as renames and delistings
        
        :returns: false.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...

    def ToString(self) -> str:
        """Converts the instance to string"""
        ...


class BrainCompanyFilingLanguageMetrics10K(QuantConnect.DataSource.BrainCompanyFilingLanguageMetricsBase[QuantConnect_DataSource_BrainCompanyFilingLanguageMetrics10K]):
    """Brain sentiment on only 10-K SEC reports"""

    DataSourceId: int
    """Data source ID"""

    @property
    def ReportType(self) -> str:
        """This property is protected."""
        ...

    @ReportType.setter
    def ReportType(self, value: str):
        """This property is protected."""
        ...


class BrainCompanyFilingLanguageMetricsAll(QuantConnect.DataSource.BrainCompanyFilingLanguageMetricsBase[QuantConnect_DataSource_BrainCompanyFilingLanguageMetricsAll]):
    """Brain sentiment on 10-K/10-Q SEC reports"""

    DataSourceId: int
    """Data source ID"""

    @property
    def ReportType(self) -> str:
        """This property is protected."""
        ...

    @ReportType.setter
    def ReportType(self, value: str):
        """This property is protected."""
        ...


class BrainSentimentIndicatorBase(typing.Generic[QuantConnect_DataSource_BrainSentimentIndicatorBase_T], QuantConnect.Data.BaseData):
    """Brain sentiment on news"""

    @property
    def TotalArticleMentions(self) -> int:
        ...

    @TotalArticleMentions.setter
    def TotalArticleMentions(self, value: int):
        ...

    @property
    def SentimentalArticleMentions(self) -> float:
        ...

    @SentimentalArticleMentions.setter
    def SentimentalArticleMentions(self, value: float):
        ...

    @property
    def Sentiment(self) -> float:
        ...

    @Sentiment.setter
    def Sentiment(self, value: float):
        ...

    @property
    def TotalBuzzVolume(self) -> typing.Optional[float]:
        ...

    @TotalBuzzVolume.setter
    def TotalBuzzVolume(self, value: typing.Optional[float]):
        ...

    @property
    def SentimentalBuzzVolume(self) -> typing.Optional[float]:
        ...

    @SentimentalBuzzVolume.setter
    def SentimentalBuzzVolume(self, value: typing.Optional[float]):
        ...

    @property
    def LookbackDays(self) -> int:
        """This property is protected."""
        ...

    @LookbackDays.setter
    def LookbackDays(self, value: int):
        """This property is protected."""
        ...

    def CloneData(self) -> QuantConnect_DataSource_BrainSentimentIndicatorBase_T:
        """
        Clones the data
        
        This method is protected.
        
        :returns: A clone of the object.
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The NodaTime.DateTimeZone of this data type.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def IsSparseData(self) -> bool:
        """
        Indicates whether the data is sparse.
        If true, we disable logging for missing files
        
        :returns: true.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: Line of data
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: New instance.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates whether the data source is tied to an underlying symbol and requires that corporate events be applied to it as well, such as renames and delistings
        
        :returns: false.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...

    def ToString(self) -> str:
        """Converts the instance to string"""
        ...


class BrainStockRanking5Day(QuantConnect.DataSource.BrainStockRankingBase[QuantConnect_DataSource_BrainStockRanking5Day]):
    """Brain universe stock rankings on expected returns in the next 5 days"""

    DataSourceId: int
    """Data source ID"""

    @property
    def LookbackDays(self) -> int:
        """This property is protected."""
        ...

    @LookbackDays.setter
    def LookbackDays(self, value: int):
        """This property is protected."""
        ...


class BrainSentimentIndicator7Day(QuantConnect.DataSource.BrainSentimentIndicatorBase[QuantConnect_DataSource_BrainSentimentIndicator7Day]):
    """Brain sentiment indicator on 7 days of news"""

    DataSourceId: int
    """Data source ID"""

    @property
    def LookbackDays(self) -> int:
        """This property is protected."""
        ...

    @LookbackDays.setter
    def LookbackDays(self, value: int):
        """This property is protected."""
        ...


class BrainStockRanking3Day(QuantConnect.DataSource.BrainStockRankingBase[QuantConnect_DataSource_BrainStockRanking3Day]):
    """Brain universe stock rankings on expected returns in the next 3 days"""

    DataSourceId: int
    """Data source ID"""

    @property
    def LookbackDays(self) -> int:
        """This property is protected."""
        ...

    @LookbackDays.setter
    def LookbackDays(self, value: int):
        """This property is protected."""
        ...


class BrainSentimentIndicator30Day(QuantConnect.DataSource.BrainSentimentIndicatorBase[QuantConnect_DataSource_BrainSentimentIndicator30Day]):
    """Brain sentiment indicator on 30 days of news"""

    DataSourceId: int
    """Data source ID"""

    @property
    def LookbackDays(self) -> int:
        """This property is protected."""
        ...

    @LookbackDays.setter
    def LookbackDays(self, value: int):
        """This property is protected."""
        ...


class QuiverWikipediaUniverse(QuantConnect.Data.BaseData):
    """Universe Selection helper class for QuiverWikipedia dataset"""

    @property
    def Date(self) -> datetime.datetime:
        """The date of the Page View count"""
        ...

    @Date.setter
    def Date(self, value: datetime.datetime):
        """The date of the Page View count"""
        ...

    @property
    def PageViews(self) -> typing.Optional[float]:
        """The company's Wikipedia Page Views on the given date"""
        ...

    @PageViews.setter
    def PageViews(self, value: typing.Optional[float]):
        """The company's Wikipedia Page Views on the given date"""
        ...

    @property
    def WeekPercentChange(self) -> typing.Optional[float]:
        """
        The view count % change over the week prior to the date.
        Represented as a whole number (e.g. 100% = 100.0)
        """
        ...

    @WeekPercentChange.setter
    def WeekPercentChange(self, value: typing.Optional[float]):
        """
        The view count % change over the week prior to the date.
        Represented as a whole number (e.g. 100% = 100.0)
        """
        ...

    @property
    def MonthPercentChange(self) -> typing.Optional[float]:
        """
        The view count % change over the month prior to the date
        Represented as a whole number (e.g. 100% = 100.0)
        """
        ...

    @MonthPercentChange.setter
    def MonthPercentChange(self, value: typing.Optional[float]):
        """
        The view count % change over the month prior to the date
        Represented as a whole number (e.g. 100% = 100.0)
        """
        ...

    @property
    def Period(self) -> datetime.timedelta:
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @Period.setter
    def Period(self, value: datetime.timedelta):
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the URL string source of the file. This will be converted to a stream
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: String URL of source file.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: Line of data
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: New instance.
        """
        ...


class QuiverWikipedia(QuantConnect.Data.BaseData):
    """Wikipedia Page Views for the specified company"""

    DataSourceId: int
    """Data source ID"""

    @property
    def Date(self) -> datetime.datetime:
        """The date of the Page View count"""
        ...

    @Date.setter
    def Date(self, value: datetime.datetime):
        """The date of the Page View count"""
        ...

    @property
    def PageViews(self) -> typing.Optional[float]:
        """The company's Wikipedia Page Views on the given date"""
        ...

    @PageViews.setter
    def PageViews(self, value: typing.Optional[float]):
        """The company's Wikipedia Page Views on the given date"""
        ...

    @property
    def WeekPercentChange(self) -> typing.Optional[float]:
        """
        The view count % change over the week prior to the date.
        Represented as a whole number (e.g. 100% = 100.0)
        """
        ...

    @WeekPercentChange.setter
    def WeekPercentChange(self, value: typing.Optional[float]):
        """
        The view count % change over the week prior to the date.
        Represented as a whole number (e.g. 100% = 100.0)
        """
        ...

    @property
    def MonthPercentChange(self) -> typing.Optional[float]:
        """
        The view count % change over the month prior to the date
        Represented as a whole number (e.g. 100% = 100.0)
        """
        ...

    @MonthPercentChange.setter
    def MonthPercentChange(self, value: typing.Optional[float]):
        """
        The view count % change over the month prior to the date
        Represented as a whole number (e.g. 100% = 100.0)
        """
        ...

    @property
    def Period(self) -> datetime.timedelta:
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @Period.setter
    def Period(self, value: datetime.timedelta):
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @overload
    def __init__(self) -> None:
        """Required for successful Json.NET deserialization"""
        ...

    @overload
    def __init__(self, csvLine: str) -> None:
        """
        Creates a new instance of QuiverWikipedia from a CSV line
        
        :param csvLine: CSV line
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The DateTimeZone of this data type.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the Subscription Data Source gained from the URL
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Subscription Data Source.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reader converts each line of the data source into BaseData objects.
        
        :param config: Subscription data config setup object
        :param line: Content of the source document
        :param date: Date of the requested data
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Quiver Wikipedia object.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates if there is support for mapping
        
        :returns: True indicates mapping should be used.
        """
        ...

    def ToString(self) -> str:
        """Formats a string with the Quiver Wikipedia information."""
        ...


class USEnergy(QuantConnect.Data.BaseData):
    """United States Energy Information Administration (EIA). This loads U.S. Energy data from QuantConnect's cache."""

    class Petroleum(System.Object):
        """Petroleum"""

        class UnitedStates(System.Object):
            """United States"""

            WeeklyRefinerAndBlenderAdjustedNetProductionOfFinishedMotorGasoline: str = "PET.WGFRPUS2.W"
            """U.S. Refiner and Blender Adjusted Net Production of Finished Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfFinishedMotorGasoline: str = "PET.WGFSTUS1.W"
            """U.S. Ending Stocks of Finished Motor Gasoline in Thousand Barrels (Mbbl)"""

            WeeklyProductSuppliedOfFinishedMotorGasoline: str = "PET.WGFUPUS2.W"
            """U.S. Product Supplied of Finished Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfCrudeOilInSpr: str = "PET.WCSSTUS1.W"
            """U.S. Ending Stocks of Crude Oil in SPR in Thousand Barrels (Mbbl)"""

            WeeklyRefinerAndBlenderNetProductionOfDistillateFuelOilGreaterThan500PpmSulfur: str = "PET.WDGRPUS2.W"
            """U.S.  Refiner and Blender Net Production of Distillate Fuel Oil Greater than 500 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfDistillateFuelOilGreaterThan500PpmSulfur: str = "PET.WDGSTUS1.W"
            """U.S. Ending Stocks of Distillate Fuel Oil, Greater Than 500 ppm Sulfur in Thousand Barrels (Mbbl)"""

            WeeklyExportsOfTotalDistillate: str = "PET.WDIEXUS2.W"
            """U.S. Exports of Total Distillate in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfDistillateFuelOil: str = "PET.WDIIMUS2.W"
            """U.S. Imports of Distillate Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfDistillateFuelOil: str = "PET.WDIRPUS2.W"
            """U.S. Refiner and Blender Net Production of Distillate Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfKeroseneTypeJetFuel: str = "PET.WKJSTUS1.W"
            """U.S. Ending Stocks of Kerosene-Type Jet Fuel in Thousand Barrels (Mbbl)"""

            WeeklyProductSuppliedOfKeroseneTypeJetFuel: str = "PET.WKJUPUS2.W"
            """U.S. Product Supplied of Kerosene-Type Jet Fuel in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfTotalGasoline: str = "PET.WGTIMUS2.W"
            """U.S. Imports of Total Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfTotalGasoline: str = "PET.WGTSTUS1.W"
            """U.S. Ending Stocks of Total Gasoline in Thousand Barrels (Mbbl)"""

            WeeklyGrossInputsIntoRefineries: str = "PET.WGIRIUS2.W"
            """U.S. Gross Inputs into Refineries in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfReformulatedMotorGasoline: str = "PET.WGRIMUS2.W"
            """U.S. Imports of Reformulated Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfReformulatedMotorGasoline: str = "PET.WGRRPUS2.W"
            """U.S. Refiner and Blender Net Production of Reformulated Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfReformulatedMotorGasoline: str = "PET.WGRSTUS1.W"
            """U.S. Ending Stocks of Reformulated Motor Gasoline in Thousand Barrels (Mbbl)"""

            WeeklyEndingStocksOfDistillateFuelOil: str = "PET.WDISTUS1.W"
            """U.S. Ending Stocks of Distillate Fuel Oil in Thousand Barrels (Mbbl)"""

            WeeklyProductSuppliedOfDistillateFuelOil: str = "PET.WDIUPUS2.W"
            """U.S. Product Supplied of Distillate Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfMilitaryKeroseneTypeJetFuel: str = "PET.WKMRPUS2.W"
            """U.S.  Refiner and Blender Net Production of Military Kerosene-Type Jet Fuel in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyOperableCrudeOilDistillationCapacity: str = "PET.WOCLEUS2.W"
            """U. S. Operable Crude Oil Distillation Capacity in Thousand Barrels per Calendar Day (Mbbl/d)"""

            WeeklyPropyleneNonfuelUseStocksAtBulkTerminals: str = "PET.WPLSTUS1.W"
            """U.S. Propylene Nonfuel Use Stocks at Bulk Terminals in Thousand Barrels (Mbbl)"""

            WeeklyEndingStocksOfPropaneAndPropylene: str = "PET.WPRSTUS1.W"
            """U.S. Ending Stocks of Propane and Propylene in Thousand Barrels (Mbbl)"""

            WeeklyPercentUtilizationOfRefineryOperableCapacity: str = "PET.WPULEUS3.W"
            """U.S. Percent Utilization of Refinery Operable Capacity in Percent (%)"""

            WeeklyExportsOfResidualFuelOil: str = "PET.WREEXUS2.W"
            """U.S. Exports of Residual Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfResidualFuelOil: str = "PET.WREIMUS2.W"
            """U.S. Imports of Residual Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfCommercialKeroseneTypeJetFuel: str = "PET.WKCRPUS2.W"
            """U.S.  Refiner and Blender Net Production of Commercial Kerosene-Type Jet Fuel in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyExportsOfKeroseneTypeJetFuel: str = "PET.WKJEXUS2.W"
            """U.S. Exports of Kerosene-Type Jet Fuel in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfKeroseneTypeJetFuel: str = "PET.WKJIMUS2.W"
            """U.S. Imports of Kerosene-Type Jet Fuel in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfKeroseneTypeJetFuel: str = "PET.WKJRPUS2.W"
            """U.S. Refiner and Blender Net Production of Kerosene-Type Jet Fuel in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksExcludingSprOfCrudeOil: str = "PET.WCESTUS1.W"
            """U.S. Ending Stocks excluding SPR of Crude Oil in Thousand Barrels (Mbbl)"""

            WeeklyExportsOfCrudeOil: str = "PET.WCREXUS2.W"
            """U.S. Exports of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyFieldProductionOfCrudeOil: str = "PET.WCRFPUS2.W"
            """U.S. Field Production of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfCrudeOil: str = "PET.WCRIMUS2.W"
            """U.S. Imports of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyNetImportsOfCrudeOil: str = "PET.WCRNTUS2.W"
            """U.S. Net Imports of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetInputOfCrudeOil: str = "PET.WCRRIUS2.W"
            """U.S. Refiner Net Input of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfResidualFuelOil: str = "PET.WRERPUS2.W"
            """U.S. Refiner and Blender Net Production of Residual Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfResidualFuelOil: str = "PET.WRESTUS1.W"
            """U.S. Ending Stocks of Residual Fuel Oil in Thousand Barrels (Mbbl)"""

            WeeklyProductSuppliedOfResidualFuelOil: str = "PET.WREUPUS2.W"
            """U.S. Product Supplied of Residual Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyExportsOfTotalPetroleumProducts: str = "PET.WRPEXUS2.W"
            """U.S. Exports of Total Petroleum Products in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfTotalPetroleumProducts: str = "PET.WRPIMUS2.W"
            """U.S. Imports of Total Petroleum Products in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyNetImportsOfTotalPetroleumProducts: str = "PET.WRPNTUS2.W"
            """U.S. Net Imports of Total Petroleum Products in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyProductSuppliedOfPetroleumProducts: str = "PET.WRPUPUS2.W"
            """U.S. Product Supplied of Petroleum Products in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksExcludingSprOfCrudeOilAndPetroleumProducts: str = "PET.WTESTUS1.W"
            """U.S. Ending Stocks excluding SPR of Crude Oil and Petroleum Products in Thousand Barrels (Mbbl)"""

            WeeklyExportsOfCrudeOilAndPetroleumProducts: str = "PET.WTTEXUS2.W"
            """U.S. Exports of Crude Oil and Petroleum Products in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfCrudeOilAndPetroleumProducts: str = "PET.WTTIMUS2.W"
            """U.S. Imports of Crude Oil and Petroleum Products in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyNetImportsOfCrudeOilAndPetroleumProducts: str = "PET.WTTNTUS2.W"
            """U.S. Net Imports of Crude Oil and Petroleum Products in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfCrudeOilAndPetroleumProducts: str = "PET.WTTSTUS1.W"
            """U.S. Ending Stocks of Crude Oil and Petroleum Products in Thousand Barrels (Mbbl)"""

            WeeklyEndingStocksOfUnfinishedOils: str = "PET.WUOSTUS1.W"
            """U.S. Ending Stocks of Unfinished Oils in Thousand Barrels (Mbbl)"""

            WeeklyRefinerAndBlenderNetProductionOfOtherFinishedConventionalMotorGasoline: str = "PET.WG6TP_NUS_2.W"
            """U.S. Refiner and Blender Net Production of Other Finished Conventional Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfDistillateFuelOil0To15PpmSulfur: str = "PET.WD0TP_NUS_2.W"
            """U.S. Refiner and Blender Net Production of Distillate Fuel Oil, 0 to 15 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfDistillateFuelOilGreaterThan15To500PpmSulfur: str = "PET.WD1ST_NUS_1.W"
            """U.S. Ending Stocks of Distillate Fuel Oil, Greater than 15 to 500 ppm Sulfur in Thousand Barrels (Mbbl)"""

            WeeklyProductionOfDistillateFuelOilGreaterThan15To500PpmSulfur: str = "PET.WD1TP_NUS_2.W"
            """U.S. Production of Distillate Fuel Oil, Greater than 15 to 500 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfReformulatedMotorGasolineWithFuelAlcohol: str = "PET.WG1ST_NUS_1.W"
            """U.S. Ending Stocks of Reformulated Motor Gasoline with Fuel ALcohol in Thousand Barrels (Mbbl)"""

            WeeklyEndingStocksOfCrudeOil: str = "PET.WCRSTUS1.W"
            """U.S. Ending Stocks of Crude Oil in Thousand Barrels (Mbbl)"""

            WeeklyCrudeOilImportsBySpr: str = "PET.WCSIMUS2.W"
            """U.S. Crude Oil Imports by SPR in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfGasolineBlendingComponents: str = "PET.WBCIMUS2.W"
            """U.S. Imports of Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfGasolineBlendingComponents: str = "PET.WBCSTUS1.W"
            """U.S. Ending Stocks of Gasoline Blending Components in Thousand Barrels (Mbbl)"""

            WeeklyCommercialCrudeOilImportsExcludingSpr: str = "PET.WCEIMUS2.W"
            """U.S. Commercial Crude Oil Imports Excluding SPR in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerBlenderAndGasPlantNetProductionOfPropaneAndPropylene: str = "PET.WPRTP_NUS_2.W"
            """U.S. Refiner, Blender, and Gas Plant Net Production of Propane and Propylene in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfFinishedReformulatedMotorGasolineWithEthanol: str = "PET.WG1TP_NUS_2.W"
            """U.S. Refiner and Blender Net Production of Finished Reformulated Motor Gasoline with Ethanol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfReformulatedMotorGasolineNonOxygentated: str = "PET.WG3ST_NUS_1.W"
            """U.S. Ending Stocks of Reformulated Motor Gasoline, Non-Oxygentated in Thousand Barrels (Mbbl)"""

            WeeklyEndingStocksOfConventionalMotorGasoline: str = "PET.WG4ST_NUS_1.W"
            """U.S. Ending Stocks of Conventional Motor Gasoline in Thousand Barrels (Mbbl)"""

            WeeklyRefinerAndBlenderNetProductionOfConventionalMotorGasoline: str = "PET.WG4TP_NUS_2.W"
            """U.S. Refiner and Blender Net Production of Conventional Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfConventionalMotorGasolineWithFuelEthanol: str = "PET.WG5ST_NUS_1.W"
            """U.S. Ending Stocks of Conventional Motor Gasoline with Fuel Ethanol in Thousand Barrels (Mbbl)"""

            WeeklyRefinerAndBlenderNetProductionOfFinishedConventionalMotorGasolineWithEthanol: str = "PET.WG5TP_NUS_2.W"
            """U.S. Refiner and Blender Net Production of Finished Conventional Motor Gasoline with Ethanol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfOtherConventionalMotorGasoline: str = "PET.WG6ST_NUS_1.W"
            """U.S. Ending Stocks of Other Conventional Motor Gasoline in Thousand Barrels (Mbbl)"""

            WeeklyRefinerAndBlenderNetInputOfConventionalCbobGasolineBlendingComponents: str = "PET.WO6RI_NUS_2.W"
            """U.S. Refiner and Blender Net Input of Conventional CBOB Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfConventionalCbobGasolineBlendingComponents: str = "PET.WO6ST_NUS_1.W"
            """U.S. Ending Stocks of Conventional CBOB Gasoline Blending Components in Thousand Barrels (Mbbl)"""

            WeeklyRefinerAndBlenderNetInputOfConventionalGtabGasolineBlendingComponents: str = "PET.WO7RI_NUS_2.W"
            """U.S. Refiner and Blender Net Input of Conventional GTAB Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfConventionalGtabGasolineBlendingComponents: str = "PET.WO7ST_NUS_1.W"
            """U.S. Ending Stocks of Conventional GTAB Gasoline Blending Components in Thousand Barrels (Mbbl)"""

            WeeklyRefinerAndBlenderNetInputOfConventionalOtherGasolineBlendingComponents: str = "PET.WO9RI_NUS_2.W"
            """U.S. Refiner and Blender Net Input of Conventional Other Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfConventionalOtherGasolineBlendingComponents: str = "PET.WO9ST_NUS_1.W"
            """U.S. Ending Stocks of Conventional Other Gasoline Blending Components in Thousand Barrels (Mbbl)"""

            WeeklyNo2HeatingOilWholesaleResalePrice: str = "PET.W_EPD2F_PWR_NUS_DPG.W"
            """U.S. No. 2 Heating Oil Wholesale/Resale Price in Dollars per Gallon ($/gal)"""

            WeeklyCrudeOilStocksInTransitOnShipsFromAlaska: str = "PET.W_EPC0_SKA_NUS_MBBL.W"
            """U.S. Crude Oil Stocks in Transit (on Ships) from Alaska in Thousand Barrels (Mbbl)"""

            WeeklyDaysOfSupplyOfCrudeOilExcludingSpr: str = "PET.W_EPC0_VSD_NUS_DAYS.W"
            """U.S. Days of Supply of Crude Oil excluding SPR in Number of Days (Days)"""

            WeeklyDaysOfSupplyOfTotalDistillate: str = "PET.W_EPD0_VSD_NUS_DAYS.W"
            """U.S. Days of Supply of Total Distillate in Number of Days (Days)"""

            WeeklyWeeklyNo2HeatingOilResidentialPrice: str = "PET.W_EPD2F_PRS_NUS_DPG.W"
            """U.S. Weekly No. 2 Heating Oil Residential Price in Dollars per Gallon ($/gal)"""

            WeeklyProductSuppliedOfPropaneAndPropylene: str = "PET.WPRUP_NUS_2.W"
            """U.S. Product Supplied of Propane and Propylene in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyProductSuppliedOfOtherOils: str = "PET.WWOUP_NUS_2.W"
            """U.S. Product Supplied of Other Oils in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetInputOfGasolineBlendingComponents: str = "PET.WBCRI_NUS_2.W"
            """U.S. Refiner and Blender Net Input of Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfDistillateFuelOil0To15PpmSulfur: str = "PET.WD0ST_NUS_1.W"
            """U.S. Ending Stocks of Distillate Fuel Oil, 0 to 15 ppm Sulfur in Thousand Barrels (Mbbl)"""

            WeeklyDaysOfSupplyOfKeroseneTypeJetFuel: str = "PET.W_EPJK_VSD_NUS_DAYS.W"
            """U.S. Days of Supply of Kerosene-Type Jet Fuel in Number of Days (Days)"""

            WeeklyDaysOfSupplyOfTotalGasoline: str = "PET.W_EPM0_VSD_NUS_DAYS.W"
            """U.S. Days of Supply of Total Gasoline in Number of Days (Days)"""

            WeeklyEndingStocksOfAsphaltAndRoadOil: str = "PET.W_EPPA_SAE_NUS_MBBL.W"
            """U.S. Ending Stocks of Asphalt and Road Oil in Thousand Barrels (Mbbl)"""

            WeeklyEndingStocksOfKerosene: str = "PET.W_EPPK_SAE_NUS_MBBL.W"
            """U.S. Ending Stocks of Kerosene in Thousand Barrels (Mbbl)"""

            WeeklySupplyAdjustmentOfDistillateFuelOilGreaterThan15To500PpmSulfur: str = "PET.W_EPDM10_VUA_NUS_2.W"
            """U.S. Supply Adjustment of Distillate Fuel Oil, Greater than 15 to 500 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfConventionalMotorGasolineWithFuelEthanol: str = "PET.WG5IM_NUS-Z00_2.W"
            """U.S. Imports of Conventional Motor Gasoline with Fuel Ethanol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfOtherConventionalMotorGasoline: str = "PET.WG6IM_NUS-Z00_2.W"
            """U.S. Imports of Other Conventional Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfDistillateFuelOil0To15PpmSulfur: str = "PET.WD0IM_NUS-Z00_2.W"
            """U.S. Imports of Distillate Fuel Oil, 0 to 15 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfDistillateFuelOilGreaterThan15To500PpmSulfur: str = "PET.WD1IM_NUS-Z00_2.W"
            """U.S. Imports of Distillate Fuel Oil, Greater than 15 to 500 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfDistillateFuelOilGreaterThan500To2000PpmSulfur: str = "PET.WD2IM_NUS-Z00_2.W"
            """U.S. Imports of Distillate Fuel Oil, Greater than 500 to 2000 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfPropaneAndPropylene: str = "PET.WPRIM_NUS-Z00_2.W"
            """U.S. Imports of Propane and Propylene in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfConventionalGtabGasolineBlendingComponents: str = "PET.WO7IM_NUS-Z00_2.W"
            """U.S. Imports of Conventional GTAB Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfDistillateFuelOilGreaterThan2000PpmSulfur: str = "PET.WD3IM_NUS-Z00_2.W"
            """U.S. Imports of Distillate Fuel Oil, Greater than 2000 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfReformulatedMotorGasolineWithFuelAlcohol: str = "PET.WG1IM_NUS-Z00_2.W"
            """U.S. Imports of Reformulated Motor Gasoline with Fuel ALcohol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfConventionalMotorGasoline: str = "PET.WG4IM_NUS-Z00_2.W"
            """U.S. Imports of Conventional Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfConventionalOtherGasolineBlendingComponents: str = "PET.WO9IM_NUS-Z00_2.W"
            """U.S. Imports of Conventional Other Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfConventionalCbobGasolineBlendingComponents: str = "PET.WO6IM_NUS-Z00_2.W"
            """U.S. Imports of Conventional CBOB Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfKerosene: str = "PET.W_EPPK_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Kerosene in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfKerosene: str = "PET.W_EPPK_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Kerosene in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfOtherOilsExcludingFuelEthanol: str = "PET.W_EPPO6_SAE_NUS_MBBL.W"
            """U.S. Ending Stocks of Other Oils (Excluding Fuel Ethanol) in Thousand Barrels (Mbbl)"""

            WeeklyRefinerNetProductionOfResidualFuelOil: str = "PET.W_EPPR_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Residual Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfReformulatedMotorGasoline: str = "PET.W_EPM0R_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Reformulated Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfReformulatedMotorGasoline: str = "PET.W_EPM0R_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Reformulated Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfFuelEthanol: str = "PET.W_EPOOXE_SAE_NUS_MBBL.W"
            """U.S. Ending Stocks of Fuel Ethanol in Thousand Barrels (Mbbl)"""

            WeeklyBlenderNetProductionOfDistillateFuelOil: str = "PET.W_EPD0_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Distillate Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfDistillateFuelOil: str = "PET.W_EPD0_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Distillate Fuel Oil in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfKeroseneTypeJetFuel: str = "PET.W_EPJK_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Kerosene-Type Jet Fuel in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfKeroseneTypeJetFuel: str = "PET.W_EPJK_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Kerosene-Type Jet Fuel in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyPropaneResidentialPrice: str = "PET.W_EPLLPA_PRS_NUS_DPG.W"
            """U.S. Propane Residential Price in Dollars per Gallon ($/gal)"""

            WeeklyPropaneWholesaleResalePrice: str = "PET.W_EPLLPA_PWR_NUS_DPG.W"
            """U.S. Propane Wholesale/Resale Price in Dollars per Gallon ($/gal)"""

            WeeklyRefinerAndBlenderNetInputOfMotorGasolineBlendingComponentsRbob: str = "PET.W_EPOBGRR_YIR_NUS_MBBLD.W"
            """U.S. Refiner and Blender Net Input of Motor Gasoline Blending Components, RBOB in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfNgplsLrgsExcludingPropanePropylene: str = "PET.W_EPL0XP_SAE_NUS_MBBL.W"
            """U.S. Ending Stocks of NGPLs/LRGs (Excluding Propane/Propylene) in Thousand Barrels (Mbbl)"""

            WeeklyDaysOfSupplyOfPropanePropylene: str = "PET.W_EPLLPZ_VSD_NUS_DAYS.W"
            """U.S. Days of Supply of Propane/Propylene in Number of Days (Days)"""

            WeeklyBlenderNetProductionOfConventionalMotorGasoline: str = "PET.W_EPM0C_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Conventional Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfConventionalMotorGasoline: str = "PET.W_EPM0C_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Conventional Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklySupplyAdjustmentOfFinishedMotorGasoline: str = "PET.W_EPM0F_VUA_NUS_MBBLD.W"
            """U.S. Supply Adjustment of Finished Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfFinishedMotorGasoline: str = "PET.W_EPM0F_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Finished Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfFinishedMotorGasoline: str = "PET.W_EPM0F_YPR_NUS_MBBLD.W"
            """U.S. Refiner and Blender Net Production of Finished Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfFinishedMotorGasoline: str = "PET.W_EPM0F_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Finished Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfDistillateFuelOilGreaterThan500PpmSulfur: str = "PET.W_EPD00H_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Distillate Fuel Oil, Greater Than 500 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfDistillateFuelOilGreaterThan500PpmSulfur: str = "PET.W_EPD00H_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Distillate Fuel Oil, Greater Than 500 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfDistillateFuelOilGreaterThan15To500PpmSulfur: str = "PET.W_EPDM10_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Distillate Fuel Oil, Greater than 15 to 500 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfDistillateFuelOilGreaterThan15To500PpmSulfur: str = "PET.W_EPDM10_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Distillate Fuel Oil, Greater than 15 to 500 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfDistillateFuelOil0To15PpmSulfur: str = "PET.W_EPDXL0_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Distillate Fuel Oil, 0 to 15 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfDistillateFuelOil0To15PpmSulfur: str = "PET.W_EPDXL0_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Distillate Fuel Oil, 0 to 15 ppm Sulfur in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfConventionalMotorGasolineWithFuelEthanol: str = "PET.W_EPM0CA_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Conventional Motor Gasoline with Fuel Ethanol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfConventionalMotorGasolineWithFuelEthanol: str = "PET.W_EPM0CA_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Conventional Motor Gasoline with Fuel Ethanol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfOtherConventionalMotorGasoline: str = "PET.W_EPM0CO_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Other Conventional Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfOtherConventionalMotorGasoline: str = "PET.W_EPM0CO_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Other Conventional Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfReformulatedMotorGasolineWithFuelAlcohol: str = "PET.W_EPM0RA_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Reformulated Motor Gasoline with Fuel ALcohol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfReformulatedMotorGasolineWithFuelAlcohol: str = "PET.W_EPM0RA_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Reformulated Motor Gasoline with Fuel ALcohol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyOxygenatePlantProductionOfFuelEthanol: str = "PET.W_EPOOXE_YOP_NUS_MBBLD.W"
            """U.S. Oxygenate Plant Production of Fuel Ethanol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfMotorGasolineFinishedConventionalEd55AndLower: str = "PET.W_EPM0CAL55_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Motor Gasoline, Finished, Conventional, Ed55 and Lower in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfFinishedConventionalMotorGasolineEd55AndLower: str = "PET.W_EPM0CAL55_YPT_NUS_MBBLD.W"
            """U.S. Refiner and Blender Net Production of Finished Conventional Motor Gasoline, Ed 55 and Lower in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfMotorGasolineFinishedConventionalEd55AndLower: str = "PET.W_EPM0CAL55_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Motor Gasoline, Finished, Conventional, Ed55 and Lower in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyExportsOfFinishedMotorGasoline: str = "PET.W_EPM0F_EEX_NUS-Z00_MBBLD.W"
            """U.S. Exports of Finished Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfFinishedMotorGasoline: str = "PET.W_EPM0F_IM0_NUS-Z00_MBBLD.W"
            """U.S. Imports of Finished Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfOtherReformulatedMotorGasoline: str = "PET.W_EPM0RO_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Other Reformulated Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfOtherFinishedReformulatedMotorGasoline: str = "PET.W_EPM0RO_YPT_NUS_MBBLD.W"
            """U.S. Refiner and Blender Net Production of Other Finished Reformulated Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfOtherReformulatedMotorGasoline: str = "PET.W_EPM0RO_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Other Reformulated Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfMotorGasolineBlendingComponentsRbob: str = "PET.W_EPOBGRR_SAE_NUS_MBBL.W"
            """U.S. Ending Stocks of Motor Gasoline Blending Components, RBOB in Thousand Barrels (Mbbl)"""

            WeeklyRefinerAndBlenderNetInputOfFuelEthanol: str = "PET.W_EPOOXE_YIR_NUS_MBBLD.W"
            """U.S. Refiner and Blender Net Input of Fuel Ethanol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfMotorGasolineFinishedConventionalGreaterThanEd55: str = "PET.W_EPM0CAG55_IM0_NUS-Z00_MBBLD.W"
            """U.S. Imports of Motor Gasoline, Finished, Conventional, Greater than Ed55 in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfMotorGasolineFinishedConventionalEd55AndLower: str = "PET.W_EPM0CAL55_IM0_NUS-Z00_MBBLD.W"
            """U.S. Imports of Motor Gasoline, Finished, Conventional, Ed55 and Lower in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyCrudeOilImportsForSprByOthers: str = "PET.W_EPC0_IMU_NUS-Z00_MBBLD.W"
            """U.S. Crude Oil Imports for SPR by Others in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfConventionalMotorGasolineGreaterThanEd55: str = "PET.W_EPM0CAG55_SAE_NUS_MBBL.W"
            """U.S. Ending Stocks of Conventional Motor Gasoline, Greater than Ed55 in Thousand Barrels (Mbbl)"""

            WeeklyImportsOfFuelEthanol: str = "PET.W_EPOOXE_IM0_NUS-Z00_MBBLD.W"
            """U.S. Imports of Fuel Ethanol in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfLiquefiedPetroleumGassesLessPropanePropylene: str = "PET.W_EPL0XP_IM0_NUS-Z00_MBBLD.W"
            """U.S. Imports of Liquefied Petroleum Gasses Less Propane/Propylene in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyExportsOfPropaneAndPropylene: str = "PET.W_EPLLPZ_EEX_NUS-Z00_MBBLD.W"
            """U.S. Exports of Propane and Propylene in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfOtherReformulatedMotorGasoline: str = "PET.W_EPM0RO_IM0_NUS-Z00_MBBLD.W"
            """U.S. Imports of Other Reformulated Motor Gasoline in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyBlenderNetProductionOfMotorGasolineFinishedConventionalGreaterThanEd55: str = "PET.W_EPM0CAG55_YPB_NUS_MBBLD.W"
            """U.S. Blender Net Production of Motor Gasoline, Finished, Conventional, Greater Than Ed55 in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerAndBlenderNetProductionOfFinishedConventionalMotorGasolineGreaterThanEd55: str = "PET.W_EPM0CAG55_YPT_NUS_MBBLD.W"
            """U.S. Refiner and Blender Net Production of Finished Conventional Motor Gasoline, Greater than Ed 55 in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRefinerNetProductionOfFinishedConventionalMotorGasolineGreaterThanEd55: str = "PET.W_EPM0CAG55_YPY_NUS_MBBLD.W"
            """U.S. Refiner Net Production of Finished Conventional Motor Gasoline, Greater than Ed 55 in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfConventionalMotorGasolineEd55AndLower: str = "PET.W_EPM0CAL55_SAE_NUS_MBBL.W"
            """U.S. Ending Stocks of Conventional Motor Gasoline, Ed55 and Lower in Thousand Barrels (Mbbl)"""

            WeeklyImportsOfKerosene: str = "PET.W_EPPK_IM0_NUS-Z00_MBBLD.W"
            """U.S. Imports of Kerosene in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyExportsOfOtherOils: str = "PET.W_EPPO4_EEX_NUS-Z00_MBBLD.W"
            """U.S. Exports of Other Oils in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfOtherOilsExcludingFuelEthanol: str = "PET.W_EPPO6_IM0_NUS-Z00_MBBLD.W"
            """U.S. Imports of Other Oils (Excluding Fuel Ethanol) in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsFromAllCountriesOfMotorGasolineBlendingComponentsRbob: str = "PET.W_EPOBGRR_IM0_NUS-Z00_MBBLD.W"
            """U.S. Imports from  All Countries of Motor Gasoline Blending Components, RBOB in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyRegularAllFormulationsRetailGasolinePrices: str = "PET.EMM_EPMR_PTE_NUS_DPG.W"
            """U.S. Regular All Formulations Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyMidgradeAllFormulationsRetailGasolinePrices: str = "PET.EMM_EPMM_PTE_NUS_DPG.W"
            """U.S. Midgrade All Formulations Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyPremiumAllFormulationsRetailGasolinePrices: str = "PET.EMM_EPMP_PTE_NUS_DPG.W"
            """U.S. Premium All Formulations Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyAllGradesAllFormulationsRetailGasolinePrices: str = "PET.EMM_EPM0_PTE_NUS_DPG.W"
            """U.S. All Grades All Formulations Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyAllGradesReformulatedRetailGasolinePrices: str = "PET.EMM_EPM0R_PTE_NUS_DPG.W"
            """U.S. All Grades Reformulated Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyMidgradeReformulatedRetailGasolinePrices: str = "PET.EMM_EPMMR_PTE_NUS_DPG.W"
            """U.S. Midgrade Reformulated Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyPremiumReformulatedRetailGasolinePrices: str = "PET.EMM_EPMPR_PTE_NUS_DPG.W"
            """U.S. Premium Reformulated Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyRegularConventionalRetailGasolinePrices: str = "PET.EMM_EPMRU_PTE_NUS_DPG.W"
            """U.S. Regular Conventional Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyRegularReformulatedRetailGasolinePrices: str = "PET.EMM_EPMRR_PTE_NUS_DPG.W"
            """U.S. Regular Reformulated Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyNo2DieselRetailPrices: str = "PET.EMD_EPD2D_PTE_NUS_DPG.W"
            """U.S. No 2 Diesel Retail Prices in Dollars per Gallon ($/gal)"""

            WeeklyPremiumConventionalRetailGasolinePrices: str = "PET.EMM_EPMPU_PTE_NUS_DPG.W"
            """U.S. Premium Conventional Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyMidgradeConventionalRetailGasolinePrices: str = "PET.EMM_EPMMU_PTE_NUS_DPG.W"
            """U.S. Midgrade Conventional Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyAllGradesConventionalRetailGasolinePrices: str = "PET.EMM_EPM0U_PTE_NUS_DPG.W"
            """U.S. All Grades Conventional Retail Gasoline Prices in Dollars per Gallon ($/gal)"""

            WeeklyNo2DieselUltraLowSulfur015PpmRetailPrices: str = "PET.EMD_EPD2DXL0_PTE_NUS_DPG.W"
            """U.S. No 2 Diesel Ultra Low Sulfur (0-15 ppm) Retail Prices in Dollars per Gallon ($/gal)"""

            WeeklyEndingStocksExcludingSprAndIncludingLeaseStockOfCrudeOil: str = "PET.W_EPC0_SAX_NUS_MBBL.W"
            """U.S. Ending Stocks excluding SPR and including Lease Stock of Crude Oil in Thousand Barrels (Mbbl)"""

            WeeklyNo2DieselLowSulfur15500PpmRetailPrices: str = "PET.EMD_EPD2DM10_PTE_NUS_DPG.W"
            """U.S. No 2 Diesel Low Sulfur (15-500 ppm) Retail Prices in Dollars per Gallon ($/gal)"""

            WeeklyImportsOfReformulatedRbobWithAlcoholGasolineBlendingComponents: str = "PET.WO3IM_NUS-Z00_2.W"
            """U.S. Imports of Reformulated RBOB with Alcohol Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyImportsOfReformulatedRbobWithEtherGasolineBlendingComponents: str = "PET.WO4IM_NUS-Z00_2.W"
            """U.S. Imports of Reformulated RBOB with Ether Gasoline Blending Components in Thousand Barrels per Day (Mbbl/d)"""

            WeeklyEndingStocksOfReformulatedGtabGasolineBlendingComponents: str = "PET.WO2ST_NUS_1.W"
            """U.S. Ending Stocks of Reformulated GTAB Gasoline Blending Components in Thousand Barrels (Mbbl)"""

            WeeklyEndingStocksOfReformulatedRbobWithAlcoholGasolineBlendingComponents: str = "PET.WO3ST_NUS_1.W"
            """U.S. Ending Stocks of Reformulated RBOB with Alcohol Gasoline Blending Components in Thousand Barrels (Mbbl)"""

            WeeklyEndingStocksOfReformulatedRbobWithEtherGasolineBlendingComponents: str = "PET.WO4ST_NUS_1.W"
            """U.S. Ending Stocks of Reformulated RBOB with Ether Gasoline Blending Components in Thousand Barrels (Mbbl)"""

        class EquatorialGuinea(System.Object):
            """Equatorial Guinea"""

            WeeklyImportsFromEquatorialGuineaOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NEK_MBBLD.W"
            """U.S. Imports from Equatorial Guinea of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Iraq(System.Object):
            """Iraq"""

            WeeklyImportsFromIraqOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NIZ_MBBLD.W"
            """U.S. Imports from Iraq of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Kuwait(System.Object):
            """Kuwait"""

            WeeklyImportsFromKuwaitOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NKU_MBBLD.W"
            """U.S. Imports from Kuwait of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Mexico(System.Object):
            """Mexico"""

            WeeklyImportsFromMexicoOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NMX_MBBLD.W"
            """U.S. Imports from Mexico of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Nigeria(System.Object):
            """Nigeria"""

            WeeklyImportsFromNigeriaOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NNI_MBBLD.W"
            """U.S. Imports from Nigeria of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Norway(System.Object):
            """Norway"""

            WeeklyImportsFromNorwayOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NNO_MBBLD.W"
            """U.S. Imports from Norway of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Russia(System.Object):
            """Russia"""

            WeeklyImportsFromRussiaOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NRS_MBBLD.W"
            """U.S. Imports from Russia of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class SaudiArabia(System.Object):
            """Saudi Arabia"""

            WeeklyImportsFromSaudiArabiaOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NSA_MBBLD.W"
            """U.S. Imports from Saudi Arabia of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class UnitedKingdom(System.Object):
            """United Kingdom"""

            WeeklyImportsFromUnitedKingdomOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NUK_MBBLD.W"
            """U.S. Imports from United Kingdom of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Venezuela(System.Object):
            """Venezuela"""

            WeeklyImportsFromVenezuelaOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NVE_MBBLD.W"
            """U.S. Imports from Venezuela of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Algeria(System.Object):
            """Algeria"""

            WeeklyImportsFromAlgeriaOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NAG_MBBLD.W"
            """U.S. Imports from Algeria of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Angola(System.Object):
            """Angola"""

            WeeklyImportsFromAngolaOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NAO_MBBLD.W"
            """U.S. Imports from Angola of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Brazil(System.Object):
            """Brazil"""

            WeeklyImportsFromBrazilOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NBR_MBBLD.W"
            """U.S. Imports from Brazil of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Canada(System.Object):
            """Canada"""

            WeeklyImportsFromCanadaOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NCA_MBBLD.W"
            """U.S. Imports from Canada of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Congo(System.Object):
            """Congo"""

            WeeklyImportsFromCongoBrazzavilleOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NCF_MBBLD.W"
            """U.S. Imports from Congo (Brazzaville) of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Colombia(System.Object):
            """Colombia"""

            WeeklyImportsFromColombiaOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NCO_MBBLD.W"
            """U.S. Imports from Colombia of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

        class Ecuador(System.Object):
            """Ecuador"""

            WeeklyImportsFromEcuadorOfCrudeOil: str = "PET.W_EPC0_IM0_NUS-NEC_MBBLD.W"
            """U.S. Imports from Ecuador of Crude Oil in Thousand Barrels per Day (Mbbl/d)"""

    DataSourceId: int
    """Data source ID"""

    def Clone(self) -> QuantConnect.Data.BaseData:
        """
        Clones the data
        
        :returns: A clone of the object.
        """
        ...

    def DefaultResolution(self) -> int:
        """
        Gets the default resolution for this data and security type
        
        :returns: This method returns the int value of a member of the QuantConnect.Resolution enum.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Determines the location of the data
        
        :param config: Subscription configuration
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: Location of the data as a SubscriptionDataSource.
        """
        ...

    def IsSparseData(self) -> bool:
        """
        Indicates whether the data is sparse.
        If true, we disable logging for missing files
        
        :returns: true.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Parses the data from the line provided and loads it into LEAN
        
        :param config: Subscription configuration
        :param line: Line of data
        :param date: Date
        :param isLiveMode: Is live mode
        :returns: New instance of USEnergy.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates whether the data source is tied
        to an underlying symbol and requires that corporate
        events be applied to it as well, such as renames and delistings
        
        :returns: false.
        """
        ...

    def SupportedResolutions(self) -> System.Collections.Generic.List[QuantConnect.Resolution]:
        """Gets the supported resolution for this data and security type"""
        ...

    def ToString(self) -> str:
        """Converts the instance to string"""
        ...


class QuiverEventsBeta(QuantConnect.Data.BaseData):
    """Political beta for the specified company"""

    DataSourceId: int
    """Data source ID"""

    @property
    def Date(self) -> datetime.datetime:
        """The date of the events beta calculation"""
        ...

    @Date.setter
    def Date(self, value: datetime.datetime):
        """The date of the events beta calculation"""
        ...

    @property
    def EventName(self) -> str:
        """Event name (e.g. PresidentialElection2020)"""
        ...

    @EventName.setter
    def EventName(self, value: str):
        """Event name (e.g. PresidentialElection2020)"""
        ...

    @property
    def FirstEventName(self) -> str:
        """Name for first outcome (e.g. TrumpVictory)"""
        ...

    @FirstEventName.setter
    def FirstEventName(self, value: str):
        """Name for first outcome (e.g. TrumpVictory)"""
        ...

    @property
    def SecondEventName(self) -> str:
        """Name for second outcome (e.g. BidenVictory)"""
        ...

    @SecondEventName.setter
    def SecondEventName(self, value: str):
        """Name for second outcome (e.g. BidenVictory)"""
        ...

    @property
    def FirstEventBeta(self) -> typing.Optional[float]:
        """Correlation between daily excess returns and daily changes in first event odds"""
        ...

    @FirstEventBeta.setter
    def FirstEventBeta(self, value: typing.Optional[float]):
        """Correlation between daily excess returns and daily changes in first event odds"""
        ...

    @property
    def FirstEventOdds(self) -> float:
        """Odds of the first event happening, based on betting markets"""
        ...

    @FirstEventOdds.setter
    def FirstEventOdds(self, value: float):
        """Odds of the first event happening, based on betting markets"""
        ...

    @property
    def SecondEventBeta(self) -> typing.Optional[float]:
        """Correlation between daily excess returns and daily changes in second event odds"""
        ...

    @SecondEventBeta.setter
    def SecondEventBeta(self, value: typing.Optional[float]):
        """Correlation between daily excess returns and daily changes in second event odds"""
        ...

    @property
    def SecondEventOdds(self) -> float:
        """Odds of the second event happening, based on betting markets"""
        ...

    @SecondEventOdds.setter
    def SecondEventOdds(self, value: float):
        """Odds of the second event happening, based on betting markets"""
        ...

    @property
    def Period(self) -> datetime.timedelta:
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @Period.setter
    def Period(self, value: datetime.timedelta):
        """The period of time that occurs between the starting time and ending time of the data point"""
        ...

    @property
    def EndTime(self) -> datetime.datetime:
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @EndTime.setter
    def EndTime(self, value: datetime.datetime):
        """The time the data point ends at and becomes available to the algorithm"""
        ...

    @overload
    def __init__(self) -> None:
        """Required for successful Json.NET deserialization"""
        ...

    @overload
    def __init__(self, csvLine: str) -> None:
        """
        Creates a new instance of QuiverPoliticalBeta from a CSV line
        
        :param csvLine: CSV line
        """
        ...

    def DataTimeZone(self) -> typing.Any:
        """
        Specifies the data time zone for this data type. This is useful for custom data types
        
        :returns: The DateTimeZone of this data type.
        """
        ...

    def GetSource(self, config: QuantConnect.Data.SubscriptionDataConfig, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.SubscriptionDataSource:
        """
        Return the Subscription Data Source gained from the URL
        
        :param config: Configuration object
        :param date: Date of this source file
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Subscription Data Source.
        """
        ...

    def Reader(self, config: QuantConnect.Data.SubscriptionDataConfig, line: str, date: typing.Union[datetime.datetime, datetime.date], isLiveMode: bool) -> QuantConnect.Data.BaseData:
        """
        Reader converts each line of the data source into BaseData objects.
        
        :param config: Subscription data config setup object
        :param line: Content of the source document
        :param date: Date of the requested data
        :param isLiveMode: true if we're in live mode, false for backtesting mode
        :returns: Quiver Political Beta object.
        """
        ...

    def RequiresMapping(self) -> bool:
        """
        Indicates if there is support for mapping
        
        :returns: True indicates mapping should be used.
        """
        ...

    def ToString(self) -> str:
        """Formats a string with the Quiver Events Beta information."""
        ...


class NullData(QuantConnect.Data.BaseData):
    """Represents a custom data type place holder"""


