import json
import threading
import webbrowser

import pkce
import pytest
import requests
from requests import Response

from sym.flow.cli.errors import InvalidTokenError

from ....helpers.login.login_flow import BrowserRedirectFlow, LoginError
from ....models import SymOrganization


@pytest.fixture
def browser_redirect_flow(mocker):
    flow = BrowserRedirectFlow(11001)
    mocker.patch.object(pkce, "generate_code_verifier", return_value="test-verifier")
    mocker.patch.object(pkce, "get_code_challenge", return_value="test-challenge")
    mocker.patch.object(webbrowser, "open")
    return flow


@pytest.fixture
def wait_for_code_tester(browser_redirect_flow, global_options):
    def fixture(expected_state: str, expected_code: str, do_test):
        def run_test():
            code = browser_redirect_flow.wait_for_code(expected_state, global_options)
            assert code == expected_code

        th = threading.Thread(target=run_test)
        th.start()
        do_test()
        th.join()

    return fixture


def test_redirect_uri(browser_redirect_flow):
    assert browser_redirect_flow.redirect_url == "http://localhost:11001/callback"


def test_get_auth_code(mocker, browser_redirect_flow, global_options):
    mocker.patch.object(browser_redirect_flow, "wait_for_code", return_value="1234XYZ")
    mocker.patch.object(browser_redirect_flow, "gen_state", return_value="foo")
    code, verifier = browser_redirect_flow.get_auth_code(global_options, SymOrganization)
    assert code == "1234XYZ"
    assert verifier == "test-verifier"
    webbrowser.open.assert_called_once_with(
        "https://auth.com/authorize?response_type=code&client_id=P5juMqe7UUpKo6634ZeuUgZF3QTXyIfj&code_challenge_method=S256&code_challenge=test-challenge&redirect_uri=http://localhost:11001/callback&scope=admin offline_access&audience=https://api.com&state=foo"
    )


def test_wait_for_code(wait_for_code_tester):
    def do_test():
        r = requests.get(
            "http://localhost:11001/callback?code=test_wait_for_code&state=foo"
        )
        assert r.status_code == 200
        assert "Login Successful!" in r.text

    wait_for_code_tester(
        expected_state="foo", expected_code="test_wait_for_code", do_test=do_test
    )


def test_wait_for_code_missing_code(wait_for_code_tester):
    def do_test():
        r = requests.get("http://localhost:11001/callback?state=foo")
        assert r.status_code == 401
        assert "Missing code in query" in r.text

    wait_for_code_tester(expected_state="foo", expected_code="", do_test=do_test)


def test_wait_for_code_missing_state(wait_for_code_tester):
    def do_test():
        r = requests.get("http://localhost:11001/callback?code=test_wait_for_code")
        assert r.status_code == 401
        assert "Missing state in query" in r.text

    wait_for_code_tester(
        expected_state="", expected_code="test_wait_for_code", do_test=do_test
    )


def test_wait_for_code_invalid_state(wait_for_code_tester):
    def do_test():
        r = requests.get(
            "http://localhost:11001/callback?code=test_wait_for_code&state=bar"
        )
        assert r.status_code == 401
        assert "Invalid state" in r.text

    wait_for_code_tester(
        expected_state="foo", expected_code="test_wait_for_code", do_test=do_test
    )


def test_get_token_from_code_ok(mocker, browser_redirect_flow, global_options):
    response_json = json.loads(
        """
        {
        "access_token": "test-access-token",
        "token_type": "Bearer",
        "expires_in": "86400",
        "refresh_token": "test-refresh-token",
        "scope": "test-scope offline_access"
    }
    """
    )
    response = Response()
    response.status_code = 200

    mocker.patch.object(requests, "post", return_value=response)
    mocker.patch.object(response, "json", return_value=response_json)
    token = browser_redirect_flow.get_token_from_code(
        global_options, SymOrganization, "1234XYZ", "test-verifier"
    )
    requests.post.assert_called_once_with(
        "https://auth.com/oauth/token",
        headers={"content-type": "application/x-www-form-urlencoded"},
        data={
            "grant_type": "authorization_code",
            "client_id": "P5juMqe7UUpKo6634ZeuUgZF3QTXyIfj",
            "code_verifier": "test-verifier",
            "code": "1234XYZ",
            "redirect_uri": "http://localhost:11001/callback",
        },
    )
    assert token["access_token"] == "test-access-token"
    assert token["refresh_token"] == "test-refresh-token"


def test_get_token_from_code_missing_refresh_token(
    mocker, browser_redirect_flow, global_options
):
    with pytest.raises(InvalidTokenError):
        response_json = json.loads(
            """
            {
            "access_token": "test-access-token",
            "token_type": "Bearer",
            "expires_in": "86400",
            "scope": "test-scope offline_access"
        }
        """
        )
        response = Response()
        response.status_code = 200
        mocker.patch.object(requests, "post", return_value=response)
        mocker.patch.object(response, "json", return_value=response_json)
        browser_redirect_flow.get_token_from_code(
            global_options, SymOrganization, "1234XYZ", "test-verifier"
        )


def test_get_token_from_code_error_response(
    mocker, browser_redirect_flow, global_options
):
    with pytest.raises(LoginError):
        response = Response()
        response.status_code = 403
        mocker.patch.object(requests, "post", return_value=response)
        browser_redirect_flow.get_token_from_code(
            global_options, SymOrganization, "1234XYZ", "test-verifier"
        )


def test_login(mocker, global_options, browser_redirect_flow, auth_token):
    mocker.patch.object(
        browser_redirect_flow,
        "get_auth_code",
        return_value=("test-code", "test-verifier"),
    )
    mocker.patch.object(
        browser_redirect_flow, "get_token_from_code", return_value=auth_token
    )
    token = browser_redirect_flow.login(global_options, SymOrganization)
    assert token == auth_token
    browser_redirect_flow.get_auth_code.assert_called_once_with(
        global_options, SymOrganization
    )
    browser_redirect_flow.get_token_from_code.assert_called_once_with(
        global_options, SymOrganization, "test-code", "test-verifier"
    )
