# SpectroscPy 0.3.0
# SpectroscPy is a script package developed by and containing contributions from

    # Karen Oda Hjorth Minde Dundas
    # Magnus Ringholm
    # Yann Cornation
    # Benedicte Ofstad

# The package is released under a LGPL licence.
# For questions, please contact on karen.o.dundas@uit.no

# Test the anharmonic module

from spectroscpy import get_X, anharm_corrected_vibrational_energies, \
                        get_anharm_fundamental_transition_moment, \
                        get_red_prefactor, transform_to_reduced_nm, transform_to_reduced_nm_prep, \
                        get_overtone_transition_moment, get_anharm_corrected_transition_moment
import numpy as np

harmonic_wn = [0.01890185, 0.01886798, 0.00811702, 0.00724063, 0.00677518, 0.00083381]

# This is too much to print out, so have chosen the largest order of magnitude I saw 
quartic_ff = np.ones((6, 6, 6, 6))
quartic_ff = np.multiply(quartic_ff, 1.0e-7)

# This is too much to print out, so have chosen the largest order of magnitude I saw 
dipole_cubic = np.ones((6, 6, 6, 3))
dipole_cubic = np.multiply(dipole_cubic, 1.0e-6)

cubic_ff = [[[-4.17533855e-08,  1.79199260e-05,  3.45610224e-07,
         -1.74214184e-10,  8.53739921e-08,  2.22782019e-07],
        [ 1.79199260e-05,  1.37992016e-08,  6.46776591e-11,
          1.10267242e-07,  2.98094279e-11,  4.53558670e-11],
        [ 3.45610224e-07,  6.46776591e-11,  1.48181617e-09,
         -2.10866805e-06, -2.06517836e-10, -6.92083908e-11],
        [-1.74214184e-10,  1.10267242e-07, -2.10866805e-06,
          1.82431354e-09,  2.80643940e-07,  6.02636335e-08],
        [ 8.53739921e-08,  2.98094279e-11, -2.06517836e-10,
          2.80643940e-07,  3.14368806e-11,  1.06391023e-10],
        [ 2.22782019e-07,  4.53558670e-11, -6.92083908e-11,
          6.02636335e-08,  1.06391023e-10,  2.45798979e-09]],

       [[ 1.79199260e-05,  1.37992016e-08,  6.46776591e-11,
          1.10267242e-07,  2.98094279e-11,  4.53558670e-11],
        [ 1.37992016e-08,  1.79944485e-05,  2.62465739e-07,
          1.67910837e-10,  4.74057689e-08,  1.65548602e-07],
        [ 6.46776591e-11,  2.62465739e-07, -1.91109082e-06,
         -1.62774614e-09,  2.53042624e-07,  8.63394926e-08],
        [ 1.10267242e-07,  1.67910837e-10, -1.62774614e-09,
         -2.32671317e-06,  2.05474698e-10,  4.79367599e-11],
        [ 2.98094279e-11,  4.74057689e-08,  2.53042624e-07,
          2.05474698e-10, -3.69281473e-08, -1.35875678e-07],
        [ 4.53558670e-11,  1.65548602e-07,  8.63394926e-08,
          4.79367599e-11, -1.35875678e-07, -3.14911792e-06]],

       [[ 3.45610224e-07,  6.46776591e-11,  1.48181617e-09,
         -2.10866805e-06, -2.06517836e-10, -6.92083908e-11],
        [ 6.46776591e-11,  2.62465739e-07, -1.91109082e-06,
         -1.62774614e-09,  2.53042624e-07,  8.63394926e-08],
        [ 1.48181617e-09, -1.91109082e-06,  2.10057316e-07,
         -8.32012011e-13,  3.38527255e-07,  2.42349274e-07],
        [-2.10866805e-06, -1.62774614e-09, -8.32012011e-13,
          2.96758887e-07,  1.01335408e-12, -8.82123348e-13],
        [-2.06517836e-10,  2.53042624e-07,  3.38527255e-07,
          1.01335408e-12,  5.23791728e-10, -4.69690927e-08],
        [-6.92083908e-11,  8.63394926e-08,  2.42349274e-07,
         -8.82123348e-13, -4.69690927e-08,  1.02063685e-07]],

       [[-1.74214184e-10,  1.10267242e-07, -2.10866805e-06,
          1.82431354e-09,  2.80643940e-07,  6.02636335e-08],
        [ 1.10267242e-07,  1.67910837e-10, -1.62774614e-09,
         -2.32671317e-06,  2.05474698e-10,  4.79367599e-11],
        [-2.10866805e-06, -1.62774614e-09, -8.32012011e-13,
          2.96758887e-07,  1.01335408e-12, -8.82123348e-13],
        [ 1.82431354e-09, -2.32671317e-06,  2.96758887e-07,
         -5.28411318e-12,  2.91781087e-07, -2.23621540e-07],
        [ 2.80643940e-07,  2.05474698e-10,  1.01335408e-12,
          2.91781087e-07, -1.94557930e-12, -6.29532479e-13],
        [ 6.02636335e-08,  4.79367599e-11, -8.82123348e-13,
         -2.23621540e-07, -6.29532478e-13, -3.93126776e-13]],

       [[ 8.53739921e-08,  2.98094279e-11, -2.06517836e-10,
          2.80643940e-07,  3.14368806e-11,  1.06391023e-10],
        [ 2.98094279e-11,  4.74057689e-08,  2.53042624e-07,
          2.05474698e-10, -3.69281473e-08, -1.35875678e-07],
        [-2.06517836e-10,  2.53042624e-07,  3.38527255e-07,
          1.01335408e-12,  5.23791728e-10, -4.69690927e-08],
        [ 2.80643940e-07,  2.05474698e-10,  1.01335408e-12,
          2.91781087e-07, -1.94557930e-12, -6.29532479e-13],
        [ 3.14368806e-11, -3.69281473e-08,  5.23791728e-10,
         -1.94557930e-12,  9.61199530e-07,  1.67458926e-08],
        [ 1.06391023e-10, -1.35875678e-07, -4.69690927e-08,
         -6.29532478e-13,  1.67458926e-08, -2.03938490e-08]],

       [[ 2.22782019e-07,  4.53558670e-11, -6.92083908e-11,
          6.02636335e-08,  1.06391023e-10,  2.45798979e-09],
        [ 4.53558670e-11,  1.65548602e-07,  8.63394926e-08,
          4.79367599e-11, -1.35875678e-07, -3.14911792e-06],
        [-6.92083908e-11,  8.63394926e-08,  2.42349274e-07,
         -8.82123348e-13, -4.69690927e-08,  1.02063685e-07],
        [ 6.02636335e-08,  4.79367599e-11, -8.82123348e-13,
         -2.23621540e-07, -6.29532479e-13, -3.93126776e-13],
        [ 1.06391023e-10, -1.35875678e-07, -4.69690927e-08,
         -6.29532479e-13,  1.67458926e-08, -2.03938490e-08],
        [ 2.45798979e-09, -3.14911792e-06,  1.02063685e-07,
         -3.93126776e-13, -2.03938490e-08, -7.35868174e-08]]]

dipole_gradient = np.array([[-3.55354944e-03, -2.04896120e-03,  4.65337200e-04],
       [-1.33397734e-03,  2.30954115e-03, -1.03890030e-06],
       [ 5.38119448e-04, -9.34229755e-04,  5.72686209e-07],
       [ 2.05564446e-04,  1.21485592e-04,  5.08962388e-03],
       [ 4.87101854e-05, -8.45631401e-05,  7.73630603e-08],
       [ 4.34177587e-03, -7.53776306e-03,  4.54332366e-06]])

dipole_hess = np.array([[[-2.87324643e-06,  4.86055888e-06, -2.18621160e-07],
        [ 3.54598739e-05,  2.05226403e-05,  1.38808597e-04],
        [-1.93796288e-06, -1.08971343e-06,  5.21175726e-05],
        [ 4.76752177e-06, -8.29841792e-06, -5.32476755e-08],
        [-5.30195291e-06, -3.07727014e-06, -4.77222118e-05],
        [-2.81316060e-05, -1.60741513e-05,  1.94858897e-05]],

       [[ 3.54598739e-05,  2.05226403e-05,  1.38808597e-04],
        [ 5.06742372e-06, -8.67026317e-06,  2.20446215e-07],
        [-2.66791363e-06,  4.62838940e-06,  3.77937381e-08],
        [ 1.20287972e-05,  6.96541104e-06,  7.50387329e-05],
        [ 3.01907714e-06, -5.25093002e-06, -3.35003016e-08],
        [ 6.56752736e-05, -1.14069189e-04,  8.38974324e-08]],

       [[-1.93796288e-06, -1.08971343e-06,  5.21175726e-05],
        [-2.66791363e-06,  4.62838940e-06,  3.77937381e-08],
        [-5.90095443e-08,  1.02687275e-07, -3.15565205e-11],
        [ 3.00629863e-05,  1.73214399e-05,  8.56122034e-06],
        [ 2.80813272e-06, -4.87487302e-06,  2.97275015e-09],
        [-1.32582208e-05,  2.30177023e-05, -1.37647379e-08]],

       [[ 4.76752177e-06, -8.29841792e-06, -5.32476755e-08],
        [ 1.20287972e-05,  6.96541104e-06,  7.50387329e-05],
        [ 3.00629863e-05,  1.73214399e-05,  8.56122034e-06],
        [ 1.70194288e-05, -2.95479478e-05,  1.75838951e-08],
        [-6.15872244e-07, -3.46792493e-07,  1.34587785e-05],
        [ 1.97337023e-07,  1.25458004e-07,  1.95867505e-05]],

       [[-5.30195291e-06, -3.07727014e-06, -4.77222118e-05],
        [ 3.01907714e-06, -5.25093002e-06, -3.35003016e-08],
        [ 2.80813272e-06, -4.87487302e-06,  2.97275015e-09],
        [-6.15872244e-07, -3.46792493e-07,  1.34587785e-05],
        [ 1.54687727e-06, -2.68548887e-06,  1.78831114e-09],
        [ 1.83609012e-06, -3.18768509e-06,  2.00379408e-09]],

       [[-2.81316060e-05, -1.60741513e-05,  1.94858897e-05],
        [ 6.56752736e-05, -1.14069189e-04,  8.38974324e-08],
        [-1.32582208e-05,  2.30177023e-05, -1.37647379e-08],
        [ 1.97337023e-07,  1.25458004e-07,  1.95867505e-05],
        [ 1.83609012e-06, -3.18768509e-06,  2.00379408e-09],
        [ 4.36167116e-05, -7.57235039e-05,  4.57200201e-08]]])

polarizability_gradient = \
     np.array([[[[ 6.04610202e-02, -3.51511055e-02,  2.05867010e-02],
        [-3.51511055e-02, -6.06465271e-02, -3.58100073e-02],
        [ 2.05867010e-02, -3.58100073e-02, -8.24129166e-06]],
       [[ 1.20853575e-01,  4.99878490e-02,  7.95877729e-02],
        [ 4.99878490e-02,  6.28477170e-02,  4.58251237e-02],
        [ 7.95877729e-02,  4.58251237e-02,  6.62798317e-02]],
       [[-2.34431047e-02, -7.82917443e-03,  4.02920687e-02],
        [-7.82917443e-03, -1.43042013e-02,  2.32435281e-02],
        [ 4.02920687e-02,  2.32435281e-02,  4.87524759e-02]],
       [[-1.55313827e-02,  9.01591539e-03,  1.14149851e-02],
        [ 9.01591539e-03,  1.55074857e-02, -1.98366345e-02],
        [ 1.14149851e-02, -1.98366345e-02,  2.34202541e-05]],
       [[ 4.39674438e-03,  4.92827640e-04,  7.19780552e-03],
        [ 4.92827640e-04,  3.83525060e-03,  4.19756005e-03],
        [ 7.19780552e-03,  4.19756005e-03,  8.94657717e-02]],
       [[-2.41613385e-02, -4.31408089e-02, -1.19041040e-02],
        [-4.31408089e-02,  2.58697434e-02, -6.88886076e-03],
        [-1.19041040e-02, -6.88886076e-03, -2.42915073e-03]]], 
       [[[ 6.04610200e-02, -3.51511054e-02,  2.05867014e-02],
        [-3.51511054e-02, -6.06465273e-02, -3.58100074e-02],
        [ 2.05867014e-02, -3.58100074e-02, -8.24116238e-06]],
       [[ 1.20853575e-01,  4.99878487e-02,  7.95877733e-02],
        [ 4.99878487e-02,  6.28477171e-02,  4.58251237e-02],
        [ 7.95877733e-02,  4.58251237e-02,  6.62798311e-02]],
       [[-2.34431046e-02, -7.82917446e-03,  4.02920685e-02],
        [-7.82917446e-03, -1.43042012e-02,  2.32435281e-02],
        [ 4.02920685e-02,  2.32435281e-02,  4.87524752e-02]],
       [[-1.55313822e-02,  9.01591527e-03,  1.14149849e-02],
        [ 9.01591527e-03,  1.55074855e-02, -1.98366344e-02],
        [ 1.14149849e-02, -1.98366344e-02,  2.34202945e-05]],
       [[ 4.39674437e-03,  4.92827626e-04,  7.19780546e-03],
        [ 4.92827626e-04,  3.83525056e-03,  4.19755994e-03],
        [ 7.19780546e-03,  4.19755994e-03,  8.94657709e-02]],
       [[-2.41613390e-02, -4.31408092e-02, -1.19041042e-02],
        [-4.31408092e-02,  2.58697431e-02, -6.88886095e-03],
        [-1.19041042e-02, -6.88886095e-03, -2.42915100e-03]]], 
       [[[ 6.24435726e-02, -3.63042282e-02,  2.14401945e-02],
        [-3.63042282e-02, -6.26361513e-02, -3.72962094e-02],
        [ 2.14401945e-02, -3.72962094e-02, -9.06377656e-06]],
       [[ 1.25458452e-01,  5.20423354e-02,  8.34157406e-02],
        [ 5.20423354e-02,  6.50703814e-02,  4.80299099e-02],
        [ 8.34157406e-02,  4.80299099e-02,  6.96537637e-02]],
       [[-2.37970681e-02, -7.93763434e-03,  4.05413248e-02],
        [-7.93763434e-03, -1.45319890e-02,  2.33876188e-02],
        [ 4.05413248e-02,  2.33876188e-02,  4.94476269e-02]],
       [[-1.59236762e-02,  9.24345865e-03,  1.14034738e-02],
        [ 9.24345865e-03,  1.58998009e-02, -1.98171324e-02],
        [ 1.14034738e-02, -1.98171324e-02,  2.33748102e-05]],
       [[ 4.60303582e-03,  4.14937046e-04,  7.97344582e-03],
        [ 4.14937046e-04,  4.13298957e-03,  4.64696407e-03],
        [ 7.97344582e-03,  4.64696407e-03,  9.41809158e-02]],
       [[-2.45703252e-02, -4.36328511e-02, -1.21236513e-02],
        [-4.36328511e-02,  2.60312669e-02, -7.01564868e-03],
        [-1.21236513e-02, -7.01564868e-03, -2.52655828e-03]]]])

CO2_harmonic_energies = [5.09398229e-20, 3.00200304e-20, 1.53545862e-20, 1.53545609e-20]

CO2_dipole_gradient = [[ 1.90029408e-30,  7.43813358e-32,  4.16193632e-47],
                    [-2.38814033e-36, -4.13591034e-36,  2.67300403e-45],
                    [ 1.62029042e-41, -4.14038006e-40, -8.53961356e-31],
                    [ 3.34023331e-32, -8.53306057e-31,  5.00490180e-40]]

CO2_dipole_hessian = [[[ 1.01842877e-37,  1.94801492e-37,  9.57619100e-46],
                    [-3.81161936e-32, -1.49214252e-33, -1.01145395e-46],
                    [ 6.54512908e-46, -2.22965388e-46, -2.54513087e-36],
                    [-5.62132272e-37, -1.53212084e-38,  1.68570933e-45]],
                   [[-3.81161936e-32, -1.49214252e-33, -1.01145395e-46],
                    [ 6.37812508e-39,  3.46967569e-38, -7.38406808e-47],
                    [-2.24877574e-43,  5.75867043e-42,  1.18778736e-32],
                    [-4.64498778e-34,  1.18683239e-32, -6.96139284e-42]],
                   [[ 6.54512908e-46, -2.22965388e-46, -2.54513087e-36],
                    [-2.24877574e-43,  5.75867043e-42,  1.18778736e-32],
                    [ 9.25293149e-36,  7.02964301e-37,  1.18489856e-45],
                    [-5.57037747e-45,  2.88253685e-46,  2.18904701e-37]],
                   [[-5.62132272e-37, -1.53212084e-38,  1.68570933e-45],
                    [-4.64498778e-34,  1.18683239e-32, -6.96139284e-42],
                    [-5.57037747e-45,  2.88253685e-46,  2.18904701e-37],
                    [ 4.88171971e-38,  7.82005447e-37,  3.27032012e-47]]]

CO2_dipole_cubic = [[[[-3.41161401e-32, -1.33547194e-33, -1.46736816e-45],
                   [ 4.60042124e-38,  6.86525655e-38,  3.03074980e-46],
                   [-7.71883025e-44,  1.99305503e-42,  4.11073771e-33],
                   [-1.60557794e-34,  4.10757085e-33, -2.40927667e-42]],
                  [[ 4.60042124e-38,  6.86525655e-38,  3.03074980e-46],
                   [-3.56402768e-33, -1.39520811e-34, -4.40240241e-46],
                   [-4.98722204e-46,  7.34481188e-47,  1.08879713e-37],
                   [-1.04436314e-37, -9.01371177e-39,  3.10810364e-47]],
                  [[-7.71883025e-44,  1.99305503e-42,  4.11073771e-33],
                   [-4.98722204e-46,  7.34481188e-47,  1.08879713e-37],
                   [-1.70510336e-32, -6.67345719e-34,  2.68528252e-47],
                   [ 1.71963893e-42,  6.73931291e-44,  2.89521221e-38]],
                  [[-1.60557794e-34,  4.10757085e-33, -2.40927667e-42],
                   [-1.04436314e-37, -9.01371177e-39,  3.10810364e-47],
                   [ 1.71963893e-42,  6.73931291e-44,  2.89521221e-38],
                   [-1.70513305e-32, -6.67387874e-34, -4.45856210e-47]]],
                 [[[ 4.60042124e-38,  6.86525655e-38,  3.03074980e-46],
                   [-3.56402768e-33, -1.39520811e-34, -4.40240241e-46],
                   [-4.98722204e-46,  7.34481188e-47,  1.08879713e-37],
                   [-1.04436314e-37, -9.01371177e-39,  3.10810364e-47]],
                  [[-3.56402768e-33, -1.39520811e-34, -4.40240241e-46],
                   [-2.36670327e-38,  3.18254520e-38, -3.13041382e-46],
                   [-4.57481866e-45,  1.17989635e-43,  2.43365484e-34],
                   [-9.56990345e-36,  2.43185457e-34, -1.42660922e-43]],
                  [[-4.98722204e-46,  7.34481188e-47,  1.08879713e-37],
                   [-4.57481866e-45,  1.17989635e-43,  2.43365484e-34],
                   [-3.90307816e-37,  1.13902941e-38, -8.63364712e-47],
                   [ 2.28239615e-46, -3.18396624e-47,  1.61637636e-38]],
                  [[-1.04436314e-37, -9.01371177e-39,  3.10810364e-47],
                   [-9.56990345e-36,  2.43185457e-34, -1.42660922e-43],
                   [ 2.28239615e-46, -3.18396624e-47,  1.61637636e-38],
                   [ 2.55070217e-38,  6.32756488e-38, -1.64696784e-47]]],
                 [[[-7.71883025e-44,  1.99305503e-42,  4.11073771e-33],
                   [-4.98722204e-46,  7.34481188e-47,  1.08879713e-37],
                   [-1.70510336e-32, -6.67345719e-34,  2.68528252e-47],
                   [ 1.71963893e-42,  6.73931291e-44,  2.89521221e-38]],
                  [[-4.98722204e-46,  7.34481188e-47,  1.08879713e-37],
                   [-4.57481866e-45,  1.17989635e-43,  2.43365484e-34],
                   [-3.90307816e-37,  1.13902941e-38, -8.63364712e-47],
                   [ 2.28239615e-46, -3.18396624e-47,  1.61637636e-38]],
                  [[-1.70510336e-32, -6.67345719e-34,  2.68528252e-47],
                   [-3.90307816e-37,  1.13902941e-38, -8.63364712e-47],
                   [-2.62564050e-43,  6.71396835e-42,  1.38475232e-32],
                   [-1.80445878e-34,  4.61235851e-33, -3.63633936e-42]],
                  [[ 1.71963893e-42,  6.73931291e-44,  2.89521221e-38],
                   [ 2.28239615e-46, -3.18396624e-47,  1.61637636e-38],
                   [-1.80445878e-34,  4.61235851e-33, -3.63633936e-42],
                   [-5.12246045e-44,  1.30765878e-42,  4.61590500e-33]]],
                 [[[-1.60557794e-34,  4.10757085e-33, -2.40927667e-42],
                   [-1.04436314e-37, -9.01371177e-39,  3.10810364e-47],
                   [ 1.71963893e-42,  6.73931291e-44,  2.89521221e-38],
                   [-1.70513305e-32, -6.67387874e-34, -4.45856210e-47]],
                  [[-1.04436314e-37, -9.01371177e-39,  3.10810364e-47],
                   [-9.56990345e-36,  2.43185457e-34, -1.42660922e-43],
                   [ 2.28239615e-46, -3.18396624e-47,  1.61637636e-38],
                   [ 2.55070217e-38,  6.32756488e-38, -1.64696784e-47]],
                  [[ 1.71963893e-42,  6.73931291e-44,  2.89521221e-38],
                   [ 2.28239615e-46, -3.18396624e-47,  1.61637636e-38],
                   [-1.80445878e-34,  4.61235851e-33, -3.63633936e-42],
                   [-5.12246045e-44,  1.30765878e-42,  4.61590500e-33]],
                  [[-1.70513305e-32, -6.67387874e-34, -4.45856210e-47],
                   [ 2.55070217e-38,  6.32756488e-38, -1.64696784e-47],
                   [-5.12246045e-44,  1.30765878e-42,  4.61590500e-33],
                   [-5.41494951e-34,  1.38372914e-32, -8.11588926e-42]]]]

CO2_cubic_ff = \
[[[-4.02458165e-26, -1.07453825e-20, -3.46715173e-35, -2.22589684e-26],
  [-1.07453825e-20,  1.68628067e-26,  4.16906857e-35,  3.92492320e-27],
  [-3.46715173e-35,  4.16906857e-35,  3.05970934e-25, -1.78387439e-34],
  [-2.22589684e-26,  3.92492320e-27, -1.78387439e-34,  1.44335455e-26]],
 [[-1.07453825e-20,  1.68628067e-26,  4.16906857e-35,  3.92492320e-27],
  [ 1.68628067e-26, -5.39835069e-21, -2.02125067e-35,  3.10023009e-26],
  [ 4.16906857e-35, -2.02125067e-35,  3.27060412e-21, -3.29901516e-31],
  [ 3.92492320e-27,  3.10023009e-26, -3.29901516e-31,  3.27059333e-21]],
 [[-3.46715173e-35,  4.16906857e-35,  3.05970934e-25, -1.78387439e-34],
  [ 4.16906857e-35, -2.02125067e-35,  3.27060412e-21, -3.29901516e-31],
  [ 3.05970934e-25,  3.27060412e-21,  5.10350632e-35, -2.97159440e-27],
  [-1.78387439e-34, -3.29901516e-31, -2.97159440e-27,  1.65359218e-35]],
 [[-2.22589684e-26,  3.92492320e-27, -1.78387439e-34,  1.44335455e-26],
  [ 3.92492320e-27,  3.10023009e-26, -3.29901516e-31,  3.27059333e-21],
  [-1.78387439e-34, -3.29901516e-31, -2.97159440e-27,  1.65359218e-35],
  [ 1.44335455e-26,  3.27059333e-21,  1.65359218e-35, -8.97708537e-27]]]

CO2_quartic_ff = \
[[[[ 3.39211819e-21,  1.41972489e-27, -7.21999201e-35, -5.17962700e-28],
   [ 1.41972489e-27,  1.53963238e-21, -1.46883591e-34, -4.76626303e-27],
   [-7.21999201e-35, -1.46883591e-34, -2.27675488e-21,  2.29626573e-31],
   [-5.17962700e-28, -4.76626303e-27,  2.29626573e-31, -2.27676704e-21]],
  [[ 1.41972489e-27,  1.53963238e-21, -1.46883591e-34, -4.76626303e-27],
   [ 1.53963238e-21, -4.57558099e-27, -1.45337703e-35, -4.80498827e-27],
   [-1.46883591e-34, -1.45337703e-35, -8.02212886e-26,  3.84770482e-35],
   [-4.76626303e-27, -4.80498827e-27,  3.84770482e-35,  6.48807319e-28]],
  [[-7.21999201e-35, -1.46883591e-34, -2.27675488e-21,  2.29626573e-31],
   [-1.46883591e-34, -1.45337703e-35, -8.02212886e-26,  3.84770482e-35],
   [-2.27675488e-21, -8.02212886e-26,  9.37612774e-35,  3.39580699e-27],
   [ 2.29626573e-31,  3.84770482e-35,  3.39580699e-27, -1.63144665e-35]],
  [[-5.17962700e-28, -4.76626303e-27,  2.29626573e-31, -2.27676704e-21],
   [-4.76626303e-27, -4.80498827e-27,  3.84770482e-35,  6.48807319e-28],
   [ 2.29626573e-31,  3.84770482e-35,  3.39580699e-27, -1.63144665e-35],
   [-2.27676704e-21,  6.48807319e-28, -1.63144665e-35,  1.49041207e-26]]],
 [[[ 1.41972489e-27,  1.53963238e-21, -1.46883591e-34, -4.76626303e-27],
   [ 1.53963238e-21, -4.57558099e-27, -1.45337703e-35, -4.80498827e-27],
   [-1.46883591e-34, -1.45337703e-35, -8.02212886e-26,  3.84770482e-35],
   [-4.76626303e-27, -4.80498827e-27,  3.84770482e-35,  6.48807319e-28]],
  [[ 1.53963238e-21, -4.57558099e-27, -1.45337703e-35, -4.80498827e-27],
   [-4.57558099e-27,  7.60712619e-22,  2.56477455e-35, -1.29364448e-26],
   [-1.45337703e-35,  2.56477455e-35, -9.25673736e-22,  9.33708800e-32],
   [-4.80498827e-27, -1.29364448e-26,  9.33708800e-32, -9.25674582e-22]],
  [[-1.46883591e-34, -1.45337703e-35, -8.02212886e-26,  3.84770482e-35],
   [-1.45337703e-35,  2.56477455e-35, -9.25673736e-22,  9.33708800e-32],
   [-8.02212886e-26, -9.25673736e-22, -1.75146046e-35,  1.41867393e-27],
   [ 3.84770482e-35,  9.33708800e-32,  1.41867393e-27, -9.47294874e-36]],
  [[-4.76626303e-27, -4.80498827e-27,  3.84770482e-35,  6.48807319e-28],
   [-4.80498827e-27, -1.29364448e-26,  9.33708800e-32, -9.25674582e-22],
   [ 3.84770482e-35,  9.33708800e-32,  1.41867393e-27, -9.47294874e-36],
   [ 6.48807319e-28, -9.25674582e-22, -9.47294874e-36,  4.01148058e-27]]],
 [[[-7.21999201e-35, -1.46883591e-34, -2.27675488e-21,  2.29626573e-31],
   [-1.46883591e-34, -1.45337703e-35, -8.02212886e-26,  3.84770482e-35],
   [-2.27675488e-21, -8.02212886e-26,  9.37612774e-35,  3.39580699e-27],
   [ 2.29626573e-31,  3.84770482e-35,  3.39580699e-27, -1.63144665e-35]],
  [[-1.46883591e-34, -1.45337703e-35, -8.02212886e-26,  3.84770482e-35],
   [-1.45337703e-35,  2.56477455e-35, -9.25673736e-22,  9.33708800e-32],
   [-8.02212886e-26, -9.25673736e-22, -1.75146046e-35,  1.41867393e-27],
   [ 3.84770482e-35,  9.33708800e-32,  1.41867393e-27, -9.47294874e-36]],
  [[-2.27675488e-21, -8.02212886e-26,  9.37612774e-35,  3.39580699e-27],
   [-8.02212886e-26, -9.25673736e-22, -1.75146046e-35,  1.41867393e-27],
   [ 9.37612774e-35, -1.75146046e-35,  1.48983250e-21, -1.50295002e-31],
   [ 3.39580699e-27,  1.41867393e-27, -1.50295002e-31,  4.96601907e-22]],
  [[ 2.29626573e-31,  3.84770482e-35,  3.39580699e-27, -1.63144665e-35],
   [ 3.84770482e-35,  9.33708800e-32,  1.41867393e-27, -9.47294874e-36],
   [ 3.39580699e-27,  1.41867393e-27, -1.50295002e-31,  4.96601907e-22],
   [-1.63144665e-35, -9.47294874e-36,  4.96601907e-22, -1.50277569e-31]]],
 [[[-5.17962700e-28, -4.76626303e-27,  2.29626573e-31, -2.27676704e-21],
   [-4.76626303e-27, -4.80498827e-27,  3.84770482e-35,  6.48807319e-28],
   [ 2.29626573e-31,  3.84770482e-35,  3.39580699e-27, -1.63144665e-35],
   [-2.27676704e-21,  6.48807319e-28, -1.63144665e-35,  1.49041207e-26]],
  [[-4.76626303e-27, -4.80498827e-27,  3.84770482e-35,  6.48807319e-28],
   [-4.80498827e-27, -1.29364448e-26,  9.33708800e-32, -9.25674582e-22],
   [ 3.84770482e-35,  9.33708800e-32,  1.41867393e-27, -9.47294874e-36],
   [ 6.48807319e-28, -9.25674582e-22, -9.47294874e-36,  4.01148058e-27]],
  [[ 2.29626573e-31,  3.84770482e-35,  3.39580699e-27, -1.63144665e-35],
   [ 3.84770482e-35,  9.33708800e-32,  1.41867393e-27, -9.47294874e-36],
   [ 3.39580699e-27,  1.41867393e-27, -1.50295002e-31,  4.96601907e-22],
   [-1.63144665e-35, -9.47294874e-36,  4.96601907e-22, -1.50277569e-31]],
  [[-2.27676704e-21,  6.48807319e-28, -1.63144665e-35,  1.49041207e-26],
   [ 6.48807319e-28, -9.25674582e-22, -9.47294874e-36,  4.01148058e-27],
   [-1.63144665e-35, -9.47294874e-36,  4.96601907e-22, -1.50277569e-31],
   [ 1.49041207e-26,  4.01148058e-27, -1.50277569e-31,  1.48981461e-21]]]]


def test_get_X():

    # Make a test that forces an Xii resonance

    ref_X = \
[[4.46606822e-09, 1.78849028e-08, 2.55713148e-08, 2.57082539e-08,
  2.50063078e-08, 2.57513309e-08],
 [1.78849028e-08, 4.45717390e-09, 2.56029074e-08, 2.56968264e-08,
  2.50081682e-08, 2.57655254e-08],
 [2.55713148e-08, 2.56029074e-08, 6.16796955e-09, 2.47175678e-08,
  2.49810161e-08, 2.49247507e-08],
 [2.57082539e-08, 2.56968264e-08, 2.47175678e-08, 6.16061652e-09,
  2.49843927e-08, 2.48962899e-08],
 [2.50063078e-08, 2.50081682e-08, 2.49810161e-08, 2.49843927e-08,
  6.23573488e-09, 2.49982787e-08],
 [2.57513309e-08, 2.57655254e-08, 2.49247507e-08, 2.48962899e-08,
  2.49982787e-08, 6.15025956e-09]]

    X, fermi_resonance = get_X(harmonic_wn, cubic_ff, quartic_ff, 0, 'No coriolis', True)

    assert np.allclose(ref_X, X)
    assert [] == fermi_resonance

    ref_X = [[-2.45912144e-22, -4.08435886e-22, -2.76518361e-22, -2.76522367e-22],
             [-4.08435886e-22, -5.35761054e-23, -1.06401892e-22, -1.06402462e-22],
             [-2.76518361e-22, -1.06401892e-22,  4.30696759e-23,  3.50698206e-23],
             [-2.76522367e-22, -1.06402462e-22,  3.50698206e-23,  4.30688838e-23]]

    ref_fermi_resonances = [[1, 2, 2, True], [1, 3, 3, True]]


    X, fermi_resonance = get_X(CO2_harmonic_energies, CO2_cubic_ff, CO2_quartic_ff, 0, 'No coriolis', \
                               True)

    assert np.allclose(ref_X, X)
    assert np.allclose(ref_fermi_resonances, fermi_resonance)

    ref_X = [[-2.45912144e-22, -4.08435886e-22, -2.76518361e-22, -2.76522367e-22],
             [-4.08435886e-22, -5.35761054e-23, -2.04664993e-21, -2.04678017e-21],
             [-2.76518361e-22, -2.04664993e-21,  5.28131686e-22,  3.50698206e-23],
             [-2.76522367e-22, -2.04678017e-21,  3.50698206e-23,  5.28163311e-22]]

    X, fermi_resonance = get_X(CO2_harmonic_energies, CO2_cubic_ff, CO2_quartic_ff, 0, 'No coriolis', \
                               False)

    assert np.allclose(ref_X, X)
    assert np.allclose([], fermi_resonance)


def test_get_anharm_corrected_transition_moment():

    ref_ftm = [[-2.51377315e-03, -1.44417272e-03,  3.30043311e-04],
               [-9.36638235e-04,  1.63737371e-03, -3.90543564e-07],
               [ 3.81621907e-04, -6.59548248e-04,  1.75283820e-06],
               [ 1.46422579e-04,  8.69970676e-05,  3.59992900e-03],
               [ 3.54952984e-05, -5.87228268e-05,  5.13191496e-07],
               [ 3.07102120e-03, -5.32866291e-03,  4.19493500e-06]]

    ref_overtone_tm = [[-1.16292209e-06,  1.97523478e-06, -7.75321336e-08],
                       [ 1.64181669e-06, -2.80678078e-06,  7.78650425e-08],
                       [-2.01423153e-07,  3.49764798e-07, -2.00349931e-10],
                       [ 5.87806609e-06, -1.02050911e-05,  6.07003033e-09],
                       [ 5.45940216e-07, -9.47790129e-07,  6.31446143e-10],
                       [ 1.52940640e-05, -2.65522106e-05,  1.60313973e-08]]

    ref_combotone_tm = [[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 1.71669992e-05,  9.93710756e-06,  6.94794802e-05],
                         [-1.00243489e-06, -5.64184892e-07,  2.60055222e-05],
                         [ 2.37231789e-06, -4.12931652e-06, -2.65838692e-08],
                         [-2.66012720e-06, -1.54390248e-06, -2.38513419e-05],
                         [-1.42979179e-05, -8.17089633e-06,  9.77665125e-06]],
                        [[ 1.71669992e-05,  9.93710756e-06,  6.94794802e-05],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [-1.34886498e-06,  2.34001663e-06,  1.88399559e-08],
                         [ 6.00021825e-06,  3.47451750e-06,  3.74527275e-05],
                         [ 1.50810684e-06, -2.62299596e-06, -1.67444120e-08],
                         [ 3.27586134e-05, -5.68978255e-05,  4.18924783e-08]],
                        [[-1.00243489e-06, -5.64184892e-07,  2.60055222e-05],
                         [-1.34886498e-06,  2.34001663e-06,  1.88399559e-08],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 1.44492888e-05,  8.32543434e-06,  4.38679513e-06],
                         [ 1.43209894e-06, -2.48610883e-06,  1.51631088e-09],
                         [-6.58588945e-06,  1.14338146e-05, -6.83713585e-09]],
                        [[ 2.37231789e-06, -4.12931652e-06, -2.65838692e-08],
                         [ 6.00021825e-06,  3.47451750e-06,  3.74527275e-05],
                         [ 1.44492888e-05,  8.32543434e-06,  4.38679513e-06],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [-2.47810937e-07, -1.38744053e-07,  6.75904714e-06],
                         [ 9.25702721e-08,  5.90140392e-08,  9.46979712e-06]],
                        [[-2.66012720e-06, -1.54390248e-06, -2.38513419e-05],
                         [ 1.50810684e-06, -2.62299596e-06, -1.67444120e-08],
                         [ 1.43209894e-06, -2.48610883e-06,  1.51631088e-09],
                         [-2.47810937e-07, -1.38744053e-07,  6.75904714e-06],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 9.24745377e-07, -1.60547499e-06,  1.00710120e-09]],
                        [[-1.42979179e-05, -8.17089633e-06,  9.77665125e-06],
                         [ 3.27586134e-05, -5.68978255e-05,  4.18924783e-08],
                         [-6.58588945e-06,  1.14338146e-05, -6.83713585e-09],
                         [ 9.25702721e-08,  5.90140392e-08,  9.46979712e-06],
                         [ 9.24745377e-07, -1.60547499e-06,  1.00710120e-09],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]]

    harmonic_transition_moment = [[-1.54955410e-31, -8.93466247e-32,  2.02914051e-32], \
                                  [-5.82213640e-32,  1.00799772e-31, -4.53416792e-35], \
                                  [ 3.58077398e-32, -6.21658556e-32,  3.81090999e-35], \
                                  [ 1.44829270e-32,  8.55919892e-33,  3.58586562e-31], \
                                  [ 3.54777519e-33, -6.15910227e-33,  5.63465208e-36], \
                                  [ 9.01428369e-31, -1.56497103e-30,  9.43272714e-34]]

    transition_moments = \
        get_anharm_corrected_transition_moment(dipole_gradient, dipole_hess, dipole_cubic, cubic_ff, \
                                               quartic_ff, harmonic_transition_moment, harmonic_wn, \
                                               'No coriolis', 0, \
                                               'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(harmonic_transition_moment, transition_moments.harmonic)
    assert np.allclose(ref_ftm, transition_moments.fundamental)
    assert np.allclose(ref_overtone_tm, transition_moments.overtones)
    assert np.allclose(ref_combotone_tm, transition_moments.combotones)

    # CO2
    harmonic_transition_moment = [[ 1.34371083e-30,  5.25955469e-32,  2.94293339e-47],
                                  [-1.68867022e-36, -2.92453025e-36,  1.89009928e-45],
                                  [ 1.14571834e-41, -2.92769082e-40, -6.03841866e-31],
                                  [ 2.36190162e-32, -6.03378499e-31,  3.53900000e-40]]

    # VPT2
    ref_ftm = [[ 1.32853011e-30,  5.20012975e-32, -3.44446578e-46],
               [ 4.55714886e-36,  7.24575525e-37, -3.73461131e-47],
               [ 2.84903522e-39, -7.27823881e-38, -6.00866116e-31],
               [ 2.35025971e-32, -6.00405062e-31,  7.28986103e-38]]
    ref_overtone_tm = [[-1.12592159e-37,  1.21877780e-37,  3.16289799e-46],
                       [ 6.25968499e-37,  7.96578806e-38, -2.99396437e-47],
                       [-5.04965541e-36, -3.41065392e-36,  2.29011571e-45],
                       [-2.26506530e-36, -3.10718563e-36,  2.08596210e-45]]
    ref_combotone_tm = [[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [-1.50403619e-31, -5.88720350e-33, -9.43194007e-47],
                         [-6.06833280e-46, -1.00838717e-46, -1.75486652e-36],
                         [-8.78672647e-37, -5.38211005e-38,  1.13738075e-45]],
                        [[-1.50403619e-31, -5.88720350e-33, -9.43194007e-47],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 6.05542068e-44, -1.63788039e-42, -5.82264169e-33],
                         [ 2.27443419e-34, -5.81835666e-33,  4.59888452e-42]],
                        [[-6.06833280e-46, -1.00838717e-46, -1.75486652e-36],
                         [ 6.05542068e-44, -1.63788039e-42, -5.82264169e-33],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 2.72984690e-45,  6.98297355e-46,  1.36997132e-37]],
                        [[-8.78672647e-37, -5.38211005e-38,  1.13738075e-45],
                         [ 2.27443419e-34, -5.81835666e-33,  4.59888452e-42],
                         [ 2.72984690e-45,  6.98297355e-46,  1.36997132e-37],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]]

    transition_moments = get_anharm_corrected_transition_moment(CO2_dipole_gradient, \
                                                                CO2_dipole_hessian, \
                                                                CO2_dipole_cubic, CO2_cubic_ff, \
                                                                CO2_quartic_ff, \
                                                                harmonic_transition_moment, \
                                                                CO2_harmonic_energies, 'No coriolis', \
                                                                0, 'Anharmonic: VPT2')

    assert np.allclose(np.multiply(harmonic_transition_moment, 1.0e40), \
           np.multiply(transition_moments.harmonic, 1.0e40))
    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(transition_moments.fundamental, 1.0e40))
    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), \
           np.multiply(transition_moments.overtones, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), \
           np.multiply(transition_moments.combotones, 1.0e40))

    # DVPT2 frequency, VPT2 intensity
    transition_moments = get_anharm_corrected_transition_moment(CO2_dipole_gradient, \
                                                                CO2_dipole_hessian, \
                                                                CO2_dipole_cubic, CO2_cubic_ff, \
                                                                CO2_quartic_ff, \
                                                                harmonic_transition_moment, \
                                                                CO2_harmonic_energies, 'No coriolis', \
                                                                0, 'Anharmonic: Freq DVPT2, Int VPT2')

    assert np.allclose(np.multiply(harmonic_transition_moment, 1.0e40), \
           np.multiply(transition_moments.harmonic, 1.0e40))
    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(transition_moments.fundamental, 1.0e40))
    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), \
           np.multiply(transition_moments.overtones, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), \
           np.multiply(transition_moments.combotones, 1.0e40))

    # DVPT2
    ref_ftm = [[ 1.32853011e-30,  5.20012975e-32, -3.44446578e-46],
               [-1.53619477e-36, -2.87420688e-36,  2.44445677e-45],
               [ 2.84903522e-39, -7.27823881e-38, -6.00866116e-31],
               [ 2.35025971e-32, -6.00405062e-31,  7.28986103e-38]]
    ref_overtone_tm = [[-1.12592159e-37,  1.21877780e-37,  3.16289799e-46],
                       [ 6.25968499e-37,  7.96578806e-38, -2.99396437e-47],
                       [-3.04608728e-36,  5.92335191e-38,  4.75566692e-47],
                       [-2.61356657e-37,  3.62945153e-37, -1.56754210e-46]]
    ref_combotone_tm = \
[[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
  [-1.50403619e-31, -5.88720350e-33, -9.43194007e-47],
  [-6.06833280e-46, -1.00838717e-46, -1.75486652e-36],
  [-8.78672647e-37, -5.38211005e-38,  1.13738075e-45]],
 [[-1.50403619e-31, -5.88720350e-33, -9.43194007e-47],
  [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
  [ 6.05542068e-44, -1.63788039e-42, -5.82264169e-33],
  [ 2.27443419e-34, -5.81835666e-33,  4.59888452e-42]],
 [[-6.06833280e-46, -1.00838717e-46, -1.75486652e-36],
  [ 6.05542068e-44, -1.63788039e-42, -5.82264169e-33],
  [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
  [ 2.72984690e-45,  6.98297355e-46,  1.36997132e-37]],
 [[-8.78672647e-37, -5.38211005e-38,  1.13738075e-45],
  [ 2.27443419e-34, -5.81835666e-33,  4.59888452e-42],
  [ 2.72984690e-45,  6.98297355e-46,  1.36997132e-37],
  [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]]
 
    transition_moments = get_anharm_corrected_transition_moment(CO2_dipole_gradient, \
                                                                CO2_dipole_hessian, \
                                                                CO2_dipole_cubic, CO2_cubic_ff, \
                                                                CO2_quartic_ff, \
                                                                harmonic_transition_moment, \
                                                                CO2_harmonic_energies, 'No coriolis', \
                                                                0, 'Anharmonic: DVPT2')

    assert np.allclose(np.multiply(harmonic_transition_moment, 1.0e40), \
           np.multiply(transition_moments.harmonic, 1.0e40))
    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(transition_moments.fundamental, 1.0e40))
    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), \
           np.multiply(transition_moments.overtones, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), \
           np.multiply(transition_moments.combotones, 1.0e40))

    # DVPT2 with 1-1 checks
    ref_ftm = [[ 1.32853011e-30,  5.20012975e-32, -3.44446578e-46],
               [-1.53619477e-36, -2.87420688e-36,  2.44445677e-45],
               [ 1.14070630e-41, -2.91485021e-40, -6.00866116e-31],
               [ 2.35025971e-32, -6.00405062e-31,  3.51997146e-40]]

    transition_moments = get_anharm_corrected_transition_moment(CO2_dipole_gradient, \
                                                                CO2_dipole_hessian, \
                                                                CO2_dipole_cubic, CO2_cubic_ff, \
                                                                CO2_quartic_ff, \
                                                                harmonic_transition_moment, \
                                                                CO2_harmonic_energies, 'No coriolis', \
                                                                0, 'Anharmonic: DVPT2, w/ 1-1 checks')

    assert np.allclose(np.multiply(harmonic_transition_moment, 1.0e40), \
           np.multiply(transition_moments.harmonic, 1.0e40))
    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(transition_moments.fundamental, 1.0e40))
    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), \
           np.multiply(transition_moments.overtones, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), \
           np.multiply(transition_moments.combotones, 1.0e40))

    # GVPT2 frequencies, DVPT2 intensities
    transition_moments = get_anharm_corrected_transition_moment(CO2_dipole_gradient, \
                                                                CO2_dipole_hessian, \
                                                                CO2_dipole_cubic, CO2_cubic_ff, \
                                                                CO2_quartic_ff, \
                                                                harmonic_transition_moment, \
                                                                CO2_harmonic_energies, 'No coriolis', \
                                                                0, 'Anharmonic: DVPT2, w/ 1-1 checks')

    assert np.allclose(np.multiply(harmonic_transition_moment, 1.0e40), \
           np.multiply(transition_moments.harmonic, 1.0e40))
    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(transition_moments.fundamental, 1.0e40))
    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), \
           np.multiply(transition_moments.overtones, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), \
           np.multiply(transition_moments.combotones, 1.0e40))

    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks

    transition_moments = \
        get_anharm_corrected_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                               CO2_dipole_cubic, CO2_cubic_ff, CO2_quartic_ff, \
                                               harmonic_transition_moment, CO2_harmonic_energies, \
                                               'No coriolis', 0, \
                                               'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(np.multiply(harmonic_transition_moment, 1.0e40), \
           np.multiply(transition_moments.harmonic, 1.0e40))
    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(transition_moments.fundamental, 1.0e40))
    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), \
           np.multiply(transition_moments.overtones, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), \
           np.multiply(transition_moments.combotones, 1.0e40))
    
    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks with forced removal of all i,j
    transition_moments = \
        get_anharm_corrected_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                               CO2_dipole_cubic, CO2_cubic_ff, CO2_quartic_ff, \
                                               harmonic_transition_moment, CO2_harmonic_energies, \
                                               'No coriolis', 0, \
                                               'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks and forced removal')

    assert np.allclose(np.multiply(harmonic_transition_moment, 1.0e40), \
           np.multiply(transition_moments.harmonic, 1.0e40))
    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(transition_moments.fundamental, 1.0e40))
    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), \
           np.multiply(transition_moments.overtones, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), \
           np.multiply(transition_moments.combotones, 1.0e40))
    


def test_get_overtone_transition_moment():

    # GVPT2 frequencies, DVPT2 intensities, 1-1 checks.  This is default.
    ref_overtone_tm = [[-1.16292209e-06,  1.97523478e-06, -7.75321336e-08],
                       [ 1.64181669e-06, -2.80678078e-06,  7.78650425e-08],
                       [-2.01423153e-07,  3.49764798e-07, -2.00349931e-10],
                       [ 5.87806609e-06, -1.02050911e-05,  6.07003033e-09],
                       [ 5.45940216e-07, -9.47790129e-07,  6.31446143e-10],
                       [ 1.52940640e-05, -2.65522106e-05,  1.60313973e-08]]
    ref_combotone_tm = [[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 1.71669992e-05,  9.93710756e-06,  6.94794802e-05],
                         [-1.00243489e-06, -5.64184892e-07,  2.60055222e-05],
                         [ 2.37231789e-06, -4.12931652e-06, -2.65838692e-08],
                         [-2.66012720e-06, -1.54390248e-06, -2.38513419e-05],
                         [-1.42979179e-05, -8.17089633e-06,  9.77665125e-06]],
                        [[ 1.71669992e-05,  9.93710756e-06,  6.94794802e-05],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [-1.34886498e-06,  2.34001663e-06,  1.88399559e-08],
                         [ 6.00021825e-06,  3.47451750e-06,  3.74527275e-05],
                         [ 1.50810684e-06, -2.62299596e-06, -1.67444120e-08],
                         [ 3.27586134e-05, -5.68978255e-05,  4.18924783e-08]],
                        [[-1.00243489e-06, -5.64184892e-07,  2.60055222e-05],
                         [-1.34886498e-06,  2.34001663e-06,  1.88399559e-08],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 1.44492888e-05,  8.32543434e-06,  4.38679513e-06],
                         [ 1.43209894e-06, -2.48610883e-06,  1.51631088e-09],
                         [-6.58588945e-06,  1.14338146e-05, -6.83713585e-09]],
                        [[ 2.37231789e-06, -4.12931652e-06, -2.65838692e-08],
                         [ 6.00021825e-06,  3.47451750e-06,  3.74527275e-05],
                         [ 1.44492888e-05,  8.32543434e-06,  4.38679513e-06],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [-2.47810937e-07, -1.38744053e-07,  6.75904714e-06],
                         [ 9.25702721e-08,  5.90140392e-08,  9.46979712e-06]],
                        [[-2.66012720e-06, -1.54390248e-06, -2.38513419e-05],
                         [ 1.50810684e-06, -2.62299596e-06, -1.67444120e-08],
                         [ 1.43209894e-06, -2.48610883e-06,  1.51631088e-09],
                         [-2.47810937e-07, -1.38744053e-07,  6.75904714e-06],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 9.24745377e-07, -1.60547499e-06,  1.00710120e-09]],
                        [[-1.42979179e-05, -8.17089633e-06,  9.77665125e-06],
                         [ 3.27586134e-05, -5.68978255e-05,  4.18924783e-08],
                         [-6.58588945e-06,  1.14338146e-05, -6.83713585e-09],
                         [ 9.25702721e-08,  5.90140392e-08,  9.46979712e-06],
                         [ 9.24745377e-07, -1.60547499e-06,  1.00710120e-09],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]]

    overtone_tm, combotone_tm = \
        get_overtone_transition_moment(dipole_gradient, dipole_hess, cubic_ff, \
                                       harmonic_wn, 'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_overtone_tm, overtone_tm)
    assert np.allclose(ref_combotone_tm, combotone_tm)

    # CO2
    # VPT2
    ref_overtone_tm = [[-1.12592159e-37,  1.21877780e-37,  3.16289799e-46],
                       [ 6.25968499e-37,  7.96578806e-38, -2.99396437e-47],
                       [-5.04965541e-36, -3.41065392e-36,  2.29011571e-45],
                       [-2.26506530e-36, -3.10718563e-36,  2.08596210e-45]]
    ref_combotone_tm = [[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [-1.50403619e-31, -5.88720350e-33, -9.43194007e-47],
                         [-6.06833280e-46, -1.00838717e-46, -1.75486652e-36],
                         [-8.78672647e-37, -5.38211005e-38,  1.13738075e-45]],
                        [[-1.50403619e-31, -5.88720350e-33, -9.43194007e-47],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 6.05542068e-44, -1.63788039e-42, -5.82264169e-33],
                         [ 2.27443419e-34, -5.81835666e-33,  4.59888452e-42]],
                        [[-6.06833280e-46, -1.00838717e-46, -1.75486652e-36],
                         [ 6.05542068e-44, -1.63788039e-42, -5.82264169e-33],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                         [ 2.72984690e-45,  6.98297355e-46,  1.36997132e-37]],
                        [[-8.78672647e-37, -5.38211005e-38,  1.13738075e-45],
                         [ 2.27443419e-34, -5.81835666e-33,  4.59888452e-42],
                         [ 2.72984690e-45,  6.98297355e-46,  1.36997132e-37],
                         [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]]

    overtone_tm, combotone_tm = get_overtone_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                               CO2_cubic_ff, CO2_harmonic_energies, \
                                                               'Anharmonic: VPT2')

    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), np.multiply(overtone_tm, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), np.multiply(combotone_tm, 1.0e40))

    # DVPT2 frequency, VPT2 intensity
    overtone_tm, combotone_tm = get_overtone_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                               CO2_cubic_ff, CO2_harmonic_energies, \
                                                               'Anharmonic: Freq DVPT2, Int VPT2')

    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), np.multiply(overtone_tm, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), np.multiply(combotone_tm, 1.0e40))

    # DVPT2
    ref_overtone_tm = [[-1.12592159e-37,  1.21877780e-37,  3.16289799e-46],
                       [ 6.25968499e-37,  7.96578806e-38, -2.99396437e-47],
                       [-3.04608728e-36,  5.92335191e-38,  4.75566692e-47],
                       [-2.61356657e-37,  3.62945153e-37, -1.56754210e-46]]
    ref_combotone_tm = \
[[[ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
  [-1.50403619e-31, -5.88720350e-33, -9.43194007e-47],
  [-6.06833280e-46, -1.00838717e-46, -1.75486652e-36],
  [-8.78672647e-37, -5.38211005e-38,  1.13738075e-45]],
 [[-1.50403619e-31, -5.88720350e-33, -9.43194007e-47],
  [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
  [ 6.05542068e-44, -1.63788039e-42, -5.82264169e-33],
  [ 2.27443419e-34, -5.81835666e-33,  4.59888452e-42]],
 [[-6.06833280e-46, -1.00838717e-46, -1.75486652e-36],
  [ 6.05542068e-44, -1.63788039e-42, -5.82264169e-33],
  [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
  [ 2.72984690e-45,  6.98297355e-46,  1.36997132e-37]],
 [[-8.78672647e-37, -5.38211005e-38,  1.13738075e-45],
  [ 2.27443419e-34, -5.81835666e-33,  4.59888452e-42],
  [ 2.72984690e-45,  6.98297355e-46,  1.36997132e-37],
  [ 0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]]

    overtone_tm, combotone_tm = get_overtone_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                               CO2_cubic_ff, CO2_harmonic_energies, \
                                                               'Anharmonic: DVPT2')

    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), np.multiply(overtone_tm, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), np.multiply(combotone_tm, 1.0e40))

    # DVTP2 with 1-1 checks
    overtone_tm, combotone_tm = get_overtone_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                               CO2_cubic_ff, CO2_harmonic_energies, \
                                                               'Anharmonic: DVPT2, w/ 1-1 checks')

    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), np.multiply(overtone_tm, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), np.multiply(combotone_tm, 1.0e40))

    # GVPT2 frequencies, DVPT2 intensities
    overtone_tm, combotone_tm = get_overtone_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                               CO2_cubic_ff, CO2_harmonic_energies, \
                                                               'Anharmonic: Freq GVPT2, Int DVPT2')

    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), np.multiply(overtone_tm, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), np.multiply(combotone_tm, 1.0e40))

    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks
    overtone_tm, combotone_tm = \
        get_overtone_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                       CO2_cubic_ff, CO2_harmonic_energies, \
                                       'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(np.multiply(ref_overtone_tm, 1.0e40), np.multiply(overtone_tm, 1.0e40))
    assert np.allclose(np.multiply(ref_combotone_tm, 1.0e40), np.multiply(combotone_tm, 1.0e40))


def test_get_anharm_fundamental_transition_moment():

    ref_ftm = [[-2.51377354e-03, -1.44417205e-03,  3.30043304e-04],
               [-9.36639274e-04,  1.63737312e-03, -3.90407951e-07],
               [ 3.81621917e-04, -6.59548266e-04,  1.75283821e-06],
               [ 1.46422582e-04,  8.69970691e-05,  3.59992900e-03],
               [ 3.54952991e-05, -5.87228279e-05,  5.13191492e-07],
               [ 3.07102123e-03, -5.32866297e-03,  4.19493504e-06]]

    ftm = get_anharm_fundamental_transition_moment(dipole_gradient, dipole_hess, dipole_cubic, \
                                                   cubic_ff, quartic_ff, harmonic_wn, 'No coriolis', 0, \
                                                   'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(ref_ftm, ftm)

    # VPT2
    ref_ftm = [[ 1.32853011e-30,  5.20012975e-32, -3.44446578e-46],
               [ 4.55714886e-36,  7.24575525e-37, -3.73461131e-47],
               [ 2.84903522e-39, -7.27823881e-38, -6.00866116e-31],
               [ 2.35025971e-32, -6.00405062e-31,  7.28986103e-38]]

    ftm = get_anharm_fundamental_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                   CO2_dipole_cubic, CO2_cubic_ff, CO2_quartic_ff, \
                                                   CO2_harmonic_energies, 'No coriolis', 0, \
                                                   'Anharmonic: VPT2')

    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(ftm, 1.0e40))

    # DVPT frequency, VPT2 intensity
    ftm = get_anharm_fundamental_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                   CO2_dipole_cubic, CO2_cubic_ff, CO2_quartic_ff, \
                                                   CO2_harmonic_energies, 'No coriolis', 0, \
                                                   'Anharmonic: Freq DVPT2, Int VPT2')

    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(ftm, 1.0e40))

    # DVPT2
    ref_ftm = [[ 1.32853011e-30,  5.20012975e-32, -3.44446578e-46],
               [-1.53619477e-36, -2.87420688e-36,  2.44445677e-45],
               [ 2.84903522e-39, -7.27823881e-38, -6.00866116e-31],
               [ 2.35025971e-32, -6.00405062e-31,  7.28986103e-38]]

    ftm = get_anharm_fundamental_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                   CO2_dipole_cubic, CO2_cubic_ff, CO2_quartic_ff, \
                                                   CO2_harmonic_energies, 'No coriolis', 0, \
                                                   'Anharmonic: DVPT2')

    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(ftm, 1.0e40))

    # DVPT2 with 1-1 checks
    ref_ftm = [[ 1.32853011e-30,  5.20012975e-32, -3.44446578e-46],
               [-1.53619477e-36, -2.87420688e-36,  2.44445677e-45],
               [ 1.14070630e-41, -2.91485021e-40, -6.00866116e-31],
               [ 2.35025971e-32, -6.00405062e-31,  3.51997146e-40]]

    ftm = get_anharm_fundamental_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                   CO2_dipole_cubic, CO2_cubic_ff, CO2_quartic_ff, \
                                                   CO2_harmonic_energies, 'No coriolis', 0, \
                                                   'Anharmonic: DVPT2, w/ 1-1 checks')

    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(ftm, 1.0e40))

    # GVPT2 frequencies, DVPT2 intensities
    ref_ftm = [[ 1.32853011e-30,  5.20012975e-32, -3.44446578e-46],
               [-1.53619477e-36, -2.87420688e-36,  2.44445677e-45],
               [ 2.84903522e-39, -7.27823881e-38, -6.00866116e-31],
               [ 2.35025971e-32, -6.00405062e-31,  7.28986103e-38]]

    ftm = get_anharm_fundamental_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                   CO2_dipole_cubic, CO2_cubic_ff, CO2_quartic_ff, \
                                                   CO2_harmonic_energies, 'No coriolis', 0, \
                                                   'Anharmonic: Freq GVPT2, Int DVPT2')

    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(ftm, 1.0e40))

    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks
    ref_ftm = [[ 1.32853011e-30,  5.20012975e-32, -3.44446578e-46],
               [-1.53619477e-36, -2.87420688e-36,  2.44445677e-45],
               [ 1.14070630e-41, -2.91485021e-40, -6.00866116e-31],
               [ 2.35025971e-32, -6.00405062e-31,  3.51997146e-40]]

    ftm = get_anharm_fundamental_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                   CO2_dipole_cubic, CO2_cubic_ff, CO2_quartic_ff, \
                                                   CO2_harmonic_energies, 'No coriolis', 0, \
                                                   'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(ftm, 1.0e40))

    # GVPT2 frequencies, DVPT2 intensities, w/ 1-1 checks with forced removal
    ftm = get_anharm_fundamental_transition_moment(CO2_dipole_gradient, CO2_dipole_hessian, \
                                                   CO2_dipole_cubic, CO2_cubic_ff, CO2_quartic_ff, \
                                                   CO2_harmonic_energies, 'No coriolis', 0, \
                                                   'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks and forced removal')

    assert np.allclose(np.multiply(ref_ftm, 1.0e40), np.multiply(ftm, 1.0e40))
    


def test_anharm_corrected_vibrational_energies():

    # CO2
    # VPT2
    ref_fwn = [4.99672603e-20, 2.76619452e-20, 1.52668003e-20, 1.52667712e-20]
    ref_overtones = [9.94426963e-20, 5.52167382e-20, 3.15898640e-20, 3.15898689e-20]
    ref_combotones = [[0.00000000e+00, 7.72207696e-20, 6.49575423e-20, 6.49575091e-20],
                      [7.72207696e-20, 0.00000000e+00, 4.08820956e-20, 4.08819362e-20],
                      [6.49575423e-20, 4.08820956e-20, 0.00000000e+00, 3.05686413e-20],
                      [6.49575091e-20, 4.08819362e-20, 3.05686413e-20, 0.00000000e+00]]
    anharmonic_energies = anharm_corrected_vibrational_energies(CO2_harmonic_energies, CO2_cubic_ff, \
                                                                CO2_quartic_ff, 0, 'No coriolis', \
                                                                'Anharmonic: VPT2')

    assert np.allclose(np.multiply(CO2_harmonic_energies, 1.0e20), \
                       np.multiply(anharmonic_energies.harmonic, 1.0e20))
    assert np.allclose(np.multiply(ref_fwn, 1.0e20), np.multiply(anharmonic_energies.fundamental, 1.0e20))
    assert np.allclose(np.multiply(ref_overtones, 1.0e20), \
                       np.multiply(anharmonic_energies.overtones, 1.0e20))
    assert np.allclose(np.multiply(ref_combotones, 1.0e20), \
                       np.multiply(anharmonic_energies.combotones, 1.0e20))

    # DVPT2 both frequencies and intensities
    ref_fwn = [4.99672603e-20, 2.96022581e-20, 1.52668003e-20, 1.52667712e-20]
    ref_overtones = [9.94426963e-20, 5.90973639e-20, 3.06197400e-20, 3.06196801e-20]
    ref_combotones = [[0.00000000e+00, 7.91610825e-20, 6.49575423e-20, 6.49575091e-20],
                      [7.91610825e-20, 0.00000000e+00, 4.47626565e-20, 4.47626268e-20],
                      [6.49575423e-20, 4.47626565e-20, 0.00000000e+00, 3.05686413e-20],
                      [6.49575091e-20, 4.47626268e-20, 3.05686413e-20, 0.00000000e+00]]

    anharmonic_energies = anharm_corrected_vibrational_energies(CO2_harmonic_energies, CO2_cubic_ff, \
                                                                CO2_quartic_ff, 0, 'No coriolis', \
                                                                'Anharmonic: DVPT2')

    assert np.allclose(np.multiply(CO2_harmonic_energies, 1.0e20), \
                       np.multiply(anharmonic_energies.harmonic, 1.0e20))
    assert np.allclose(np.multiply(ref_fwn, 1.0e20), np.multiply(anharmonic_energies.fundamental, 1.0e20))
    assert np.allclose(np.multiply(ref_overtones, 1.0e20), \
                       np.multiply(anharmonic_energies.overtones, 1.0e20))
    assert np.allclose(np.multiply(ref_combotones, 1.0e20), \
                       np.multiply(anharmonic_energies.combotones, 1.0e20))

    #DVPT2 Frequencies
    anharmonic_energies = anharm_corrected_vibrational_energies(CO2_harmonic_energies, CO2_cubic_ff, \
                                                                CO2_quartic_ff, 0, 'No coriolis', \
                                                                'Anharmonic: Freq DVPT2, Int VPT2')

    assert np.allclose(np.multiply(CO2_harmonic_energies, 1.0e20), \
                       np.multiply(anharmonic_energies.harmonic, 1.0e20))
    assert np.allclose(np.multiply(ref_fwn, 1.0e20), \
                       np.multiply(anharmonic_energies.fundamental, 1.0e20))
    assert np.allclose(np.multiply(ref_overtones, 1.0e20), \
                       np.multiply(anharmonic_energies.overtones, 1.0e20))
    assert np.allclose(np.multiply(ref_combotones, 1.0e20), \
                       np.multiply(anharmonic_energies.combotones, 1.0e20))

    # DVPT2 for both frequencies and intensities, with 1-1 resonance checks
    anharmonic_energies = anharm_corrected_vibrational_energies(CO2_harmonic_energies, CO2_cubic_ff, \
                                                                CO2_quartic_ff, 0, 'No coriolis', \
                                                                'Anharmonic: DVPT2, w/ 1-1 checks')

    assert np.allclose(np.multiply(CO2_harmonic_energies, 1.0e20), \
                       np.multiply(anharmonic_energies.harmonic, 1.0e20))
    assert np.allclose(np.multiply(ref_fwn, 1.0e20), \
                       np.multiply(anharmonic_energies.fundamental, 1.0e20))
    assert np.allclose(np.multiply(ref_overtones, 1.0e20), \
                       np.multiply(anharmonic_energies.overtones, 1.0e20))
    assert np.allclose(np.multiply(ref_combotones, 1.0e20), \
                       np.multiply(anharmonic_energies.combotones, 1.0e20))

    # GVPT2 frequencies, DVPT2 intensities
    ref_fwn = [4.99672603e-20, 2.88476931e-20, 1.52668003e-20, 1.52667712e-20]
    ref_overtones = [9.94426963e-20, 5.90973639e-20, 3.06197101e-20, 3.13742751e-20]
    ref_combotones = [[0.00000000e+00, 7.91610825e-20, 6.49575423e-20, 6.49575091e-20],
                      [7.91610825e-20, 0.00000000e+00, 4.47626565e-20, 4.47626268e-20],
                      [6.49575423e-20, 4.47626565e-20, 0.00000000e+00, 3.05686413e-20],
                      [6.49575091e-20, 4.47626268e-20, 3.05686413e-20, 0.00000000e+00]]

    anharmonic_energies = anharm_corrected_vibrational_energies(CO2_harmonic_energies, CO2_cubic_ff, \
                                                                CO2_quartic_ff, 0, 'No coriolis', \
                                                                'Anharmonic: Freq GVPT2, Int DVPT2')

    assert np.allclose(np.multiply(CO2_harmonic_energies, 1.0e20), \
                       np.multiply(anharmonic_energies.harmonic, 1.0e20))
    assert np.allclose(np.multiply(ref_fwn, 1.0e20), np.multiply(anharmonic_energies.fundamental, 1.0e20))
    assert np.allclose(np.multiply(ref_overtones, 1.0e20), \
                       np.multiply(anharmonic_energies.overtones, 1.0e20))
    assert np.allclose(np.multiply(ref_combotones, 1.0e20), \
                       np.multiply(anharmonic_energies.combotones, 1.0e20))

    # GVPT2 frequencies, DVPT2 intensities, with 1-1 checks
    anharmonic_energies = \
        anharm_corrected_vibrational_energies(CO2_harmonic_energies, CO2_cubic_ff, \
                                              CO2_quartic_ff, 0, 'No coriolis', \
                                              'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks')

    assert np.allclose(np.multiply(CO2_harmonic_energies, 1.0e20), \
                       np.multiply(anharmonic_energies.harmonic, 1.0e20))
    assert np.allclose(np.multiply(ref_fwn, 1.0e20), np.multiply(anharmonic_energies.fundamental, 1.0e20))
    assert np.allclose(np.multiply(ref_overtones, 1.0e20), \
                       np.multiply(anharmonic_energies.overtones, 1.0e20))
    assert np.allclose(np.multiply(ref_combotones, 1.0e20), \
                       np.multiply(anharmonic_energies.combotones, 1.0e20))
    
    # GVPT2 frequencies, DVPT2 intensities, with 1-1 checks and forced removal of all i,j
    anharmonic_energies = \
        anharm_corrected_vibrational_energies(CO2_harmonic_energies, CO2_cubic_ff, \
                                              CO2_quartic_ff, 0, 'No coriolis', \
                                              'Anharmonic: Freq GVPT2, Int DVPT2, w/ 1-1 checks and forced removal')

    assert np.allclose(np.multiply(CO2_harmonic_energies, 1.0e20), \
                       np.multiply(anharmonic_energies.harmonic, 1.0e20))
    assert np.allclose(np.multiply(ref_fwn, 1.0e20), np.multiply(anharmonic_energies.fundamental, 1.0e20))
    assert np.allclose(np.multiply(ref_overtones, 1.0e20), \
                       np.multiply(anharmonic_energies.overtones, 1.0e20))
    assert np.allclose(np.multiply(ref_combotones, 1.0e20), \
                       np.multiply(anharmonic_energies.combotones, 1.0e20))
    


def test_get_red_prefactor():

    ref_prefactor = [1.72101890e-21, 1.72256291e-21, 2.62627069e-21, 2.78067106e-21, 2.87459970e-21, \
                     8.19415633e-21]

    prefactor = get_red_prefactor(harmonic_wn)

    assert np.allclose(ref_prefactor, prefactor)


def test_transform_to_reduced_nm():

    # Dipole gradient
    out_mat = np.zeros(18)
    wn_prefactor = [1.72101890e-21, 1.72256291e-21, 2.62627069e-21, 2.78067106e-21, 2.87459970e-21, \
                    8.19415633e-21]

    ref_out_mat = [-6.11572575e-24, -3.52630095e-24,  8.00854116e-25, -2.29785989e-24, \
                    3.97832992e-24, -1.78957112e-27,  1.41324733e-24, -2.45354022e-24, \
                    1.50402901e-27,  5.71607106e-25,  3.37811470e-25,  1.41525698e-23, \
                    1.40022284e-25, -2.43085177e-25,  2.22387830e-28,  3.55771902e-23, \
                   -6.17656089e-23,  3.72287043e-26]

    offset = transform_to_reduced_nm(1, 1, out_mat, dipole_gradient, 0, wn_prefactor)

    assert np.allclose(ref_out_mat, out_mat)

    # Dipole hessian
    out_mat = np.zeros(108)
    ref_out_mat = [-8.51028600e-48,  1.43965188e-47, -6.47535337e-49,  1.05123042e-46, \
                    6.08406668e-47,  4.11506876e-46, -8.75932380e-48, -4.92535377e-48, \
                    2.35564210e-46,  2.28153923e-47, -3.97128046e-47, -2.54821408e-49, \
                   -2.62300357e-47, -1.52239952e-47, -2.36093255e-46, -3.96720289e-46, \
                   -2.26682470e-46,  2.74795822e-46,  1.05123042e-46,  6.08406668e-47, \
                    4.11506876e-46,  1.50361761e-47, -2.57266041e-47,  6.54113075e-49, \
                   -1.20694184e-47,  2.09384471e-47,  1.70975715e-49,  5.76165052e-47, \
                    3.33634888e-47,  3.59426588e-46,  1.49495003e-47, -2.60009190e-47, \
                   -1.65882734e-49,  9.27003188e-46, -1.61008087e-45,  1.18420804e-48, \
                   -8.75932380e-48, -4.92535377e-48,  2.35564210e-46, -1.20694184e-47, \
                    2.09384471e-47,  1.70975715e-49, -4.07006396e-49,  7.08264709e-49, \
                   -2.17654717e-52,  2.19543823e-46,  1.26494923e-46,  6.25208363e-47, \
                    2.11999332e-47, -3.68027414e-47,  2.24427087e-50, -2.85317874e-46, \
                    4.95342625e-46, -2.96218159e-49,  2.28153923e-47, -3.97128046e-47, \
                   -2.54821408e-49,  5.76165052e-47,  3.33634888e-47,  3.59426588e-46, \
                    2.19543823e-46,  1.26494923e-46,  6.25208363e-47,  1.31596462e-46, \
                   -2.28468619e-46,  1.35960990e-49, -4.92286158e-48, -2.77202205e-48, \
                    1.07580272e-46,  4.49637407e-48,  2.85859241e-48,  4.46289073e-46, \
                   -2.62300357e-47, -1.52239952e-47, -2.36093255e-46,  1.49495003e-47, \
                   -2.60009190e-47, -1.65882734e-49,  2.11999332e-47, -3.68027414e-47, \
                    2.24427087e-50, -4.92286158e-48, -2.77202205e-48,  1.07580272e-46, \
                    1.27823472e-47, -2.21910631e-47,  1.47773934e-50,  4.32489547e-47, \
                   -7.50856651e-47,  4.71992079e-50, -3.96720289e-46, -2.26682470e-46, \
                    2.74795822e-46,  9.27003188e-46, -1.61008087e-45,  1.18420804e-48, \
                   -2.85317874e-46,  4.95342625e-46, -2.96218159e-49,  4.49637407e-48, \
                    2.85859241e-48,  4.46289073e-46,  4.32489547e-47, -7.50856651e-47, \
                    4.71992079e-50,  2.92860912e-45, -5.08439394e-45,  3.06983408e-48]

    offset = transform_to_reduced_nm(1, 1, out_mat, dipole_hess, 0, wn_prefactor)

    assert np.allclose(ref_out_mat, out_mat)

    # Polarizability gradient
    out_mat = np.zeros(54)
    ref_out_mat = [ 1.04054558e-22, -6.04957169e-23,  3.54301015e-23, -6.04957169e-23, \
                   -1.04373819e-22, -6.16296994e-23,  3.54301015e-23, -6.16296994e-23, \
                   -1.41834187e-26,  2.08177886e-22,  8.61072146e-23,  1.37094946e-22, \
                    8.61072146e-23,  1.08259146e-22,  7.89366584e-23,  1.37094946e-22, \
                    7.89366584e-23,  1.14171180e-22, -6.15679388e-23, -2.05615313e-23, \
                    1.05817879e-22, -2.05615313e-23, -3.75667046e-23,  6.10437966e-23, \
                    1.05817879e-22,  6.10437966e-23,  1.28037199e-22, -4.31876664e-23, \
                    2.50702950e-23,  3.17413187e-23,  2.50702950e-23,  4.31212167e-23, \
                   -5.51591555e-23,  3.17413187e-23, -5.51591555e-23,  6.51240228e-26, \
                    1.26388801e-23,  1.41668219e-24,  2.06908096e-23,  1.41668219e-24, \
                    1.10248102e-23,  1.20663049e-23,  2.06908096e-23,  1.20663049e-23, \
                    2.57178280e-22, -1.97981785e-22, -3.53502532e-22, -9.75440891e-23, \
                   -3.53502532e-22,  2.11980722e-22, -5.64484020e-23, -9.75440891e-23, \
                   -5.64484020e-23, -1.99048408e-23]

    offset = transform_to_reduced_nm(1, 2, out_mat, polarizability_gradient[0], 0, wn_prefactor)

    assert np.allclose(ref_out_mat, out_mat)


def test_transform_to_reduced_nm_prep():

    ref_transformed = \
       [[[-6.11572573e-24, -3.52630094e-24,  8.00854114e-25],
       [-2.29785989e-24,  3.97832992e-24, -1.78957112e-27],
       [ 1.41324733e-24, -2.45354022e-24,  1.50402901e-27],
       [ 5.71607106e-25,  3.37811470e-25,  1.41525698e-23],
       [ 1.40022284e-25, -2.43085177e-25,  2.22387830e-28],
       [ 3.55771902e-23, -6.17656089e-23,  3.72287043e-26]]] 

    transformed = transform_to_reduced_nm_prep([dipole_gradient], harmonic_wn, 1)

    assert np.allclose(ref_transformed, transformed)

    ref_transformed = \
        [[[[ 1.04054558e-22, -6.04957168e-23,  3.54301014e-23],
        [-6.04957168e-23, -1.04373819e-22, -6.16296992e-23],
        [ 3.54301014e-23, -6.16296992e-23, -1.41834187e-26]],
       [[ 2.08177886e-22,  8.61072145e-23,  1.37094946e-22],
        [ 8.61072145e-23,  1.08259146e-22,  7.89366583e-23],
        [ 1.37094946e-22,  7.89366583e-23,  1.14171180e-22]],
       [[-6.15679388e-23, -2.05615313e-23,  1.05817879e-22],
        [-2.05615313e-23, -3.75667046e-23,  6.10437966e-23],
        [ 1.05817879e-22,  6.10437966e-23,  1.28037199e-22]],
       [[-4.31876664e-23,  2.50702950e-23,  3.17413187e-23],
        [ 2.50702950e-23,  4.31212167e-23, -5.51591555e-23],
        [ 3.17413187e-23, -5.51591555e-23,  6.51240228e-26]],
       [[ 1.26388801e-23,  1.41668219e-24,  2.06908096e-23],
        [ 1.41668219e-24,  1.10248102e-23,  1.20663049e-23],
        [ 2.06908096e-23,  1.20663049e-23,  2.57178281e-22]],
       [[-1.97981785e-22, -3.53502532e-22, -9.75440891e-23],
        [-3.53502532e-22,  2.11980722e-22, -5.64484020e-23],
        [-9.75440891e-23, -5.64484020e-23, -1.99048408e-23]]], 
       [[[ 1.04054558e-22, -6.04957166e-23,  3.54301021e-23],
        [-6.04957166e-23, -1.04373819e-22, -6.16296994e-23],
        [ 3.54301021e-23, -6.16296994e-23, -1.41831962e-26]],
       [[ 2.08177886e-22,  8.61072140e-23,  1.37094946e-22],
        [ 8.61072140e-23,  1.08259146e-22,  7.89366583e-23],
        [ 1.37094946e-22,  7.89366583e-23,  1.14171179e-22]],
       [[-6.15679385e-23, -2.05615314e-23,  1.05817879e-22],
        [-2.05615314e-23, -3.75667044e-23,  6.10437966e-23],
        [ 1.05817879e-22,  6.10437966e-23,  1.28037197e-22]],
       [[-4.31876650e-23,  2.50702947e-23,  3.17413182e-23],
        [ 2.50702947e-23,  4.31212162e-23, -5.51591552e-23],
        [ 3.17413182e-23, -5.51591552e-23,  6.51241352e-26]],
       [[ 1.26388800e-23,  1.41668215e-24,  2.06908094e-23],
        [ 1.41668215e-24,  1.10248101e-23,  1.20663045e-23],
        [ 2.06908094e-23,  1.20663045e-23,  2.57178278e-22]],
       [[-1.97981789e-22, -3.53502535e-22, -9.75440908e-23],
        [-3.53502535e-22,  2.11980719e-22, -5.64484036e-23],
        [-9.75440908e-23, -5.64484036e-23, -1.99048430e-23]]], 
       [[[ 1.07466568e-22, -6.24802627e-23,  3.68989799e-23],
        [-6.24802627e-23, -1.07798000e-22, -6.41874811e-23],
        [ 3.68989799e-23, -6.41874811e-23, -1.55989307e-26]],
       [[ 2.16110076e-22,  8.96461966e-23,  1.43688861e-22],
        [ 8.96461966e-23,  1.12087825e-22,  8.27345413e-23],
        [ 1.43688861e-22,  8.27345413e-23,  1.19982990e-22]],
       [[-6.24975425e-23, -2.08463764e-23,  1.06472493e-22],
        [-2.08463764e-23, -3.81649368e-23,  6.14222178e-23],
        [ 1.06472493e-22,  6.14222178e-23,  1.29862853e-22]],
       [[-4.42785056e-23,  2.57030180e-23,  3.17093096e-23],
        [ 2.57030180e-23,  4.42121162e-23, -5.51049266e-23],
        [ 3.17093096e-23, -5.51049266e-23,  6.49976583e-26]],
       [[ 1.32318854e-23,  1.19277791e-24,  2.29204650e-23],
        [ 1.19277791e-24,  1.18806906e-23,  1.33581615e-23],
        [ 2.29204650e-23,  1.33581615e-23,  2.70732432e-22]],
       [[-2.01333086e-22, -3.57534403e-22, -9.93430940e-23],
        [-3.57534403e-22,  2.13304270e-22, -5.74873220e-23],
        [-9.93430940e-23, -5.74873220e-23, -2.07030135e-23]]]]

    transformed = transform_to_reduced_nm_prep(polarizability_gradient, harmonic_wn, 2)

    assert np.allclose(ref_transformed, transformed)
