from dataclasses import field, dataclass
from typing import Optional


@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for training and eval.
    """
    notation: str = field(
        default="BIO",
        metadata={"help": "NER notation e.g BIO"},
    )
    ner_types: Optional[str] = field(
        default=None,
        metadata={"help": "Pass a list of NER types"},
    )
    ner_priorities: Optional[str] = field(
        default=None,
        metadata={"help": "Pass a list of NER priorities - that indicate the importance of the ner types - needs to "
                          "be in the same order as the ner_types list"
                  },
    )
    ner_type_maps: Optional[str] = field(
        default=None,
        metadata={
            "help": "List that contains the mappings between the original NER types to another set of NER types. Used "
                    "mainly for evaluation. to map ner token labels to another set of ner token"
        },
    )
    fix_spans: bool = field(
        default=True,
        metadata={
            "help": "Whether to fix token span alignments. Required to align tokens and labels (token level)"
        },
    )
    ner_augment: Optional[str] = field(
        default=None,
        metadata={
            "help": "Whether to replace/augment NER tokens (augment data)"
        },
    )
    character_noise_augment: Optional[float] = field(
        default=None,
        metadata={
            "help": "Add noise by replacing a character in a token"
        },
    )
    train_file: Optional[str] = field(
        default=None,
        metadata={"help": "The input training data file (a text file)."}
    )
    validation_file: Optional[str] = field(
        default=None,
        metadata={"help": "An optional input evaluation data file to evaluate the perplexity on (a text file)."},
    )
    test_file: Optional[str] = field(
        default=None,
        metadata={"help": "An optional input evaluation data file to evaluate the perplexity on (a text file)."},
    )
    train_on_fly: Optional[str] = field(
        default=True,
        metadata={"help": "Whether to run text transformations (e.g tokenization) on the fly"}
    )
    validation_on_fly: Optional[str] = field(
        default=False,
        metadata={"help": "Whether to run text transformations (e.g tokenization) on the fly"}
    )
    test_on_fly: Optional[str] = field(
        default=False,
        metadata={"help": "Whether to run text transformations (e.g tokenization) on the fly"}
    )
    overwrite_cache: bool = field(
        default=False, metadata={"help": "Overwrite the cached training and evaluation sets"}
    )
    max_seq_length: Optional[int] = field(
        default=None,
        metadata={
            "help": "The maximum total input sequence length after tokenization. Sequences longer "
            "than this will be truncated."
        },
    )
    preprocessing_num_workers: Optional[int] = field(
        default=None,
        metadata={"help": "The number of processes to use for the preprocessing."},
    )
    pad_to_max_length: bool = field(
        default=True,
        metadata={
            "help": "Whether to pad all samples to `max_seq_length`. "
            "If False, will pad the samples dynamically when batching to the maximum length in the batch."
        },
    )
    text_column_name: Optional[str] = field(
        default=None,
        metadata={"help": "The column that contains the ICD10 codes"}
    )
    note_id_column_name: Optional[str] = field(
        default=None,
        metadata={"help": "The column that contains the note ids"}
    )
    max_train_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of training examples to this "
            "value if set."
        },
    )
    max_eval_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of evaluation examples to this "
            "value if set."
        },
    )
    max_predict_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of test examples to this "
                    "value if set."
        },
    )
    label_all_tokens: bool = field(
        default=False,
        metadata={
            "help": (
                "Whether to put the label for one word on all tokens of generated by that word or just on the "
                "one (in which case the other tokens will have a padding index)."
            )
        },
    )
    return_entity_level_metrics: bool = field(
        default=True,
        metadata={"help": "Whether to return all the entity levels during evaluation or just the overall ones."},
    )

    def __post_init__(self):
        if self.train_file is None and self.validation_file is None and self.test_file is None:
            raise ValueError("Need either a dataset name or a training/validation file.")
        else:
            if self.train_file is not None:
                if type(self.train_file) == list:
                    extension = self.train_file[0].split(".")[-1]
                else:
                    extension = self.train_file.split(".")[-1]
                if extension not in ["csv", "json", "txt", "parquet", "zst", "jsonl"]:
                    raise ValueError("`train_file` should be a csv, a parquet, a json or a txt file.")
            if self.validation_file is not None:
                if type(self.validation_file) == list:
                    extension = self.validation_file[0].split(".")[-1]
                else:
                    extension = self.validation_file.split(".")[-1]
                if extension not in ["csv", "json", "txt", "parquet", "zst", "jsonl"]:
                    raise ValueError("`validation_file` should be a csv, a parquet, a json or a txt file.")
