import os
import logging
from typing import Union, List, Optional, Sequence, NoReturn, Dict
from argparse import ArgumentParser, ArgumentDefaultsHelpFormatter

from datasets import Dataset
from transformers import (
    set_seed,
    HfArgumentParser,
    TrainingArguments,
    DataCollatorForTokenClassification,
    Trainer,
    PreTrainedModel,
    PreTrainedTokenizer,
    PreTrainedTokenizerFast, PretrainedConfig
)
from transformers.trainer_utils import get_last_checkpoint

from .arguments import ModelArguments, DataTrainingArguments
from .deidentification import TextDeidentification, DeidentificationStrategy
from .model_outputs import NoteAggregate
from .sequence_datasets.builder import DatasetBuilder, SequenceDataset
from .sequence_datasets.labels import AlignTokenLabels, SpanFixer, LabelInfo
from .sequence_datasets.text_tokenizers import SpacyTokenizer
from .sequence_datasets.text_transformations import TextTransform
from .model_outputs import ArgmaxProcess, ThresholdProcess
from .metrics import SequenceMetrics
from .model_helpers import ModelHelpers

from .utils import setup_logging, log_training_args, get_ner_augmenter, get_deidentification_level_type

logger = logging.getLogger(__name__)


class SequenceTagger(object):
    """
    The SequenceTagger class can be used to train sequence tagging models, evaluate models, and
    run predictions. The class can also be used to de-identify and augment text.
    TODO: Add a more detailed description
    """

    def __init__(
            self,
            text_tokenizer: SpacyTokenizer,
            config: PretrainedConfig,
            tokenizer: Union[PreTrainedTokenizer, PreTrainedTokenizerFast],
            model: PreTrainedModel,
            notation: str,
            ner_types: Sequence[str],
            fix_spans: bool,
            ner_priorities: Sequence[str],
            fp16: bool,
            ner_augment: str,
            recall_threshold: Optional[float] = None,
            pad_to_max_length: bool = True,
            max_seq_length: Optional[int] = None,
            label_all_tokens: bool = False,
            ignore_label: int = -100,
            token_ignore_label: str = 'NA',
            seed: int = 41
    ):
        """
        Initialize the variables and load the relevant objects.

        Args:
            text_tokenizer (SpacyTokenizer): The SpacyTokenizer object for word tokenization of text.
            config (PretrainedConfig): The HuggingFace model config object.
            tokenizer (Union[PreTrainedTokenizer, PreTrainedTokenizerFast]): The Huggingface tokenizer object.
            model (PreTrainedModel): The HuggingFace model object.
            notation (str): The NER notation (e.g. BIO, BILOU etc).
            ner_types (List[str]): The list of NER labels (e.g. [AGE, PHONE, ...]).
            fix_spans (bool): Fix any token spans mis-alignments.
            ner_priorities (List[str]): THe list of priorities corresponding to the NER types. These will be used
            when fixing span and token mis-alignments.
            fp16 (bool): Whether to use mixed-precision.
            ner_augment (str): The type of NER augment that will be used.
            recall_threshold (Optional[float], defaults to `None`): Apply a recall threshold on the softmax
            probabilities of the linear classification layer.
            pad_to_max_length (bool, defaults to `True`): Whether to pad all samples to `max_seq_length`. If False,
            will pad the samples dynamically when batching to the maximum length in the batch.
            max_seq_length (Optional[int], defaults to `None`): The maximum total input sequence length after
            tokenization. Sequences longer than this will be truncated.
            label_all_tokens (bool, defaults to `False`): Whether to put the label for one word on all tokens of
            generated by that word or just on the one (in which case the other tokens will have a padding index).
            ignore_label (int, defaults to `-100`): Use this value for padded tokens and any tokens that should not be
            involved in the backward pass.
            token_ignore_label (str, defaults to `NA`): This indicates tokens added for context and that should not be
            part of the backward pass or the final output.
            seed (int, defaults to `41): Reproducible seed.
        """

        # Set seed before initializing model.
        set_seed(seed)

        # Initialize NER related variables
        self._notation = notation
        self._ner_types = ner_types
        self._ner_priorities = ner_priorities
        self._label_all_tokens = label_all_tokens

        # Align tokens with labels (token level labels)
        self._align_token_labels = AlignTokenLabels(notation=self._notation)

        # Create the span fixer object
        # Set it to None, if we don't wish to fix spans (the input file may already contain fixed spans)
        if fix_spans:
            self._span_fixer = SpanFixer(
                ner_types=self._ner_types,
                ner_priorities=self._ner_priorities,
            )
        else:
            self._span_fixer = None

        # This object will have label information
        if ner_types is not None:
            self._label_info = LabelInfo(ner_types=self._ner_types, notation=self._notation)
        else:
            # TODO: Fix this
            self._label_info = None

        # Set the b_to_i label mapping if we are labeling all tokens (i.e. labeling the subwords)
        if self._label_all_tokens:
            self._b_to_i_label = self.get_b_to_i_label(label_list=self._label_info.get_label_list())
        else:
            self._b_to_i_label = None

        # Initialize variables that will be set in the set_trainer function to None
        self._trainer = None
        self.__train_samples = None
        self.__eval_samples = None

        # Initialize the tokenizers (HuggingFace and SpaCy), Huggingface config, Huggingface model and data collator
        self._text_tokenizer = text_tokenizer
        self._config = config
        self._tokenizer = tokenizer
        self._model = model
        self._data_collator = self.get_data_collator(fp16=fp16)

        # Set the label2id and id2label mapping of the HuggingFace config object
        if self._label_info is not None:
            self._model.config.label2id = self._label_info.get_label_to_id()
            self._model.config.id2label = self._label_info.get_id_to_label()
            self._label_list = self._label_info.get_label_list()
        else:
            self._label_list = list(self._config.label2id)

        # Set the label_to_id
        self.label_to_id = self._model.config.label2id

        # Create the NER augment object and set the TextTransform
        self._text_transform = self.get_text_transform(
            ner_augmenter=self.get_ner_augmenter(ner_augment=ner_augment),
            pad_to_max_length=pad_to_max_length,
            max_seq_length=max_seq_length
        )

        # Based on the recall threshold, initialize the appropriate post process method
        if recall_threshold is None:
            self._post_process = ArgmaxProcess(
                label_list=self._label_list,
                ignore_label=ignore_label,
                token_ignore_label=token_ignore_label
            )
        else:
            self._post_process = ThresholdProcess(
                label_list=self._label_list,
                recall_threshold=recall_threshold,
                ignore_label=ignore_label,
                token_ignore_label=token_ignore_label
            )

        # Initialize the NoteAggregate object to aggregate sequences/sentences back
        # to the note/document level
        self._note_aggregate = NoteAggregate()

    def get_label_list(self) -> Sequence[str]:
        """
        Return the NER label list.

        Returns:
            _label_list (Sequence[str]): The NER label list.
        """

        return self._label_list

    def get_data_collator(self, fp16: bool) -> DataCollatorForTokenClassification:
        """
        Get the data collator object.

        Args:
            fp16 (bool): Whether to use mixed precision training.

        Returns:
            (DataCollatorForTokenClassification): The data collator object.
        """
        # Data collator
        return DataCollatorForTokenClassification(
            self._tokenizer, pad_to_multiple_of=8 if fp16 else None
        )

    def get_text_transform(
            self,
            ner_augmenter,
            pad_to_max_length=True,
            max_seq_length=None
    ):

        return TextTransform(
            tokenizer=self._tokenizer,
            text_tokenizer=self._text_tokenizer,
            span_fixer=self._span_fixer,
            align_token_labels=self._align_token_labels,
            ner_augmenter=ner_augmenter,
            character_noise_augmenter=None,
            truncation=True,
            pad_to_max_length=pad_to_max_length,
            max_seq_length=max_seq_length,
            label_to_id=self.label_to_id,
            label_all_tokens=self._label_all_tokens,
            b_to_i_label=self._b_to_i_label
        )

    def get_train_dataset(
            self,
            train_file: Optional[str],
            training_args: TrainingArguments,
            train_on_fly: bool = True,
            preprocessing_num_workers: Optional[int] = None,
            overwrite_cache: bool = False,
            max_train_samples: Optional[int] = None
    ) -> Optional[Dataset]:
        """
        Get the training dataset.

        Args:
            train_file (Optional[str]): The file that contains the training data.
            training_args (TrainingArguments): The HuggingFace training arguments object.
            train_on_fly (bool, defaults to `True`): Option to return text transformations on the fly (e.g tokenization
            and augmentation etc.).
            preprocessing_num_workers (Optional[int], defaults to `None`): The number of processes to use for the
            preprocessing.
            overwrite_cache (bool, defaults to `False`): Overwrite the cached training sets.
            max_train_samples (Optional[int], defaults to `None`): For debugging purposes or quicker training, truncate
            the number of training examples to this value if set.

        Returns:
            train_dataset (Optional[Dataset]): The training dataset if the train file is not None else None.
        """

        # Return none if train file is not passed
        if train_file is None:
            return None

        # Create a sequence dataset object to load the training dataset
        raw_sequence_datasets = DatasetBuilder(
            train_file=train_file,
            validation_file=None,
            test_file=None
        ).raw_datasets
        sequence_datasets = SequenceDataset(
            sentence_datasets=raw_sequence_datasets,
        )
        # Get the train split of the dataset
        train_dataset = sequence_datasets.get_train_dataset(max_train_samples=max_train_samples)

        # Apply any text transformations on the fly or initially
        if train_on_fly:
            train_dataset.set_transform(self._text_transform.encode_train)
        else:
            remove_columns = train_dataset.column_names
            # Run the tokenization process
            with training_args.main_process_first(desc="Train dataset map tokenization"):
                train_dataset = train_dataset.map(
                    self._text_transform.encode_train,
                    batched=True,
                    num_proc=preprocessing_num_workers,
                    load_from_cache_file=not overwrite_cache,
                    remove_columns=remove_columns,
                    desc="Running text transformations on train dataset",
                )
        return train_dataset

    def get_validation_dataset(
            self,
            validation_file: Optional[str],
            training_args: TrainingArguments,
            validation_on_fly: bool = False,
            preprocessing_num_workers: Optional[int] = None,
            overwrite_cache: bool = False,
            max_eval_samples: Optional[int] = None
    ):
        """
        Get the validation dataset.

        Args:
            validation_file (Optional[str]): The file that contains the validation data.
            training_args (TrainingArguments): The HuggingFace training arguments object.
            validation_on_fly (bool, defaults to `True`): Option to return text transformations on the fly
            (e.g tokenization and augmentation etc.).
            preprocessing_num_workers (Optional[int], defaults to `None`): The number of processes to use for the
            preprocessing.
            overwrite_cache (bool, defaults to `False`): Overwrite the cached validation sets.
            max_eval_samples (Optional[int], defaults to `None`): For debugging purposes or quicker training, truncate
            the number of validation examples to this value if set.

        Returns:
            eval_dataset (Optional[Dataset]): The validation dataset if the validation file is not None else None.
        """

        # Return none if validation file is not passed
        if validation_file is None:
            return None

        # Create a sequence dataset object to load the validation dataset
        raw_sequence_datasets = DatasetBuilder(
            train_file=None,
            validation_file=validation_file,
            test_file=None
        ).raw_datasets
        sequence_datasets = SequenceDataset(
            sentence_datasets=raw_sequence_datasets,
        )
        # Get the validation split of the dataset
        eval_dataset = sequence_datasets.get_eval_dataset(max_eval_samples=max_eval_samples)

        # Apply any text transformations on the fly or initially
        if validation_on_fly:
            eval_dataset.set_transform(self._text_transform.encode_validation)
        else:
            remove_columns = eval_dataset.column_names
            with training_args.main_process_first(desc="Validation dataset map tokenization"):
                eval_dataset = eval_dataset.map(
                    self._text_transform.encode_validation,
                    batched=True,
                    num_proc=preprocessing_num_workers,
                    load_from_cache_file=not overwrite_cache,
                    remove_columns=remove_columns,
                    desc="Running text transformations on validation dataset",
                )
        return eval_dataset

    def get_test_dataset(
            self,
            test_file: Optional[str],
            training_args: TrainingArguments,
            test_on_fly: bool = False,
            preprocessing_num_workers: Optional[int] = None,
            overwrite_cache: bool = False,
            max_predict_samples: Optional[int] = None
    ):
        """
        Get the test dataset.

        Args:
            test_file (Optional[str]): The file that contains the test data.
            training_args (TrainingArguments): The HuggingFace training arguments object.
            test_on_fly (bool, defaults to `True`): Option to return text transformations on the fly
            (e.g tokenization and augmentation etc.).
            preprocessing_num_workers (Optional[int], defaults to `None`): The number of processes to use for the
            preprocessing.
            overwrite_cache (bool, defaults to `False`): Overwrite the cached validation sets.
            max_predict_samples (Optional[int], defaults to `None`): For debugging purposes or quicker training,
            truncate the number of test examples to this value if set.

        Returns:
            test_dataset (Optional[Dataset]): The test dataset if the test file is not None else None.
        """

        # Return none if test file is not passed
        if test_file is None:
            return None

        # Create a sequence dataset object to load the test dataset
        raw_sequence_datasets = DatasetBuilder(
            train_file=None,
            validation_file=None,
            test_file=test_file
        ).raw_datasets
        sequence_datasets = SequenceDataset(
            sentence_datasets=raw_sequence_datasets,
        )
        test_dataset = sequence_datasets.get_test_dataset(max_test_samples=max_predict_samples)

        # Apply any text transformations on the fly or initially
        if test_on_fly:
            test_dataset.set_transform(self._text_transform.encode_test)
        else:
            with training_args.main_process_first(desc="Test dataset map tokenization"):
                test_dataset = test_dataset.map(
                    self._text_transform.encode_test,
                    batched=True,
                    num_proc=preprocessing_num_workers,
                    load_from_cache_file=not overwrite_cache,
                    desc="Running text transformations on test dataset",
                )
        return test_dataset

    def set_trainer(
            self,
            training_args: TrainingArguments,
            train_dataset: Optional[Dataset],
            eval_dataset: Optional[Dataset],
            do_train: bool,
            do_eval: bool,
            do_predict: bool,
            train_on_fly: bool = True
    ) -> NoReturn:
        """
        Set the HuggingFace trainer object. If the models are being trained/evaluated, the
        function expects the train and validation datasets to be passed. Based on the
        parameters passed, the HuggingFace trainer is set and can be used to train,
        evaluate and run predictions on the model.

        Args:
            training_args (TrainingArguments): The HuggingFace training arguments object.
            train_dataset (Optional[Dataset]): The training dataset if do_train is set to True.
            eval_dataset (Optional[Dataset]): The validation dataset if do_eval is set to True.
            do_train (bool): Whether to train the model.
            do_eval (bool): Whether to evaluate the model on the validation dataset.
            do_predict (bool): Whether to run predictions on the test dataset.
            train_on_fly (bool, defaults to `True`): Option to return text transformations on the fly (e.g tokenization
            and augmentation etc.).
        """

        # Compute metrics will be None if do_eval is False
        compute_metrics = None

        # Set the remove columns parameter based on the dataset inputs
        # We will remove columns based on whether we're running things
        # on the fly or not
        # TODO Change these conditions based on on-the-fly parameters
        if do_predict and not train_on_fly:
            # Remove unused columns from test set - running forward pass
            training_args.remove_unused_columns = True
        else:
            # Setting to False so that unused columns can be used for the on the fly transformations
            training_args.remove_unused_columns = False

        # Check if train dataset is passed when do_train is True
        if do_train and train_dataset is None:
            raise ValueError("--do_train requires a train dataset")

        # Check if validation dataset is passed when do_eval is True
        # Set the metrics function for validation dataset
        if do_eval:
            if eval_dataset is None:
                raise ValueError("--do_eval requires a validation dataset")
            compute_metrics = SequenceMetrics(
                post_process=self._post_process,
                ner_types=self._ner_types,
                label_list=self._label_list,
                return_entity_level_metrics=True
            ).compute_metrics

        # Initialize our Trainer
        self._trainer = Trainer(
            model=self._model,
            args=training_args,
            train_dataset=train_dataset,
            eval_dataset=eval_dataset,
            tokenizer=self._tokenizer,
            data_collator=self._data_collator,
            compute_metrics=compute_metrics,
        )

        # Set the number of training and evaluation samples
        self.__train_samples = len(train_dataset) if train_dataset is not None else None
        self.__eval_samples = len(eval_dataset) if eval_dataset is not None else None

    def run_train(self, resume_from_checkpoint, last_checkpoint):
        if self.__train_samples is None:
            return None
        checkpoint = None
        if resume_from_checkpoint is not None:
            checkpoint = resume_from_checkpoint
        elif last_checkpoint is not None:
            checkpoint = last_checkpoint
        train_result = self._trainer.train(resume_from_checkpoint=checkpoint)
        self._trainer.save_model()  # Saves the tokenizer too for easy upload
        metrics = train_result.metrics

        metrics["train_samples"] = self.__train_samples

        self._trainer.log_metrics("train", metrics)
        self._trainer.save_metrics("train", metrics)
        self._trainer.save_state()
        return metrics

    def run_eval(self):
        if self.__eval_samples is None:
            return None
        logging.info("*** Evaluate ***")
        metrics = self._trainer.evaluate()

        metrics["eval_samples"] = self.__eval_samples
        self._trainer.log_metrics("eval", metrics)
        self._trainer.save_metrics("eval", metrics)
        return metrics

    def run_predict(self, test_dataset):
        if test_dataset is None:
            return None
        return self._trainer.predict(test_dataset)

    def get_logits(self, model_predictions, model_labels):
        logits = self._post_process.get_true_logits(model_predictions=model_predictions, model_labels=model_labels)
        true_labels = self._post_process.get_true_labels(model_labels=model_labels)
        return logits, true_labels

    def get_logits_aggregate(self, test_dataset, model_predictions, model_labels):
        logits = self._post_process.get_true_logits(model_predictions=model_predictions, model_labels=model_labels)
        true_labels = self._post_process.get_true_labels(model_labels=model_labels)
        return self._note_aggregate.aggregate(
            test_dataset['note_id'],
            test_dataset['global_current_chunk_start'],
            logits,
            true_labels
        )

    def predict_aggregate(self, test_dataset):
        model_output = self.run_predict(test_dataset=test_dataset)
        return self.aggregate_chunks(
            test_dataset=test_dataset,
            model_predictions=model_output.predictions,
            model_labels=model_output.label_ids
        )

    def predict_aggregate_de_identify(self, test_dataset, ner_augment='mimic'):
        model_output = self.run_predict(test_dataset=test_dataset)
        return self.de_identify_chunks(
            test_dataset=test_dataset,
            model_predictions=model_output.predictions,
            model_labels=model_output.label_ids,
            ner_augment=ner_augment
        )

    def aggregate_chunks(self, test_dataset, model_predictions, model_labels):
        true_predictions, true_labels = self._post_process.decode(
            model_predictions=model_predictions, model_labels=model_labels
        )
        true_tokens = self._post_process.filter_by_dataset_labels(
            dataset_lists=test_dataset['tokens'], dataset_labels=test_dataset['labels_list']
        )
        true_whitespaces = self._post_process.filter_by_dataset_labels(
            dataset_lists=test_dataset['whitespaces'], dataset_labels=test_dataset['labels_list']
        )
        true_token_positions = self._post_process.filter_by_dataset_labels(
            dataset_lists=test_dataset['token_positions'], dataset_labels=test_dataset['labels_list']
        )
        return self._note_aggregate.aggregate(
            test_dataset['note_id'],
            test_dataset['global_current_chunk_start'],
            true_predictions,
            true_labels,
            true_tokens,
            true_whitespaces,
            true_token_positions
        )

    def de_identify_chunks(
            self,
            test_dataset,
            model_predictions,
            model_labels,
            ner_augment='mimic',
            deidentification_level='exact'
    ):
        if ner_augment is None:
            raise ValueError('NER augment cannot be None')
        ner_augmenter = self.get_ner_augmenter(ner_augment=ner_augment)
        text_deidentification = TextDeidentification(
            ner_augmenter=ner_augmenter
        )
        aggregated_chunks = self.aggregate_chunks(
            test_dataset=test_dataset,
            model_predictions=model_predictions,
            model_labels=model_labels
        )
        for (
                note_id,
                note_predictions,
                note_labels,
                note_tokens,
                note_whitespaces,
                note_token_positions
        ) in aggregated_chunks:
            text = text_deidentification.get_text(
                tokens=note_tokens, whitespaces=note_whitespaces, token_positions=note_token_positions
            )
            spans = [
                span for span in text_deidentification.get_ner_spans(
                    tokens=note_tokens,
                    predictions=note_predictions,
                    token_positions=note_token_positions,
                    whitespaces=note_whitespaces,
                    deidentification_level=get_deidentification_level_type(
                        deidentification_level=deidentification_level
                    )
                )
            ]
            augmented_text, augmented_spans = text_deidentification.de_identify_text(
                text=text, entities=spans, deidentification_strategy=DeidentificationStrategy.AUGMENT
            )
            informative_text, _ = text_deidentification.de_identify_text(
                text=text, entities=spans, deidentification_strategy=DeidentificationStrategy.INFORMATIVE
            )
            de_identified_text, _ = text_deidentification.de_identify_text(
                text=text, entities=spans, deidentification_strategy=DeidentificationStrategy.REMOVE
            )

            yield (
                {
                    'text': text,
                    'note_id': note_id,
                    'spans': spans,
                    'augmented_text': augmented_text,
                    'augmented_spans': augmented_spans,
                    'informative_text': informative_text,
                    'de_identified_text': de_identified_text
                }
            )

    @staticmethod
    def get_b_to_i_label(label_list: Sequence[str]) -> Sequence[int]:
        b_to_i_label = []
        for idx, label in enumerate(label_list):
            if label.startswith("B-") and label.replace("B-", "I-") in label_list:
                b_to_i_label.append(label_list.index(label.replace("B-", "I-")))
            else:
                b_to_i_label.append(idx)
        return b_to_i_label

    def get_ner_augmenter(self, ner_augment):
        # Create the NER augment object
        if ner_augment is not None:
            return get_ner_augmenter(
                text_tokenizer=self._text_tokenizer,
                notation=self._notation,
                augment_type=ner_augment
            )
        else:
            return None

    @staticmethod
    def log_args(training_args: TrainingArguments) -> NoReturn:
        """
        Log the training arguments.

        Args:
            training_args (TrainingArguments): The HuggingFace training arguments.
        """

        # Setup logging
        setup_logging(logger=logger, log_level=training_args.get_process_log_level())

        # Log the training arguments
        log_training_args(training_args=training_args)

    @staticmethod
    def get_checkpoint(
            output_dir: str,
            resume_from_checkpoint: Union[str, bool],
            do_train: bool, overwrite_output_dir: bool
    ):
        last_checkpoint = None
        if os.path.isdir(output_dir) and do_train and not overwrite_output_dir:
            last_checkpoint = get_last_checkpoint(output_dir)
            if last_checkpoint is None and len(os.listdir(output_dir)) > 0:
                raise ValueError(
                    f"Output directory ({output_dir}) already exists and is not empty. "
                    "Use --overwrite_output_dir to overcome."
                )
            elif last_checkpoint is not None and resume_from_checkpoint is None:
                logging.info(
                    f"Checkpoint detected, resuming training at {last_checkpoint}. To avoid this behavior, change "
                    "the `--output_dir` or add `--overwrite_output_dir` to train from scratch."
                )
        return last_checkpoint


def main():
    cli_parser = ArgumentParser(
        description='configuration arguments provided at run time from the CLI',
        formatter_class=ArgumentDefaultsHelpFormatter
    )
    cli_parser.add_argument(
        '--config_file',
        type=str,
        required=True,
        help='The file that contains training configurations'
    )

    args = cli_parser.parse_args()

    # Wan DB project name
    os.environ["WANDB_PROJECT"] = f"deidentification_2022"

    # Huggingface parser
    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, TrainingArguments))
    model_args, data_args, training_args = parser.parse_json_file(json_file=args.config_file)

    # Setup training args, read and store all the other arguments
    training_args = training_args
    do_train = training_args.do_train
    do_eval = training_args.do_eval
    do_predict = training_args.do_predict
    output_dir = training_args.output_dir
    overwrite_output_dir = training_args.overwrite_output_dir
    resume_from_checkpoint = training_args.resume_from_checkpoint
    seed = training_args.seed

    notation = data_args.notation
    ner_types = data_args.ner_types
    ner_priorities = data_args.ner_priorities
    fix_spans = data_args.fix_spans
    recall_threshold = model_args.recall_threshold

    train_file = data_args.train_file
    validation_file = data_args.validation_file
    test_file = data_args.test_file
    max_train_samples = data_args.max_train_samples
    max_eval_samples = data_args.max_eval_samples
    max_predict_samples = data_args.max_predict_samples
    train_on_fly = data_args.train_on_fly
    validation_on_fly = data_args.validation_on_fly
    test_on_fly = data_args.test_on_fly

    pad_to_max_length = data_args.pad_to_max_length
    max_seq_length = data_args.max_seq_length
    label_all_tokens = data_args.label_all_tokens
    preprocessing_num_workers = data_args.preprocessing_num_workers
    overwrite_cache = data_args.overwrite_cache

    ner_augment = data_args.ner_augment

    model_name_or_path = model_args.model_name_or_path
    config_name = (
        model_args.config_name if model_args.config_name else model_args.model_name_or_path
    )
    tokenizer_name = (
        model_args.tokenizer_name if model_args.tokenizer_name else model_args.model_name_or_path
    )
    config_overrides = model_args.config_overrides
    spacy_model = model_args.spacy_model
    cache_dir = model_args.cache_dir
    model_revision = model_args.model_revision
    use_auth_token = model_args.use_auth_token

    # Load the model helpers object
    model_helpers = ModelHelpers()

    # Load the SpaCy tokenizer object
    text_tokenizer = model_helpers.get_text_tokenizer(spacy_model=spacy_model)

    # Load the HuggingFace model config
    config = model_helpers.get_config(
        notation=notation,
        ner_types=ner_types,
        config_name=config_name,
        config_overrides=config_overrides,
        cache_dir=cache_dir,
        model_revision=model_revision,
        use_auth_token=use_auth_token
    )

    # Load the HuggingFace tokenizer
    tokenizer = model_helpers.get_tokenizer(
        tokenizer_name=tokenizer_name,
        model_type=config.model_type,
        cache_dir=cache_dir,
        model_revision=model_revision,
        use_auth_token=use_auth_token
    )

    # Load the HuggingFace model
    model = model_helpers.get_model(
        model_name_or_path=model_name_or_path,
        config=config,
        from_tf=bool(".ckpt" in model_name_or_path),
        cache_dir=cache_dir,
        model_revision=model_revision,
        use_auth_token=use_auth_token
    )

    # Load the sequence tagger object
    sequence_tagger = SequenceTagger(
        text_tokenizer=text_tokenizer,
        config=config,
        tokenizer=tokenizer,
        model=model,
        notation=notation,
        ner_types=ner_types,
        ner_priorities=ner_priorities,
        fix_spans=fix_spans,
        fp16=training_args.fp16,
        ner_augment=ner_augment,
        recall_threshold=recall_threshold,
        pad_to_max_length=pad_to_max_length,
        max_seq_length=max_seq_length,
        label_all_tokens=label_all_tokens,
        seed=seed
    )

    # Load the train dataset
    train_dataset = sequence_tagger.get_train_dataset(
        train_file=train_file,
        training_args=training_args,
        train_on_fly=train_on_fly,
        preprocessing_num_workers=preprocessing_num_workers,
        overwrite_cache=overwrite_cache,
        max_train_samples=max_train_samples
    )

    # Load the evaluation dataset - monitor performance on the validation dataset during the course of training
    eval_dataset = sequence_tagger.get_validation_dataset(
        validation_file=validation_file,
        training_args=training_args,
        validation_on_fly=validation_on_fly,
        preprocessing_num_workers=preprocessing_num_workers,
        overwrite_cache=overwrite_cache,
        max_eval_samples=max_eval_samples
    )

    test_dataset = sequence_tagger.get_test_dataset(
        test_file=test_file,
        training_args=training_args,
        test_on_fly=test_on_fly,
        preprocessing_num_workers=preprocessing_num_workers,
        overwrite_cache=overwrite_cache,
        max_predict_samples=max_predict_samples
    )

    # Set the HuggingFace trainer object based on the training arguments and datasets
    sequence_tagger.set_trainer(
        training_args=training_args,
        train_dataset=train_dataset,
        eval_dataset=eval_dataset,
        do_train=do_train,
        do_eval=do_eval,
        do_predict=do_predict
    )

    # Detecting last checkpoint.
    last_checkpoint = sequence_tagger.get_checkpoint(
        output_dir=output_dir,
        resume_from_checkpoint=resume_from_checkpoint,
        do_train=do_train,
        overwrite_output_dir=overwrite_output_dir
    )

    train_metrics = sequence_tagger.run_train(
        resume_from_checkpoint=training_args.resume_from_checkpoint,
        last_checkpoint=last_checkpoint
    )

    eval_metrics = sequence_tagger.run_eval()

    model_output = sequence_tagger.run_predict(test_dataset=test_dataset)


if __name__ == '__main__':
    main()
