import json
from jsonschema.validators import validator_for
from .. import settings
from ..type import Type


class GeojsonType(Type):
    """Geojson type implementation.

    API      | Usage
    -------- | --------
    Public   | `from frictionless import types`

    """

    code = "geojson"
    builtin = True
    constraints = [
        "required",
        "enum",
    ]

    # Read

    def read_cell(self, cell):
        if isinstance(cell, str):
            try:
                cell = json.loads(cell)
            except Exception:
                return None
        if not isinstance(cell, dict):
            return None
        if self.field.format in ["default", "topojson"]:
            try:
                VALIDATORS[self.field.format].validate(cell)
            except Exception:
                return None
        return cell

    # Write

    def write_cell(self, cell):
        return json.dumps(cell)


# Internal


VALIDATORS = {
    "default": validator_for(settings.GEOJSON_PROFILE)(settings.GEOJSON_PROFILE),
    "topojson": validator_for(settings.TOPOJSON_PROFILE)(settings.TOPOJSON_PROFILE),
}
